
/**
 * Namespace for ForgeJS
 * @namespace FORGE
 * @type {Object}
 */
var FORGE = FORGE || {};

/**
 * Version number of ForgeJS.
 * @name FORGE.VERSION
 * @type {string}
 * @const
 */
FORGE.VERSION = '0.9.5';

/**
 * Array of {@link FORGE.Viewer} uids.
 * @name FORGE.VIEWERS
 * @type {Array<string>}
 */
FORGE.VIEWERS = [];

/**
 * Global debug switch.
 * @name FORGE.DEBUG
 * @type {boolean}
 * @const
 */
FORGE.DEBUG = false;

/**
 * Global warning switch.
 * @name  FORGE.WARNING
 * @type {boolean}
 * @const
 */
FORGE.WARNING = true;


/**
 * Most of the ForgeJS objects extends FORGE.BaseObject.<br>
 * Its main purpose is to name objects with a class name (a type) and to have a destroy method that handles an "alive" flag.
 *
 * @constructor FORGE.BaseObject
 * @param {string} className - The class name of the object.
 *
 * @todo  See if we can trace inheritance.
 */
FORGE.BaseObject = function(className)
{
    /**
     * The unique identifier of this object.
     * @name  FORGE.BaseObject#_uid
     * @type {string}
     * @private
     */
    this._uid = "";

    /**
     * Array of tags that can be used to identify / classify this object.
     * @name  FORGE.BaseObject#_tags
     * @type {?Array<string>}
     * @private
     */
    this._tags = null;

    /**
     * Custom data associated to this object.
     * @name  FORGE.BaseObject#_data
     * @type {?*}
     * @private
     */
    this._data = null;

    /**
     * Internal refernce to the onDestroy {@link FORGE.EventDispatcher}.
     * @name FORGE.BaseObject#_onDestroy
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onDestroy = null;

    /**
     * Internal reference to the alive flag.
     * @name FORGE.BaseObject#_alive
     * @type {boolean}
     * @private
     */
    this._alive = true;

    /**
     * Internal reference to the debug flag.
     * @name FORGE.BaseObject#_debug
     * @type {boolean}
     * @private
     */
    this._debug = false;

    /**
     * Internal reference to the warning flag.
     * @name  FORGE.BaseObject#_warning
     * @type {boolean}
     * @private
     */
    this._warning = false;

    /**
     * Array to store log if debug is enabled.
     * @name FORGE.BaseObject#_logs
     * @type {?Array}
     * @private
     */
    this._logs = null;

    /**
     * Internal reference to the class name of the object that extends this base object.
     * @name FORGE.BaseObject#_className
     * @type {string}
     * @private
     */
    this._className = className || "BaseObject";

    /**
     * Inheritance chain.
     * @name FORGE.BaseObject#_inheritance
     * @type {Array<String>}
     * @private
     * @todo  Code this mechanism of type chain inheritance
     */
    this._inheritance = ["BaseObject"];
};

FORGE.BaseObject.prototype.constructor = FORGE.BaseObject;


/**
 * Registers the object in the uid index and bind to tag manager if tags set.
 * If no uid set, use a generated one.
 * @method FORGE.BaseObject#_register
 * @private
 */
FORGE.BaseObject.prototype._register = function()
{
    this.log("register");

    //Generate a uid if undefined
    if(typeof this._uid !== "string" || this._uid === "")
    {
        this._uid = FORGE.UID.generate();
    }
    //Register in UID table
    var registered = FORGE.UID.register(this);

    //If this object have tags associated to it.
    if(this._tags !== null)
    {
        //Maybe there is a single string typed tag, convert it to an array.
        if(typeof this._tags === "string")
        {
            this._tags = [this._tags];
        }

        //Register tags if it is an Array
        if(Array.isArray(this._tags) === true)
        {
            FORGE.Tags.register(this);
        }
    }

    return registered;
};

/**
 * Unregisters the object in the uid index.
 * @method FORGE.BaseObject#_unregister
 * @private
 */
FORGE.BaseObject.prototype._unregister = function()
{
    this.log("unregister");

    if(this._uid !== "" && FORGE.UID.exists(this._uid) === true)
    {
        FORGE.UID.unregister(this);
    }
};

/**
 * That method describe how to output the log, can be overwritted by a debug plugin for example.
 * @method FORGE.BaseObject#_stdout
 * @private
 * @param {*} value - The value you want to stdout.
 * @param {string} mode - The console method to use (default is log)
 */
FORGE.BaseObject.prototype._stdout = function(value, mode)
{
    var m = mode || "log";
    var consoleLog = [];
    if (FORGE.Device.chrome === true || FORGE.Device.firefox === true)
    {
        consoleLog = [ "%c[ForgeJS]%c "
            + "FORGE." + this._className + ": " + value + " %c(@"
            + window.performance.now().toFixed(2) + "ms)",
            "background: #e2edff; color: #4286f4; font-weight: 700;",
            "font-weight: 400;",
            "color: #AAA;"
            ];
    }
    else
    {
        consoleLog = [ "[ForgeJS] FORGE." + this._className + ": " + value + " (@"
            + window.performance.now().toFixed(2) + "ms)"
            ];
    }
    console[m].apply(console, consoleLog);

    if(typeof value === "object" && value !== null)
    {
        console[m](value);
    }
};

/**
 * Basic log method, log a string in the console if debug is enabled.
 * @method FORGE.BaseObject#log
 * @param {*} value - The value you want to log in the console.
 */
FORGE.BaseObject.prototype.log = function(value)
{
    if(window["FORGE"]["DEBUG"] === true || window["FORGE"][this._className]["DEBUG"] === true || this._debug === true)
    {
        this._stdout(value, "log");

        if(this._logs === null)
        {
            this._logs = [];
        }

        this._logs.push(value);
    }
};

/**
 * Basic warn method, log a warn string in the console if warning is enabled.
 * @method FORGE.BaseObject#warn
 * @param {?(string|Object)} value - The value you want to warn in the console.
 */
FORGE.BaseObject.prototype.warn = function(value)
{
    if(window["FORGE"]["WARNING"] === true || window["FORGE"][this._className]["WARNING"] === true || this._warning === true)
    {
        this._stdout(value, "warn");
    }
};

/**
 * Basic destroy method, prevent double destroy, change the alive flag.
 * @method FORGE.BaseObject#destroy
 */
FORGE.BaseObject.prototype.destroy = function()
{
    if(this._alive === false)
    {
        return;
    }

    this.log("destroy");

    this._unregister();

    if(this._onDestroy !== null)
    {
        this._onDestroy.dispatch();
        this._onDestroy.destroy();
        this._onDestroy = null;
    }

    this._data = null;

    this._alive = false;
};

/**
 * Get the class name of the object.
 * @name FORGE.BaseObject#className
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "className",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._className;
    }

});

/**
 * Get the uid of the object.
 * @name FORGE.BaseObject#uid
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "uid",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._uid;
    }
});

/**
 * Get the tags associated to this object.
 * @name FORGE.BaseObject#tags
 * @readonly
 * @type {Array}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "tags",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._tags;
    }
});

/**
 * Get the alive flag value of the object.
 * @name FORGE.BaseObject#alive
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "alive",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._alive;
    }
});

/**
* Get and set any custom data you want to associate to this object.
* @name FORGE.BaseObject#data
* @type {*}
*/
Object.defineProperty(FORGE.BaseObject.prototype, "data",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._data;
    },

    /** @this {FORGE.BaseObject} */
    set: function(value)
    {
        this._data = value;
    }
});

/**
 * Get and set the debug flag.
 * @name FORGE.BaseObject#debug
 * @type {boolean}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "debug",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._debug;
    },

    /** @this {FORGE.BaseObject} */
    set: function(value)
    {
        this._debug = Boolean(value);

        if(this._debug === true)
        {
            console.log("Enabling debug for a FORGE."+this._className+" instance :");
            console.log(this);
        }
    }
});

/**
 * Get and set the warning flag.
 * @name FORGE.BaseObject#warning
 * @type {boolean}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "warning",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        return this._warning;
    },

    /** @this {FORGE.BaseObject} */
    set: function(value)
    {
        this._warning = Boolean(value);

        if(this._warning === true)
        {
            console.log("Enabling warning for a FORGE."+this._className+" instance :");
            console.log(this);
        }
    }
});

/**
 * Get the onDestroy {@link FORGE.EventDispatcher}, this event is emitted at the end of the destroy sequence.
 * @name FORGE.BaseObject#onDestroy
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.BaseObject.prototype, "onDestroy",
{
    /** @this {FORGE.BaseObject} */
    get: function()
    {
        if(this._onDestroy === null)
        {
            this._onDestroy = new FORGE.EventDispatcher(this);
        }

        return this._onDestroy;
    }
});

/**
 * The main viewer class.
 *
 * @constructor FORGE.Viewer
 * @param {HTMLElement|string} parent - The parent element.
 * @param {(MainConfig|string)} config - Object that represents the project configuration, it can be an object or a json configuration url.
 * @param {?ViewerCallbacks} callbacks - On boot callbacks
 * @extends {FORGE.BaseObject}
 *
 * @todo Create a FORGE.Config Object to define default values, that can be overrided by the config param.
 */
FORGE.Viewer = function(parent, config, callbacks)
{
    FORGE.BaseObject.call(this, "Viewer");

    /**
     * Either the parent id or the parent DOM Element of the viewer.
     * @name FORGE.Viewer#_parent
     * @type {Element|HTMLElement|string}
     * @private
     */
    this._parent = parent;

    /**
     * The main config of the FORGE project
     * @name  FORGE.Viewer#_mainConfig
     * @type {(MainConfig|string)}
     * @private
     */
    this._mainConfig = config;

    /**
     * The viewer configuration or its URL.
     * After the config loading it will be a MainConfig in all cases.
     * @name FORGE.Viewer#_config
     * @type {?ViewerConfig}
     * @private
     */
    this._config = null;

    /**
     * Reference to the DisplayList manager
     * @name FORGE.DispalyList
     * @type {FORGE.DisplayList}
     * @private
     */
    this._display = null;

    /**
     * This is a relative div between the parent and the viewer container.
     * @name  FORGE.Viewer#_relative
     * @type {?Element}
     * @private
     */
    this._relative = null;

    /**
     * The viewer container reference.
     * @name FORGE.Viewer#_container
     * @type {FORGE.DisplayObjectContainer}
     * @private
     */
    this._container = null;

    /**
     * The canvas container reference.
     * @name FORGE.Viewer#_canvasContainer
     * @type {FORGE.DisplayObjectContainer}
     * @private
     */
    this._canvasContainer = null;

    /**
     * The DOM hotspot container reference.
     * @name FORGE.Viewer#_domHotspotContainer
     * @type {FORGE.DisplayObjectContainer}
     * @private
     */
    this._domHotspotContainer = null;

    /**
     * The DOM hotspot style container reference.
     * @name FORGE.Viewer#_domHotspotStyle
     * @type {Element|HTMLStyleElement}
     * @private
     */
    this._domHotspotStyle = null;

    /**
     * The plugins container reference.
     * @name FORGE.Viewer#_pluginContainer
     * @type {FORGE.DisplayObjectContainer}
     * @private
     */
    this._pluginContainer = null;

    /**
     * The canvas reference.
     * @name FORGE.Viewer#_canvas
     * @type {FORGE.Canvas}
     * @private
     */
    this._canvas = null;

     /**
     * System module reference.
     * @name FORGE.Viewer#_system
     * @type {FORGE.System}
     * @private
     */
    this._system = null;

    /**
     * Audio / mixer interface reference.
     * @name FORGE.Viewer#_audio
     * @type {FORGE.SoundManager}
     * @private
     */
    this._audio = null;

    /**
     * Playlists interface reference.
     * @name FORGE.Viewer#_playlists
     * @type {FORGE.PlaylistManager}
     * @private
     */
    this._playlists = null;

    /**
     * Hotspots interface reference.
     * @name  FORGE.Viewer#_hotspots
     * @type {FORGE.HotspotManager}
     * @private
     */
    this._hotspots = null;

    /**
     * Action manager reference.
     * @name FORGE.Viewer#_actions
     * @type {FORGE.ActionManager}
     * @private
     */
    this._actions = null;

    /**
     * Director's cut track manager
     * @name  FORGE.Viewer#_director
     * @type {FORGE.Director}
     * @private
     */
    this._director = null;

    /**
     * Controller manager.
     * @name FORGE.Viewer#_controllers
     * @type {FORGE.ControllerManager}
     * @private
     */
    this._controllers = null;

    /**
     * Post processing.
     * @name  FORGE.Viewer#_postProcessing
     * @type {FORGE.PostProcessing}
     * @private
     */
    this._postProcessing = null;

    /**
     * Story reference.
     * @name FORGE.Viewer#_story
     * @type {FORGE.Story}
     * @private
     */
    this._story = null;

    /**
     * History manager reference.
     * @name FORGE.Viewer#_history
     * @type {FORGE.History}
     * @private
     */
    this._history = null;

    /**
     * Loader reference.
     * @name FORGE.Viewer#_load
     * @type {FORGE.Loader}
     * @private
     */
    this._load = null;

    /**
     * Cache reference.
     * @name FORGE.Viewer#_cache
     * @type {FORGE.Cache}
     * @private
     */
    this._cache = null;

    /**
     * Keyboard interface.
     * @name FORGE.Viewer#_keyboard
     * @type {FORGE.Keyboard}
     * @private
     */
    this._keyboard = null;

    /**
     * Gyroscope interface.
     * @name FORGE.Viewer#_gyroscope
     * @type {FORGE.Gyroscope}
     * @private
     */
    this._gyroscope = null;

    /**
     * Gamepads manager.
     * @name FORGE.Viewer#_gamepad
     * @type {FORGE.GamepadsManager}
     * @private
     */
    this._gamepad = null;

    /**
     * Plugins interface reference.
     * @name FORGE.Viewer#_plugins
     * @type {FORGE.PluginManager}
     * @private
     */
    this._plugins = null;

    /**
     * Main loop reference.
     * @name FORGE.Viewer#_raf
     * @type {FORGE.RequestAnimationFrame}
     * @private
     */
    this._raf = null;

    /**
     * Handle viewer clock reference.
     * @name FORGE.Viewer#_clock
     * @type {FORGE.Clock}
     * @private
     */
    this._clock = null;

    /**
     * Tween Manager reference.
     * @name FORGE.Viewer#_tween
     * @type {FORGE.TweenManager}
     * @private
     */
    this._tween = null;

    /**
     * i18n and locales interface reference.
     * @name FORGE.Viewer#_i18n
     * @type {FORGE.LocaleManager}
     * @private
     */
    this._i18n = null;

    /**
     * Renderer reference.
     * @name FORGE.Viewer#_renderManager
     * @type {FORGE.RenderManager}
     * @private
     */
    this._renderManager = null;

    /**
     * Paused state of the main loop.
     * @name FORGE.Viewer#_paused
     * @type {boolean}
     * @private
     */
    this._paused = false;

    /**
     * Flag to know if the viewer is ready
     * @name  FORGE.Viewer#_ready
     * @type {boolean}
     * @private
     */
    this._ready = false;

    /**
     * Event dispatcher for the viewer on ready event. Dispatched after the boot sequence.
     * @name  FORGE.Viewer#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    /**
     * Event dispatcher for the on pause event.
     * @name  FORGE.Viewer#_onPause
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onPause = null;

    /**
     * Event dispatcher for the on resume event.
     * @name  FORGE.Viewer#_onResume
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onResume = null;

    /**
     * Event dispatcher for the on config load complete event.
     * @name  FORGE.Viewer#_onConfigLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onConfigLoadComplete = null;

    /**
     * Event dispatcher for the on main config load complete event.
     * @name  FORGE.Viewer#_onMainConfigLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onMainConfigLoadComplete = null;

    /**
     * Callback function for the viewer.
     * @name FORGE.Viewer#_callbacks
     * @type {?ViewerCallbacks}
     * @private
     */
    this._callbacks = callbacks || null;

    var bootBind = this._boot.bind(this);
    window.setTimeout(bootBind, 0);
};

FORGE.Viewer.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Viewer.prototype.constructor = FORGE.Viewer;

/**
 * Viewer default configuration
 * @name  FORGE.Viewer.DEFAULT_CONFIG
 * @type {ViewerConfig}
 * @const
 */
FORGE.Viewer.DEFAULT_CONFIG =
{
    background: "#000",
    autoResume: true,
    autoPause: true,

    webgl:
    {
        antialias: true,
        alpha: true,
        premultipliedAlpha: false,
        stencil: false
    }
};

/**
 * Boot sequence.
 * @method FORGE.Viewer#_boot
 * @param {Function} callback - Callback function.
 * @private
 */
FORGE.Viewer.prototype._boot = function(callback)
{
    /**
     * WARNING THE ORDER OF THE BOOT SEQUENCE MATTERS A LOT!
     * DO NOT CHANGE ANYTHING UNLESS YOU ARE SURE OF WHAT YOURE DOING
     */

    if(this._alive === false)
    {
        return;
    }

    this._uid = "FORGE-instance-" + String(FORGE.VIEWERS.length);
    FORGE.VIEWERS.push(this._uid);
    this._register();

    this.log("FORGE.Viewer.boot();");

    this._display = new FORGE.DisplayList(this);

    this._createContainers();
    this._createCanvas();

    this._system = new FORGE.System(this);
    this._clock = new FORGE.Clock(this);
    this._audio = new FORGE.SoundManager(this);
    this._raf = new FORGE.RequestAnimationFrame(this);
    this._i18n = new FORGE.LocaleManager(this);
    this._story = new FORGE.Story(this);
    this._history = new FORGE.History(this);
    this._renderManager = new FORGE.RenderManager(this);
    this._controllers = new FORGE.ControllerManager(this);
    this._playlists = new FORGE.PlaylistManager(this);
    this._plugins = new FORGE.PluginManager(this);
    this._hotspots = new FORGE.HotspotManager(this);
    this._actions = new FORGE.ActionManager(this);
    this._director = new FORGE.Director(this);
    this._postProcessing = new FORGE.PostProcessing(this);

    this._keyboard = new FORGE.Keyboard(this);
    this._gyroscope = new FORGE.Gyroscope(this);
    this._gamepad = new FORGE.GamepadsManager(this);
    this._cache = new FORGE.Cache(this);
    this._load = new FORGE.Loader(this);
    this._tween = new FORGE.TweenManager(this);

    this._system.boot();
    this._audio.boot();
    this._raf.boot();
    this._story.boot();
    this._playlists.boot();
    this._plugins.boot();
    this._hotspots.boot();

    this.log("ForgeJS " + FORGE.VERSION);

    //Call the boot method argument callback
    if (typeof callback === "function")
    {
        callback.call();
    }

    //Call the viewer constructor callback
    if (this._callbacks !== null && typeof this._callbacks.boot === "function")
    {
        this._callbacks.boot.call();
    }

    this._ready = true;

    if (this._onReady !== null)
    {
        this._onReady.dispatch();
    }

    this._loadConfig(this._mainConfig);
};

/**
 * Load the main configuration of the viewer.
 * @method FORGE.Viewer#_loadConfig
 * @param  {?(MainConfig|string)} config - The config object or its URL
 * @private
 */
FORGE.Viewer.prototype._loadConfig = function(config)
{
    if (typeof config === "string")
    {
        this._load.json(this._uid + "-configuration", config, this._mainConfigLoadComplete, this);
    }
    else if (config !== null && typeof config === "object")
    {
        this._parseMainConfig(config);
    }
};

/**
 * Event handler for the configuration JSON load complete.
 * @method FORGE.Story#_mainConfigLoadComplete
 * @param  {FORGE.File} file - The {@link FORGE.File} that describes the loaded JSON file.
 * @private
 */
FORGE.Viewer.prototype._mainConfigLoadComplete = function(file)
{
    this.log("FORGE.Viewer._mainConfigLoadComplete();");

    var json = this._cache.get(FORGE.Cache.types.JSON, file.key);
    var config = /** @type {MainConfig} */ (json.data);

    this._parseMainConfig(config);
};

/**
 * Parse the Global Main Configuration
 * @param  {MainConfig} config - Main configuration to parse.
 * @private
 */
FORGE.Viewer.prototype._parseMainConfig = function(config)
{
    // Final assignement of the config
    this._mainConfig = config;

    this._parseConfig(config.viewer);

    if (typeof config.i18n !== "undefined")
    {
        this._i18n.addConfig(config.i18n); //force the parse of the main config
    }

    this._history.addConfig(config.history);

    if (typeof config.audio !== "undefined")
    {
        this._audio.addConfig(config.audio);
    }

    if (typeof config.playlists !== "undefined")
    {
        this._playlists.addConfig(config.playlists);
    }

    if (typeof config.actions !== "undefined")
    {
        this._actions.addConfig(config.actions);
    }

    if (typeof config.director !== "undefined")
    {
        this._director.load(config.director);
    }

    if (typeof config.hotspots !== "undefined")
    {
        this._hotspots.addTracks(config.hotspots);
    }

    if (typeof config.postProcessing !== "undefined")
    {
        this._postProcessing.addConfig(config.postProcessing);
    }

    if (typeof config.plugins !== "undefined")
    {
        this._plugins.addConfig(config.plugins);
    }

    this._controllers.addConfig(config.controllers);

    if (typeof config.story !== "undefined")
    {
        this._story.load(config.story);
    }

    this._raf.start();

    if (this._onMainConfigLoadComplete !== null)
    {
        this._onMainConfigLoadComplete.dispatch();
    }
};

/**
 * Parse the Viewer config.
 * @method FORGE.Viewer#_parseConfig
 * @param {ViewerConfig} config - The viewer configuration to parse
 * @private
 */
FORGE.Viewer.prototype._parseConfig = function(config)
{
    this._config = /** @type {ViewerConfig} */ (FORGE.Utils.extendSimpleObject(FORGE.Viewer.DEFAULT_CONFIG, config));

    if (this._onConfigLoadComplete !== null)
    {
        this._onConfigLoadComplete.dispatch();
    }
};

/**
 * Create all containers into the DOM.
 * @method FORGE.Viewer#_createContainers
 * @private
 */
FORGE.Viewer.prototype._createContainers = function()
{
    if (typeof this._parent === "string" && this._parent !== "")
    {
        this._parent = document.getElementById(this._parent);
    }

    if (typeof this._parent === "undefined" || this._parent === null || FORGE.Dom.isHtmlElement(this._parent) === false)
    {
        throw "FORGE.Viewer.boot : Viewer parent is invalid";
    }

    this._relative = document.createElement("div");
    this._relative.style.width = "100%";
    this._relative.style.height = "100%";
    this._relative.style.position = "relative";
    this._parent.appendChild(this._relative);

    this._container = new FORGE.DisplayObjectContainer(this, null, null, /** @type {Element} */ (this._relative));
    this._container.id = "FORGE-main-container-" + this._uid;

    this._canvasContainer = new FORGE.DisplayObjectContainer(this);
    this._canvasContainer.id = "FORGE-canvas-container-" + this._uid;
    this._container.index = 0;
    this._canvasContainer.maximize(true);
    this._container.addChild(this._canvasContainer);

    this._domHotspotContainer = new FORGE.DisplayObjectContainer(this);
    this._domHotspotContainer.id = "FORGE-dom-hotspot-container-" + this._uid;
    this._container.index = 0;
    this._domHotspotContainer.maximize(true);
    this._container.addChild(this._domHotspotContainer);

    this._domHotspotStyle = document.createElement("style");
    this._domHotspotStyle.type = "text/css";
    document.head.appendChild(this._domHotspotStyle);

    this._pluginContainer = new FORGE.DisplayObjectContainer(this);
    this._pluginContainer.id = "FORGE-plugin-container-" + this._uid;
    this._container.index = 0;
    this._pluginContainer.maximize(true);
    this._container.addChild(this._pluginContainer);
};

/**
 * Create the canvas into the DOM.
 * @method FORGE.Viewer#_createCanvas
 * @private
 */
FORGE.Viewer.prototype._createCanvas = function()
{
    this._canvas = new FORGE.Canvas(this);
    this._canvas.maximize(true);
    this._canvasContainer.addChild(this._canvas);
};

/**
 * Update class informations on main loop.
 * @method FORGE.Viewer#_updateLogic
 * @private
 */
FORGE.Viewer.prototype._updateLogic = function()
{
    this._display.update();
    this._keyboard.update();
    this._gamepad.update();
    this._audio.update();
    this._plugins.update();
    this._tween.update();
    this._hotspots.update();
    this._controllers.update();
    this._renderManager.update();

    if (this._callbacks !== null && typeof this._callbacks.update === "function")
    {
        this._callbacks.update.call();
    }
};

/**
 * Update scene rendering on main loop.
 * @method FORGE.Viewer#_updateRendering
 * @private
 */
FORGE.Viewer.prototype._updateRendering = function()
{
    if (this._callbacks !== null && typeof this._callbacks.beforeRender === "function")
    {
        this._callbacks.beforeRender.call();
    }

    if (this._renderManager !== null)
    {
        this._renderManager.render();
    }

    if (this._callbacks !== null && typeof this._callbacks.afterRender === "function")
    {
        this._callbacks.afterRender.call();
    }
};

/**
 * Update method called by the viewer main loop.
 * @method FORGE.Viewer#update
 * @param  {number} time - Time in ms
 */
FORGE.Viewer.prototype.update = function(time)
{
    if (this._paused === true)
    {
        return;
    }

    //Update the global clock
    this._clock.update(time);

    this._updateLogic();
    this._updateRendering();
};

/**
 * Pause the refresh on the main loop.
 * @method FORGE.Viewer#pause
 * @param {boolean} internal - Internal lib usage
 */
FORGE.Viewer.prototype.pause = function(internal)
{
    if(this._paused === true)
    {
        return;
    }

    this._paused = true;

    // Pause all media if autoPause is true
    if (internal !== true || (this._config !== null && this._config.autoPause === true))
    {
        this._audio.pauseAll();
    }

    if (this._onPause !== null)
    {
        this._onPause.dispatch({
            "internal": internal
        });
    }
};

/**
 * Resume the refresh on the main loop.
 * @method FORGE.Viewer#resume
 * @param {boolean} internal - Internal lib usage
 */
FORGE.Viewer.prototype.resume = function(internal)
{
    if(this._paused === false)
    {
        return;
    }

    this._paused = false;

    // Resume all media if autoResume is true
    if (internal !== true || (this._config !== null && this._config.autoResume === true))
    {
        this._audio.resumeAll();
    }

    if (this._onResume !== null)
    {
        this._onResume.dispatch({
            "internal": internal
        });
    }
};

/**
 * Destroy method.
 * @method FORGE.Viewer#destroy
 */
FORGE.Viewer.prototype.destroy = function()
{

    /**
     * WARNING THE ORDER OF THE DESTROY SEQUENCE MATTERS A LOT!
     * DO NOT CHANGE ANYTHING UNLESS YOU ARE SURE OF WHAT YOURE DOING
     */

    if(this._raf !== null)
    {
        this._raf.destroy();
        this._raf = null;
    }

    if(this._system !== null)
    {
        this._system.destroy();
        this._system = null;
    }

    if(this._plugins !== null)
    {
        this._plugins.destroy();
        this._plugins = null;
    }

    if(this._load !== null)
    {
        this._load.destroy();
        this._load = null;
    }

    if(this._history !== null)
    {
        this._history.destroy();
        this._history = null;
    }

    if(this._clock !== null)
    {
        this._clock.destroy();
        this._clock = null;
    }

    if (this._controllers !== null)
    {
        this._controllers.destroy();
        this._controllers = null;
    }

    if(this._keyboard !== null)
    {
        this._keyboard.destroy();
        this._keyboard = null;
    }

    if(this._gyroscope !== null)
    {
        this._gyroscope.destroy();
        this._gyroscope = null;
    }

    if(this._gamepad !== null)
    {
        this._gamepad.destroy();
        this._gamepad = null;
    }

    if(this._hotspots !== null)
    {
        this._hotspots.destroy();
        this._hotspots = null;
    }

    if(this._director !== null)
    {
        this._director.destroy();
        this._director = null;
    }

    if(this._renderManager !== null)
    {
        this._renderManager.destroy();
        this._renderManager = null;
    }

    if(this._canvas !== null)
    {
        this._canvas.destroy();
        this._canvas = null;
    }

    if(this._canvasContainer !== null)
    {
        this._canvasContainer.destroy();
        this._canvasContainer = null;
    }

    if(this._domHotspotContainer !== null)
    {
        this._domHotspotContainer.destroy();
        this._domHotspotContainer = null;
    }

    if(this._pluginContainer !== null)
    {
        this._pluginContainer.destroy();
        this._pluginContainer = null;
    }

    if(this._container !== null)
    {
        this._container.destroy();
        this._container = null;
    }

    if(this._postProcessing !== null)
    {
        this._postProcessing.destroy();
        this._postProcessing = null;
    }

    if(this._actions !== null)
    {
        this._actions.destroy();
        this._actions = null;
    }

    if(this._display !== null)
    {
        this._display.destroy();
        this._display = null;
    }

    if(this._playlists !== null)
    {
        this._playlists.destroy();
        this._playlists = null;
    }

    if(this._audio !== null)
    {
        this._audio.destroy();
        this._audio = null;
    }

    if(this._story !== null)
    {
        this._story.destroy();
        this._story = null;
    }

    if(this._tween !== null)
    {
        this._tween.destroy();
        this._tween = null;
    }

    if(this._cache !== null)
    {
        this._cache.destroy();
        this._cache = null;
    }

    if(this._i18n !== null)
    {
        this._i18n.destroy();
        this._i18n = null;
    }

    this._parent = null;
    this._callbacks = null;

    if (this._onReady !== null)
    {
        this._onReady.destroy();
        this._onReady = null;
    }

    if (this._onPause !== null)
    {
        this._onPause.destroy();
        this._onPause = null;
    }

    if (this._onResume !== null)
    {
        this._onResume.destroy();
        this._onResume = null;
    }

    if (this._onConfigLoadComplete !== null)
    {
        this._onConfigLoadComplete.destroy();
        this._onConfigLoadComplete = null;
    }

    if (this._onMainConfigLoadComplete !== null)
    {
        this._onMainConfigLoadComplete.destroy();
        this._onMainConfigLoadComplete = null;
    }

    FORGE.VIEWERS.splice(this._uid, 1);

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the viewer ready status.
 * @name FORGE.Viewer#ready
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "ready",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get the main configuration.
 * @name FORGE.Viewer#mainConfig
 * @type {MainConfig}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "mainConfig",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._mainConfig;
    }
});

/**
 * Get the viewer configuration.
 * @name FORGE.Viewer#config
 * @type {ViewerConfig}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "config",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._config;
    }
});

/**
 * Get the viewer parent element.
 * @name FORGE.Viewer#parent
 * @type {HTMLElement|String}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "parent",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._parent;
    }
});

/**
 * Get the viewer display list manager.
 * @name FORGE.Viewer#display
 * @type {FORGE.DisplayList}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "display",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._display;
    }
});

/**
 * Get and set the fullscreen property of the viewer main container.
 * @name FORGE.Viewer#fullscreen
 * @type {boolean}
 */
Object.defineProperty(FORGE.Viewer.prototype, "fullscreen",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._container.isFullscreen();
    },

    /** @this {FORGE.Viewer} */
    set: function(value)
    {
        if (typeof value !== "boolean" || this._container.isFullscreen() === value)
        {
            return;
        }

        this._container.fullscreen = value;
    }
});

/**
 * Get the viewer container.
 * @name FORGE.Viewer#container
 * @type {FORGE.DisplayObjectContainer}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "container",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._container;
    }
});

/**
 * Get the viewer canvas container.
 * @name FORGE.Viewer#canvasContainer
 * @type {FORGE.DisplayObjectContainer}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "canvasContainer",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._canvasContainer;
    }
});

/**
 * Get the viewer canvas.
 * @name FORGE.Viewer#canvas
 * @type {FORGE.Canvas}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "canvas",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._canvas;
    }
});

/**
 * Get the viewer DOM hotspot container.
 * @name FORGE.Viewer#domHotspotContainer
 * @type {FORGE.DisplayObjectContainer}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "domHotspotContainer",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._domHotspotContainer;
    }
});

/**
 * Get the viewer DOM hotspot style container.
 * @name FORGE.Viewer#domHotspotStyle
 * @type {HTMLStyleElement}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "domHotspotStyle",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._domHotspotStyle;
    }
});

/**
 * Get the viewer plugin container.
 * @name FORGE.Viewer#pluginContainer
 * @type {FORGE.DisplayObjectContainer}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "pluginContainer",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._pluginContainer;
    }
});

/**
 * Get the viewer request animation frame interface.
 * @name FORGE.Viewer#raf
 * @type {FORGE.RequestAnimationFrame}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "raf",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._raf;
    }
});

/**
 * Get the viewer audio/sound interface.
 * @name FORGE.Viewer#audio
 * @type {FORGE.SoundManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "audio",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._audio;
    }
});

/**
 * Get the viewer playlists for sounds.
 * @name FORGE.Viewer#playlists
 * @type {FORGE.PlaylistManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "playlists",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._playlists;
    }
});

/**
 * Get the viewer hotspots module.
 * @name FORGE.Viewer#hotspots
 * @type {FORGE.HotspotManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "hotspots",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._hotspots;
    }
});

/**
 * Get the viewer action manager.
 * @name FORGE.Viewer#actions
 * @type {FORGE.ActionManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "actions",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._actions;
    }
});

/**
 * Get the director's cut track manager.
 * @name FORGE.Viewer#director
 * @type {FORGE.Director}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "director",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._director;
    }
});

/**
 * Get the story module.
 * @name FORGE.Viewer#story
 * @type {FORGE.Story}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "story",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._story;
    }
});

/**
 * Get the history module.
 * @name FORGE.Viewer#history
 * @type {FORGE.History}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "history",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._history;
    }
});

/**
 * Get the viewer loader.
 * @name FORGE.Viewer#load
 * @type {FORGE.Loader}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "load",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._load;
    }
});

/**
 * Get the viewer cache.
 * @name FORGE.Viewer#cache
 * @type {FORGE.Cache}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "cache",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._cache;
    }
});

/**
 * Get the viewer plugins interface.
 * @name FORGE.Viewer#plugins
 * @type {FORGE.PluginManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "plugins",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._plugins;
    }
});

/**
 * Get the viewer clock interface.
 * @name FORGE.Viewer#clock
 * @type {FORGE.Clock}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "clock",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._clock;
    }
});

/**
 * Get the viewer tween interface.
 * @name FORGE.Viewer#tween
 * @type {FORGE.TweenManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "tween",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._tween;
    }
});

/**
 * Get the viewer i18n and locales interface.
 * @name FORGE.Viewer#i18n
 * @type {FORGE.LocaleManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "i18n",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._i18n;
    }
});

/**
 * Get the viewer render manager.
 * @name FORGE.Viewer#renderer
 * @type {FORGE.RenderManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "renderer",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._renderManager;
    }
});

/**
 * Get the view.
 * @name FORGE.Viewer#view
 * @type {FORGE.ViewManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "view",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._renderManager.view;
    }
});

/**
 * Get the camera.
 * @name FORGE.Viewer#camera
 * @type {FORGE.Camera}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "camera",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._renderManager.camera;
    }
});

/**
 * Get controller manager.
 * @name FORGE.Viewer#controllers
 * @readonly
 * @type {FORGE.ControllerManager}
 */
Object.defineProperty(FORGE.Viewer.prototype, "controllers",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._controllers;
    }
});

/**
 * Get the postProcessing object.
 * @name FORGE.Viewer#postProcessing
 * @type {FORGE.PostProcessing}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "postProcessing",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._postProcessing;
    }
});

/**
 * Get the viewer keyboard interface.
 * @name FORGE.Viewer#keyboard
 * @type {FORGE.Keyboard}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "keyboard",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._keyboard;
    }
});

/**
 * Get the viewer gyroscope interface.
 * @name FORGE.Viewer#gyroscope
 * @type {FORGE.Gyroscope}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "gyroscope",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._gyroscope;
    }
});

/**
 * Get the viewer gamepad interface.
 * @name FORGE.Viewer#gamepad
 * @type {FORGE.GamepadsManager}
 * @readonly
 */
Object.defineProperty(FORGE.Viewer.prototype, "gamepad",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._gamepad;
    }
});

/**
 * Get and set the viewer width.
 * @name FORGE.Viewer#width
 * @type {number}
 */
Object.defineProperty(FORGE.Viewer.prototype, "width",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._container.width;
    },

    /** @this {FORGE.Viewer} */
    set: function(value)
    {
        this._container.width = value;
    }
});

/**
 * Get and set the viewer height.
 * @name FORGE.Viewer#height
 * @type {number}
 */
Object.defineProperty(FORGE.Viewer.prototype, "height",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        return this._container.height;
    },

    /** @this {FORGE.Viewer} */
    set: function(value)
    {
        this._container.height = value;
    }
});


/**
 * Get the "onReady" {@link FORGE.EventDispatcher} of the viewer.
 * @name FORGE.Viewer#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Viewer.prototype, "onReady",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        if (this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});

/**
 * Get the "onPause" {@link FORGE.EventDispatcher} of the viewer.
 * @name FORGE.Viewer#onPause
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Viewer.prototype, "onPause",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        if (this._onPause === null)
        {
            this._onPause = new FORGE.EventDispatcher(this);
        }

        return this._onPause;
    }
});

/**
 * Get the "onResume" {@link FORGE.EventDispatcher} of the viewer.
 * @name FORGE.Viewer#onResume
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Viewer.prototype, "onResume",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        if (this._onResume === null)
        {
            this._onResume = new FORGE.EventDispatcher(this);
        }

        return this._onResume;
    }
});

/**
 * Get the "onConfigLoadComplete" {@link FORGE.EventDispatcher} of the viewer.
 * @name FORGE.Viewer#onConfigLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Viewer.prototype, "onConfigLoadComplete",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        if (this._onConfigLoadComplete === null)
        {
            this._onConfigLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onConfigLoadComplete;
    }
});

/**
 * Get the "onMainConfigLoadComplete" {@link FORGE.EventDispatcher} of the viewer.
 * @name FORGE.Viewer#onMainConfigLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Viewer.prototype, "onMainConfigLoadComplete",
{
    /** @this {FORGE.Viewer} */
    get: function()
    {
        if (this._onMainConfigLoadComplete === null)
        {
            this._onMainConfigLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onMainConfigLoadComplete;
    }
});


/**
 * Manage Tags inside FORGE.
 * Tags is singleton, so if you have multiple instances in the same page you MUST avoid Tags conflict.
 * @constructor
 * @extends {FORGE.BaseObject}
 */
FORGE.Tags = (function(c)
{
    var Tmp = c();
    Tmp.prototype = Object.create(FORGE.BaseObject.prototype);
    Tmp.prototype.constructor = Tmp;

    /**
     * Register a tagged object.
     * @method FORGE.Tags.register
     * @param  {Object} object - The object you want to register into the tag index.
     */
    Tmp.prototype.register = function(object)
    {
        if(typeof object === "object" && typeof object.uid === "string" && Array.isArray(object.tags))
        {
            var uid = object.uid;
            var tags = object.tags;
            var tag;

            if(FORGE.UID.exists(uid) === true)
            {
                //this._objects[uid] = object;
                for(var i = 0, ii = tags.length; i < ii; i++)
                {
                    tag = tags[i];

                    //Skip if a tag is not a string, this is an error!
                    if(typeof tag !== "string")
                    {
                        this.warn("Tags: An object has a tag that isn't a string!");
                        continue;
                    }

                    //If the tag is unknow, add an entry to the tags array
                    if(typeof this._allTags[tag] === "undefined")
                    {
                        this._allTags[tag] = [];

                        if(this._onRegister !== null)
                        {
                            this._onRegister.dispatch({"tag": tag});
                        }
                    }

                    if(this._allTags[tag].indexOf(uid) === -1)
                    {
                        this._allTags[tag].push(uid);
                    }
                }
            }
            else
            {
                this.warn("Tags: The object you want to register in tags is not known by UID!");
                this.warn(object);
            }
        }
        else
        {
            this.warn("Tags: No uid or no tags on the object you try to register!");
            this.warn(object);
        }
    };

    /**
     * Get uids associated to a tag or to an array of tags.<br>
     * You can filter the className of objets to get.
     * @method FORGE.Tags.getUids
     * @param  {string|Array} value - The uid or array of uids of object(s) you want to get.
     * @param  {string} className  - Filter you result by className of objects.
     * @return {Array<string>}  Returns an array of uids that matches the request.
     */
    Tmp.prototype.getUids = function(value, className)
    {
        var uids = [];

        if(typeof value === "string")
        {
            uids = this._allTags[value];
        }
        else if(Array.isArray(value))
        {
            var tag;
            for(var i = 0, ii = value.length; i < ii; i++)
            {
                tag = value[i];
                if(this.exists(tag) === true)
                {
                    uids.concat(this.get[tag]);
                }
            }
        }

        if(typeof className === "string")
        {
            uids = FORGE.UID.filterType(uids, className);
        }

        return uids;
    };

    /**
     * Get objects associated to a tag or to an array of tags.<br>
     * You can filter the className of objets to get.
     * @method FORGE.Tags.get
     * @param  {(string|Array)} value - The uid or array of uids of object(s) you want to get.
     * @param {string} className - Filter you result by className of objects.
     * @return {(Object|Array|undefined)} Returns an object or an array of objects that matches the request.
     */
    Tmp.prototype.get = function(value, className)
    {
        if(typeof value === "string")
        {
            return FORGE.UID.get(this._allTags[value], className);
        }
        else if(Array.isArray(value))
        {
            var uids = [];
            var tag;
            for(var i = 0, ii = value.length; i < ii; i++)
            {
                tag = value[i];
                if(this.exists(tag) === true)
                {
                    uids.concat(this.get[tag]);
                }
            }

            return FORGE.UID.get(uids, className);
        }

        return undefined;
    };

    /**
     * Tell if this tag have at least an object of a specific className.
     * @method  FORGE.Tags.hasTypeOf
     * @param  {string}  tag - The tag you want to check if it reference an onbject of specified className.
     * @param {string} className - The className you want to check.
     * @return {boolean} Returns true if the tag have at least an object of the asked className.
     */
    Tmp.prototype.hasTypeOf = function(tag, className)
    {
        if(this.exists(tag) === true)
        {
            var uids = this._allTags[tag];

            for (var i = 0, ii = this._allTags.length; i < ii; i++)
            {
                if(FORGE.UID.isTypeOf(uids[i], className) === true)
                {
                    return true;
                }
            }
        }

        return false;
    };

    /**
     * Does a tag exists ?
     * @method FORGE.Tags.exists
     * @param  {string} tag - The tag you want to check.
     * @return {boolean} Returns true if uid is already registered, false if not.
     */
    Tmp.prototype.exists = function(tag)
    {
        return (typeof this._allTags[tag] !== "undefined" && this._allTags[tag] !== null);
    };

    /**
     * Get the list of all existing tags.
     * @name  FORGE.Tags#list
     * @readonly
     */
    Object.defineProperty(Tmp.prototype, "list",
    {
        get: function()
        {
            return Object.keys(this._allTags);
        }
    });

    /**
     * Get the on add event dispatcher.
     * @name  FORGE.Tags#onRegister
     * @readonly
     */
    Object.defineProperty(Tmp.prototype, "onRegister",
    {
        get: function()
        {
            if(this._onRegister === null)
            {
                this._onRegister = new FORGE.EventDispatcher(this);
            }

            return this._onRegister;
        }
    });

    return new Tmp();
})(function()
{
    return function()
    {
        /**
         * The object that reference all tags objects couples.
         * @name FORGE.Tags#_allTags
         * @type {Object}
         * @private
         */
        this._allTags = {};

        /**
         * Event dispatcher for add tag event.
         * @name  FORGE.Tags#_onRegister
         * @type {FORGE.EventDispatcher}
         * @private
         */
        this._onRegister = null;

        FORGE.BaseObject.call(this, "Tags");
    };
});
/**
 * Manage UIDs inside FORGE.<br>
 * UID is singleton, so if you have multiple instances in the same page you MUST avoid UID conflict.
 * @constructor
 * @extends {FORGE.BaseObject}
 */
FORGE.UID = (function(c)
{
    var Tmp = c();
    Tmp.prototype = Object.create(FORGE.BaseObject.prototype);
    Tmp.prototype.constructor = Tmp;

    /**
     * Generate a uid.
     * @method FORGE.UID.generate
     * @return {string} Return a UID string.
     */
    Tmp.prototype.generate = function()
    {
        // see http://stackoverflow.com/a/2117523/1233787
        var uid = "xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx".replace(/[xy]/g, function(c)
        {
            var h = Math.random() * 16 | 0;
            h = c == "x" ? h : (h & 0x3 | 0x8);
            return h.toString(16);
        });

        if (FORGE.UID.exists(uid) === false)
        {
            return uid;
        }
        else
        {
            return FORGE.UID.generate();
        }
    };

    /**
     * Validate recursively a uid.
     * @method FORGE.UID.validate
     * @param  {Object} object - The object you want to validate into the index.
     * @return {boolean} Returns the validate status.
     */
    Tmp.prototype.validate = function(object)
    {
        var uids = [];

        var validateRecursive = function(object)
        {
            var uid;

            for (var i in object)
            {
                if (i === "uid")
                {
                    uid = object[i];

                    if (typeof uid === "string")
                    {
                        if (FORGE.UID.exists(uid) || uids.indexOf(uid) !== -1)
                        {
                            this.warn("UID configuration not valid, uid " + uid + " already exists!");
                            return false;
                        }
                        else
                        {
                            uids.push(object[i]);
                        }
                    }
                    else
                    {
                        this.warn("Found a uid in configuration that is not a string!");
                        return false;
                    }
                }
                else if (typeof(object[i]) === "object")
                {
                    return validateRecursive.apply(this, [object[i]]);
                }
            }

            return true;
        };

        return validateRecursive.apply(this, [object]);
    };


    /**
     * Register an object into the uid index.
     * @method FORGE.UID.register
     * @param  {Object} object - The object you want to register into the index.
     * @return {boolean} Return true if the object is added to UID, false if not.
     */
    Tmp.prototype.register = function(object)
    {
        if (typeof object === "object" && typeof object.uid === "string")
        {
            var uid = object.uid;

            if (FORGE.UID.exists(uid) === false)
            {
                this._objects[uid] = object;
                return true;
            }
            else
            {
                this.warn("The uid you want to register already exists!");
                this.warn(object);
            }
        }
        else
        {
            this.warn("No uid on the object you try to register!");
            this.warn(object);
        }

        return false;
    };

    /**
     * Unregister an object from the uid index.
     * @method FORGE.UID.unregister
     * @param  {Object} object - The object you want to unregister from the index.
     */
    Tmp.prototype.unregister = function(object)
    {
        if (typeof object === "object" && typeof object.uid === "string")
        {
            var uid = object.uid;
            this._objects[uid] = null;
            delete this._objects[uid];
            return;
        }

        this.warn("No uid on the object you try to unregister!");
    };

    /**
     * Get all uids or uids of a specific className.
     * @method  FORGE.UID.getuids
     * @param  {string} className - Type of uids you want to get, if undefined this will return all the uids.
     * @return {Array<string>} Returns an array of uids.
     */
    Tmp.prototype.getUids = function(className)
    {
        if (typeof className === "undefined" || className === null)
        {
            return Object.keys(this._objects);
        }
        else
        {
            return FORGE.UID.filterType(Object.keys(this._objects), className);
        }
    };

    /**
     * Filter an array of uids and return only uids of a specific className.
     * @method FORGE.UID.filterType
     * @param  {Array<string>} uids - Array of uids to filter by className.
     * @param  {string} className - Class name of object to filter.
     * @return {Array<string>} Returns an array of uids filtered by className.
     */
    Tmp.prototype.filterType = function(uids, className)
    {
        var filter = function(uid)
        {
            if (FORGE.UID.isTypeOf(uid, className))
            {
                return true;
            }

            return false;
        };

        return uids.filter(filter);
    };

    /**
     * Get a registered object from its uid.
     * @method FORGE.UID.get
     * @param  {string|Array} value - The uid or array of uids of object(s) you want to get.
     * @param  {string=} className - The className of the object you want
     * @return {*} Returns the object(s) related to the filters.
     */
    Tmp.prototype.get = function(value, className)
    {
        //If no value is passed, the whole uids array is used as values
        if (typeof value === "undefined" || value === null)
        {
            //If no value and no className, return all the objects.
            if (typeof className !== "string")
            {
                return this._objects;
            }

            value = Object.keys(this._objects);
        }

        if (typeof value === "string")
        {
            return this._objects[value];
        }
        else if (Array.isArray(value))
        {
            if (typeof className === "string")
            {
                value = FORGE.UID.filterType(value, className);
            }

            var result = [];
            var uid;
            for (var i = 0, ii = value.length; i < ii; i++)
            {
                uid = value[i];
                if (FORGE.UID.exists(uid) === true)
                {
                    result.push(this._objects[uid]);
                }
            }

            return result;
        }

        return undefined;
    };

    /**
     * Tell if this uid matches an object of a specific className.
     * @method  FORGE.UID.isTypeOf
     * @param {string}  uid - uid of the object you want to check the className of.
     * @param {string} className - The className you want to check.
     * @return {boolean} Returns true if the object is of the asked className.
     */
    Tmp.prototype.isTypeOf = function(uid, className)
    {
        var object = FORGE.UID.get(uid);
        return FORGE.Utils.isTypeOf(object, className);
    };

    /**
     * Does a uid exists?
     * @method FORGE.UID.exists
     * @param  {string} uid - The uid you want to check.
     * @return {boolean} Return true if uid is already registered, false if not.
     */
    Tmp.prototype.exists = function(uid)
    {
        return (typeof this._objects[uid] !== "undefined" && this._objects[uid] !== null);
    };

    return new Tmp();

})(function()
{
    return function()
    {
        /**
         * The object that reference all uids objects couples.
         * @name FORGE.UID#_objects
         * @type {Object}
         * @private
         */
        this._objects = {};

        FORGE.BaseObject.call(this, "UID");
    };
});


/**
 * FORGE.EventDispatcher can dispatch and reference listeners.
 *
 * @constructor FORGE.EventDispatcher
 * @param {Object} emitter - The object that wiil be considered as the emitter of the event.
 * @param {boolean=} memorize - Does the dispatcher should memorize the previous dispatcher ?
 * @extends {FORGE.BaseObject}
 */
FORGE.EventDispatcher = function(emitter, memorize)
{
    /**
     * The emitter reference.
     * @name  FORGE.EventDispatcher#_emitter
     * @type {Object}
     * @private
     */
    this._emitter = emitter;

    /**
     * Does the dispatcher should memorize the previous dispatch?<br>
     * If true, will redispatch with the previous data when you add a listener to it.
     * @name  FORGE.EventDispatcher#_memorize
     * @type {boolean}
     * @private
     */
    this._memorize = memorize || false;

    /**
     * A backup of previous dispatched data for memorized dispatcher.
     * @name FORGE.EventDispatcher#_previousData
     * @type {*}
     * @private
     */
    this._previousData = null;

    /**
     * Array of {@link FORGE.Listener}.
     * @name FORGE.EventDispatcher#_listeners
     * @type {?Array<FORGE.Listener>}
     * @private
     */
    this._listeners = null;

    /**
     * Is this event dipatcher is active?<br>
     * If not, it will ignore all dispatch calls.
     * @name FORGE.EventDispatcher#_active
     * @type {boolean}
     * @private
     */
    this._active = true;

    /**
     * Dispatched flag, set to true at the first dispatch
     * @name  FORGE.EventDispatcher#_dispatched
     * @type {boolean}
     * @private
     */
    this._dispatched = false;

    FORGE.BaseObject.call(this, "EventDispatcher");
};

FORGE.EventDispatcher.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.EventDispatcher.prototype.constructor = FORGE.EventDispatcher;

/**
 * Create the listeners array an push a new listener into it.
 *
 * @method FORGE.EventDispatcher#_addListener
 * @private
 * @param {Function} listener - The listener to add.
 * @param {boolean} isOnce - Is the dispatcher should trigger this listener only one time?
 * @param {Object} context - The context in which the listener will be executed.
 * @param {number=} priority - The priority of the event.
 */
FORGE.EventDispatcher.prototype._addListener = function(listener, isOnce, context, priority)
{
    if(typeof listener !== "function")
    {
        this.warn("You're trying to add an undefined listener");
        return;
    }

    if(this.has(listener, context) === true)
    {
        this.warn("You're trying to add a duplicate listener for this context");
        return;
    }

    var lis = new FORGE.Listener(this, listener, isOnce, context, priority);

    //register the listener with priority
    this._registerListener(lis);

    if(this._memorize === true && this._active === true && this._dispatched === true)
    {
        lis.execute(this._previousData);
    }

    return listener;
};

/**
 * Internal method to insert the listener into the array according to its priority.
 *
 * @method FORGE.EventDispatcher#_registerListener
 * @private
 * @param  {FORGE.Listener} listener - The object which handle the listener and it's context.
 */
FORGE.EventDispatcher.prototype._registerListener = function(listener)
{
    if(this._listeners === null)
    {
        this._listeners = [];
    }

    if (this._listeners.length === 0)
    {
        this._listeners.push(listener);
        return;
    }

    var n = this._listeners.length;
    do
    {
        n--;
    }
    while(this._listeners[n] && listener._priority <= this._listeners[n]._priority);

    this._listeners.splice(n + 1, 0, listener);
};

/**
 * Internal method to get the index of a couple listener + context.
 *
 * @method FORGE.EventDispatcher#_indexOfListener
 * @private
 * @param  {Function} listener - The listener function you need to find its index.
 * @param {Object} context - The context associated to the listener function.
 * @return {number} - The index of the couple listener + context if found, -1 if not.
 */
FORGE.EventDispatcher.prototype._indexOfListener = function(listener, context)
{
    if(this._listeners === null)
    {
        return -1;
    }

    if ( typeof context === "undefined" )
    {
        context = null;
    }

    var _listener;

    for ( var i = 0, ii = this._listeners.length; i < ii; i++ )
    {
        _listener = this._listeners[i];

        if(_listener.listener === listener && _listener.context === context)
        {
            return i;
        }
    }

    return -1;
};

/**
 * Add an event listener function.
 *
 * @method FORGE.EventDispatcher#add
 * @param {Function} listener - Event handler callback function.
 * @param {Object} context - The context for the listener call.
 * @param {number=} priority - Priority level for the event to be execute.
 */
FORGE.EventDispatcher.prototype.add = function(listener, context, priority)
{
    this._addListener(listener, false, context, priority);
};

/**
 * Add an event listener function that will be triggered only once.
 *
 * @method FORGE.EventDispatcher#addOnce
 * @param {Function} listener - Event handler callback function.
 * @param {Object} context - The context for the listener call.
 * @param {number=} priority - Priority level for the event to be execute.
 */
FORGE.EventDispatcher.prototype.addOnce = function(listener, context, priority)
{
    this._addListener(listener, true, context, priority);
};

/**
 * Remove a {@link FORGE.Listener} from this event dispatcher.
 *
 * @method FORGE.EventDispatcher#remove
 * @param  {Function} listener - The listener handler to be removed.
 * @param  {Object} context - The context of the handler to be removed.
 */
FORGE.EventDispatcher.prototype.remove = function(listener, context)
{
    var i = this._indexOfListener(listener, context);

    if(i !== -1)
    {
        this._listeners[i].destroy();
        this._listeners.splice(i, 1);
    }
};

/**
 * Check if this event dispatcher has a specific listener.
 *
 * @method FORGE.EventDispatcher#has
 * @param  {Function} listener - listener function to check.
 * @param  {Object} context - listener context to check.
 * @return {boolean} Returns true if the dispatcher has the listener, false if not.
 */
FORGE.EventDispatcher.prototype.has = function(listener, context)
{
    return this._indexOfListener(listener, context) !== -1;
};

/**
 * Dispatch the event, will trigger all the listeners methods.
 *
 * @method FORGE.EventDispatcher#dispatch
 * @param {*=} data - Any object or data you want to associate with the dispatched event.
 * @param {boolean=} async - Does the dispatch need to be async ?
 */
FORGE.EventDispatcher.prototype.dispatch = function(data, async)
{
    this._dispatched = true;

    if(this._memorize === true)
    {
        this._previousData = data === undefined ? null : data;
    }

    if(this._active === false || this._listeners === null)
    {
        return;
    }

    var n = this._listeners.length;

    while(n--)
    {
        this._listeners[n].execute(data, async);
    }

};

/**
 * Destroy method.
 * @method FORGE.EventDispatcher#destroy
 */
FORGE.EventDispatcher.prototype.destroy = function()
{
    if(this._alive === false)
    {
        return;
    }

    if(this._listeners !== null)
    {
        var n = this._listeners.length;
        while(n--)
        {
            this._listeners[n].destroy();
        }

        this._listeners = null;
    }

    this._emitter = null;
    this._previousData = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
* Get the emitter object associated to this event dispatcher.
* @name FORGE.EventDispatcher#emitter
* @readonly
* @type {Object}
*/
Object.defineProperty(FORGE.EventDispatcher.prototype, "emitter",
{
    /** @this {FORGE.EventDispatcher} */
    get: function()
    {
        return this._emitter;
    }
});

/**
* Get and set the memorize flag associated to this event dispatcher.
* @name FORGE.EventDispatcher#memorized
* @type {boolean}
*/
Object.defineProperty(FORGE.EventDispatcher.prototype, "memorize",
{
    /** @this {FORGE.EventDispatcher} */
    get: function()
    {
        return this._memorize;
    },

    /** @this {FORGE.EventDispatcher} */
    set: function(value)
    {
        this._memorize = Boolean(value);
    }
});

/**
* Get and set the active flag associated to this event dispatcher.<br>
* If active is false, this dispatcher will not dispatch any event.
* @name FORGE.EventDispatcher#active
* @type {boolean}
*/
Object.defineProperty(FORGE.EventDispatcher.prototype, "active",
{
    /** @this {FORGE.EventDispatcher} */
    get: function()
    {
        return this._active;
    },

    /** @this {FORGE.EventDispatcher} */
    set: function(value)
    {
        this._active = Boolean(value);
    }
});

/**
 * FORGE.Event reference an emitter and eventually some data.
 *
 * @constructor FORGE.Event
 * @param {Object} emitter - The object that will be considered as the emitter of the event.
 * @param {Object} data - Any data associated to this event.
 */
FORGE.Event = function(emitter, data)
{
    /**
     * The object that will be considered as the emitter of the event.
     * @name FORGE.Event#_emitter
     * @type {Object}
     * @private
     */
    this._emitter = emitter;

    /**
     * Any data associated to this event.
     * @name FORGE.Event#_data
     * @type {?Object}
     * @private
     */
    this._data = data || null;
};


FORGE.Event.prototype.constructor = FORGE.Event;

/**
 * Get the event emitter.
 * @name  FORGE.Event#emitter
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Event.prototype, "emitter",
{
    /** @this {FORGE.Event} */
    get: function ()
    {
        return this._emitter;
    }
});

/**
 * Get the data associated to the event.
 * @name  FORGE.Event#data
 * @readonly
 * @type {?Object}
 */
Object.defineProperty(FORGE.Event.prototype, "data",
{
    /** @this {FORGE.Event} */
    get: function ()
    {
        return this._data;
    }
});


/**
 * Object that handle listener function and its context.
 *
 * @constructor FORGE.Listener
 * @param {Function} listener - The handler function
 * @param {boolean} isOnce - If this listener will trigger only once, then delete itself from its dispatcher.
 * @param {Object} context - The context for listener execution.
 * @param {number=} priority - The priority of the listener.
 */
FORGE.Listener = function(dispatcher, listener, isOnce, context, priority)
{
    /**
     * Reference to the {@link FORGE.EvenDispatcher} this listener is attached to.
     * @name FORGE.Listener#_dispatcher
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._dispatcher = dispatcher;

    /**
     * The callback function that will be triggered when event occurs.
     * @name FORGE.Listener#_listener
     * @type {Function}
     * @private
     */
    this._listener = listener;

    /**
     * The context for listener execution.
     * @name FORGE.Listener#_context
     * @type {?Object}
     * @private
     */
    this._context = context || null;

    /**
     * If this listener will trigger only once, then delete itself from its dispatcher.
     * @name FORGE.Listener#_isOnce
     * @type {boolean}
     * @default false
     * @private
     */
    this._isOnce = isOnce || false;

    /**
     * The priority level of the event listener.<br>
     * Listeners with higher priority will be executed before listeners with lower priority.<br>
     * Listeners with same priority level will be executed at the same order as they were added. (default = 0).
     * @name FORGE.Listener#_priority
     * @type {number}
     * @private
     */
    this._priority = priority || 0;

    /**
     * The number of times the listener has been called.
     * @name  FORGE.Listener#_callCount
     * @type {number}
     * @private
     */
    this._callCount = 0;

    /**
     * The active state of the listener. Will be executed only if active.
     * @name  FORGE.Listener#_active
     * @type {boolean}
     * @private
     */
    this._active = true;

    /**
     * Is the async process is busy?
     * @name  FORGE.Listener#_asyncBusy
     * @type {boolean}
     * @private
     */
    this._asyncBusy = false;
};

FORGE.Listener.prototype.constructor = FORGE.Listener;

FORGE.Listener.prototype._execute = function(data)
{
    var event = new FORGE.Event(this._dispatcher.emitter, data);
    this._listener.call(this._context, event);

    this._callCount++;

    if(this._isOnce === true)
    {
        this.detach();
    }

    this._asyncBusy = false; //reset the async busy flag to false in all cases
};

/**
* Call listener passing a data object.<br>
* If listener was added using EventDispatcher.addOnce() it will be automatically removed.
*
* @method FORGE.Listener#execute
* @param {*=} data - Data that should be passed to the listener.
* @param {boolean=} async - Execute the listener in async mode (with a setTimeout at 0).
*/
FORGE.Listener.prototype.execute = function(data, async)
{
    if(this._active === true && this._listener !== null)
    {
        if(async === true && this._asyncBusy === false)
        {
            this._asyncBusy = true;
            var executeBind = this._execute.bind(this, data);
            window.setTimeout(executeBind, 0);
        }
        else if(async !== true)
        {
            this._execute(data);
        }
    }
};

/**
 * Detach the listener from its event dispatcher.
 * @method FORGE.Listener#detach
 */
FORGE.Listener.prototype.detach = function()
{
    return this._dispatcher.remove(this._listener, this._context);
};

/**
 * Destroy method.
 * @method FORGE.Listener#destroy
 */
FORGE.Listener.prototype.destroy = function()
{
    this._dispatcher = null;
    this._listener = null;
    this._context = null;
};

/**
 * Get the listener function.
 * @name FORGE.Listener#listener
 * @readonly
 * @type {Function}
 */
Object.defineProperty(FORGE.Listener.prototype, "listener", {

    /** @this {FORGE.Listener} */
    get: function ()
    {
        return this._listener;
    }
});

/**
 * Get the context object.
 * @name FORGE.Listener#context
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Listener.prototype, "context", {

    /** @this {FORGE.Listener} */
    get: function ()
    {
        return this._context;
    }
});

/**
 * Get the priority number.
 * @name FORGE.Listener#priority
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Listener.prototype, "priority", {

    /** @this {FORGE.Listener} */
    get: function ()
    {
        return this._priority;
    }
});

/**
 * Get the call count property value.
 * @name FORGE.Listener#callCount
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Listener.prototype, "callCount", {

    /** @this {FORGE.Listener} */
    get: function ()
    {
        return this._callCount;
    }
});

/**
 * The FORGE.Story manages groups and scenes of the project's story.
 *
 * @constructor FORGE.Story
 * @param {FORGE.Viewer} viewer {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.Story = function(viewer)
{
    /**
    * The viewer reference.
    * @name FORGE.Story#_viewer
    * @type {FORGE.Viewer}
    * @private
    */
    this._viewer = viewer;

    /**
     * The config object.
     * @name FORGE.Story#_config
     * @type {?StoryConfig}
     * @private
     */
    this._config = null;

    /**
     * The internationalizable name of the story.
     * @name FORGE.Story#_name
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._name = null;

    /**
     * The internationalizable slug name of the story.
     * @name FORGE.Story#_slug
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._slug = null;

    /**
     * The internationalizable description of the story.
     * @name FORGE.Story#_description
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._description = null;

    /**
     * The default uid to load, this can be a scene or a group uid.
     * @name FORGE.Story#_default
     * @type {string}
     * @private
     */
    this._default = "";

    /**
     * Array of {@link FORGE.Scene} uid of the story.
     * @name FORGE.Story#_scenes
     * @type {?Array<string>}
     * @private
     */
    this._scenes = null;

    /**
     * Uid of the current scene.
     * @name FORGE.Story#_sceneUid
     * @type {string}
     * @private
     */
    this._sceneUid = "";

    /**
     * Array of {@link FORGE.Group} uid of the story.
     * @name FORGE.Story#_groups
     * @type {?Array<string>}
     * @private
     */
    this._groups = null;

    /**
     * Uid of the current group.
     * @name FORGE.Story#_groupUid
     * @type {?string}
     * @private
     */
    this._groupUid = "";

    /**
     * Story events from the json configuration
     * @name FORGE.Story#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = {};

    /**
     * On ready event dispatcher
     * @name FORGE.Story#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    /**
     * On scene load requset event dispatcher.
     * @name  FORGE.Story#_onSceneLoadRequest
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onSceneLoadRequest = null;

    /**
     * On scene load start event dispatcher.
     * @name  FORGE.Story#_onSceneLoadStart
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onSceneLoadStart = null;

    /**
     * On scene load progress event dispatcher.
     * @name FORGE.Story#_onSceneLoadProgress
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onSceneLoadProgress = null;

    /**
     * On scene load complete event dispatcher.
     * @name FORGE.Story#_onSceneLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onSceneLoadComplete = null;

    /**
     * On scene load error event dispatcher.
     * @name FORGE.Story#_onSceneLoadError
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onSceneLoadError = null;

    /**
     * On scene preview event dispatcher.
     * @name FORGE.Story#_onScenePreview
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onScenePreview = null;

    /**
     * On group change event dispatcher.
     * @name FORGE.Story#_onGroupChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onGroupChange = null;

    FORGE.BaseObject.call(this, "Story");

};

FORGE.Story.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Story.prototype.constructor = FORGE.Story;

/**
 * Event handler for the configuration JSON load complete.
 * @method FORGE.Story#_configLoadComplete
 * @private
 * @param  {FORGE.File} file - The {@link FORGE.File} that describes the loaded JSON file.
 */
FORGE.Story.prototype._configLoadComplete = function(file)
{
    this.log("FORGE.Story.loadComplete();");

    var json = this._viewer.cache.get(FORGE.Cache.types.JSON, file.key);
    var config = /** @type {StoryConfig} */ (json.data);

    this._parseConfig(config);
};

/**
 * Parse the story configuration.
 * @method FORGE.Story#_parseConfig
 * @private
 * @param  {StoryConfig} config - The story configuration to parse.
 */
FORGE.Story.prototype._parseConfig = function(config)
{
    if(FORGE.UID.validate(config) !== true)
    {
        throw "Story configuration is not valid, you have duplicate uids";
    }

    this._config = config;

    this._uid = this._config.uid;
    this._register();

    this._default = this._config.default;

    // Set the keys for the locale strings
    this._name.key = this._config.name;
    this._slug.key = this._config.slug;
    this._description.key = this._config.description;

    if(typeof this._config.scenes !== "undefined" && this._config.scenes.length > 0)
    {
        this._createScenes(this._config.scenes);
    }

    if(typeof this._config.groups !== "undefined" && this._config.groups.length > 0)
    {
        this._createGroups(this._config.groups);
    }

    if(typeof this._config.events === "object" && this._config.events !== null)
    {
        this._createEvents(this._config.events);
    }

    this._checkStoryScenes();
};

/**
 * Create events dispatchers.
 * @method FORGE.Story#_createEvents
 * @private
 * @param {StoryEventsConfig} events - The events config of the story.
 */
FORGE.Story.prototype._createEvents = function(events)
{
    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all events.
 * @method FORGE.Story#_clearEvents
 * @private
 */
FORGE.Story.prototype._clearEvents = function()
{
    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Check if all scenes have been loaded.
 * @method FORGE.Story#_checkStoryScenes
 * @private
 */
FORGE.Story.prototype._checkStoryScenes = function()
{
    if(typeof this._config.scenes === "undefined" || this._scenes.length === this._config.scenes.length)
    {
        this._setStoryReady();
    }
};

/**
 * Activate the story and load the first scene.
 * @method FORGE.Story#_setStoryReady
 * @private
 */
FORGE.Story.prototype._setStoryReady = function()
{
    if(this._onReady !== null)
    {
        this._onReady.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onReady, "ActionEventDispatcher") === true)
    {
        this._events.onReady.dispatch();
    }

    // If slug name in URL load the associated object
    // NB: I couldn't find another way to correctly access the property without minification
    var hashParameters = FORGE.URL.parse()["hashParameters"];

    if(hashParameters !== null && typeof hashParameters.uid === "string" && FORGE.UID.exists(hashParameters.uid))
    {
        this._loadUid(hashParameters.uid);

        // this._viewer.i18n.onLocaleChangeComplete.addOnce(this._localeChangeCompleteHandler, this);
    }
    //Else if default uid
    else if(this._default !== "" && this._default !== null && FORGE.UID.exists(this._default))
    {
        this._loadUid(this._default);
    }
    //Else load scene index 0
    else
    {
        this.loadScene(0);
    }
};

/**
 * Load an object by its uid. It can be a scene uid or a group uid.
 * @method FORGE.Story._loadUid
 * @private
 * @param  {string} uid - Uid of the object you want to load.
 */
FORGE.Story.prototype._loadUid = function(uid)
{
    if(FORGE.UID.isTypeOf(uid, "Scene") === true || FORGE.UID.isTypeOf(uid, "Group") === true)
    {
        FORGE.UID.get(uid).load();
    }
};

/**
 * Create {@link FORGE.Scene}s from scenes configuration object.
 * @method FORGE.Story#_createScenes
 * @private
 * @param  {Array<SceneConfig>} config - The object that describes the scenes, issued from the main configuration.
 */
FORGE.Story.prototype._createScenes = function(config)
{
    var scene;
    for(var i = 0, ii = config.length; i < ii; i++)
    {
        scene = new FORGE.Scene(this._viewer);
        scene.addConfig(config[i]);

        //Scene is not booted at creation if the scene config is in an external file
        if(scene.booted === true)
        {
            this._addScene(scene);
        }
    }
};

/**
 * Add a scene into the scenes array.
 * @param {FORGE.Scene} scene - The scene to add.
 * @private
 */
FORGE.Story.prototype._addScene = function(scene)
{
    scene.onLoadRequest.add(this._sceneLoadRequestHandler, this);
    scene.onLoadStart.add(this._sceneLoadStartHandler, this);
    scene.onLoadComplete.add(this._sceneLoadCompleteHandler, this);

    this._scenes.push(scene.uid);
};

/**
 * Handler for the onBackgroundReady event of the FORGE.Scene that is being loaded.
 * @method FORGE.Story#_backgroundReadyHandler
 * @private
 */
FORGE.Story.prototype._backgroundReadyHandler = function()
{
    this._viewer.renderer.pickingManager.start();
};

/**
 * Internal envent handler for scene load request.
 * @method FORGE.Story#_sceneLoadRequestHandler
 * @param  {FORGE.Event} event - The {@link FORGE.Event} emitted by the scene that its load method is requested.
 * @private
 */
FORGE.Story.prototype._sceneLoadRequestHandler = function(event)
{
    var previousScene = this.scene;
    var nextScene = event.emitter;
    var time = NaN;

    //Unload the previous scene
    if(previousScene !== null)
    {
        // If the next scene have to be sync with the previous one
        if(nextScene.sync.indexOf(previousScene.uid) !== -1 && previousScene.media.type === FORGE.MediaType.VIDEO)
        {
            time = previousScene.media.displayObject.currentTime;
        }

        previousScene.unload();
    }

    this._sceneUid = nextScene.uid;

    nextScene.loadStart(time);

    this.log("scene load request");

    //The scene has no group so nullify the _groupUid
    if(nextScene.hasGroups() === false)
    {
        this._groupUid = null;

        if(this._onGroupChange !== null)
        {
            this._onGroupChange.dispatch(/** @type {StoryEvent} */({ groupUid: this._groupUid }));
        }

        if(FORGE.Utils.isTypeOf(this._events.onGroupChange, "ActionEventDispatcher") === true)
        {
            this._events.onGroupChange.dispatch();
        }
    }
    else if (nextScene.hasGroups() === true && nextScene.hasGroup(this._groupUid) === false)
    {
        this._setGroupUid(nextScene.groups[0].uid);
    }

    if(this._onSceneLoadRequest !== null)
    {
        this._onSceneLoadRequest.dispatch(/** @type {StoryEvent} */({ sceneUid: this._sceneUid }));
    }

    if(FORGE.Utils.isTypeOf(this._events.onSceneLoadRequest, "ActionEventDispatcher") === true)
    {
        this._events.onSceneLoadRequest.dispatch();
    }
};

/**
 * Internal envent handler for scene load start, updates the group index, re-dispatch scene load start at the story level.
 * @method FORGE.Story#_sceneLoadStartHandler
 * @private
 */
FORGE.Story.prototype._sceneLoadStartHandler = function()
{
    this.log("scene load start");

    if(this._onSceneLoadStart !== null)
    {
        this._onSceneLoadStart.dispatch(/** @type {StoryEvent} */({ sceneUid: this._sceneUid }));
    }

    if(FORGE.Utils.isTypeOf(this._events.onSceneLoadStart, "ActionEventDispatcher") === true)
    {
        this._events.onSceneLoadStart.dispatch();
    }
};

/**
 * Internal event handler for scene load complete, re-dispatch the load complete event at the story level.
 * @method FORGE.Story#_sceneLoadCompleteHandler
 * @private
 */
FORGE.Story.prototype._sceneLoadCompleteHandler = function()
{
    this.log("scene load complete");

    if(this._onSceneLoadComplete !== null)
    {
        this._onSceneLoadComplete.dispatch(/** @type {StoryEvent} */({ sceneUid: this._sceneUid }));
    }

    if(FORGE.Utils.isTypeOf(this._events.onSceneLoadComplete, "ActionEventDispatcher") === true)
    {
        this._events.onSceneLoadComplete.dispatch();
    }
};

/**
 * Create {@link FORGE.Group}s from groups configuration object.
 * @method FORGE.Story#_createGroups
 * @private
 * @param  {Object} config - The object that describes the groups, issued from the main configuration.
 */
FORGE.Story.prototype._createGroups = function(config)
{
    var group;
    for(var i = 0, ii = config.length; i < ii; i++)
    {
        group = new FORGE.Group(this._viewer, config[i]);
        this._groups.push(group.uid);
    }
};

/**
 * Internal method to set the current {@link FORGE.Group} uid. Dispatch "onGroupChange" if its a valid operation.
 * @method FORGE.Story#_setGroupUid
 * @private
 * @param {string} uid - Uid of the group to set
 */
FORGE.Story.prototype._setGroupUid = function(uid)
{
    if(FORGE.UID.isTypeOf(uid, "Group") === true && uid !== this._groupUid)
    {
        this._groupUid = uid;

        if(this._onGroupChange !== null)
        {
            this._onGroupChange.dispatch(/** @type {StoryEvent} */({ groupUid: this._groupUid }));
        }

        if(FORGE.Utils.isTypeOf(this._events.onGroupChange, "ActionEventDispatcher") === true)
        {
            this._events.onGroupChange.dispatch();
        }
    }
};

/**
 * Boot sequence.
 * @method FORGE.Story#boot
 */
FORGE.Story.prototype.boot = function()
{
    this.log("FORGE.Story.boot();");

    this._scenes = [];
    this._groups = [];

    this._name = new FORGE.LocaleString(this._viewer);
    this._slug = new FORGE.LocaleString(this._viewer);
    this._description = new FORGE.LocaleString(this._viewer);
};

/**
 * Load a JSON story configuration.
 * @method FORGE.Story#load
 * @param  {(string|StoryConfig)} config - The URL of the configuration JSON file to load or a story configuration object.
 */
FORGE.Story.prototype.load = function(config)
{
    this.log("load");

    if(typeof config === "string")
    {
        this._viewer.load.json("forge.story.config", config, this._configLoadComplete, this);
    }
    else if (typeof config === "object")
    {
        this._parseConfig(config);
    }
};

/**
 * Know if the story have any {@link FORGE.Scene}.
 * @method FORGE.Story#hasScenes
 * @return {boolean} Returns true if the story have at least a {@link FORGE.Scene}, false if not.
 */
FORGE.Story.prototype.hasScenes = function()
{
    return this._scenes.length !== 0;
};

/**
 * Know if the story have any {@link FORGE.Group}.
 * @method FORGE.Story#hasGroups
 * @return {boolean} Returns true if the story have at least a {@link FORGE.Group}, false if not.
 */
FORGE.Story.prototype.hasGroups = function()
{
    return this._groups.length !== 0;
};

/**
 * Load the next scene of the story.
 * @method FORGE.Story#nextScene
 */
FORGE.Story.prototype.nextScene = function()
{
    var index = this._scenes.indexOf(this._sceneUid);
    var uid;

    if(index + 1 < this._scenes.length)
    {
        uid = this._scenes[index + 1];
    }
    else
    {
        uid = this._scenes[0];
    }

    this.loadScene(uid);
};

/**
 * Load the previous scene of the story.
 * @method FORGE.Story#previousScene
 */
FORGE.Story.prototype.previousScene = function()
{
    var index = this._scenes.indexOf(this._sceneUid);
    var uid;

    if(index - 1 >= 0)
    {
        uid = this._scenes[index - 1];
    }
    else
    {
        uid = this._scenes[this._scenes.length - 1];
    }

    this.loadScene(uid);
};

/**
 * Load a {@link FORGE.Scene}.
 * @method FORGE.Story#loadScene
 * @param  {(FORGE.Scene|number|string)} value - Either the {@link FORGE.Scene} itself its index in the main _scenes Array or its uid.
 */
FORGE.Story.prototype.loadScene = function(value)
{
    var uid;

    // use the index of the group array
    if (typeof value === "number")
    {
        if(value >= 0 && value < this._scenes.length)
        {
            uid = this._scenes[value];
        }
        else
        {
            this.warn("Load scene error: index "+value+" is out of bounds");
        }
    }
    // use the uid
    else if (typeof value === "string" && FORGE.UID.isTypeOf(value, "Scene"))
    {
        uid = value;
    }
    // use a Group object directly
    else if (typeof value === "object" && FORGE.Utils.isTypeOf(value, "Scene"))
    {
        uid = value.uid;
    }

    //If uid is defined and if it's not the current scene
    if(typeof uid !== "undefined" && uid !== this._sceneUid)
    {
        // Disable picking while the scene is loading
        this._viewer.renderer.pickingManager.stop();

        // Read it when the background is ready again
        this._viewer.renderer.onBackgroundReady.addOnce(this._backgroundReadyHandler, this);

        this._loadUid(uid);
    }
};

/**
 * Internal method to load a {@link FORGE.Group}.
 * @method FORGE.Story#loadGroup
 * @param  {(FORGE.Group|number|string)} value - Either the {@link FORGE.Group} itself its index in the main _groups Array or its uid.
 */
FORGE.Story.prototype.loadGroup = function(value)
{
    var uid;

    // use the index of the group array
    if (typeof value === "number")
    {
        if(value >= 0 && value < this._groups.length)
        {
            uid = this._groups[value];
        }
        else
        {
            this.warn("Load group, index "+value+" is out of bounds");
        }
    }
    // use the uid
    else if (typeof value === "string" && FORGE.UID.isTypeOf(value, "Group"))
    {
        uid = value;
    }
    // use a Group object directly
    else if (typeof value === "object" && FORGE.Utils.isTypeOf(value, "Group"))
    {
        uid = value.uid;
    }

    //If uid is defined and if it's not the current scene
    if(typeof uid !== "undefined" && uid !== this._groupUid)
    {
        this._setGroupUid(uid);
        this._loadUid(uid);
    }
};

/**
 * Event handler for scene loaded from an external json file into config.
 * @method  FORGE.Story#_sceneConfigComplete
 * @param  {FORGE.Scene} scene - The scene that has finish to load its configuration.
 * @private
 */
FORGE.Story.prototype.notifySceneConfigLoadComplete = function(scene)
{
    this._addScene(scene);

    //check if all scenes are loaded
    this._checkStoryScenes();
};

/**
 * Destroy method.
 * @method FORGE.Story#destroy
 */
FORGE.Story.prototype.destroy = function()
{
    this._viewer = null;
    this._config = null;

    this._name.destroy();
    this._name = null;

    this._slug.destroy();
    this._slug = null;

    this._description.destroy();
    this._description = null;

    for(var i = 0, ii = this._scenes.length; i < ii; i++)
    {
        FORGE.UID.get(this._scenes[i]).destroy();
    }
    this._scenes = null;

    for(var j = 0, jj = this._groups.length; j < jj; j++)
    {
        FORGE.UID.get(this._groups[j]).destroy();
    }
    this._groups = null;

    // Events
    if(this._onReady !== null)
    {
        this._onReady.destroy();
        this._onReady = null;
    }

    if(this._onSceneLoadStart !== null)
    {
        this._onSceneLoadStart.destroy();
        this._onSceneLoadStart = null;
    }

    if(this._onSceneLoadProgress !== null)
    {
        this._onSceneLoadProgress.destroy();
        this._onSceneLoadProgress = null;
    }

    if(this._onSceneLoadComplete !== null)
    {
        this._onSceneLoadComplete.destroy();
        this._onSceneLoadComplete = null;
    }

    if(this._onSceneLoadError !==null)
    {
        this._onSceneLoadError.destroy();
        this._onSceneLoadError = null;
    }

    if(this._onScenePreview !== null)
    {
        this._onScenePreview.destroy();
        this._onScenePreview = null;
    }

    if(this._onGroupChange !== null)
    {
        this._onGroupChange.destroy();
        this._onGroupChange = null;
    }

    this._clearEvents();
    this._events = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
* Get the story config object.
* @name FORGE.Story#config
* @readonly
* @type {Object}
*/
Object.defineProperty(FORGE.Story.prototype, "config",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._config;
    }
});

/**
* Get the name of the story.
* @name FORGE.Story#name
* @readonly
* @type {string}
*/
Object.defineProperty(FORGE.Story.prototype, "name",
{
    /** @this {FORGE.Story} */
    get: function ()
    {
        return this._name.value;
    }
});

/**
* Get the slug name of the story.
* @name FORGE.Story#slug
* @readonly
* @type {string}
*/
Object.defineProperty(FORGE.Story.prototype, "slug",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._slug.value;
    }
});

/**
* Get the description of the story.
* @name FORGE.Story#description
* @readonly
* @type {string}
*/
Object.defineProperty(FORGE.Story.prototype, "description",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._description.value;
    }
});


/**
* Get the Array of {@link FORGE.Scene} that compose the story.
* @name FORGE.Story#scenes
* @readonly
* @type {?Array<FORGE.Scene>}
*/
Object.defineProperty(FORGE.Story.prototype, "scenes",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return FORGE.UID.get(this._scenes);
    }
});

/**
 * Get the current {@link FORGE.Scene} object, or set the current scene passing the {@link FORGE.Scene} object itself, its index or uid.
 * @name FORGE.Story#scene
 * @type  {FORGE.Scene}
 */
Object.defineProperty(FORGE.Story.prototype, "scene",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._sceneUid === null || this._sceneUid === "")
        {
            return null;
        }

        return FORGE.UID.get(this._sceneUid);
    },

    /** @this {FORGE.Story} */
    set: function(value)
    {
        this.loadScene(value);
    }
});

/**
* Get all the sceneUids.
* @name FORGE.Story#sceneUids
* @readonly
* @type {Array<string>}
*/
Object.defineProperty(FORGE.Story.prototype, "sceneUids",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._scenes;
    }
});

/**
* Get the current sceneUid.
* @name FORGE.Story#sceneUid
* @type {string}
*/
Object.defineProperty(FORGE.Story.prototype, "sceneUid",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._sceneUid;
    },

    /** @this {FORGE.Story} */
    set: function(value)
    {
        this.loadScene(value);
    }
});

/**
* Get the Array of {@link FORGE.Group} that compose the story.
* @name FORGE.Story#groups
* @readonly
* @type {?Array<FORGE.Group>}
*/
Object.defineProperty(FORGE.Story.prototype, "groups",
{
    /** @this {FORGE.Story} */
    get: function ()
    {
        return FORGE.UID.get(this._groups);
    }
});

/**
 * Get the current {@link FORGE.Group} object, or set the current scene passing the {@link FORGE.Group} object itself, its index or uid.
 * @name FORGE.Story#group
 * @type  {(FORGE.Group)}
 */
Object.defineProperty(FORGE.Story.prototype, "group",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._groupUid === null || this._groupUid === "")
        {
            return null;
        }

        return FORGE.UID.get(this._groupUid);
    },

    /** @this {FORGE.Story} */
    set: function(value)
    {
        this.loadGroup(value);
    }
});

/**
* Get all the group Uids.
* @name FORGE.Story#groupUids
* @readonly
* @type {Array<string>}
*/
Object.defineProperty(FORGE.Story.prototype, "groupUids",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._groups;
    }
});

/**
* Get the current groupUid.
* @name FORGE.Story#groupUid
* @type {string}
*/
Object.defineProperty(FORGE.Story.prototype, "groupUid",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        return this._groupUid;
    },

    /** @this {FORGE.Story} */
    set: function(value)
    {
        this.loadGroup(value);
    }
});

/**
 * Get the onReady {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Story.prototype, "onReady",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});

/**
 * Get the onSceneLoadRequest {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onSceneLoadRequest
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Story.prototype, "onSceneLoadRequest",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onSceneLoadRequest === null)
        {
            this._onSceneLoadRequest = new FORGE.EventDispatcher(this);
        }

        return this._onSceneLoadRequest;
    }
});

/**
 * Get the onSceneLoadStart {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onSceneLoadStart
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Story.prototype, "onSceneLoadStart",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onSceneLoadStart === null)
        {
            this._onSceneLoadStart = new FORGE.EventDispatcher(this);
        }

        return this._onSceneLoadStart;
    }
});

/**
 * Get the onSceneLoadProgress {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onSceneLoadProgress
 * @readonly
 * @type {FORGE.EventDispatcher}
 * @todo  This event is currently not dispatched
 */
Object.defineProperty(FORGE.Story.prototype, "onSceneLoadProgress",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onSceneLoadProgress === null)
        {
            this._onSceneLoadProgress = new FORGE.EventDispatcher(this);
        }

        return this._onSceneLoadProgress;
    }
});

/**
 * Get the onSceneLoadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onSceneLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Story.prototype, "onSceneLoadComplete",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onSceneLoadComplete === null)
        {
            this._onSceneLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onSceneLoadComplete;
    }
});

/**
 * Get the onSceneLoadError {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onSceneLoadError
 * @readonly
 * @type {FORGE.EventDispatcher}
 * @todo  This event is currently not dispatched
 */
Object.defineProperty(FORGE.Story.prototype, "onSceneLoadError",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onSceneLoadError === null)
        {
            this._onSceneLoadError = new FORGE.EventDispatcher(this);
        }

        return this._onSceneLoadError;
    }
});

/**
 * Get the onScenePreview {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onScenePreview
 * @readonly
 * @type {FORGE.EventDispatcher}
 * @todo  This event is currently not dispatched
 */
Object.defineProperty(FORGE.Story.prototype, "onScenePreview",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onScenePreview === null)
        {
            this._onScenePreview = new FORGE.EventDispatcher(this);
        }

        return this._onScenePreview;
    }
});

/**
 * Get the onGroupChange {@link FORGE.EventDispatcher}.
 * @name  FORGE.Story#onGroupChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Story.prototype, "onGroupChange",
{
    /** @this {FORGE.Story} */
    get: function()
    {
        if(this._onGroupChange === null)
        {
            this._onGroupChange = new FORGE.EventDispatcher(this);
        }

        return this._onGroupChange;
    }
});

/**
 * A FORGE.Group is an object that represents a group of {@link FORGE.Scene} objects.
 *
 * @constructor FORGE.Group
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {GroupConfig} config - The group config object.
 * @extends {FORGE.BaseObject}
 */
FORGE.Group = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.Group#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The config object.
     * @name FORGE.Group#_config
     * @type {GroupConfig}
     * @private
     */
    this._config = config;

    /**
     * The internationalizable name of the group.
     * @name FORGE.Group#_name
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._name = null;

    /**
     * The color associated to this group (hexa code like #ffffff).<br>
     * Default value is the white code #ffffff.
     * @name  FORGE.Group#_color
     * @type {string}
     * @private
     */
    this._color = "#ffffff";

    /**
     * The color alpha associated to this group is a number between 0 and 1.<br>
     * Default value is 1.
     * @name  FORGE.Group#_alpha
     * @type {number}
     * @private
     */
    this._alpha = 1;

    /**
     * The internationalizable slug name of the group.
     * @name FORGE.Group#_slug
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._slug = null;

    /**
     * The internationalizable description of the group.
     * @name FORGE.Group#_description
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._description = null;

    /**
     * Array of children that can be {@link FORGE.Scene} or {@link FORGE.Group}.
     * @name FORGE.Group#children
     * @type {Array<string>}
     * @private
     */
    this._children = null;

    /**
     * The default child uid to load.
     * @name  FORGE.group#_default
     * @type {string}
     * @private
     */
    this._default = "";

    /**
     * Parents of this group, these are {@link FORGE.Group}.
     * @name  FORGE.Group#parents
     * @type {Array<FORGE.Group>}
     * @private
     */
    this._parents = null;

    FORGE.BaseObject.call(this, "Group");

    this._boot();
};

FORGE.Group.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Group.prototype.constructor = FORGE.Group;

/**
 * Boot sequence.
 * @method FORGE.Group#_boot
 * @private
 */
FORGE.Group.prototype._boot = function()
{
    this.log("FORGE.Group._boot();");

    this._uid = this._config.uid;
    this._tags = this._config.tags;
    this._register();

    this._parents = [];
    this._children = [];

    this._name = new FORGE.LocaleString(this._viewer, this._config.name);
    this._slug = new FORGE.LocaleString(this._viewer, this._config.slug);
    this._description = new FORGE.LocaleString(this._viewer, this._config.description);

    if (typeof this._config.color === "string")
    {
        this._color = this._config.color;
    }

    if (typeof this._config.alpha === "number")
    {
        this._alpha = this._config.alpha;
    }

    this._parseChildren(this._config);
};

/**
 * Parse scenes in config.
 * @method FORGE.Group#_parseChildren
 * @private
 * @param  {GroupConfig} config - The object that describes the scenes config.
 */
FORGE.Group.prototype._parseChildren = function(config)
{

    if (typeof config.children !== "undefined" && FORGE.Utils.isArrayOf(config.children, "string") === true)
    {
        this._children = config.children;
    }
    else
    {
        this.warn("A group has no children in its configuration, or configuration is not valid!");
    }

    //Parse the default child of the group
    if (typeof config.default === "string")
    {
        if (this._children.indexOf(config.default) !== -1)
        {
            this._default = config.default;
        }
        else
        {
            this.warn("A group has a default child that is not in its children array!");
        }
    }
};

/**
 * Load the group with a specific scene, by default the scene is scene index 0.
 * @method FORGE.Group#load
 * @param {number|string|FORGE.Scene|FORGE.Group=} value - The numeric index of the child or the uid of the child you want to load.<br>
 * If no value passed in arguments, the group will load its default child if it is set.<br>
 * If no default child set, it will load its first child no matter the type of this child
 */
FORGE.Group.prototype.load = function(value)
{
    this.log("FORGE.Group.load();");

    if (this._children === null || this._children.length === 0)
    {
        throw "Group.load() : can't load a group that have no children";
    }

    var uid = "";

    // If no value passed in arguments, the group will load its default child if it is set.
    if (typeof value === "undefined" || value === null)
    {
        if (typeof this._default === "string" && this._default !== "")
        {
            uid = this._default;
        }
        else
        {
            uid = this._children[0];
        }
    }
    else if (typeof value === "number" && value >= 0 && value < this._children.length)
    {
        uid = this._children[value];
    }
    else if (typeof value === "string")
    {
        uid = value;
    }
    else if (typeof value === "object" && this.hasChild(value) === true)
    {
        uid = value.uid;
    }

    if (this._children.indexOf(uid) === -1)
    {
        throw "Group.load() : uid \"" + uid + "\" is not in children of the group!";
    }

    if (FORGE.UID.isTypeOf(uid, "Scene") === true || FORGE.UID.isTypeOf(uid, "Group") === true)
    {
        var child = FORGE.UID.get(uid);
        child.load();
    }
    else
    {
        throw "Impossible to load group child with uid " + uid + ", it seems to be neither a scene or a group!";
    }
};

/**
 * Know if a {@link FORGE.Scene}, a {@link FORGE.Group} or a uid is part of this group?
 * @method FORGE.Group#hasScene
 * @param {(FORGE.Scene|FORGE.Group|string)} value - Either the {@link FORGE.Scene} or a {@link FORGE.Group} or a uid string.
 * @return {boolean} Returns true if the child is part of this group, false if not.
 */
FORGE.Group.prototype.hasChild = function(value)
{
    if (typeof value === "string")
    {
        return this._children.indexOf(value) !== -1;
    }
    else if (FORGE.Utils.isTypeOf(value, "Scene") === true || FORGE.Utils.isTypeOf(value, "Group") === true)
    {
        return this.hasChild(value.uid);
    }

    return false;
};


/**
 * Know if a {@link FORGE.Scene} is part of this group?
 * @method FORGE.Group#hasScene
 * @param {(FORGE.Scene|string)} value - Either the {@link FORGE.Scene} or a uid string.
 * @return {boolean} Returns true if the scene is part of this group, false if not.
 */
FORGE.Group.prototype.hasScene = function(value)
{
    if (typeof value === "string" && FORGE.UID.isTypeOf(value, "Scene"))
    {
        return this._children.indexOf(value) !== -1;
    }
    else if (FORGE.Utils.isTypeOf(value, "Scene") === true)
    {
        return this.hasScene(value.uid);
    }

    return false;
};

/**
 * Know if a {@link FORGE.Group} is part of this group?
 * @method FORGE.Group#hasGroup
 * @param {(FORGE.Group|string)} value - Either the {@link FORGE.Group} or a uid string.
 * @return {boolean} Returns true if the group is part of this group, false if not.
 */
FORGE.Group.prototype.hasGroup = function(value)
{
    if (typeof value === "string" && FORGE.UID.isTypeOf(value, "Group"))
    {
        return this._children.indexOf(value) !== -1;
    }
    else if (FORGE.Utils.isTypeOf(value, "Group") === true)
    {
        return this.hasGroup(value.uid);
    }

    return false;
};

/**
 * Know if this group have any children.
 * @method FORGE.Group#hasChildren
 * @return {boolean} Returns true if this group have at least a children, false if not.
 */
FORGE.Group.prototype.hasChildren = function()
{
    return this._children.length !== 0;
};

/**
 * Know if this group have any object of a specified className.
 * @method FORGE.Group#hasTypeOfChild
 * @param {string} className - the className of the object you want to know if this group has in its children array.
 * @return {boolean} Returns true if this group have at least an object of the requested className in its children, false if not.
 */
FORGE.Group.prototype.hasTypeOfChild = function(className)
{
    var uid;
    for (var i = 0, ii = this._children.length; i < ii; i++)
    {
        uid = this._children[i];
        if (FORGE.UID.isTypeOf(uid, className))
        {
            return true;
        }
    }

    return false;
};

/**
 * Know if this group have any {@link FORGE.Scene}.
 * @method FORGE.Group#hasScenes
 * @return {boolean} Returns true if this group have at least a {@link FORGE.Scene}, false if not.
 */
FORGE.Group.prototype.hasScenes = function()
{
    return this.hasTypeOfChild("Scene");
};

/**
 * Know if this group have any {@link FORGE.Group}.
 * @method FORGE.Group#hasGroups
 * @return {boolean} Returns true if this group have at least a {@link FORGE.Group}, false if not.
 */
FORGE.Group.prototype.hasGroups = function()
{
    return this.hasTypeOfChild("Group");
};

/**
 * Get children uids of a specified className (or not).
 * @method FORGE.Group#getChildrenUids
 * @param {string=} className - the className of the object uids you want to get.
 * @return {Array} Returns array of children uids of the specified className.
 */
FORGE.Group.prototype.getChildrenUids = function(className)
{
    //If no className specified, return the complete array of children uids.
    if (typeof className !== "string")
    {
        return this._children;
    }

    var children = [];

    for (var i = 0, ii = this._children.length; i < ii; i++)
    {
        if (FORGE.UID.isTypeOf(this._children[i], className))
        {
            children.push(this._children[i]);
        }
    }

    return children;
};

/**
 * Get children objects of a specified className.<br>
 * If you do not specify className this method will return all the children objects.
 * @method FORGE.Group#getChildren
 * @param {string=} className - the className of the object you want to get.
 * @return {Array} Returns array of children objects of the specified className.
 */
FORGE.Group.prototype.getChildren = function(className)
{
    var uids = this.getChildrenUids(className);
    return FORGE.UID.get(uids);
};

/**
 * Load the next scene of this group.<br>
 * If this group has no scene, you can't use this method.<br>
 * If the current scene of the story is not one of this group, the group will load either the default child or its first found scene.<br>
 * If the current scene is part of this group, nextScene will loop forward through its scenes.
 * @method FORGE.Group#nextScene
 */
FORGE.Group.prototype.nextScene = function()
{
    if (this.hasScenes() === false)
    {
        this.warn("Can't do Group.nextScene() on this group that have no scenes!");
        return;
    }

    var scenesUids = this.scenesUids;
    var index = scenesUids.indexOf(this._viewer.story.sceneUid);
    var uid; //Default uid to load is the first scene.

    if (index === -1)
    {
        if (FORGE.UID.isTypeOf(this._default, "Scene"))
        {
            uid = this._default;
        }
        else
        {
            uid = scenesUids[0];
        }
    }

    if (index + 1 < scenesUids.length)
    {
        uid = scenesUids[index + 1];
    }
    else if (index !== -1)
    {
        uid = scenesUids[0];
    }

    this.load(uid);
};

/**
 * Load the previous scene of this group.<br>
 * If this group has no scene, you can't use this method.<br>
 * If the current scene of the story is not one of this group, the group will load either the default child or its first found scene.<br>
 * If the current scene is part of this group, previousScene will loop backward through its scenes.
 * @method FORGE.Group#previousScene
 */
FORGE.Group.prototype.previousScene = function()
{
    if (this.hasScenes() === false)
    {
        this.warn("Can't do Group.previousScene() on this group that have no scenes!");
        return;
    }

    var scenesUids = this.scenesUids;
    var index = scenesUids.indexOf(this._viewer.story.sceneUid);
    var uid; //Default uid to load is the first scene.

    if (index === -1)
    {
        if (FORGE.UID.isTypeOf(this._default, "Scene"))
        {
            uid = this._default;
        }
        else
        {
            uid = scenesUids[0];
        }
    }

    if (index - 1 >= 0)
    {
        uid = scenesUids[index - 1];
    }
    else if (index !== -1)
    {
        uid = scenesUids[scenesUids.length - 1];
    }

    this.load(uid);
};

/**
 * Destroy method.
 * @method FORGE.Group#destroy
 */
FORGE.Group.prototype.destroy = function()
{
    this._viewer = null;

    this._children = null;

    this._name.destroy();
    this._name = null;

    this._slug.destroy();
    this._slug = null;

    this._description.destroy();
    this._description = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the group config object.
 * @name FORGE.Group#config
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Group.prototype, "config",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._config;
    }
});

/**
 * Get the count of how many times this group has been viewed.
 * @name FORGE.Group#viewCount
 * @readonly
 * @type {number}
 */
// Object.defineProperty(FORGE.Group.prototype, "viewCount",
// {
//     /** @this {FORGE.Group} */
//     get: function()
//     {
//         var count = 0;

//         for(var i = 0, ii = this._scenes.length; i < ii; i++)
//         {
//             count += this._scenes[i].viewCount;
//         }
//         return count;
//     }
// });

/**
 * Know if this group has been viewed at least one time.
 * @name FORGE.Group#viewed
 * @readonly
 * @type {boolean}
 */
// Object.defineProperty(FORGE.Group.prototype, "viewed",
// {
//     /** @this {FORGE.Group} */
//     get: function()
//     {
//         return this.viewCount !== 0;
//     }
// });


/**
 * Get the name of this group.
 * @name FORGE.Group#name
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Group.prototype, "name",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._name.value;
    }
});

/**
 * Get the color associated to this group.
 * @name FORGE.Group#color
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Group.prototype, "color",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._color;
    }
});

/**
 * Get the alpha associated to this group.
 * @name FORGE.Group#alpha
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Group.prototype, "alpha",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._alpha;
    }
});

/**
 * Get the slug name of this group.
 * @name FORGE.Group#slug
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Group.prototype, "slug",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._slug.value;
    }
});

/**
 * Get the description of this group.
 * @name FORGE.Group#description
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Group.prototype, "description",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this._description.value;
    }
});

/**
 * Get the Array of children uids that compose this group.
 * @name FORGE.Group#childrenUids
 * @readonly
 * @type {?Array<string>}
 */
Object.defineProperty(FORGE.Group.prototype, "childrenUids",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildrenUids();
    }
});

/**
 * Get the Array of children objects that compose this group.
 * @name FORGE.Group#children
 * @readonly
 * @type {?Array<Object>}
 */
Object.defineProperty(FORGE.Group.prototype, "children",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildren();
    }
});

/**
 * Get the Array of {@link FORGE.Scene} uids that compose this group.
 * @name FORGE.Group#scenesUids
 * @readonly
 * @type {?Array<string>}
 */
Object.defineProperty(FORGE.Group.prototype, "scenesUids",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildrenUids("Scene");
    }
});

/**
 * Get the Array of {@link FORGE.Scene} objects that compose this group.
 * @name FORGE.Group#scenes
 * @readonly
 * @type {?Array<FORGE.Scene>}
 */
Object.defineProperty(FORGE.Group.prototype, "scenes",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildren("Scene");
    }
});

/**
 * Get the Array of {@link FORGE.Group} uids that compose this group.
 * @name FORGE.Group#groupsUids
 * @readonly
 * @type {?Array<string>}
 */
Object.defineProperty(FORGE.Group.prototype, "groupsUids",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildrenUids("Group");
    }
});

/**
 * Get the Array of {@link FORGE.Group} uids that compose this group.
 * @name FORGE.Group#groups
 * @readonly
 * @type {?Array<FORGE.Group>}
 */
Object.defineProperty(FORGE.Group.prototype, "groups",
{
    /** @this {FORGE.Group} */
    get: function()
    {
        return this.getChildren("Group");
    }
});
/**
 * A FORGE.Scene is an object that represents a scene of a {@link FORGE.Story}.
 *
 * @constructor FORGE.Scene
 * @param {FORGE.Viewer} viewer {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.Scene = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.Scene#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The scene config object.
     * @name FORGE.Scene#_sceneConfig
     * @type {?SceneConfig}
     * @private
     */
    this._config = null;

    /**
     * The internationalizable name of the scene.
     * @name FORGE.Scene#_name
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._name = null;

    /**
     * The internationalizable slug name of the scene.
     * @name FORGE.Scene#_slug
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._slug = null;

    /**
     * The internationalizable description of the scene.
     * @name FORGE.Scene#_description
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._description = null;

    /**
     * The array of scene uids to be sync with
     * @name FORGE.Scene#_sync
     * @type {Array<string>}
     * @private
     */
    this._sync = [];

    /**
     * The number of times this has been viewed.
     * @name  FORGE.Scene#_viewCount
     * @type {number}
     * @private
     */
    this._viewCount = 0;

    /**
     * Array of group uids this scene belongs to. aka "parents".
     * @name FORGE.Scene#_groups
     * @type {?Array<string>}
     * @private
     */
    this._groups = null;

    /**
     * Is booted flag.
     * @name FORGE.Scene#_booted
     * @type {boolean}
     * @private
     */
    this._booted = false;

    /**
     * Use external config file flag.
     * @name FORGE.Scene#_useExternalConfig
     * @type {boolean}
     * @private
     */
    this._useExternalConfig = false;

    /**
     * Scene events from the json configuration
     * @name FORGE.Story#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = {};

    /**
     * The media of the scene
     * @name FORGE.Scene#_media
     * @type {FORGE.Media}
     * @private
     */
    this._media = null;

    /**
     * Load request event dispatcher.
     * @name  FORGE.Scene#_onLoadRequest
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadRequest = null;

    /**
     * Load start event dispatcher.
     * @name  FORGE.Scene#_onLoadStart
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadStart = null;

    /**
     * Load complete event dispatcher.
     * @name  FORGE.Scene#_onLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadComplete = null;

    /**
     * Unload start event dispatcher.
     * @name  FORGE.Scene#_onUnloadStart
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onUnloadStart = null;

    /**
     * Unload complete event dispatcher.
     * @name  FORGE.Scene#_onUnloadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onUnloadComplete = null;

    /**
     * Load complete event dispatcher for scene configuration file.
     * @name  FORGE.Scene#_onConfigLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onConfigLoadComplete = null;

    /**
     * media create event dispatcher.
     * @name  FORGE.Scene#_onMediaCreate
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onMediaCreate = null;

    FORGE.BaseObject.call(this, "Scene");
};

FORGE.Scene.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Scene.prototype.constructor = FORGE.Scene;

/**
 * Parse scene configuration.
 * @method FORGE.Scene#_parseConfig
 * @private
 * @param {SceneConfig} config - The configuration object to parse.
 */
FORGE.Scene.prototype._parseConfig = function(config)
{
    this._config = config;

    this._uid = config.uid;
    this._tags = config.tags;
    this._register();

    this._name = new FORGE.LocaleString(this._viewer, this._config.name);
    this._slug = new FORGE.LocaleString(this._viewer, this._config.slug);
    this._description = new FORGE.LocaleString(this._viewer, this._config.description);
    this._sync = (FORGE.Utils.isArrayOf(this._config.sync, "string") === true) ? this._config.sync : [];

    if(typeof config.events === "object" && config.events !== null)
    {
        this._createEvents(config.events);
    }

    if (this._booted === false && typeof config.url === "string" && config.url !== "")
    {
        //use an external config json file
        this._useExternalConfig = true;
        this._viewer.load.json(this._uid, config.url, this._configLoadComplete, this);
    }
    else
    {
        this._booted = true;
    }
};

/**
 * Event handler for the load of the scene config json file.
 * @method FORGE.Scene#_configLoadComplete
 * @param {FORGE.File} file - The file data.
 * @private
 *
 * @todo the "story.config" cache file is not updated in this case, a cache entry is added for each scene UID.
 */
FORGE.Scene.prototype._configLoadComplete = function(file)
{
    this.log("config load complete");

    this._booted = true;

    //extend the config
    if (typeof file.data === "string")
    {
        file.data = /** @type {Object} */ (JSON.parse(file.data));
    }

    // extend init config
    this._config = /** @type {SceneConfig} */ (FORGE.Utils.extendSimpleObject(this._config, file.data));

    this._viewer.story.notifySceneConfigLoadComplete(this);

    if (this._onConfigLoadComplete !== null)
    {
        this._onConfigLoadComplete.dispatch();
    }
};

/**
 * Create events dispatchers.
 * @method FORGE.Scene#_createEvents
 * @private
 * @param {SceneEventsConfig} events - The events config of the scene.
 */
FORGE.Scene.prototype._createEvents = function(events)
{
    this.log("create events");

    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all events.
 * @method FORGE.Scene#_clearEvents
 * @private
 */
FORGE.Scene.prototype._clearEvents = function()
{
    this.log("clear events");

    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Create the scene media
 * @param  {SceneMediaConfig} media - media configuration
 * @private
 */
FORGE.Scene.prototype._createMedia = function(media)
{
    this.log("create media");

    if(this._media === null)
    {
        this._media = new FORGE.Media(this._viewer, media);

        if(this._onMediaCreate !== null)
        {
            this._onMediaCreate.dispatch({ media: this._media });
        }
    }
};

/**
 * Add a scene configuration object.
 * @method  FORGE.Scene#addConfig
 * @param {SceneConfig} config - The scene configuration object to add.
 */
FORGE.Scene.prototype.addConfig = function(config)
{
    this._parseConfig(config);
};

/**
 * Load just emmit a load request. The story will trigger the loadStart.
 * @method FORGE.Scene#load
 */
FORGE.Scene.prototype.load = function()
{
    this.log("load");

    if (this._viewer.story.scene === this)
    {
        return;
    }

    if (this._onLoadRequest !== null)
    {
        this._onLoadRequest.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onLoadRequest, "ActionEventDispatcher") === true)
    {
        this._events.onLoadRequest.dispatch();
    }
};

/**
 * Create the media and start to load.
 * @method FORGE.Scene#loadStart
 * @param {number} time - The time of the media (if video)
 */
FORGE.Scene.prototype.loadStart = function(time)
{
    if(typeof time === "number" && isNaN(time) === false && typeof this._config.media !== "undefined")
    {
        if(typeof this._config.media.options === "undefined")
        {
            this._config.media.options = {};
        }

        this._config.media.options.startTime = time;
    }

    this._createMedia(this._config.media);

    if (this._onLoadStart !== null)
    {
        this._onLoadStart.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onLoadStart, "ActionEventDispatcher") === true)
    {
        this._events.onLoadStart.dispatch();
    }

    this._viewCount++;

    if (this._onLoadComplete !== null)
    {
        this._onLoadComplete.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onLoadComplete, "ActionEventDispatcher") === true)
    {
        this._events.onLoadComplete.dispatch();
    }
};

/**
 * Unload the scene.
 * @method FORGE.Scene#unload
 * @todo cleanup if useless
 */
FORGE.Scene.prototype.unload = function()
{
    this.log("unload");

    if (this._onUnloadStart !== null)
    {
        this._onUnloadStart.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onUnloadStart, "ActionEventDispatcher") === true)
    {
        this._events.onUnloadStart.dispatch();
    }

    this._media.destroy();
    this._media = null;

    if (this._onUnloadComplete !== null)
    {
        this._onUnloadComplete.dispatch();
    }

    if(FORGE.Utils.isTypeOf(this._events.onUnloadComplete, "ActionEventDispatcher") === true)
    {
        this._events.onUnloadComplete.dispatch();
    }
};

/**
 * Know if a {@link FORGE.Group} is related to this scene?
 * @method FORGE.Scene#hasGroup
 * @param {(FORGE.Group|string)} value - Either the {@link FORGE.Group} itself or its index or its uid.
 * @return {boolean} Returns true if the {@link FORGE.Group} is related to this scene, false if not.
 */
FORGE.Scene.prototype.hasGroup = function(value)
{
    if (typeof value === "string" && FORGE.UID.isTypeOf(value, "Group") === true)
    {
        return FORGE.UID.get( /** @type {string} */ (value)).hasScene(this);
    }
    else if (FORGE.Utils.isTypeOf(value, "Group") === true)
    {
        return value.hasScene(this);
    }

    return false;
};

/**
 * Know if this scene is related to any {@link FORGE.Group}.
 * @method FORGE.Scene#hasGroups
 * @return {boolean} Returns true if this scene is related to at least a {@link FORGE.Group}, false if not.
 */
FORGE.Scene.prototype.hasGroups = function()
{
    var groups = this._viewer.story.groups;
    var group;
    for (var i = 0, ii = groups.length; i < ii; i++)
    {
        group = groups[i];
        if (group.hasScene(this) === true)
        {
            return true;
        }
    }

    return false;
};

/**
 * Know if the scene has sound source?
 * @method FORGE.Scene#hasSoundSource
 * @return {boolean} Returns true if the scene has a sound source, false if not.
 */
FORGE.Scene.prototype.hasSoundSource = function()
{
    if (typeof this._config.sound !== "undefined" && typeof this._config.sound.source !== "undefined" && ((typeof this._config.sound.source.url !== "undefined" && this._config.sound.source.url !== "") || (typeof this._config.sound.source.target !== "undefined" && this._config.sound.source.target !== "")))
    {
        return true;
    }
    return false;
};

/**
 * Know if the scene has sound target as source?
 * @method FORGE.Scene#hasSoundTarget
 * @param {string} uid - The target source UID to verify.
 * @return {boolean} Returns true if the scene has a sound source target, false if not.
 */
FORGE.Scene.prototype.hasSoundTarget = function(uid)
{
    if (typeof this._config.sound !== "undefined" && typeof this._config.sound.source !== "undefined" && typeof this._config.sound.source.target !== "undefined" && this._config.sound.source.target !== "" && this._config.sound.source.target === uid)
    {
        return true;
    }

    return false;
};

/**
 * Know if an ambisonic sound is attached to the scene?
 * @method FORGE.Scene#isAmbisonic
 * @return {boolean} Returns true if the scene has an ambisonic sound source, false if not.
 */
FORGE.Scene.prototype.isAmbisonic = function()
{
    //@todo real check of the UID target object rather then the isAmbisonic method of the FORGE.Scene
    if (this.hasSoundSource() === true && this._config.sound.type === FORGE.SoundType.AMBISONIC)
    {
        return true;
    }

    return false;
};

/**
 * Destroy method
 * @method FORGE.Scene#destroy
 */
FORGE.Scene.prototype.destroy = function()
{
    this._viewer = null;

    this._name.destroy();
    this._name = null;

    this._slug.destroy();
    this._slug = null;

    this._description.destroy();
    this._description = null;

    if (this._media !== null)
    {
        this._media.destroy();
        this._media = null;
    }

    if (this._onLoadStart !== null)
    {
        this._onLoadStart.destroy();
        this._onLoadStart = null;
    }

    if (this._onLoadComplete !== null)
    {
        this._onLoadComplete.destroy();
        this._onLoadComplete = null;
    }

    if (this._onUnloadStart !== null)
    {
        this._onUnloadStart.destroy();
        this._onUnloadStart = null;
    }

    if (this._onUnloadComplete !== null)
    {
        this._onUnloadComplete.destroy();
        this._onUnloadComplete = null;
    }

    this._clearEvents();
    this._events = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
* Get the booted status of the scene.
* @name FORGE.Scene#booted
* @type {boolean}
* @readonly
*/
Object.defineProperty(FORGE.Scene.prototype, "booted",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._booted;
    }
});

/**
* Get the group config object.
* @name FORGE.Scene#config
* @readonly
* @type {SceneConfig}
*/
Object.defineProperty(FORGE.Scene.prototype, "config",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._config;
    }
});

/**
 * Get the count of how many times this group has been viewed.
 * @name FORGE.Scene#viewCount
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Scene.prototype, "viewCount",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._viewCount;
    }
});

/**
 * Know if this scene has been viewed at least one time.
 * @name FORGE.Scene#viewed
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Scene.prototype, "viewed",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._viewCount !== 0;
    }
});

/**
 * Get the name of this scene.
 * @name FORGE.Scene#name
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Scene.prototype, "name",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._name.value;
    }
});

/**
 * Get the slug name of this scene.
 * @name FORGE.Scene#slug
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Scene.prototype, "slug",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._slug.value;
    }
});

/**
 * Get the description of this scene.
 * @name FORGE.Scene#description
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Scene.prototype, "description",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._description.value;
    }
});

/**
 * Get the sync array.
 * @name FORGE.Scene#sync
 * @readonly
 * @type {Array<string>}
 */
Object.defineProperty(FORGE.Scene.prototype, "sync",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._sync;
    }
});

/**
 * Get the Array of groups uids to which this scene belongs to.
 * @name FORGE.Scene#groupsUid
 * @readonly
 * @type {?Array<FORGE.Group>}
 */
Object.defineProperty(FORGE.Scene.prototype, "groupsUid",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        var groups = this._viewer.story.groups;
        var group;
        var result = [];

        for (var i = 0, ii = groups.length; i < ii; i++)
        {
            group = groups[i];

            if (group.hasScene(this) === true)
            {
                result.push(group.uid);
            }
        }

        return result;
    }
});

/**
 * Get the Array of {@link FORGE.Group} to which this scene belongs to.
 * @name FORGE.Scene#groups
 * @readonly
 * @type {?Array<FORGE.Group>}
 */
Object.defineProperty(FORGE.Scene.prototype, "groups",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return FORGE.UID.get(this.groupsUid);
    }
});

/**
 * Get the thumbnails Array.
 * @name  FORGE.Scene#thumbnails
 * @readonly
 * @type {Array}
 *
 * @todo  Define what is a thumbnail array, maybe with a thumbnail object descriptor
 */
Object.defineProperty(FORGE.Scene.prototype, "thumbnails",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._config.thumbnails;
    }
});

/**
 * Get the scene media.
 * @name  FORGE.Scene#media
 * @readonly
 * @type {FORGE.Media}
 */
Object.defineProperty(FORGE.Scene.prototype, "media",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return this._media;
    }
});

/**
 * Get the background of the scene.
 * @name  FORGE.Scene#background
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Scene.prototype, "background",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        return (typeof this._config.background !== "undefined") ? this._config.background : this._viewer.config.background;
    }
});

/**
 * Get the onLoadRequest {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onLoadRequest
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onLoadRequest",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onLoadRequest === null)
        {
            this._onLoadRequest = new FORGE.EventDispatcher(this);
        }

        return this._onLoadRequest;
    }
});

/**
 * Get the onLoadStart {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onLoadStart
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onLoadStart",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onLoadStart === null)
        {
            this._onLoadStart = new FORGE.EventDispatcher(this);
        }

        return this._onLoadStart;
    }
});

/**
 * Get the onLoadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onLoadComplete",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onLoadComplete === null)
        {
            this._onLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onLoadComplete;
    }
});

/**
 * Get the onUnloadStart {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onUnloadStart
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onUnloadStart",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onUnloadStart === null)
        {
            this._onUnloadStart = new FORGE.EventDispatcher(this);
        }

        return this._onUnloadStart;
    }
});

/**
 * Get the onUnloadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onUnloadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onUnloadComplete",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onUnloadComplete === null)
        {
            this._onUnloadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onUnloadComplete;
    }
});

/**
 * Get the onConfigLoadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onConfigLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onConfigLoadComplete",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onConfigLoadComplete === null)
        {
            this._onConfigLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onConfigLoadComplete;
    }
});

/**
 * Get the onMediaCreate {@link FORGE.EventDispatcher}.
 * @name  FORGE.Scene#onMediaCreate
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Scene.prototype, "onMediaCreate",
{
    /** @this {FORGE.Scene} */
    get: function()
    {
        if (this._onMediaCreate === null)
        {
            this._onMediaCreate = new FORGE.EventDispatcher(this);
        }

        return this._onMediaCreate;
    }
});

/**
 * Math helper
 * @namespace FORGE.Math
 * @type {Object}
 */
FORGE.Math = {};

/**
 * @name FORGE.Math.DEGREES
 * @type {string}
 * @const
 */
FORGE.Math.DEGREES = "degrees";

/**
 * @name FORGE.Math.RADIANS
 * @type {string}
 * @const
 */
FORGE.Math.RADIANS = "radians";

/**
 * FORGE.Math.DEG2RAD
 * @type {number}
 * @const
 */
FORGE.Math.DEG2RAD = Math.PI / 180.0;

/**
 * FORGE.Math.RAD2DEG
 * @type {number}
 * @const
 */
FORGE.Math.RAD2DEG = 180.0 / Math.PI;

/**
 * FORGE.Math.TWOPI
 * @type {number}
 * @const
 */
FORGE.Math.TWOPI = Math.PI * 2;


/**
 * Converts angle unit degrees to radians
 *
 * @method FORGE.Math.degToRad
 * @param  {number} deg - angle in degrees
 * @return {number} Return the angle in radians
 */
FORGE.Math.degToRad = function(deg)
{
    return deg * FORGE.Math.DEG2RAD;
};

/**
 * Converts angle unit radians to degrees
 *
 * @method FORGE.Math.radToDeg
 * @param  {number} rad - angle in radians
 * @return {number} Return the angle in degrees
 */
FORGE.Math.radToDeg = function(rad)
{
    return rad * FORGE.Math.RAD2DEG;
};

/**
 * Returns the value of a number rounded to the nearest decimal value
 *
 * @method FORGE.Math.round10
 * @param  {number} value - Value to round
 * @return {number} Return the rounded value
 */
FORGE.Math.round10 = function(value)
{
    return Math.round(value * 10) / 10;
};

/**
 * Clamp a value between a min and a max value
 *
 * @method FORGE.Math.clamp
 * @param  {number} value - Value to clamp
 * @param  {?number=} min - The min value
 * @param  {?number=} max - The max value
 * @return {number} Return the clamped value
 */
FORGE.Math.clamp = function(value, min, max)
{
    min = (typeof min === "number" && isNaN(min) === false) ? min : -Infinity;
    max = (typeof max === "number" && isNaN(max) === false) ? max : Infinity;

    return Math.min(Math.max(value, min), max);
};

/**
 * Wrap a value between a min and a max value
 *
 * @method FORGE.Math.wrap
 * @param  {number} value - Value to wrap
 * @param  {number} min - The min value
 * @param  {number} max - The max value
 * @return {number} Return the wrapped value
 */
FORGE.Math.wrap = function(value, min, max)
{
    if (value === max)
    {
        return max;
    }

    var range = max - min;

    if (range === 0)
    {
        return min;
    }

    return ((value - min) % range + range) % range + min;
};

/**
 * Linear mix function
 *
 * @method FORGE.Math.mix
 * @param  {number} a - first value
 * @param  {number} b - second value
 * @param  {number} mix - factor
 * @return {number} linear mix of a and b
 */
FORGE.Math.mix = function(a, b, mix)
{
    return a * mix + b * (1 - mix);
};

/**
 * Smoothstep function
 *
 * @method FORGE.Math.smoothStep
 * @param  {number} value - Value to smooth
 * @param  {number} edge0 - low edge
 * @param  {number} edge1 - high edge
 * @return {number} smooth step result
 */
FORGE.Math.smoothStep = function(value, edge0, edge1)
{
    value = FORGE.Math.clamp((value - edge0) / (edge1 - edge0), 0.0, 1.0);
    return value * value * (3 - 2 * value);
};

/**
 * Check if a value is a power of two
 * @method FORGE.Math.isPowerOfTwo
 * @param  {number} value - value to check
 */
FORGE.Math.isPowerOfTwo = function(value)
{
    return ((value != 0) && !(value & (value - 1)));
};

/**
 * Get euler angles from rotation matrix
 *
 * @method FORGE.Math.rotationMatrixToEuler
 * @param  {THREE.Matrix4} mat - rotation matrix
 * @return {TEuler} object with keys {yaw, pitch, roll} and euler angles as values [radians]
 */
FORGE.Math.rotationMatrixToEuler = function(mat)
{
    return {
        yaw: Math.atan2(-mat.elements[2 + 0 * 4], mat.elements[2 + 2 * 4]),
        pitch: Math.asin(-mat.elements[2 + 1 * 4]),
        roll: Math.atan2(-mat.elements[0 + 1 * 4], mat.elements[1 + 1 * 4])
    };
};

/**
 * Get rotation matrix from euler angles
 *
 * @method FORGE.Math.eulerToRotationMatrix
 * @param  {number} yaw - yaw angle [rad]
 * @param  {number} pitch - pitch angle [rad]
 * @param  {number} roll - roll angle [rad]
 * @param  {boolean=} orderYPR
 * @return {THREE.Matrix4} rotation matrix
 */
FORGE.Math.eulerToRotationMatrix = function(yaw, pitch, roll, orderYPR)
{
    var cy = Math.cos(yaw);
    var sy = Math.sin(yaw);
    var cp = Math.cos(-pitch);
    var sp = Math.sin(-pitch);
    var cr = Math.cos(roll);
    var sr = Math.sin(roll);

    if (typeof orderYPR === "undefined")
    {
        orderYPR = false;
    }

    //jscs:disable
    if (orderYPR)
    {
        // M(yaw) * M(pitch) * M(roll)
        return new THREE.Matrix4().set(
             cy * cr + sy * sp * sr, -cy * sr + sy * sp * cr, sy * cp, 0,
                            cp * sr,                 cp * cr,     -sp, 0,
            -sy * cr + cy * sp * sr,  sy * sr + cy * sp * cr, cy * cp, 0,
                                  0,                       0,       0, 1
        );
    }

    // M(roll) * M(pitch) *  M(yaw)
    return new THREE.Matrix4().set(
        cr * cy - sr * sp * sy, -sr * cp, cr * sy + sr * sp * cy, 0,
        sr * cy + cr * sp * sy,  cr * cp, sr * sy - cr * sp * cy, 0,
                      -sy * cp,       sp,                cp * cy, 0,
                             0,        0,                      0, 1
     );
    //jscs:enable
};

/**
 * Converts spherical coordinates to cartesian, respecting the FORGE
 * coordinates system.
 *
 * @method FORGE.Math.sphericalToCartesian
 * @param {number} radius - radius
 * @param {number} theta - theta angle
 * @param {number} phi - phi angle
 * @param {string} [unit=radian] - The unit used for theta and phi arguments
 * @return {CartesianCoordinates} the resulting cartesian coordinates
 */
FORGE.Math.sphericalToCartesian = function(radius, theta, phi, unit)
{
    var res = {};

    if (unit === FORGE.Math.DEGREES)
    {
        theta = FORGE.Math.degToRad(theta);
        phi = FORGE.Math.degToRad(phi);
    }

    // wrap phi in [-π/2; π/2]
    phi = FORGE.Math.wrap(phi, -Math.PI / 2, Math.PI / 2);
    // invert theta if radius is negative
    theta += radius < 0 ? Math.PI : 0;
    // wrap theta in [0; 2π)
    theta = FORGE.Math.wrap(theta, 0, 2 * Math.PI);
    // abs so the radius is positive
    radius = Math.abs(radius);

    res.x = radius * Math.cos(phi) * Math.sin(theta);
    res.y = radius * Math.sin(phi);
    res.z = -radius * Math.cos(phi) * Math.cos(theta);

    return res;
};

/**
 * Converts cartesian coordinates to spherical, respecting the FORGE
 * coordinates system.
 *
 * @method FORGE.Math.cartesianToSpherical
 * @param {number} x - x
 * @param {number} y - y
 * @param {number} z - z
 * @param {string} [unit=radian] - The unit used to return spherical
 * @return {SphericalCoordinates}
 */
FORGE.Math.cartesianToSpherical = function(x, y, z, unit)
{
    var res = {};

    res.radius = Math.sqrt(x*x + y*y + z*z);

    if (res.radius === 0)
    {
        return { radius: 0, theta: 0, phi: 0 };
    }

    res.phi = Math.asin(y / res.radius);
    res.theta = Math.atan2(x, -z || 0); // we want to avoid -z = -0

    if(unit === FORGE.Math.DEGREES)
    {
        res.phi = FORGE.Math.radToDeg(res.phi);
        res.theta = FORGE.Math.radToDeg(res.theta);
    }

    return res;
};

/**
 * Quaternion helper.
 * @namespace {Object} FORGE.Quaternion
 */
FORGE.Quaternion = {};

/**
 * Get euler angles from a quaternion.
 *
 * @method FORGE.Quaternion.toEuler
 * @param {THREE.Quaternion} quat - rotation quaternion
 * @return {TEuler} euler angle object
 */
FORGE.Quaternion.toEuler = function(quat)
{
    var euler = new THREE.Euler().setFromQuaternion (quat, "XYZ");

    var result =
    {
        yaw: euler.x,
        pitch: -euler.y,
        roll: euler.z
    };

    return result;
};

/**
 * Get a quaternion from euler angles.
 *
 * @method FORGE.Quaternion.fromEuler
 * @param {TEuler|number} yaw - yaw euler angle (y axis) [radians]
 * @param {number=} pitch - pitch euler angle (x axis) [radians]
 * @param {number=} roll - roll euler angle (z axis) [radians]
 * @return {THREE.Quaternion} resulting rotation quaternion
 */
FORGE.Quaternion.fromEuler = function(yaw, pitch, roll)
{
    if(typeof yaw === "object")
    {
        pitch = yaw.pitch;
        roll = yaw.roll;
        yaw = yaw.yaw;
    }

    return new THREE.Quaternion().setFromEuler(new THREE.Euler(yaw, -pitch, roll, "ZXY"));
};

/**
 * Get a rotation matrix from a quaternion.
 *
 * @method FORGE.Quaternion.toRotationMatrix
 * @param {THREE.Quaternion} quat - quaternion
 * @return {THREE.Matrix4} rotation matrix
 */
FORGE.Quaternion.toRotationMatrix = function(quat)
{
    var euler = FORGE.Quaternion.toEuler(quat);
    return FORGE.Math.eulerToRotationMatrix(euler.yaw, euler.pitch, euler.roll);
};

/**
 * Get a quaternion from a rotation matrix.
 *
 * @method FORGE.Quaternion.fromRotationMatrix
 * @param {THREE.Matrix4} mat - rotation matrix
 * @return {THREE.Quaternion} quat quaternion
 */
FORGE.Quaternion.fromRotationMatrix = function(mat)
{
    var euler = FORGE.Math.rotationMatrixToEuler(mat);
    return FORGE.Quaternion.fromEuler(euler);
};

/**
 * Cancel roll and return new quaternion.
 *
 * @method FORGE.Quaternion.cancelRoll
 * @param  {THREE.Quaternion} quat - input quaternion
 * @return {THREE.Quaternion} quaternion without roll
 */
FORGE.Quaternion.cancelRoll = function(quat)
{
    var euler = FORGE.Quaternion.toEuler(quat);
    return FORGE.Quaternion.fromEuler(euler.yaw, euler.pitch, 0);
};

/**
 * Get difference quaternion between two rotation quaternions.
 *
 * @method FORGE.Quaternion.diffBetweenQuaternions
 * @param {THREE.Quaternion} q0 - start quaternion
 * @param {THREE.Quaternion} q1 - end quaternion
 * @return {THREE.Quaternion}
 */
FORGE.Quaternion.diffBetweenQuaternions = function(q0, q1)
{
    return new THREE.Quaternion().copy(q0).inverse().multiply(q1);
};

/**
 * Get rotation angle between two quaternions.
 *
 * @method FORGE.Quaternion.angularDistance
 * @param {THREE.Quaternion} q0 - interval start quaternion
 * @param {THREE.Quaternion} q1 - interval end quaternion
 * @return {number} angle in radians
 */
FORGE.Quaternion.angularDistance = function(q0, q1)
{
    var d = Math.abs(q0.dot(q1));

    if (d >= 1.0)
    {
        return 0;
    }

    return 2 * Math.acos(d);
};

/**
 * Multiply a quaternion with a scalar.
 *
 * @method FORGE.Quaternion.multiplyScalar
 * @param {number} scalar scalar - multiplyScalar
 * @param {THREE.Quaternion} quat - quaternion
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.multiplyScalar = function (scalar, quat)
{
    return new THREE.Quaternion(scalar * quat.x, scalar * quat.y, scalar * quat.z, scalar * quat.w);
};

/**
 * Add quaternions.
 *
 * @method FORGE.Quaternion.add
 * @param {THREE.Quaternion} q0 - first quaternion
 * @param {THREE.Quaternion} q1 - second quaternion
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.add = function (q0, q1)
{
    return new THREE.Quaternion(q0.x + q1.x, q0.y + q1.y, q0.z + q1.z, q0.w + q1.w);
};

/**
 * Quaternion log function.
 *
 * @method FORGE.Quaternion.log
 * @param {THREE.Quaternion} quat - quaternion
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.log = function(quat)
{
    var qres = new THREE.Quaternion(0, 0, 0, 0);

    if (Math.abs(quat.w) < 1)
    {
        var angle = Math.acos(quat.w);
        var sin = Math.sin(angle);

        if (sin > 0)
        {
            qres.x = angle * quat.x / sin;
            qres.y = angle * quat.y / sin;
            qres.z = angle * quat.z / sin;
        }

        return qres;
    }

    qres.x = quat.x;
    qres.y = quat.y;
    qres.z = quat.z;

    return qres;
};

/**
 * Quaternion exp function.
 *
 * @method FORGE.Quaternion.exp
 * @param {THREE.Quaternion} quat - quaternion
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.exp = function (quat)
{
    var angle = Math.sqrt(quat.x * quat.x + quat.y * quat.y + quat.z * quat.z);
    var sin = Math.sin(angle);

    var qres = new THREE.Quaternion().copy(quat);
    qres.w = Math.cos(angle);

    if (Math.abs(sin) > Number.EPSILON)
    {
        qres.x = sin * quat.x / angle;
        qres.y = sin * quat.y / angle;
        qres.z = sin * quat.z / angle;
    }

    return qres;
};

/**
 * Quaternion slerp computation.
 *
 * @method FORGE.Quaternion.slerp
 * @param {THREE.Quaternion} q0 - interval start quaternion
 * @param {THREE.Quaternion} q1 - interval end quaternion
 * @param {number} t - interpolation time
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.slerp = function(q0, q1, t)
{
    var qres = new THREE.Quaternion();
    THREE.Quaternion.slerp(q0, q1, qres, t);
    return qres;
};

/**
 * Compute squad interpolation.
 *
 * @method FORGE.Quaternion.squad
 * @param {THREE.Quaternion} q0 - interval start quaternion
 * @param {THREE.Quaternion} a0 - left quandrangle
 * @param {THREE.Quaternion} a1 - right quandrangle
 * @param {THREE.Quaternion} q1 - interval end quaternion
 * @param {THREE.Quaternion} qres - result quaternion
 * @param {number} t - interpolation time
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.squad = function(q0, a0, a1, q1, qres, t)
{
    var slerp1 = FORGE.Quaternion.slerp(q0, q1, t);
    var slerp2 = FORGE.Quaternion.slerp(a0, a1, t);
    return FORGE.Quaternion.slerp(slerp1, slerp2, 2*t*(1-t));
};

/**
 * Compute NLERP interpolation without inversion
 *
 * @method FORGE.Quaternion.nlerpNoInvert
 * @param {THREE.Quaternion} q0 interval start quaternion
 * @param {THREE.Quaternion} q1 interval end quaternion
 * @param {number} t interpolation time
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.nlerpNoInvert = function(q0, q1, t)
{
    return FORGE.Quaternion.add(
        FORGE.Quaternion.multiplyScalar(1 - t, q0),
        FORGE.Quaternion.multiplyScalar(t, q1)
    ).normalize();
};

/**
 * Compute SLERP interpolation without inversion
 *
 * @method FORGE.Quaternion.slerpNoInvert
 * @param {THREE.Quaternion} q0 interval start quaternion
 * @param {THREE.Quaternion} q1 interval end quaternion
 * @param {number} t interpolation time
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.slerpNoInvert = function(q0, q1, t)
{
    var dot = q0.dot(q1);

    if (Math.abs(dot) >= 0.95)
    {
        return FORGE.Quaternion.nlerpNoInvert(q0, q1, t);
    }

    var angle = Math.acos(dot);
    return FORGE.Quaternion.multiplyScalar(1.0 / Math.sin(angle),
        FORGE.Quaternion.add(
            FORGE.Quaternion.multiplyScalar(Math.sin(angle * (1-t)), q0),
            FORGE.Quaternion.multiplyScalar(Math.sin(angle * t), q1)
        )
    );
};

/**
 * Compute SQUAD interpolation without inversion
 *
 * @method FORGE.Quaternion.squadNoInvert
 * @param {THREE.Quaternion} q0 interval start quaternion
 * @param {THREE.Quaternion} a0 left quandrangle
 * @param {THREE.Quaternion} a1 right quandrangle
 * @param {THREE.Quaternion} q1 interval end quaternion
 * @param {number} t interpolation time
 * @return {THREE.Quaternion} result quaternion
 */
FORGE.Quaternion.squadNoInvert = function(q0, a0, a1, q1, t)
{
    var qq = FORGE.Quaternion.slerpNoInvert(q0, q1, t);
    var qa = FORGE.Quaternion.slerpNoInvert(a0, a1, t);
    return FORGE.Quaternion.slerpNoInvert(qq, qa, 2 * t * (1-t));
};

/**
 * Make a spline from three quaternions.
 *
 * @method FORGE.Quaternion.spline
 * @param {THREE.Quaternion} q0 previous quaternion
 * @param {THREE.Quaternion} q1 current quaternion
 * @param {THREE.Quaternion} q2 next quaternion
 * @return {THREE.Quaternion}
 */
FORGE.Quaternion.spline = function (q0, q1, q2)
{
    var q1Inv = new THREE.Quaternion().copy(q1).conjugate();

    var p0 = new THREE.Quaternion().copy(q1Inv).multiply(q0);
    var p2 = new THREE.Quaternion().copy(q1Inv).multiply(q2);

    var qLog0 = FORGE.Quaternion.log(p0);
    var qLog2 = FORGE.Quaternion.log(p2);

    var qLogSum = FORGE.Quaternion.add(qLog0, qLog2);
    var qLogSumQuater = FORGE.Quaternion.multiplyScalar(-0.25, qLogSum);
    var qExp = FORGE.Quaternion.exp(qLogSumQuater);

    return new THREE.Quaternion().copy(q1).multiply(qExp).normalize();
};

/**
 * Rectangle object.
 *
 * @constructor FORGE.Rectangle
 * @param {number} x - horizontal coordinate of origin
 * @param {number} y - vertical coordinate of origin
 * @param {number} width - width of the rectangle
 * @param {number} height - height of the rectangle
 */
FORGE.Rectangle = function(x, y, width, height)
{
    /**
     * Horizontal coordinate of origin
     * @name FORGE.Rectangle#_x
     * @type {number}
     * @private
     */
    this._x = typeof x === "number" ? x : 0;

    /**
     * Vertical coordinate of origin
     * @name FORGE.Rectangle#_y
     * @type {number}
     * @private
     */
    this._y = typeof y === "number" ? y : 0;

    /**
     * Width
     * @name FORGE.Rectangle#_width
     * @type {number}
     * @private
     */
    this._width = typeof width === "number" ? width : 0;

    /**
     * Height
     * @name FORGE.Rectangle#_height
     * @type {number}
     * @private
     */
    this._height = typeof height === "number" ? height : 0;

};

FORGE.Rectangle.prototype.constructor = FORGE.Rectangle;


/**
 * Get and set x origin coordinate.
 * @name FORGE.Rectangle#x
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "x",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this._x;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        if(typeof value === "number")
        {
            this._x = value;
        }
    }
});

/**
 * Get and set y origin coordinate.
 * @name FORGE.Rectangle#y
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "y",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this._y;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        if(typeof value === "number")
        {
            this._y = value;
        }
    }
});

/**
 * Get and set width of the rectangle.
 * @name FORGE.Rectangle#width
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "width",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this._width;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        if(typeof value === "number")
        {
            this._width = value;
        }
    }
});

/**
 * Get and set width of the rectangle (w is a short for width).
 * @name FORGE.Rectangle#w
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "w",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this.width;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        this.width = value;
    }
});

/**
 * Get and set height of the rectangle.
 * @name FORGE.Rectangle#height
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "height",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this._height;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        if(typeof value === "number")
        {
            this._height = value;
        }
    }
});

/**
 * Get and set height of the rectangle (h is a short for height).
 * @name FORGE.Rectangle#h
 * @type {number}
 */
Object.defineProperty(FORGE.Rectangle.prototype, "h",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return this.height;
    },

    /** @this {FORGE.Rectangle} */
    set: function(value)
    {
        this.height = value;
    }
});

/**
 * Get center point.
 * @name FORGE.Rectangle#center
 * @type {THREE.Vector2}
 * @readonly
 */
Object.defineProperty(FORGE.Rectangle.prototype, "center",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return new THREE.Vector2(this._x + 0.5 * this._width, this._y + 0.5 * this._height);
    }
});

/**
 * Get origin.
 * @name FORGE.Rectangle#origin
 * @type {THREE.Vector2}
 * @readonly
 */
Object.defineProperty(FORGE.Rectangle.prototype, "origin",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return new THREE.Vector2(this._x, this._y);
    }
});

/**
 * Get size.
 * @name FORGE.Rectangle#size
 * @type {FORGE.Size}
 * @readonly
 */
Object.defineProperty(FORGE.Rectangle.prototype, "size",
{
    /** @this {FORGE.Rectangle} */
    get: function()
    {
        return new FORGE.Size(this._width, this._height);
    }
});



/**
 * Size object.
 *
 * @constructor FORGE.Size
 * @param {number} width width property
 * @param {number} height height property
 */
FORGE.Size = function(width, height)
{
    /**
     * Width.
     * @name FORGE.Size#_width
     * @type {number}
     * @private
     */
    this._width = width || 0;

    /**
     * Height.
     * @name FORGE.Size#_height
     * @type {number}
     * @private
     */
    this._height = height || 0;
};

FORGE.Size.prototype.constructor = FORGE.Size;


/**
 * Get width.
 * @name FORGE.Size#width
 * @type {number}
 */
Object.defineProperty(FORGE.Size.prototype, "width",
{
    /** @this {FORGE.Size} */
    get: function()
    {
        return this._width;
    }
});

/**
 * Get height.
 * @name FORGE.Size#height
 * @type {number}
 */
Object.defineProperty(FORGE.Size.prototype, "height",
{
    /** @this {FORGE.Size} */
    get: function()
    {
        return this._height;
    }
});

/**
 * Get ratio.
 * @name FORGE.Size#ratio
 * @type {number}
 */
Object.defineProperty(FORGE.Size.prototype, "ratio",
{
    /** @this {FORGE.Size} */
    get: function()
    {
        return this._width / this._height;
    }
});


/**
 * @namespace {Object} FORGE.MediaType
 */
FORGE.MediaType = {};

/**
 * @name FORGE.MediaType.UNDEFINED
 * @type {string}
 * @const
 */
FORGE.MediaType.UNDEFINED = "undefined";

/**
 * @name FORGE.MediaType.IMAGE
 * @type {string}
 * @const
 */
FORGE.MediaType.IMAGE = "image";

/**
 * @name FORGE.MediaType.VIDEO
 * @type {string}
 * @const
 */
FORGE.MediaType.VIDEO = "video";

/**
 * @name FORGE.MediaType.GRID
 * @type {string}
 * @const
 */
FORGE.MediaType.GRID = "grid";


/**
 * @namespace {Object} FORGE.MediaFormat
 */
FORGE.MediaFormat = {};

/**
 * @name FORGE.MediaFormat.EQUIRECTANGULAR
 * @type {string}
 * @const
 */
FORGE.MediaFormat.EQUIRECTANGULAR = "equi";

/**
 * @name FORGE.MediaFormat.CUBE
 * @type {string}
 * @const
 */
FORGE.MediaFormat.CUBE = "cube";

/**
 * @name FORGE.MediaFormat.FLAT
 * @type {string}
 * @const
 */
FORGE.MediaFormat.FLAT = "flat";

/**
 * @name FORGE.MediaFormat.HOTSPOT
 * @type {string}
 * @const
 */
FORGE.MediaFormat.HOTSPOT = "hotspot";

/**
 * This object stores a number of tiles used for multi resolution cases with
 * tiles. It acts as a LRU map, as we can't store infinite amount of tiles.
 * The number of tiles to store is Σ(6 * 4^n), with n being the number of levels.
 *
 * There is an exception though: the level 0 of a multi resolution is always
 * kept in the cache.
 *
 * @constructor FORGE.MediaStore
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference
 * @param {SceneMediaSourceConfig} config - the config given by a media to know
 *                                          how to load each tile
 * @param {(FORGE.Image|SceneMediaPreviewConfig)} preview - the pattern of the preview
 * @extends {FORGE.BaseObject}
 */
FORGE.MediaStore = function(viewer, config, preview)
{
    /**
     * The viewer reference.
     * @name FORGE.MediaStore#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Media source configuration
     * @name FORGE.MediaStore#_config
     * @type {SceneMediaSourceConfig}
     * @private
     */
    this._config = config;

    /**
     * Pattern of the preview
     * @name FORGE.MediaStore#_preview
     * @type {(FORGE.Image|SceneMediaSourceConfig)}
     * @private
     */
    this._preview = preview;

    /**
     * A map containing all {@link FORGE.MediaTexture}, with the key being constitued
     * from the level, face, x and y properties defining the texture
     * @name FORGE.MediaStore#_textures
     * @type {?FORGE.Map}
     * @private
     */
    this._textures = null;

    /**
     * The list of currently being loaded textures
     * @name FORGE.MediaStore#_loadingTextures
     * @type {?Array<string>}
     * @private
     */
    this._loadingTextures = null;

    /**
     * Map of texture promises
     * @type {FORGE.Map}
     * @private
     */
    this._texturePromises = null;

    /**
     * LIFO stack holding texture requests
     * @type {Array}
     * @private
     */
    this._textureStack = null;

    /**
     * LIFO stack timer interval
     * @type {?number}
     * @private
     */
    this._textureStackInterval = null;

    /**
     * The current size of all loaded textures.
     * @name FORGE.MediaStore#_size
     * @type {number}
     * @private
     */
    this._size = 0;

    /**
     * The max size of the cache.
     * @name FORGE.MediaStore#_maxSize
     * @type {number}
     * @private
     */
    this._maxSize = 0;

    /**
     * The global pattern of texture file urls
     * @name FORGE.MediaStore#_pattern
     * @type {string}
     * @private
     */
    this._pattern = "";

    /**
     * Object containing patterns of texture file urls per pyramid level
     * @name FORGE.MediaStore#_patterns
     * @type {?Object}
     * @private
     */
    this._patterns = null;

    /**
     * Faces configuration
     * @type {?Object}
     * @private
     */
    this._cubeFaceConfig = null;

    FORGE.BaseObject.call(this, "MediaStore");

    this._boot();
};

FORGE.MediaStore.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.MediaStore.prototype.constructor = FORGE.MediaStore;

/**
 * Texture stack interval in milliseconds
 * @type {number}
 */
FORGE.MediaStore.TEXTURE_STACK_INTERVAL_MS = 250;

/**
 * Table describing previous cube face
 * @type {CubeFaceObject}
 */
FORGE.MediaStore.CUBE_FACE_CONFIG = {
    "front": "front",
    "right": "right",
    "back": "back",
    "left": "left",
    "down": "down",
    "up": "up"
};

/**
 * Boot routine.
 *
 * @method FORGE.MediaStore#_boot
 * @private
 */
FORGE.MediaStore.prototype._boot = function()
{
    this._register();

    this._textures = new FORGE.Map();
    this._loadingTextures = [];
    this._textureStack = [];
    this._texturePromises = new FORGE.Map();
    this._patterns = {};

    this._parseConfig(this._config);

    if (FORGE.Device.desktop === true)
    {
        this._maxSize = 150000000;
    }
    else if (FORGE.Device.iOS === true)
    {
        this._maxSize = 40000000;
    }
    else
    {
        this._maxSize = 50000000;
    }
};

/**
 * Parse config routine.
 *
 * @method FORGE.MediaStore#_parseConfig
 * @param {SceneMediaSourceConfig} config - the config of the media source
 * @private
 */
FORGE.MediaStore.prototype._parseConfig = function(config)
{
    // a pattern should contains at least {f}, {l}, {x} or {y}
    var re = /\{[lfxy].*\}/;

    // Check if there is a global pattern
    if (typeof config.pattern === "string")
    {
        if (config.pattern.match(re) === null)
        {
            throw "the pattern of the multi resolution media is wrong";
        }

        this._pattern = config.pattern;
    }

    if (this._preview !== null)
    {
        if (typeof this._preview.url !== "string" || this._preview.url.match(re) === null)
        {
            this.warn("no preview for this panorama");
        }
        else
        {
            this._patterns[FORGE.Tile.PREVIEW] = this._preview.url;
        }
    }

    // Then check if each resolution level has its own pattern
    for (var l = 0, ll = config.levels.length; l < ll; l++)
    {
        if (typeof config.levels[l].pattern === "string")
        {
            if (config.pattern.match(re) === null)
            {
                throw "the pattern of the multi resolution media is wrong";
            }

            this._patterns[l] = config.levels[l].pattern;
        }
    }

    this._cubeFaceConfig = FORGE.MediaStore.CUBE_FACE_CONFIG;
    if (typeof config.faces !== "undefined")
    {
        this._cubeFaceConfig = config.faces;
    }
};

/**
 * Create the key of an image.
 *
 * @method FORGE.MediaStore#_createKey
 * @param {FORGE.Tile} tile - tile
 * @return {string} returns the key for this image
 * @private
 */
FORGE.MediaStore.prototype._createKey = function(tile)
{
    var key = "";
    key += typeof tile.face !== "undefined" ? this._cubeFaceConfig[tile.face] + "-" : "";
    key += typeof tile.level !== "undefined" ? tile.level + "-" : "";
    key += typeof tile.x !== "undefined" ? tile.x + "-" : "";
    key += typeof tile.y !== "undefined" ? tile.y : "";

    return key;
};

/**
 * Loads an Image from parameters, but doesn't add it to the map yet.
 *
 * @method FORGE.MediaStore#_load
 * @param {FORGE.Tile} tile - tile
 * @private
 */
FORGE.MediaStore.prototype._load = function(tile)
{
    var key = this._createKey(tile);
    if (this._loadingTextures.indexOf(key) !== -1)
    {
        return;
    }

    var entry = this._texturePromises.get(key);
    if (entry.cancelled)
    {
        this.log("Load promise cancelled for tile " + tile.name);
        entry.load.reject("Tile cancelled");
        this._texturePromises.delete(key);
        return;
    }

    this.log("Push loading texture for tile " + tile.name);
    this._loadingTextures.push(key);

    var url = this._pattern;

    if (typeof this._patterns[tile.level] !== "undefined")
    {
        url = this._patterns[tile.level];
    }

    url = url.replace(/\{face\}/, this._cubeFaceConfig[tile.face]);
    url = url.replace(/\{level\}/, tile.level.toString());
    url = url.replace(/\{x\}/, tile.x.toString());
    url = url.replace(/\{y\}/, tile.y.toString());

    var config = {
        url: url
    };

    var image = new FORGE.Image(this._viewer, config);

    image.data = {
        tile: tile
    };

    image.onLoadComplete.add(this._onLoadComplete.bind(this), this);
};

/**
 * Add the loaded image to the map.
 *
 * @method FORGE.MediaStore#_onLoadComplete
 * @param {FORGE.Image} image - the loaded image
 * @private
 */
FORGE.MediaStore.prototype._onLoadComplete = function(image)
{
    if (this._textures === null)
    {
        image.emitter.destroy();
        // stop it all, it means this mediastore has been destroyed and this is
        // a late-coming tile
        return;
    }

    image = image.emitter;
    var tile = image.data.tile;
    var key = this._createKey(tile);

    this.log("Texture load complete for tile " + tile.name);

    var texture = new THREE.Texture();
    texture.image = image.element;

    var size = image.element.height * image.element.width;
    this._size += size;

    var mediaTexture = new FORGE.MediaTexture(texture, (tile.level === FORGE.Tile.PREVIEW), size);
    this._textures.set(key, mediaTexture);

    // destroy the image, it is no longer needed
    this._loadingTextures.splice(this._loadingTextures.indexOf(image.data.key), 1);

    var entry = this._texturePromises.get(key);
    entry.load.resolve(mediaTexture.texture);
    this._texturePromises.delete(key);

    image.destroy();

    this._checkSize();
};

/**
 * Discard texture.
 *
 * @method FORGE.MediaStore#_discardTexture
 * @param {string} key - texture key
 * @private
 */
FORGE.MediaStore.prototype._discardTexture = function(key)
{
    if (this._textures.has(key) === false)
    {
        return;
    }

    var texture = this._textures.get(key);

    this._size -= texture.size;
    texture.destroy();

    this._textures.delete(key);
};

/**
 * Check the current size of the store, and flush some texture if necessary.
 *
 * @method FORGE.MediaStore#_checkSize
 * @private
 */
FORGE.MediaStore.prototype._checkSize = function()
{
    if (this._size < this._maxSize)
    {
        return;
    }

    var entries = this._textures.entries(),
        time = window.performance.now(),
        force = false,
        texture;

    entries = FORGE.Utils.sortArrayByProperty(entries, "1.lastTime");

    while (this._size > this._maxSize)
    {
        // oldest are first
        texture = entries.shift();

        // if no more entries (aka all texture are level 0) remove it anyway
        if (typeof texture === "undefined")
        {
            entries = this._textures.entries();
            entries = FORGE.Utils.sortArrayByProperty(entries, "1.lastTime");
            force = true;
        }
        else
        // but don't delete if it is locked
        if (force === true || texture[1].locked !== true)
        {
            this._discardTexture(texture[0]);
        }
    }
};

/**
 * Push item from texture stack.
 *
 * @method FORGE.MediaStore#_textureStackPush
 * @param {FORGE.Tile} tile - tile requesting texture
 */
FORGE.MediaStore.prototype._textureStackPush = function(tile)
{
    // First clear interval if any and push
    if (this._textureStackInterval !== null)
    {
        window.clearTimeout(this._textureStackInterval);
        this._textureStackInterval = null;
    }

    if (tile.level === FORGE.Tile.PREVIEW)
    {
        this._load(tile);
    }
    else
    {
        this._textureStack.push(tile);

        this._textureStackInterval = window.setTimeout(this._textureStackPop.bind(this), FORGE.MediaStore.TEXTURE_STACK_INTERVAL_MS);
    }
};

/**
 * Pop item from texture stack.
 *
 * @method FORGE.MediaStore#_textureStackPop
 */
FORGE.MediaStore.prototype._textureStackPop = function()
{
    this._textureStackInterval = null;

    while (this._textureStack.length > 0)
    {
        var tile = this._textureStack.pop();
        //this.log("Texture stack length (---): " + this._textureStack.length + " (" + tile.name + ")");

        this.log("Pop texture request from stack for tile " + tile.name);
        this._load(tile);
    }
};

/**
 * Get an image from this store, given four parameters: the face associated to
 * this image, the level of quality and the x and y positions. It returns
 * either a {@link THREE.Texture} or null.
 *
 * The inner working is as follow: either the image is already loaded and
 * returned, or either the image is being loaded and nothing is returned yet.
 * If the latter, the image is added to the map once it is completely loaded
 * (the onLoadComplete event).
 *
 * @method FORGE.MediaStore#get
 * @param {FORGE.Tile} tile - tile
 * @return {Promise} returns a promise on the image
 */
FORGE.MediaStore.prototype.get = function(tile)
{
    var key = this._createKey(tile);

    // If texture is available, return a resolved promise
    if (this._textures.has(key))
    {
        this.log("Texture available, return resolved promise");
        var promise = FORGE.Utils.makePromise();
        var mediaTexture = this._textures.get(key);
        promise.resolve(mediaTexture.texture);
        return promise;
    }

    // First check if texture is already loading (pending promise)
    // Return null, and client should do nothing but wait
    var promise = this._texturePromises.get(key);
    if (promise !== undefined)
    {
        return null;
    }

    this.log("Create load promise for tile " + tile.name);

    var loadingPromise = FORGE.Utils.makePromise();

    // Texture already available
    // Return resolved promise
    if (this._textures.has(key))
    {
        this.log("and resolve it immediately");
        loadingPromise.resolve(this._textures.get(key));
        return loadingPromise;
    }

    // Create new entry in map of promises
    var entry = /** @type {!TexturePromiseObject} */ (
    {
        load: loadingPromise,
        cancelled: false
    });

    this._texturePromises.set(key, entry);

    this._textureStackPush(tile);

    return loadingPromise;
};

/**
 * Ask store if it has a texture already available
 * @method FORGE.MediaStore#has
 * @param {string} key - texture key
 */
FORGE.MediaStore.prototype.has = function(key)
{
    return this._textures.has(key);
};

/**
 * Discard texture for a given tile
 * @method FORGE.MediaStore#discardTileTexture
 * @param {FORGE.Tile} tile - tile
 */
FORGE.MediaStore.prototype.discardTileTexture = function(tile)
{
    this._discardTexture(this._createKey(tile));
};

/**
 * Destroy routine.
 *
 * @method FORGE.MediaStore#destroy
 */
FORGE.MediaStore.prototype.destroy = function()
{
    this._unregister();

    this._patterns = null;

    this._viewer = null;
    this._loadingTextures = null;

    if (this._textureStack !== null)
    {
        this._textureStack.length = 0;
        this._textureStack = null;
    }

    this._texturePromises = null;

    this._textures.clear();
    this._textures = null;
};

/**
 * This object stores a THREE.Texture used for multi resolutions scene. It is
 * simplier (in terms of memory to store this object as it is tinier than a
 * FORGE.Image. It also remove the fact that we need to create a THREE.Texture
 * in the renderer.
 *
 * @constructor FORGE.MediaTexture
 * @param {THREE.Texture} texture - the THREE.Texture to store
 * @param {boolean} locked - is the texture locked (i.e. it isn't deletable)
 * @param {number} size - the size of the texture
 */
FORGE.MediaTexture = function(texture, locked, size)
{
    /**
     * The texture
     * @name FORGE.MediaTexture#_texture
     * @type {THREE.Texture}
     * @private
     */
    this._texture = texture;

    /**
     * Can the texture be deleted ? Otherwise it is locked, e.g. it is a level 0
     * @name FORGE.MediaTexture#_locked
     * @type {boolean}
     * @private
     */
    this._locked = locked;

    /**
     * The size of the texture
     * @name FORGE.MediaTexture#_size
     * @type {number}
     * @private
     */
    this._size = size;

    /**
     * The time the texture was last used
     * @name FORGE.MediaTexture#_lastTime
     * @type {number}
     * @private
     */
    this._lastTime = window.performance.now();

    /**
     * The number of times the texture was used
     * @name FORGE.MediaTexture#_count
     * @type {number}
     * @private
     */
    this._count = 0;
};

FORGE.MediaTexture.prototype.constructor = FORGE.MediaTexture;

/**
 * Destroy routine
 * @method FORGE.MediaTexture#destroy
 */
FORGE.MediaTexture.prototype.destroy = function()
{
    if (this._texture !== null)
    {
        this._texture.dispose();
    }

    this._texture = null;
};

/**
 * Get the texture.
 * @name  FORGE.MediaTexture#texture
 * @type {THREE.Texture}
 * @readonly
 */
Object.defineProperty(FORGE.MediaTexture.prototype, "texture",
{
    /** @this {FORGE.MediaTexture} */
    get: function()
    {
        this._count++;
        this._lastTime = window.performance.now();

        return this._texture;
    }
});

/**
 * Is the texture locked ?
 * @name  FORGE.MediaTexture#locked
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.MediaTexture.prototype, "locked",
{
    /** @this {FORGE.MediaTexture} */
    get: function()
    {
        return this._locked;
    }
});

/**
 * Get the size of the texture
 * @name  FORGE.MediaTexture#size
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.MediaTexture.prototype, "size",
{
    /** @this {FORGE.MediaTexture} */
    get: function()
    {
        return this._size;
    }
});

/**
 * Get the last time when it was called.
 * @name  FORGE.MediaTexture#lastTime
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.MediaTexture.prototype, "lastTime",
{
    /** @this {FORGE.MediaTexture} */
    get: function()
    {
        return this._lastTime;
    }
});

/**
 * Get the number of times it was called.
 * @name  FORGE.MediaTexture#count
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.MediaTexture.prototype, "count",
{
    /** @this {FORGE.MediaTexture} */
    get: function()
    {
        return this._count;
    }
});

/**
 * Media class.
 *
 * @constructor FORGE.Media
 * @param {FORGE.Viewer} viewer {@link FORGE.Viewer} reference.
 * @param {SceneMediaConfig} config input media configuration from json
 * @extends {FORGE.BaseObject}
 *
 */
FORGE.Media = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.Media#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Input scene and media config
     * @name FORGE.Media#_config
     * @type {SceneMediaConfig}
     * @private
     */
    this._config = config;

    /**
     * Type of the media
     * @name FORGE.Media#_type
     * @type {string}
     * @private
     */
    this._type = "";

    /**
     * Media options
     * @name  FORGE.Media#_options
     * @type {Object}
     * @private
     */
    this._options = null;

    /**
     * Image reference.
     * @name FORGE.Media#_displayObject
     * @type {FORGE.DisplayObject}
     * @private
     */
    this._displayObject = null;

    /**
     * Media store reference, if it is a multi resolution image.
     * @name FORGE.Media#_store
     * @type {FORGE.MediaStore}
     * @private
     */
    this._store = null;

    /**
     * A preview of the media: it is always an image, never a video (so, a
     * preview for a video would be an image).
     * @name FORGE.Media#_preview
     * @type {(FORGE.Image|SceneMediaPreviewConfig)}
     * @private
     */
    this._preview = null;

    /**
     * Loaded flag
     * @name FORGE.Media#_loaded
     * @type {boolean}
     * @private
     */
    this._loaded = false;

    /**
     * Events object that will keep references of the ActionEventDispatcher
     * @name FORGE.Media#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = null;

    /**
     * On load complete event dispatcher.
     * @name  FORGE.Media#_onLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadComplete = null;

    FORGE.BaseObject.call(this, "Media");

    this._boot();
};

FORGE.Media.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Media.prototype.constructor = FORGE.Media;

/**
 * Init routine
 * @method FORGE.Media#_boot
 * @private
 */
FORGE.Media.prototype._boot = function()
{
    this._events = {};

    // This event can no be a lazzy one (memorize is true)
    this._onLoadComplete = new FORGE.EventDispatcher(this, true);

    this._parseConfig(this._config);
};

/**
 * Configuration parsing.
 * @method FORGE.Media#_parseConfig
 * @param {SceneMediaConfig} config input media configuration
 * @private
 */
FORGE.Media.prototype._parseConfig = function(config)
{
    if (typeof config === "undefined" || config === null)
    {
        this._type = FORGE.MediaType.UNDEFINED;
        this._notifyLoadComplete();

        return;
    }

    // Warning : UID is not registered and applied to the FORGE.Image|FORGE.VideoHTML5|FORGE.VideoDash objects for registration
    this._uid = config.uid;

    this._options = (typeof config.options !== "undefined") ? config.options : null;

    this._type = config.type;

    var source = config.source;

    if (typeof config.source !== "undefined" && typeof config.source.format === "undefined")
    {
        config.source.format = FORGE.MediaFormat.FLAT;
    }

    if (typeof config.events === "object" && config.events !== null)
    {
        this._createEvents(config.events);
    }

    if (this._type === FORGE.MediaType.GRID)
    {
        this._notifyLoadComplete();
        return;
    }

    if (typeof config.source === "undefined" || config.source === null)
    {
        return;
    }

    var preview = config.preview;

    if (this._type === FORGE.MediaType.IMAGE)
    {
        // Load the preview
        if (typeof preview !== "undefined")
        {
            if (typeof preview === "string")
            {
                preview = { url: preview };
            }

            var re = /\{[lfxy].*\}/;
            if (preview.url.match(re) !== null)
            {
                this._preview = /** @type {SceneMediaPreviewConfig} */ (preview);
            }
            else if (source.format === FORGE.MediaFormat.EQUIRECTANGULAR ||
                source.format === FORGE.MediaFormat.CUBE ||
                source.format === FORGE.MediaFormat.FLAT)
            {
                var previewConfig = {
                    key: this._uid + "-preview",
                    url: preview.url
                };

                this._preview = new FORGE.Image(this._viewer, previewConfig);
                this._preview.onLoadComplete.addOnce(this._onImageLoadComplete, this);
            }
        }

        var imageConfig;

        // If there isn't an URL set, it means that this is a multi resolution image.
        if (!source.url)
        {
            this._store = new FORGE.MediaStore(this._viewer, source, this._preview);
            this._notifyLoadComplete();
        }
        else if (source.format === FORGE.MediaFormat.EQUIRECTANGULAR ||
            source.format === FORGE.MediaFormat.CUBE ||
            source.format === FORGE.MediaFormat.FLAT)
        {
            imageConfig = {
                key: this._uid,
                url: source.url
            };

            this._displayObject = new FORGE.Image(this._viewer, imageConfig);
            this._displayObject.onLoadComplete.addOnce(this._onImageLoadComplete, this);
        }
        else
        {
            throw "Media format not supported";
        }

        return;
    }

    if (this._type === FORGE.MediaType.VIDEO)
    {
        // If the levels property is present, we get all urls from it and put it
        // inside source.url: it means that there is multi-quality. It is way
        // easier to handle for video than for image, as it can never be video
        // tiles to display.
        if (Array.isArray(source.levels))
        {
            source.url = [];
            for (var i = 0, ii = source.levels.length; i < ii; i++)
            {
                if(FORGE.Device.check(source.levels[i].device) === false)
                {
                    continue;
                }

                source.url.push(source.levels[i].url);
            }
        }

        if (typeof source.url !== "string" && source.url.length === 0)
        {
            return;
        }

        if (typeof source.streaming !== "undefined" && source.streaming.toLowerCase() === FORGE.VideoFormat.DASH)
        {
            this._displayObject = new FORGE.VideoDash(this._viewer, this._uid);
        }
        else
        {
            var scene = this._viewer.story.scene;

            // check of the ambisonic state of the video sound prior to the video instanciation
            this._displayObject = new FORGE.VideoHTML5(this._viewer, this._uid, null, null, (scene.hasSoundTarget(this._uid) === true && scene.isAmbisonic() === true ? true : false));
        }

        // At this point, source.url is either a streaming address, a simple
        // url, or an array of url
        this._displayObject.load(source.url);

        this._displayObject.onLoadedMetaData.addOnce(this._onLoadedMetaDataHandler, this);
        this._displayObject.onPlay.add(this._onPlayHandler, this);
        this._displayObject.onPause.add(this._onPauseHandler, this);
        this._displayObject.onSeeked.add(this._onSeekedHandler, this);
        this._displayObject.onEnded.add(this._onEndedHandler, this);


        return;
    }
};

/**
 * Create action events dispatchers.
 * @method FORGE.Media#_createEvents
 * @private
 * @param {SceneMediaEventsConfig} events - The events config of the media.
 */
FORGE.Media.prototype._createEvents = function(events)
{
    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all object events.
 * @method Media.Object3D#_clearEvents
 * @private
 */
FORGE.Media.prototype._clearEvents = function()
{
    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Internal handler on image ready.
 * @method FORGE.Media#_onImageLoadComplete
 * @private
 */
FORGE.Media.prototype._onImageLoadComplete = function()
{
    this._notifyLoadComplete();
};

/**
 * Internal handler on video metadata loaded.
 * @method FORGE.Media#_onLoadedMetaDataHandler
 * @private
 */
FORGE.Media.prototype._onLoadedMetaDataHandler = function()
{
    if (this._options !== null)
    {
        this._displayObject.volume = (typeof this._options.volume === "number") ? this._options.volume : 1;
        this._displayObject.loop = (typeof this._options.loop === "boolean") ? this._options.loop : true;
        this._displayObject.currentTime = (typeof this._options.startTime === "number") ? this._options.startTime : 0;

        if (this._options.autoPlay === true && document[FORGE.Device.visibilityState] === "visible")
        {
            this._displayObject.play();
        }

        this._displayObject.autoPause = this._options.autoPause;
        this._displayObject.autoResume = this._options.autoResume;
    }

    this._notifyLoadComplete();
};

/**
 * Method to dispatch the load complete event and set the media ready.
 * @method FORGE.Media#_onLoadedMetaDataHandler
 */
FORGE.Media.prototype._notifyLoadComplete = function()
{
    if (this._type === FORGE.MediaType.IMAGE)
    {
        if (this._store !== null)
        {
            this._loaded = true;
            this._onLoadComplete.dispatch();
        }
        else
        {
            this._loaded = this._displayObject !== null && this._displayObject.loaded && this._preview !== null && this._preview.loaded;

            if (this._preview === null || (this._displayObject !== null && this._displayObject.loaded === false) || this._preview.loaded === false)
            {
                this._onLoadComplete.dispatch();
            }
            else if (this._viewer.renderer.backgroundRenderer !== null)
            {
                this._viewer.renderer.backgroundRenderer.displayObject = this._displayObject;
            }

        }
    }
    else
    {
        this._loaded = true;
        this._onLoadComplete.dispatch();
    }
};

/**
 * Internal handler on video play.
 * @method FORGE.Media#_onPlayHandler
 * @private
 */
FORGE.Media.prototype._onPlayHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onPlay, "ActionEventDispatcher") === true)
    {
        this._events.onPlay.dispatch();
    }
};

/**
 * Internal handler on video pause.
 * @method FORGE.Media#_onPauseHandler
 * @private
 */
FORGE.Media.prototype._onPauseHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onPause, "ActionEventDispatcher") === true)
    {
        this._events.onPause.dispatch();
    }
};

/**
 * Internal handler on video seeked.
 * @method FORGE.Media#_onSeekedHandler
 * @private
 */
FORGE.Media.prototype._onSeekedHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onSeeked, "ActionEventDispatcher") === true)
    {
        this._events.onSeeked.dispatch();
    }
};

/**
 * Internal handler on video ended.
 * @method FORGE.Media#_onEndedHandler
 * @private
 */
FORGE.Media.prototype._onEndedHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onEnded, "ActionEventDispatcher") === true)
    {
        this._events.onEnded.dispatch();
    }
};

/**
 * Media destroy sequence
 *
 * @method FORGE.Media#destroy
 */
FORGE.Media.prototype.destroy = function()
{
    if (this._displayObject !== null)
    {
        this._displayObject.destroy();
        this._displayObject = null;
    }

    if (this._store !== null)
    {
        this._store.destroy();
        this._store = null;
    }

    if (this._onLoadComplete !== null)
    {
        this._onLoadComplete.destroy();
        this._onLoadComplete = null;
    }

    this._clearEvents();
    this._events = null;

    this._viewer = null;
};

/**
 * Get the media config.
 * @name  FORGE.Media#config
 * @type {SceneMediaConfig}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "config",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        return this._config;
    }
});

/**
 * Get the media type.
 * @name  FORGE.Media#type
 * @type {string}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "type",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Get the displayObject.
 * @name  FORGE.Media#displayObject
 * @type {FORGE.DisplayObject}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "displayObject",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        if (this._type === FORGE.MediaType.IMAGE && this._store === null)
        {
            if (this._displayObject !== null && this._displayObject.loaded === true)
            {
                return this._displayObject;
            }

            return this._preview;
        }

        return this._displayObject;
    }
});

/**
 * Get the media store, if this is a multi resolution media.
 * @name FORGE.Media#store
 * @type {FORGE.MediaStore}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "store",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        return this._store;
    }
});

/**
 * Get the loaded flag
 * @name FORGE.Media#loaded
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "loaded",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        return this._loaded;
    }
});

/**
 * Get the onLoadComplete {@link FORGE.EventDispatcher}.
 * @name FORGE.Media#onLoadComplete
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.Media.prototype, "onLoadComplete",
{
    /** @this {FORGE.Media} */
    get: function()
    {
        return this._onLoadComplete;
    }
});

/**
 * RenderManager class.
 *
 * @constructor FORGE.RenderManager
 * @param {FORGE.Viewer} viewer - viewer reference
 * @extends {FORGE.BaseObject}
 *
 * @todo think about how to render multiple scene at the same time, with blending / overlap / viewport layouting...
 * maybe add a layer object encapsulating background / foreground renderings to ease the process
 */
FORGE.RenderManager = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.RenderManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * WebGL Renderer
     * @name FORGE.RenderManager#_webGLRenderer
     * @type {?THREE.WebGLRenderer}
     * @private
     */
    this._webGLRenderer = null;

    /**
     * Render pipeline managing composers
     * @name FORGE.RenderManager#_renderPipeline
     * @type {FORGE.RenderPipeline}
     * @private
     */
    this._renderPipeline = null;

    /**
     * Object managing screen/VR display
     * @name FORGE.RenderManager#_renderDisplay
     * @type {FORGE.RenderDisplay}
     * @private
     */
    this._renderDisplay = null;

    /**
     * View manager reference
     * @name FORGE.RenderManager#_viewManager
     * @type {FORGE.ViewManager}
     * @private
     */
    this._viewManager = null;

    /**
     * The sound reference linked to the media.
     * @name FORGE.RenderManager#_mediaSound
     * @type {?(FORGE.Sound|Object)}
     * @private
     */
    this._mediaSound = null;

    /**
     * Picking manager
     * @name FORGE.RenderManager#_pickingManager
     * @type {FORGE.PickingManager}
     * @private
     */
    this._pickingManager = null;

    /**
     * Background renderer.
     * @name FORGE.RenderManager#_backgroundRenderer
     * @type {?(FORGE.BackgroundMeshRenderer|FORGE.BackgroundShaderRenderer|FORGE.BackgroundPyramidRenderer)}
     * @private
     */
    this._backgroundRenderer = null;

    /**
     * Type of the background renderer.
     * @name FORGE.RenderManager#_backgroundRendererType
     * @type {string}
     * @private
     */
    this._backgroundRendererType = FORGE.BackgroundType.UNDEFINED;

    /**
     * Camera reference.
     * @name FORGE.RenderManager#_camera
     * @type {?FORGE.Camera}
     * @private
     */
    this._camera = null;

    /**
     * Canvas resolution (px)
     * @name FORGE.RenderManager#_canvasResolution
     * @type {?FORGE.Size}
     * @private
     */
    this._canvasResolution = null;

    /**
     * Display resolution (px)
     * @name FORGE.RenderManager#_displayResolution
     * @type {?FORGE.Size}
     * @private
     */
    this._displayResolution = null;

    /**
     * objects renderer
     * @name  FORGE.RenderManager#_objectRenderer
     * @type {?FORGE.ObjectRenderer}
     * @private
     */
    this._objectRenderer = null;

    /**
     * Background renderer ready flag
     * @name FORGE.RenderManager#_backgroundReady
     * @type {boolean}
     * @private
     */
    this._backgroundReady = false;

    /**
     * Render pipeline renderer ready flag
     * @name FORGE.RenderManager#_renderPipelineReady
     * @type {boolean}
     * @private
     */
    this._renderPipelineReady = false;

    /**
     * Event dispatcher for background renderer ready.
     * @name FORGE.RenderManager#_onBackgroundReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onBackgroundReady = null;

    /**
     * Event dispatcher for on before render.
     * @name FORGE.RenderManager#_onBeforeRender
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onBeforeRender = null;

    /**
     * Event dispatcher for on after render.
     * @name FORGE.RenderManager#_onAfterRender
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onAfterRender = null;

    FORGE.BaseObject.call(this, "RenderManager");

    this._boot();
};

FORGE.RenderManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.RenderManager.prototype.constructor = FORGE.RenderManager;

/**
 * Render manager constant near depth
 * @type {number}
 */
FORGE.RenderManager.DEPTH_NEAR = 0.01;

/**
 * Render manager constant far depth
 * @type {number}
 */
FORGE.RenderManager.DEPTH_FAR = 10000;

/**
 * Boot sequence.
 * @method FORGE.RenderManager#_boot
 * @private
 */
FORGE.RenderManager.prototype._boot = function()
{
    this._clock = new THREE.Clock();

    this._viewer.onConfigLoadComplete.add(this._onViewerConfigLoadComplete, this, 1000);
};

/**
 * Viewer ready handler
 * @method FORGE.RenderManager#_onViewerConfigLoadComplete
 * @private
 */
FORGE.RenderManager.prototype._onViewerConfigLoadComplete = function()
{
    var canvas = this._viewer.canvas.dom;

    var options = this._viewer.config.webgl;
    options.canvas = canvas;

    // WebGLRenderer will draw any component supported by WebGL
    try
    {
        this._webGLRenderer = new THREE.WebGLRenderer(options);
    }
    catch (error)
    {
        this.destroy();
        return;
    }

    this._webGLRenderer.autoClear = false;
    this._webGLRenderer.autoClearDepth = true;
    this._webGLRenderer.autoClearColor = true;

    this._viewManager = new FORGE.ViewManager(this._viewer);
    this._viewManager.onChange.add(this._onViewChangeHandler, this);

    this._pickingManager = new FORGE.PickingManager(this._viewer);
    this._renderDisplay = new FORGE.RenderDisplay(this._viewer);
    this._objectRenderer = new FORGE.ObjectRenderer(this._viewer);
    this._renderPipeline = new FORGE.RenderPipeline(this._viewer);

    this._renderDisplay.onDisplayChange.add(this._renderDisplayChangeHandler, this);
    this._setRendererSize(this._renderDisplay.rendererSize);

    this._camera = new FORGE.Camera(this._viewer);

    this._viewer.canvas.onResize.add(this._canvasResizeHandler, this);
    this._viewer.story.onSceneLoadStart.add(this._onSceneLoadStartHandler, this);
};

/**
 * Scene has started to load.
 * Init the view, the camera and the media
 * @todo create media / background renderer
 *
 * @method FORGE.RenderManager#_onSceneLoadStartHandler
 * @private
 */
FORGE.RenderManager.prototype._onSceneLoadStartHandler = function()
{
    // Listen to scene unload
    this._viewer.story.scene.onUnloadStart.addOnce(this._onSceneUnloadStartHandler, this);

    var sceneConfig = this._viewer.story.scene.config;

    // Apply background to renderer
    this._viewer.container.background = this._viewer.story.scene.background;

    // Create render scenes before initing the view to ensure pipeline is ready when
    // enabling the picking manager
    this._objectRenderer.createRenderScenes();
    this._renderPipeline.addRenderScenes(this._objectRenderer.renderScenes);

    this._viewManager.load(sceneConfig.view);

    this._initCamera(sceneConfig);

    this._initSound(sceneConfig);

    this._setupMedia();
};

/**
 * Bind event handlers on media.
 * @method FORGE.RenderManager#_setupMedia
 * @private
 */
FORGE.RenderManager.prototype._setupMedia = function()
{
    var media = this._viewer.story.scene.media;

    media.onLoadComplete.addOnce(this._mediaLoadCompleteHandler, this);

    // If media is a video, listen to the quality change event
    if (FORGE.Utils.isTypeOf(media.displayObject, ["VideoHTML5", "VideoDash"]))
    {
        media.displayObject.onQualityChange.add(this._mediaQualityChangeHandler, this);
    }
};

/**
 * Scene has started to unload
 * @todo clean media / background renderer
 *
 * @method FORGE.RenderManager#_onSceneUnloadStartHandler
 * @private
 */
FORGE.RenderManager.prototype._onSceneUnloadStartHandler = function()
{
    this._renderPipelineReady = false;

    this._clearBackgroundRenderer();

    // Clear fx composer and hotspot renderer
    if (this._objectRenderer !== null)
    {
        this._objectRenderer.clear();
    }

    if (this._pickingManager !== null)
    {
        this._pickingManager.clear();
    }

    if (this._renderPipeline !== null)
    {
        this._renderPipeline.clear();
    }

    // Destroy media
    if (this._mediaSound !== null)
    {
        this._mediaSound.destroy();
        this._mediaSound = null;
    }
};

/**
 * Init camera with info contained in configuration
 * @method FORGE.RenderManager#_initCamera
 * @param {SceneConfig} sceneConfig - scene configuration
 * @private
 */
FORGE.RenderManager.prototype._initCamera = function(sceneConfig)
{
    var sceneCameraConfig = /** @type {CameraConfig} */ (sceneConfig.camera);
    var storyCameraConfig = /** @type {CameraConfig} */ (this._viewer.mainConfig.camera);
    var extendedCameraConfig = /** @type {CameraConfig} */ (FORGE.Utils.extendMultipleObjects(storyCameraConfig, sceneCameraConfig));

    this._camera.load(extendedCameraConfig);
};

/**
 * Init the scene sound with info contained in configuration
 * @method FORGE.RenderManager#_initSound
 * @param {SceneConfig} sceneConfig - scene configuration
 * @private
 */
FORGE.RenderManager.prototype._initSound = function(sceneConfig)
{
    var soundConfig = /** @type {SoundConfig} */ (sceneConfig.sound);

    // Create sound from the SceneConfig
    if (typeof soundConfig !== "undefined" && typeof soundConfig.source !== "undefined")
    {
        var volume, loop, startTime, autoPlay;

        if (typeof soundConfig.options !== "undefined")
        {
            volume = (typeof soundConfig.options.volume === "number") ? FORGE.Math.clamp(soundConfig.options.volume, 0, 1) : 1;
            loop = (typeof soundConfig.options.loop === "boolean") ? soundConfig.options.loop : false;
            startTime = (typeof soundConfig.options.startTime === "number") ? soundConfig.options.startTime : 0;
            autoPlay = (typeof soundConfig.options.autoPlay === "boolean") ? soundConfig.options.autoPlay : false;
        }

        if (typeof soundConfig.source.url !== "undefined" && soundConfig.source.url !== "")
        {
            // Warning : UID is not registered and applied to the FORGE.Sound object for registration
            this._mediaSound = new FORGE.Sound(this._viewer, sceneConfig.sound.uid, sceneConfig.sound.source.url, (sceneConfig.sound.type === FORGE.SoundType.AMBISONIC));

            if (typeof soundConfig.options !== "undefined" && soundConfig.options !== null)
            {
                this._mediaSound.volume = volume;
                this._mediaSound.loop = loop;
                this._mediaSound.startTime = startTime;

                if (autoPlay === true)
                {
                    this._mediaSound.play(this._mediaSound.startTime, this._mediaSound.loop, true);
                }
            }
        }
        // @todo Ability to use a target uid rather than a source url (ie. soundConfig.source.target)
    }
};

/**
 * View change handler
 * @method FORGE.RenderManager#_onViewChangeHandler
 * @private
 */
FORGE.RenderManager.prototype._onViewChangeHandler = function()
{
    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.updateAfterViewChange();
    }

    this._pickingManager.updateForViewType(this._viewManager.type);

    if (this._viewManager.type === FORGE.ViewType.RECTILINEAR)
    {
        this._renderPipeline.enablePicking(false);
    }
    else
    {
        this._renderPipeline.enablePicking(true, this._pickingManager.material, this._pickingManager.renderTarget);
    }
};

/**
 * Handler of media load complete event
 * @method FORGE.RenderManager#_mediaLoadCompleteHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.RenderManager.prototype._mediaLoadCompleteHandler = function(event)
{
    this.log("Media load is complete");

    var media = event.emitter;

    this._setBackgroundRendererType(this._renderDisplay.presentingVR);
    this._setBackgroundRenderer(this._backgroundRendererType);

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.displayObject = media.displayObject;
    }

    this._setupRenderPipeline();
};

/**
 * Handler of media quality change event
 * @method FORGE.RenderManager#_mediaQualityChangeHandler
 * @private
 */
FORGE.RenderManager.prototype._mediaQualityChangeHandler = function(event)
{
    this.log("Media quality has changed");

    this._backgroundRenderer.displayObject = event.emitter;
};

/**
 * Setup render pipeline
 * @method FORGE.RenderManager#_setupRenderPipeline
 * @private
 */
FORGE.RenderManager.prototype._setupRenderPipeline = function()
{
    var fxSet = null;

    var sceneConfig = this._viewer.story.scene.config;
    var mediaConfig = sceneConfig.media;

    if (typeof mediaConfig !== "undefined" && mediaConfig !== null &&
        typeof mediaConfig.fx !== "undefined" && mediaConfig.fx !== null)
    {
        fxSet = this._viewer.postProcessing.getFxSetByUID(mediaConfig.fx);
    }

    if(this._backgroundRenderer !== null)
    {
        this._renderPipeline.addBackground(this._backgroundRenderer.renderTarget.texture, fxSet, 1.0);
    }

    if (typeof sceneConfig.fx !== "undefined" && sceneConfig.fx !== null)
    {
        var globalFxSet = this._viewer.postProcessing.getFxSetByUID(sceneConfig.fx);
        this._renderPipeline.addGlobalFx(globalFxSet);
    }

    this._renderPipelineReady = true;
};

/**
 * Render background.
 * @method FORGE.RenderManager#_drawBackground
 * @param {THREE.PerspectiveCamera} camera - perspective camera used to render mesh, N/A with shader rendering
 * @private
 */
FORGE.RenderManager.prototype._drawBackground = function(camera)
{
    // this.log("_drawBackground");

    if (this._backgroundRenderer === null)
    {
        return;
    }

    this._backgroundRenderer.render(camera || null);
};

/**
 * Set renderer size and all objects aware of resolution
 * @method FORGE.RenderManager#_setRendererSize
 * @param {FORGE.Size} size - new renderer size
 * @private
 */
FORGE.RenderManager.prototype._setRendererSize = function(size)
{
    var vr = this._renderDisplay.presentingVR;

    var keepCanvasStyle = true;

    if (vr === true)
    {
        size = this._renderDisplay.rendererSize;
        keepCanvasStyle = false;
    }
    else
    {
        this._renderDisplay.setSize(size);
    }

    this.log("set renderer size: " + size.width + "x" + size.height);

    this._webGLRenderer.setSize(size.width, size.height, keepCanvasStyle);
    this._canvasResolution = size;

    this._displayResolution = new FORGE.Size(size.width, size.height);

    if (vr === true)
    {
        this._displayResolution.width *= 0.5;
    }

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.setSize(this._displayResolution);
    }

    this._pickingManager.setSize(this._displayResolution);

    this._renderPipeline.setSize(this._displayResolution);

    this.log("render size change in " + (vr ? "VR" : "screen") + " mode, resolution: " + this._displayResolution.width + "x" + this._displayResolution.height);
};

/**
 * Internal handler on canvas resize.
 * @method FORGE.RenderManager#_canvasResizeHandler
 * @private
 */
FORGE.RenderManager.prototype._canvasResizeHandler = function()
{
    this.log("canvas resize handler");

    var canvas = this._viewer.canvas.dom;
    this._setRendererSize(new FORGE.Size(canvas.width, canvas.height));
};

/**
 * VR Renderer display change event handler
 * @method FORGE.RenderManager#_renderDisplayChangeHandler
 * @private
 */
FORGE.RenderManager.prototype._renderDisplayChangeHandler = function()
{
    this.log("render display change handler");

    this._setRendererSize(this._renderDisplay.rendererSize);

    if (this._renderDisplay.presentingVR === true)
    {
        this._pickingManager.mode = FORGE.PickingManager.modes.GAZE;
    }
    else
    {
        this._pickingManager.mode = FORGE.PickingManager.modes.POINTER;
    }
};

/**
 * Renderer set background renderer
 *
 * @method FORGE.RenderManager#_setBackgroundRenderer
 * @param {string} type - type of background renderer
 * @private
 */
FORGE.RenderManager.prototype._setBackgroundRenderer = function(type)
{
    this.log("set background renderer");

    var displayObject = null;
    var renderTarget = null;

    if (this._backgroundRenderer !== null)
    {
        if (this._backgroundRenderer.displayObject !== null)
        {
            displayObject = this._backgroundRenderer.displayObject;
        }

        if (this._backgroundRenderer.renderTarget !== null)
        {
            renderTarget = this._backgroundRenderer.renderTarget;
        }
    }

    this._clearBackgroundRenderer();

    var config = {};
    var media = this._viewer.story.scene.media;
    var mediaConfig = media.config;

    if (typeof mediaConfig !== "undefined" && mediaConfig !== null)
    {
        config.type = mediaConfig.type;

        if (typeof mediaConfig.source !== "undefined" && mediaConfig.source !== null)
        {
            var source = mediaConfig.source;

            if (typeof source.levels === "undefined")
            {
                config.mediaFormat = mediaConfig.source.format;
                var ratio = media.displayObject.element.width / media.displayObject.element.height || 1;

                if (typeof source.fov !== "undefined")
                {
                    var vFov;
                    if (typeof source.fov === "number")
                    {
                        vFov = source.fov.vertical;
                    }
                    else if (typeof source.fov.vertical === "number")
                    {
                        vFov = source.fov.vertical;
                    }
                    else if (typeof source.fov.horizontal === "number")
                    {
                        vFov = source.fov.horizontal / ratio;
                    }
                    else if (typeof source.fov.diagonal === "number")
                    {
                        vFov = source.fov.diagonal / Math.sqrt(1 + ratio * ratio);
                    }
                    else
                    {
                        vFov = 90;
                    }

                    config.verticalFov = FORGE.Math.degToRad(vFov);
                }
            }
        }

        if (typeof mediaConfig.options !== "undefined" && mediaConfig.options !== null)
        {
            if (typeof mediaConfig.options.color !== "undefined")
            {
                config.color = mediaConfig.options.color;
            }
        }
    }

    if (type === FORGE.BackgroundType.SHADER)
    {
        this.log("Create background shader renderer");
        this._backgroundRenderer = new FORGE.BackgroundShaderRenderer(this._viewer, renderTarget, config);

        var size = this._webGLRenderer.getSize();
        this._setRendererSize(new FORGE.Size(size.width, size.height));
    }
    else if (type === FORGE.BackgroundType.PYRAMID)
    {
        this.log("Create background pyramid renderer (multiresolution image)");
        this._backgroundRenderer = new FORGE.BackgroundPyramidRenderer(this._viewer, renderTarget, mediaConfig);
    }
    else if (type === FORGE.BackgroundType.MESH)
    {
        this.log("Create background mesh renderer");

        if (typeof mediaConfig !== "undefined" && mediaConfig !== null)
        {
            if (typeof mediaConfig.source !== "undefined" && mediaConfig.source !== null)
            {
                config.order = mediaConfig.source.order || "RLUDFB";

                // Get the right tile
                if (typeof mediaConfig.source.tile === "number")
                {
                    config.tile = mediaConfig.source.tile;
                }
                else if (Array.isArray(mediaConfig.source.levels) && typeof mediaConfig.source.levels[0].tile === "number")
                {
                    config.tile = mediaConfig.source.levels[0].tile;
                }
            }
        }

        this._backgroundRenderer = new FORGE.BackgroundMeshRenderer(this._viewer, renderTarget, config);
    }
    else
    {
        //@todo not implemented
        // auto mode: try with fragment shader and fall back to mesh if fps is too low
    }

    if (displayObject !== null)
    {
        this._backgroundRenderer.displayObject = displayObject;
    }

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.updateAfterViewChange();
    }

    this._backgroundReady = true;

    if (this._onBackgroundReady !== null)
    {
        this._onBackgroundReady.dispatch();
    }
};

/**
 * Set the background renderer depending on current media format and view type.
 * @method FORGE.RenderManager#_setBackgroundRendererType
 * @param {boolean} vrEnabled - VR enabled flag
 * @private
 */
FORGE.RenderManager.prototype._setBackgroundRendererType = function(vrEnabled)
{
    this.log("set background renderer type");

    var media = this._viewer.story.scene.media;

    if(media.type === FORGE.MediaType.UNDEFINED)
    {
        this._backgroundRendererType = FORGE.BackgroundType.UNDEFINED;
        return;
    }

    if (vrEnabled === true)
    {
        this.log("VR on - background type = MESH");
        this._backgroundRendererType = FORGE.BackgroundType.MESH;
        return;
    }

    var mediaConfig = media.config;

    if (mediaConfig.type === FORGE.MediaType.GRID)
    {
        this._backgroundRendererType = FORGE.BackgroundType.MESH;
    }
    else if (typeof mediaConfig.source !== "undefined")
    {
        if (typeof mediaConfig.source.levels !== "undefined" && media.type === FORGE.MediaType.IMAGE)
        {
            this._backgroundRendererType = FORGE.BackgroundType.PYRAMID;
        }
        else if (mediaConfig.source.format === FORGE.MediaFormat.CUBE)
        {
            this._backgroundRendererType = FORGE.BackgroundType.MESH;
        }
        else
        {
            this._backgroundRendererType = FORGE.BackgroundType.SHADER;
        }
    }
    else
    {
        this._backgroundRendererType = FORGE.BackgroundType.SHADER;
    }

    if (typeof mediaConfig.source === "undefined" || typeof mediaConfig.source.format === "undefined")
    {
        this.log("VR off - view " + this._viewManager.current.type + ", background type = " + this._backgroundRendererType);
    }
    else
    {
        this.log("VR off - media " + mediaConfig.source.format + ", view " + this._viewManager.current.type +
            ", background type = " + this._backgroundRendererType);
    }
};


/**
 * Clear the background renderer.
 * @method FORGE.RenderManager#_clearBackgroundRenderer
 * @private
 */
FORGE.RenderManager.prototype._clearBackgroundRenderer = function()
{
    this.log("clear background renderer");

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.destroy();
        this._backgroundRenderer = null;
    }

    this._backgroundReady = false;
};

/**
 * Update routine
 * @method FORGE.RenderManager#update
 */
FORGE.RenderManager.prototype.update = function()
{
    this._camera.update();
};

/**
 * Render routine
 * @method FORGE.RenderManager#render
 */
FORGE.RenderManager.prototype.render = function()
{
    if (this._viewManager === null ||
        this._renderPipelineReady === false ||
        this._renderPipeline === null)
    {
        return;
    }

    if(this._onBeforeRender !== null)
    {
        this._onBeforeRender.dispatch();
    }

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.update();
    }

    var vr = this._renderDisplay.presentingVR;
    var renderParams = this._renderDisplay.getRenderParams();

    for (var i = 0, ii = renderParams.length; i < ii; i++)
    {
        var params = renderParams[i];
        var rect = params.rectangle;
        var camera = params.camera;

        this._webGLRenderer.setViewport(rect.origin.x, rect.origin.y, rect.size.width, rect.size.height);

        this._drawBackground((vr === true) ? camera : null);

        this._renderPipeline.render(camera);

        // Render perspective camera children (objects in camera local space)
        this._webGLRenderer.clearDepth();

        if (vr === true)
        {
            var scene = new THREE.Scene();
            scene.add(camera);
            //window.scene = scene;

            this._webGLRenderer.render(scene, camera);
        }
    }

    if (this._renderDisplay.presentingVR === true)
    {
        this._renderDisplay.submitFrame();
    }

    if(this._onAfterRender !== null)
    {
        this._onAfterRender.dispatch();
    }

    // @todo implement event for render stats (fps, objects count...)
};

/**
 * Enable VR display
 * @method FORGE.RenderManager#enableVR
 */
FORGE.RenderManager.prototype.enableVR = function()
{
    if (this._renderDisplay.presentingVR === true || FORGE.Device.webVR !== true)
    {
        return;
    }

    this._renderDisplay.enableVR();
    this._viewManager.enableVR();

    var sceneConfig = this._viewer.story.scene.config;

    // If we enter VR with a cubemap: do nothing. With an equi: toggle to mesh renderer
    if (typeof sceneConfig.media !== "undefined" && sceneConfig.media !== null && typeof sceneConfig.media.source !== "undefined" && sceneConfig.media.source !== null && sceneConfig.media.source.format === FORGE.MediaFormat.EQUIRECTANGULAR)
    {
        this._setBackgroundRenderer(FORGE.BackgroundType.MESH);
    }
};

/**
 * Disable VR display
 * @method FORGE.RenderManager#disableVR
 */
FORGE.RenderManager.prototype.disableVR = function()
{
    if (this._renderDisplay.presentingVR === false || FORGE.Device.webVR !== true)
    {
        return;
    }

    this._renderDisplay.disableVR();
    this._viewManager.disableVR();

    // If we exit VR with a cubemap: do nothing. With an equi: toggle to shader renderer
    var sceneConfig = this._viewer.story.scene.config;
    var mediaConfig = sceneConfig.media;

    if (typeof mediaConfig !== "undefined" && mediaConfig !== null &&
        typeof mediaConfig.source !== "undefined" && mediaConfig.source !== null &&
        mediaConfig.source.format === FORGE.MediaFormat.EQUIRECTANGULAR)
    {
        this._setBackgroundRendererType(false);
        this._setBackgroundRenderer(this._backgroundRendererType);
    }
};

/**
 * Toggle VR display
 * @method FORGE.RenderManager#toggleVR
 */
FORGE.RenderManager.prototype.toggleVR = function()
{
    if(this._renderDisplay.presentingVR === true)
    {
        this.disableVR();
    }
    else
    {
        this.enableVR();
    }
};

/**
 * Renderer destroy sequence
 *
 * @method FORGE.RenderManager#destroy
 */
FORGE.RenderManager.prototype.destroy = function()
{
    this._viewer.canvas.onResize.remove(this._canvasResizeHandler, this);
    this._viewer.story.onSceneLoadStart.remove(this._onSceneLoadStartHandler, this);
    this._viewer.onConfigLoadComplete.remove(this._onViewerConfigLoadComplete, this);

    if (this._pickingManager !== null)
    {
        this._pickingManager.destroy();
        this._pickingManager = null;
    }

    if (this._onBackgroundReady !== null)
    {
        this._onBackgroundReady.destroy();
        this._onBackgroundReady = null;
    }

    if (this._mediaSound !== null)
    {
        this._mediaSound.destroy();
        this._mediaSound = null;
    }

    if (this._objectRenderer !== null)
    {
        this._objectRenderer.destroy();
        this._objectRenderer = null;
    }

    if (this._backgroundRenderer !== null)
    {
        this._backgroundRenderer.destroy();
        this._backgroundRenderer = null;
    }

    if (this._camera !== null)
    {
        this._camera.destroy();
        this._camera = null;
    }

    if (this._viewManager !== null)
    {
        this._viewManager.destroy();
        this._viewManager = null;
    }

    if (this._renderDisplay !== null)
    {
        this._renderDisplay.onDisplayChange.remove(this._renderDisplayChangeHandler, this);
        this._renderDisplay.destroy();
        this._renderDisplay = null;
    }

    if (this._renderPipeline !== null)
    {
        this._renderPipeline.destroy();
        this._renderPipeline = null;
    }

    if(this._onBeforeRender !== null)
    {
        this._onBeforeRender.destroy();
        this._onBeforeRender = null;
    }

    if(this._onAfterRender !== null)
    {
        this._onAfterRender.destroy();
        this._onAfterRender = null;
    }

    this._clock = null;
    this._webGLRenderer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get viewer.
 * @name FORGE.RenderManager#viewer
 * @type {FORGE.Viewer}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "viewer",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._viewer;
    }
});

/**
 * Get sound linked to the media.
 * @name FORGE.RenderManager#mediaSound
 * @type {(FORGE.Sound|Object)}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "mediaSound",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._mediaSound;
    }
});

/**
 * Get WebGL renderer.
 * @name FORGE.RenderManager#WebGLRenderer
 * @type {THREE.WebGLRenderer}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "webGLRenderer",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._webGLRenderer;
    }
});

/**
 * Get FX Composer.
 * @name FORGE.RenderManager#renderPipeline
 * @type {FORGE.RenderPipeline}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "renderPipeline",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._renderPipeline;
    }
});

/**
 * Get background renderer.
 * @name FORGE.RenderManager#backgroundRenderer
 * @type {(FORGE.BackgroundMeshRenderer|FORGE.BackgroundShaderRenderer)}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "backgroundRenderer",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._backgroundRenderer;
    }
});

/**
 * Get picking manager.
 * @name FORGE.RenderManager#pickingManager
 * @type {FORGE.PickingManager}
 */
Object.defineProperty(FORGE.RenderManager.prototype, "pickingManager",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._pickingManager;
    }
});

/**
 * Get the view manager.
 * @name FORGE.RenderManager#view
 * @type {FORGE.ViewManager}
 */
Object.defineProperty(FORGE.RenderManager.prototype, "view",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._viewManager;
    }
});

/**
 * Get camera.
 * @name FORGE.RenderManager#camera
 * @type {FORGE.Camera}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "camera",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._camera;
    }
});

/**
 * Get canvas resolution in pixels.
 * @name FORGE.RenderManager#canvasResolution
 * @type {FORGE.Size}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "canvasResolution",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._canvasResolution;
    }
});

/**
 * Get display resolution in pixels.
 * @name FORGE.RenderManager#displayResolution
 * @type {FORGE.Size}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "displayResolution",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._displayResolution;
    }
});

/**
 * VR presenting status.
 * @name FORGE.RenderManager#presentingVR
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "presentingVR",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._renderDisplay.presentingVR;
    }
});

/**
 * Get the render display.
 * @name FORGE.RenderManager#display
 * @type {FORGE.RenderDisplay}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "display",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._renderDisplay;
    }
});

/**
 * Get the backgroundReady flag
 * @name FORGE.RenderManager#backgroundReady
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "backgroundReady",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._backgroundReady;
    }
});

/**
 * Get the FORGE.ObjectRenderer instance
 * @name FORGE.RenderManager#objects
 * @type {FORGE.ObjectRenderer}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "objects",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        return this._objectRenderer;
    }
});

/**
 * Get the onBackgroundReady {@link FORGE.EventDispatcher}.
 * @name FORGE.RenderManager#onBackgroundReady
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "onBackgroundReady",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        if (this._onBackgroundReady === null)
        {
            this._onBackgroundReady = new FORGE.EventDispatcher(this, true);
        }

        return this._onBackgroundReady;
    }
});

/**
 * Get the onBeforeRender {@link FORGE.EventDispatcher}.
 * @name FORGE.RenderManager#onBeforeRender
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "onBeforeRender",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        if (this._onBeforeRender === null)
        {
            this._onBeforeRender = new FORGE.EventDispatcher(this);
        }

        return this._onBeforeRender;
    }
});

/**
 * Get the onAfterRender {@link FORGE.EventDispatcher}.
 * @name FORGE.RenderManager#onAfterRender
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.RenderManager.prototype, "onAfterRender",
{
    /** @this {FORGE.RenderManager} */
    get: function()
    {
        if (this._onAfterRender === null)
        {
            this._onAfterRender = new FORGE.EventDispatcher(this);
        }

        return this._onAfterRender;
    }
});

/**
 * A render scene is an object responsible of preparing the draw of a scene
 * with a camera, applying a set of image effect.
 *
 * It creates an effect composer that will be called by render loop.
 * It writes the resulting image into a texture used in main render pipeline.
 *
 * @constructor FORGE.RenderScene
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {THREE.Scene} scene scene to render
 * @param {THREE.Camera} camera camera used to render the scene
 * @param {Array<FX>} fxConfig image fx configuration object
 * @extends {FORGE.BaseObject}
 */
FORGE.RenderScene = function(viewer, scene, camera, fxConfig)
{
    /**
     * The viewer reference.
     * @name FORGE.RenderScene#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Scene to be rendered
     * @name FORGE.RenderScene#_scene
     * @type {THREE.Scene}
     * @private
     */
    this._scene = scene;

    /**
     * Camera used to render the scene
     * @name FORGE.RenderScene#_camera
     * @type {THREE.Camera}
     * @private
     */
    this._camera = camera;

    /**
     * Set of image effect configuration
     * @name FORGE.RenderScene#_fxConfig
     * @type {Array<FX>}
     * @private
     */
    this._fxConfig = fxConfig;

    /**
     * Effect composer rendering the scene into a texture
     * @name FORGE.RenderScene#_sceneComposer
     * @type {FORGE.EffectComposer}
     * @private
     */
    this._sceneComposer = null;

    /**
     * Effect composer dedicated to object picking
     * @name FORGE.RenderScene#_pickingComposer
     * @type {FORGE.EffectComposer}
     * @private
     */
    this._pickingComposer = null;

    FORGE.BaseObject.call(this, "RenderScene");

    this._boot();
};

FORGE.RenderScene.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.RenderScene.prototype.constructor = FORGE.RenderScene;

/**
 * Boot sequence
 * @method FORGE.RenderScene#_boot
 * @private
 */
FORGE.RenderScene.prototype._boot = function()
{
    // Create main effect composer, add render pass and shader passes
    this._sceneComposer = new FORGE.EffectComposer(FORGE.EffectComposerType.RENDER,
                                                 this._viewer.renderer.webGLRenderer);

    var renderPass = new FORGE.RenderPass(this._scene, this._camera);
    renderPass.position = FORGE.PassPosition.RENDER;
    this._sceneComposer.addPass(renderPass);

    var shaderPasses = this._parseShaderPasses(this._fxConfig);
    for (var j = 0, jj = shaderPasses.length; j < jj; j++)
    {
        var shaderPass = shaderPasses[j];
        shaderPass.renderToScreen = false;
        shaderPass.position = FORGE.PassPosition.RENDER;
        this._sceneComposer.addPass(shaderPass);
    }

    // Create picking effect composer and add render pass
    this._pickingComposer = new FORGE.EffectComposer(FORGE.EffectComposerType.PICKING,
                                                   this._viewer.renderer.webGLRenderer);

    var pickingPass = new FORGE.RenderPass(this._scene, this._camera);
    pickingPass.position = FORGE.PassPosition.RENDER;
    this._pickingComposer.addPass(pickingPass);
};

/**
 * Parse shader passes
 * @method FORGE.RenderScene#_parseShaderPasses
 * @param {Array<FX>} config shader passes configuration
 * @return {Array<THREE.ShaderPass>} array of shader passes
 * @private
 */
FORGE.RenderScene.prototype._parseShaderPasses = function(config)
{
    return this._viewer.postProcessing.parseShaderPasses(config);
};

/**
 * Set size of each pass of the render scene
 * @method FORGE.RenderScene#setSize
 * @param {number} width composer width
 * @param {number} height composer height
 */
FORGE.RenderScene.prototype.setSize = function(width, height)
{
    this._sceneComposer.setSize(width, height);
    this._pickingComposer.setSize(width, height);
};

/**
 * Destroy sequence
 * @method FORGE.RenderScene#destroy
 */
FORGE.RenderScene.prototype.destroy = function()
{
    this._sceneComposer.readBuffer.dispose();
    this._sceneComposer.writeBuffer.dispose();

    while (this._sceneComposer.passes.length > 0)
    {
        var pass = this._sceneComposer.passes.pop();

        if (pass instanceof FORGE.TexturePass && pass.hasOwnProperty("tDiffuse"))
        {
            pass.uniforms["tDiffuse"].value.texture.dispose();
            pass.uniforms["tDiffuse"].value = null;
        }
    }

    this._sceneComposer = null;

    this._pickingComposer.passes.length = 0;
    this._pickingComposer = null;

    this._viewer = null;
    this._scene = null;
    this._camera = null;
    this._fxConfig = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get scene.
 * @name FORGE.RenderScene#scene
 * @type {THREE.Scene}
 */
Object.defineProperty(FORGE.RenderScene.prototype, "scene",
{
    /** @this {FORGE.RenderScene} */
    get: function()
    {
        return this._scene;
    }
});

/**
 * Get camera.
 * @name FORGE.RenderScene#camera
 * @type {THREE.Camera}
 */
Object.defineProperty(FORGE.RenderScene.prototype, "camera",
{
    /** @this {FORGE.RenderScene} */
    get: function()
    {
        return this._camera;
    }
});

/**
 * Get scene composer.
 * @name FORGE.RenderScene#sceneComposer
 * @type {FORGE.EffectComposer}
 */
Object.defineProperty(FORGE.RenderScene.prototype, "sceneComposer",
{
    /** @this {FORGE.RenderScene} */
    get: function()
    {
        return this._sceneComposer;
    }
});

/**
 * Get scene composer.
 * @name FORGE.RenderScene#pickingComposer
 * @type {FORGE.EffectComposer}
 */
Object.defineProperty(FORGE.RenderScene.prototype, "pickingComposer",
{
    /** @this {FORGE.RenderScene} */
    get: function()
    {
        return this._pickingComposer;
    }
});

/**
 * FXComposer class.
 *
 * @constructor FORGE.RenderPipeline
 * @param {FORGE.Viewer} viewer - viewer reference
 * @extends {FORGE.BaseObject}
 *
 * @todo think about how to render multiple scene at the same time, with blending / overlap / viewport layouting...
 * maybe add a layer object encapsulating background / foreground renderings to ease the process
 */
FORGE.RenderPipeline = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.RenderPipeline#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Render composer
     * @name FORGE.RenderPipeline#_renderComposer
     * @type {FORGE.EffectComposer}
     * @private
     */
    this._renderComposer = null;

    /**
     * Sub composers array, created for each pass including texture/render + some shaders
     * @name FORGE.RenderPipeline#_subComposers
     * @type {Array<FORGE.EffectComposer>}
     * @private
     */
    this._subComposers = null;

    /**
     * Internal clock used to feed time to effect shaders
     * @name FORGE.RenderPipeline#_clock
     * @type {THREE.Clock}
     * @private
     */
    this._clock = null;

    /**
     * Enabled status flag
     * @name FORGE.RenderPipeline#_enabled
     * @type boolean
     * @private
     */
    this._enabled = true;

    FORGE.BaseObject.call(this, "RenderPipeline");

    this._boot();
};

FORGE.RenderPipeline.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.RenderPipeline.prototype.constructor = FORGE.RenderPipeline;

/**
 * Boot sequence
 * @method FORGE.RenderPipeline#_boot
 * @private
 */
FORGE.RenderPipeline.prototype._boot = function()
{
    this._clock = new THREE.Clock();
    this._subComposers = [];

    this._renderComposer = new FORGE.EffectComposer(FORGE.EffectComposerType.MAIN, this._viewer.renderer.webGLRenderer);
};

/**
 * Get pass type of a THREE.Pass
 * @method FORGE.RenderPipeline#_getPassType
 * @param {THREE.Pass} pass - THREE pass object
 * @private
 * @return {string} - Returns the pass type.
 */
FORGE.RenderPipeline.prototype._getPassType = function(pass)
{
    if (pass instanceof FORGE.TexturePass)
    {
        if (pass.hasOwnProperty("map"))
        {
            var uid = pass.map.uuid.split("-")[0].slice(0, 4);
            return "Texture(" + uid + ")";
        }
        return "Texture";
    }

    if (pass instanceof FORGE.RenderPass)
    {
        return "Render";
    }

    if (pass instanceof THREE.ClearMaskPass)
    {
        return "ClearMask";
    }

    if (pass instanceof THREE.MaskPass)
    {
        return "Mask";
    }

    if (pass instanceof FORGE.ShaderPass)
    {
        return "Shader(" + pass.type.replace("Shader", "") + ")";
    }

    return "Pass";
};

/**
 * Dump some effect composer to console output
 * @method FORGE.RenderPipeline#_dumpComposer
 * @param {FORGE.EffectComposer} composer - The effect composer to dump
 * @private
 */
FORGE.RenderPipeline.prototype._dumpComposer = function(composer)
{
    this.log("COMPOSER: " + composer.name);

    if (composer.passes.length === 0 )
    {
        return;
    }

    var types = [];
    var typesLength = 0;

    for (var i = 0, ii = composer.passes.length; i < ii; i++)
    {
        var typei = this._getPassType(composer.passes[i]);
        types.push(typei);
        typesLength += typei.length;
    }

    // Header/Footer
    var hdft = "";
    for (var j = 0, jj = types.length; j < jj; j++)
    {
        var type = types[j];

        hdft += "| ";
        for (var m=0; m<type.length; m++)
        {
            hdft += "-";
        }
        hdft += " |";

        if (j < jj - 1)
        {
            for (var l=0; l<5; l++)
            {
                hdft += " ";
            }
        }
    }

    // Header
    this.log(hdft);

    // Types
    var line = "";
    for (var k = 0, kk = types.length; k < kk; k++)
    {
        var typek = types[k];
        line += "| " + typek + " |";
        if (k < kk - 1)
        {
            line += " --- ";
        }
    }

    // Add composer output texture to the line if not render to screen
    if (composer.passes[composer.passes.length - 1].renderToScreen === false)
    {
        line += " --- tex";
        if (composer.readBuffer !== null && typeof composer.readBuffer.name !== "undefined")
        {
            line += "(" + composer.readBuffer.name + ")";
        };
        // line += " --- tex";
    }

    this.log(line);

    // Footer
    this.log(hdft);
};

/**
 * Dump whole composing pipeline to console output
 * @method FORGE.RenderPipeline#_dumpPipeline
 * @private
 */
FORGE.RenderPipeline.prototype._dumpPipeline = function()
{
    for (var i = 0, ii = this._subComposers.length; i < ii; i++)
    {
        var composer = this._subComposers[i];
        this._dumpComposer(composer);
    }

    this._dumpComposer(this._renderComposer);
};

/**
 * Add a subcomposer to compositing pipeline
 * Add subcomposer to inner list and add a texture pass drawing subcomposer output
 * @method FORGE.RenderPipeline#_addSubComposer
 * @param {FORGE.EffectComposer} subComposer - new effect composer
 * @param {boolean} renderToTarget - subcomposer should render to target
 * @private
 */
FORGE.RenderPipeline.prototype._addSubComposer = function(subComposer, renderToTarget)
{
    // Add a ForgeJS object into subcomposer to set some private properties
    this._subComposers.push(subComposer);

    if (renderToTarget === false)
    {
        var additionPass = new FORGE.AdditionPass(subComposer);
        this._renderComposer.addPass(additionPass);
    }

    this._updateRenderPipeline();
};

/**
 * Add shader passes to an effect composer
 * @method FORGE.RenderPipeline#_addShaderPasses
 * @param {FORGE.EffectComposer} composer - effect composer
 * @param {Array<THREE.Pass>|THREE.Pass} passes - pass or array of passes to be added to the composer
 * @param {number=} index - index where passes should be inserted
 * @private
 */
FORGE.RenderPipeline.prototype._addShaderPasses = function(composer, passes, index)
{
    index = typeof index !== "undefined" ? index : composer.passes.length;

    if (Array.isArray(passes))
    {
        for (var i = passes.length - 1; i >= 0; i--)
        {
            var shaderPass = passes[i];
            composer.insertPass(shaderPass, index);
        }
    }
    else if (passes instanceof THREE.Pass)
    {
        var shaderThreePass = passes;
        composer.insertPass(shaderThreePass, index);
    }
};

/**
 * Change status of all shader passes in a composer
 * @method FORGE.RenderPipeline#_setComposerShaderPassesStatus
 * @param {FORGE.EffectComposer} composer - effect composer.
 * @param {boolean} status - The status you want to set.
 * @private
 */
FORGE.RenderPipeline.prototype._setComposerShaderPassesStatus = function(composer, status)
{
    if (status === false)
    {
        this._enabled = false;
    }

    for (var i = composer.passes.length - 1; i >= 0; i--)
    {
        var pass = composer.passes[i];

        // Enable/disable shader passes except for Addition passes
        if (pass instanceof THREE.ShaderPass)
        {
            if (pass instanceof FORGE.AdditionPass)
            {
                continue;
            }

            pass.enabled = status;
        }
    }

    if (composer.hasOwnProperty("enabled"))
    {
        composer.enabled = status;
    }

    this._updateRenderPipeline();
};

/**
 * Set all shader passes status
 * @method FORGE.RenderPipeline#_setAllShaderPassesStatus
 * @param {boolean} status - new shader passes status.
 * @private
 */
FORGE.RenderPipeline.prototype._setAllShaderPassesStatus = function(status)
{
    var composers = this._subComposers.concat(this._renderComposer);

    for (var i = 0, ii = composers.length; i < ii; i++)
    {
        var composer = composers[i];
        this._setComposerShaderPassesStatus(composer, status);
    }

    this._enabled = status;
};

/**
 * Update rendering pipeline
 * Called whenever pipeline has changed to optimize the render stream.
 * @method FORGE.RenderPipeline#_updateRenderPipeline
 * @private
 */
FORGE.RenderPipeline.prototype._updateRenderPipeline = function()
{
    // First check render composer order
    // background passes --> render passes --> global passes
    var passes = this._renderComposer.passes;

    // Collect background passes in reverse order and reinsert them at zero index
    var backgrounds = [];

    passes.filter(function(element, index) //array
    {
        if (element.position === FORGE.PassPosition.BACKGROUND)
        {
            backgrounds.push(index);
        }
    });

    if (backgrounds.length > 0 && backgrounds[0] > 0)
    {
        var bgdPasses = [];

        for (var i = backgrounds.length - 1; i >= 0; i--)
        {
            var index = backgrounds[i];
            var passi = passes.splice(index, 1)[0];
            bgdPasses.push(passi);
        }

        for (var j = 0, jj = bgdPasses.length; j < jj; j++)
        {
            this._renderComposer.insertPass(bgdPasses[j], 0);
        }
    }

    // Collect global passes in normal order and reinsert them at the end
    var globals = [];
    passes.filter(function(element, index) //array
    {
        if (element.position === FORGE.PassPosition.GLOBAL)
        {
            globals.push(index);
        }
    });

    // Check if last global pass is at the end of passes array
    if (globals.length > 0 && globals[globals.length - 1] < passes.length - 1)
    {
        var globalPasses = [];

        for (var k = 0, kk = globals.length; k < kk; k++)
        {
            var indexk = globals[k];
            var passk = passes.splice(indexk, 1)[0];
            globalPasses.push(passk);
        }

        for (var l = 0, ll = globalPasses.length; l < ll; l++)
        {
            var passl = globalPasses[l];
            this._renderComposer.addPass(passl);
        }
    }

    // Only first pass of each composer should clear the target
    for (var m = 0, mm = this._subComposers.length; m < mm; m++)
    {
        var composer = this._subComposers[m];

        // First check if a composer share a target with another one
        // Then only the first should clear the target
        var sharingTarget = false;
        var s, ss = this._subComposers.length;
        for (s = 0; s < ss; s++)
        {
            if (s === m)
            {
                continue;
            }

            var anotherComposer = this._subComposers[s];
            if (composer.readBuffer === anotherComposer.readBuffer)
            {
                sharingTarget = true;
                break;
            }
        }
        sharingTarget = sharingTarget && (m > s);

        for (var n = 0, nn = composer.passes.length; n < nn; n++)
        {
            var passn = composer.passes[n];

            // If sharing target with a previous subcomposer, force clear to false
            if (sharingTarget === true)
            {
                passn.clear = false;
            }
            else
            {
                passn.clear = (n === 0);
            }
        }

    }

    // Only last enabled pass of render composer should render to screen
    var rts = false;
    for (var p = passes.length - 1; p >= 0; p--)
    {
        var passp = passes[p];

        if (rts === true)
        {
            passp.renderToScreen = false;
        }
        else if (passp.enabled === true)
        {
            passp.renderToScreen = true;
            rts = true;
        }
    }

    this._dumpPipeline();
};

/**
 * Setup default background texture pass.
 * @method FORGE.RenderPipeline#_setupDefaultBackground.
 * @private
 */
FORGE.RenderPipeline.prototype._setupDefaultBackground = function()
{
    var canvas = document.createElement("canvas");
    canvas.width = canvas.height = 1;
    var context = canvas.getContext("2d");
    context.fillStyle = "rgb(0, 0, 0)";
    context.fillRect(0, 0, canvas.width, canvas.height);

    var defaultTexture = new THREE.TextureLoader().load(canvas.toDataURL());
    defaultTexture.name = "forge-default-texture";

    this.addBackground(defaultTexture, null, 0);
};

/**
 * Set all render passes camera.
 * @method FORGE.RenderPipeline#_setAllRenderPassCamera.
 * @param {THREE.Camera} camera render pass camera.
 * @private
 */
FORGE.RenderPipeline.prototype._setAllRenderPassCamera = function(camera)
{
    var composers = this._subComposers.concat(this._renderComposer);

    for (var i = 0, ii = composers.length; i < ii; i++)
    {
        var composer = composers[i];

        for (var j=0, jj=composer.passes.length; j<jj; j++)
        {
            var pass = composer.passes[j];

            if (pass instanceof FORGE.RenderPass)
            {
                pass.camera = camera;
            }
        }
    }
};

/**
 * Enable/Disable picking.
 * @method FORGE.RenderPipeline#enablePicking
 * @param {boolean} status new picking state
 * @param {THREE.Material=} material picking material
 * @param {THREE.WebGLRenderTarget=} renderTarget picking render target
 */
FORGE.RenderPipeline.prototype.enablePicking = function(status, material, renderTarget)
{
    for (var i = 0, ii = this._subComposers.length; i < ii; i++)
    {
        var composer = this._subComposers[i];
        if (composer.type === FORGE.EffectComposerType.PICKING)
        {
            composer.enabled = status;

            if (typeof material !== "undefined")
            {
                composer.passes[0].overrideMaterial = material;
            }

            if (typeof renderTarget !== "undefined")
            {
                composer.renderTarget = renderTarget;
            }
        }
    }
};

/**
 * Add background to the rendering pipeline.
 * @method FORGE.RenderPipeline#addBackground
 * @param {THREE.Texture} texture texture object used as background.
 * @param {Array<FX>} fxSet image fx set to apply to background only.
 * @param {number=} opacity texture pass opacity
 */
FORGE.RenderPipeline.prototype.addBackground = function(texture, fxSet, opacity)
{
    // Background addition will be an insertion of all passes at index 0
    // First we add all shaders passes in reverse order at index 0
    // Then we insert the texture pass so everything ends up in the right order

    if (typeof fxSet !== "undefined" && fxSet !== null && fxSet.length > 0)
    {
        var shaderPasses = this._viewer.postProcessing.parseShaderPasses(fxSet);

        for (var i = 0, ii = shaderPasses.length; i < ii; i++)
        {
            var pass = shaderPasses[i];
            pass.position = FORGE.PassPosition.BACKGROUND;
        }

        this._addShaderPasses(this._renderComposer, shaderPasses, 0);
    }

    var texturePass = new FORGE.TexturePass(texture, opacity);
    texturePass.position = FORGE.PassPosition.BACKGROUND;

    this._renderComposer.insertPass(texturePass, 0);

    this._updateRenderPipeline();
};

/**
 * Add render scenes to the rendering pipeline
 * @method FORGE.RenderPipeline#addRenderScenes
 * @param {Array<FORGE.RenderScene>} renderScenes array of render scenes
 */
FORGE.RenderPipeline.prototype.addRenderScenes = function(renderScenes)
{
    for (var i = 0, ii = renderScenes.length; i < ii; i++)
    {
        this._addSubComposer(renderScenes[i].sceneComposer, false);
        this._addSubComposer(renderScenes[i].pickingComposer, true);
    }

    this._updateRenderPipeline();
};

/**
 * Add fx at the end of the whole rendering pipeline
 * @method FORGE.RenderPipeline#addGlobalFx
 * @param {Array<FX>} fxSet set of effects
 */
FORGE.RenderPipeline.prototype.addGlobalFx = function(fxSet)
{
    var shaderPasses = this._viewer.postProcessing.parseShaderPasses(fxSet);

    for (var i = 0, ii = shaderPasses.length; i < ii; i++)
    {
        var pass = shaderPasses[i];
        pass.position = FORGE.PassPosition.GLOBAL;
    }

    this._addShaderPasses(this._renderComposer, shaderPasses);
    this._updateRenderPipeline();
};

/**
 * Get a shader pass with an UID
 * @method FORGE.RenderPipeline#getShaderPassByUID
 * @param {string} uid - the uid of the ShaderPass
 * @return {THREE.Pass} shader pass
 */
FORGE.RenderPipeline.prototype.getShaderPassByUID = function(uid)
{
    if (this._renderComposer !== null)
    {
        for (var i = 0, ii = this._renderComposer.passes.length; i < ii; i++)
        {
            var pass = this._renderComposer.passes[i];

            if (pass.uid === uid)
            {
                return pass;
            }
        }
    }

    if (this._subComposers !== null)
    {
        for (var k = 0, kk = this._subComposers.length; k < kk; k++)
        {
            var composer = this._subComposers[k];

            for (var j = 0, jj = composer.passes.length; j < jj; j++)
            {
                var subPass = composer.passes[j];

                if (subPass.uid === uid)
                {
                    return subPass;
                }
            }
        }
    }

    return null;
};

/**
 * Set size of each composers
 * @method FORGE.RenderPipeline#setSize
 * @param {FORGE.Size} size new composer size
 */
FORGE.RenderPipeline.prototype.setSize = function(size)
{
    for (var i = 0, ii = this._subComposers.length; i < ii; i++)
    {
        var composer = this._subComposers[i];
        composer.setSize(size.width, size.height);

        for (var c = 0, cc = composer.passes.length; c < cc; c++)
        {
            var pass = composer.passes[c];

            if (pass instanceof THREE.ShaderPass)
            {
                if (typeof pass.uniforms.resolution !== "undefined")
                {
                    pass.uniforms.resolution.value = new THREE.Vector2(1 / size.width, 1 / size.height);
                }
            }
        }
    }

    this._renderComposer.setSize(size.width, size.height);

    for (var c = 0, cc = this._renderComposer.passes.length; c < cc; c++)
    {
        var pass = this._renderComposer.passes[c];

        if (pass instanceof THREE.ShaderPass)
        {
            if (typeof pass.uniforms.resolution !== "undefined")
            {
                pass.uniforms.resolution.value = new THREE.Vector2(1 / size.width, 1 / size.height);
            }
        }
    }
};

/**
 * Render routine
 *
 * @method FORGE.RenderPipeline#render
 * @param {THREE.PerspectiveCamera} camera - The camera to use for render.
 */
FORGE.RenderPipeline.prototype.render = function(camera)
{
    if (this._renderComposer === null)
    {
        return;
    }

    // Create default texture and use it if there is no texture in background
    if (!(this._renderComposer.passes[0] instanceof FORGE.TexturePass))
    {
        this._setupDefaultBackground();
    }

    for (var i = 0, ii = this._renderComposer.passes.length; i < ii; i++)
    {
        var pass = this._renderComposer.passes[i];
        if (pass instanceof FORGE.AdditionPass)
        {
            // Pick right buffer as texture provider for addition pass
            var composer = pass.composer;
            var lastPass = composer.passes[composer.passes.length - 1];

            // Addition pass will blend readbuffer content unless last pass needs swap (ShaderPass for example)
            var texture = composer.readBuffer.texture;

            if (typeof lastPass !== "undefined" && lastPass.needsSwap === true)
            {
                texture = composer.writeBuffer.texture;
            }

            pass.uniforms["tAdd"].value = texture;
        }
    }

    var delta = this._clock.getDelta();

    this._setAllRenderPassCamera(camera);

    for (var j = 0, jj = this._subComposers.length; j < jj; j++)
    {
        if (this._subComposers[j].enabled)
        {
            this._subComposers[j].render(delta);
        }
    }

    this._renderComposer.render(delta);
};

/**
 * Clear composer components
 * @method FORGE.RenderPipeline#clear
 */
FORGE.RenderPipeline.prototype.clear = function()
{
    if (this._renderComposer !== null)
    {
        this._renderComposer.readBuffer.dispose();
        this._renderComposer.writeBuffer.dispose();

        for (var i = 0, ii = this._renderComposer.passes.length; i < ii; i++)
        {
            var pass = this._renderComposer.passes[i];
            if (pass instanceof FORGE.ShaderPass && pass.uniforms.hasOwnProperty("tAdd"))
            {
                pass.uniforms["tAdd"].value = null;
            }
        }

        this._renderComposer.passes = [];
    }

    this._subComposers = [];
};

/**
 * Destroy sequence
 * @method FORGE.RenderPipeline#destroy
 */
FORGE.RenderPipeline.prototype.destroy = function()
{
    this.clear();
    this._clock = null;

    for (var i = this._subComposers.length - 1; i > 0; i--)
    {
        var composer = this._subComposers[i];

        for (var j=0, jj=composer.passes.length; j<jj; j++)
        {
            var pass = composer.passes[j];

            if (typeof pass.destroy === "function")
            {
                pass.destroy();
            }
        }
        composer.passes.length = 0;
    }
    this._subComposers = null;

    for (var j=0, jj=this._renderComposer.passes.length; j<jj; j++)
    {
        var pass = this._renderComposer.passes[j];

        if (typeof pass.destroy === "function")
        {
            pass.destroy();
        }
    }
    this._renderComposer.passes.length = 0;
    this._renderComposer = null;

    this._viewer = null;
};

/**
 * FX pipeline status.
 * @name FORGE.RenderPipeline#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.RenderPipeline.prototype, "enabled",
{
    /** @this {FORGE.RenderPipeline} */
    get: function()
    {
        return this._enabled;
    },

    /** @this {FORGE.RenderPipeline} */
    set: function(status)
    {
        this._setAllShaderPassesStatus(status);
    }
});

/**
 * @constructor FORGE.RenderDisplay
 * @param {FORGE.Viewer} viewer - viewer reference
 * @extends {FORGE.BaseObject}
 */
FORGE.RenderDisplay = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.RenderDisplay#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * WebVR flag, true if runtime supports WebVR.
     * @name FORGE.RenderDisplay#_webVR
     * @type {boolean}
     * @private
     */
    this._webVR = false;

    /**
     * Presenting VR status.
     * @name FORGE.RenderDisplay#_presentingVR
     * @type {boolean}
     * @private
     */
    this._presentingVR = false;

    /**
     * WebVR VRDisplay interface.
     * @name FORGE.RenderDisplay#_vrDisplay
     * @type {VRDisplay}
     * @private
     */
    this._vrDisplay = null;

    /**
     * VRDisplay boundaries for left eye.
     * @name FORGE.RenderDisplay#_leftBounds
     * @type {Array<number>}
     * @private
     */
    this._leftBounds = null;

    /**
     * VRDisplay boundaries for right eye.
     * @name FORGE.RenderDisplay#_rightBounds
     * @type {Array<number>}
     * @private
     */
    this._rightBounds = null;

    /**
     * Renderer size.
     * @name FORGE.RenderDisplay#_rendererSize
     * @type {FORGE.Size}
     * @private
     */
    this._rendererSize = null;

    /**
     * Renderer size for screen display.
     * @name FORGE.RenderDisplay#_rendererSizeScreen
     * @type {FORGE.Size}
     * @private
     */
    this._rendererSizeScreen = null;

    /**
     * Renderer pixel ratio.
     * @name FORGE.RenderDisplay#_rendererPixelRatio
     * @type {number}
     * @private
     */
    this._rendererPixelRatio = 1;

    /**
     * WebVR frame data receiver
     * @name FORGE.RenderDisplay#_frameData
     * @type {VRFrameData}
     * @private
     */
    this._frameData = null;

    /**
     * On display change event dispatcher.
     * @name  FORGE.RenderDisplay#_onDisplayChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onDisplayChange = null;

    FORGE.BaseObject.call(this, "RenderDisplay");

    this._boot();
};

FORGE.RenderDisplay.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.RenderDisplay.prototype.constructor = FORGE.RenderDisplay;

/**
 * Boot sequence
 * @method FORGE.RenderDisplay#_boot
 * @private
 */
FORGE.RenderDisplay.prototype._boot = function()
{
    var renderer = this._viewer.renderer.webGLRenderer;
    this._rendererPixelRatio = renderer.getPixelRatio();
    var size = renderer.getSize();
    this._rendererSizeScreen = new FORGE.Size(size.width, size.height);
    this._rendererSize = this._rendererSizeScreen;

    // Bounds will be set when user request fullscreen VR display
    this._leftBounds = [ 0.0, 0.0, 0.5, 1.0 ];
    this._rightBounds = [ 0.5, 0.0, 0.5, 1.0 ];

    // Create frame data receiver
    if ("VRFrameData" in window)
    {
        this._frameData = new VRFrameData();
    }

    if(FORGE.Device.webVR === true)
    {
        var self = this;

        navigator.getVRDisplays().then(
            function(displays)
            {
                self._gotVRDisplays(displays);
            }
        );

        this._addFullscreenListener();
    }
    else
    {
        this.warn("missing api navigator.getVRDisplays");
    }
};

/**
 * VR display interface found callback
 * Just keep the first interface available
 * @method FORGE.RenderDisplay#_gotVRDisplays
 * @param {Array<VRDisplay>} displays - array of VRDisplay interfaces available
 * @param {Function=} onError - error callback
 * @private
 */
FORGE.RenderDisplay.prototype._gotVRDisplays = function(displays, onError)
{
    this._webVR = displays.length > 0;

    for (var i=0, ii=displays.length; i<ii; i++)
    {
        if ("VRDisplay" in window && displays[i] instanceof VRDisplay)
        {
            this._vrDisplay = displays[i];
            this._vrDisplay.depthNear = FORGE.RenderManager.DEPTH_NEAR;
            this._vrDisplay.depthFar = 2 * FORGE.RenderManager.DEPTH_FAR;
            break;
        }
    }

    if (typeof this._vrDisplay === "undefined")
    {
        if (typeof onError === "function")
        {
            onError("HMD not available");
        }
    }
};

/**
 * Add event listeners for fullscreen and VR display change
 * Cross browser implementation
 * @method FORGE.RenderDisplay#_addFullscreenListener
 * @private
 */
FORGE.RenderDisplay.prototype._addFullscreenListener = function()
{
    this._viewer.container.onFullscreenEnter.add(this._displayChangeHandler, this);
    this._viewer.container.onFullscreenExit.add(this._displayChangeHandler, this);

    window.addEventListener("vrdisplaypresentchange", this._displayChangeHandler.bind(this), false);
};

/**
 * Remove event listeners for fullscreen and VR display change
 * Cross browser implementation
 * @method FORGE.RenderDisplay#_addFullscreenListener
 * @private
 */
FORGE.RenderDisplay.prototype._removeFullscreenListener = function()
{
    this._viewer.container.onFullscreenEnter.remove(this._displayChangeHandler, this);
    this._viewer.container.onFullscreenExit.remove(this._displayChangeHandler, this);

    window.removeEventListener("vrdisplaypresentchange", this._displayChangeHandler.bind(this));
};

/**
 * Display change event handler
 * @method FORGE.RenderDisplay#_displayChangeHandler
 * @private
 */
FORGE.RenderDisplay.prototype._displayChangeHandler = function()
{
    var wasPresentingVR = this._presentingVR;
    var renderer = this._viewer.renderer.webGLRenderer;

    this._presentingVR = typeof this._vrDisplay !== "undefined" &&
        (this._vrDisplay.isPresenting === true ||
            (this._webVR === false && document[FORGE.Device.fullscreenElement] instanceof window.HTMLElement));

    var displaySize;

    if (this._presentingVR === true)
    {
        var eyeParamsL = this._vrDisplay.getEyeParameters("left");

        if (this._webVR === true)
        {
            this._eyeWidth = eyeParamsL.renderWidth;
            this._eyeHeight = eyeParamsL.renderHeight;

            this.log("Window size: " + window.innerWidth + "x" + window.innerHeight);
            this.log("Render size: " + this._eyeWidth + "x" + this._eyeHeight);

            this._leftBounds = [ 0.0, 0.0, 0.5, 1.0 ];
            this._rightBounds = [ 0.5, 0.0, 0.5, 1.0 ];

            if (this._vrDisplay.getLayers)
            {
                var layers = this._vrDisplay.getLayers();

                if (layers.length > 0 && layers[0].leftBounds !== null && layers[0].leftBounds.length === 4)
                {
                    this._leftBounds = layers[0].leftBounds;
                    this._rightBounds = layers[0].rightBounds;
                }
            }

            this.log("Bounds L:[" + this._leftBounds[0] + ", " + this._leftBounds[1] + ", " + this._leftBounds[2] + ", " + this._leftBounds[3] + "], " +
                            "R:[" + this._rightBounds[0] + ", " + this._rightBounds[1] + ", " + this._rightBounds[2] + ", " + this._rightBounds[3] + "]");
        }

        if (wasPresentingVR === false)
        {
            this._rendererPixelRatio = renderer.getPixelRatio();
            var size = renderer.getSize();
            this._rendererSizeScreen.width = size.width;
            this._rendererSizeScreen.height = size.height;
            this._rendererSize = new FORGE.Size(this._eyeWidth * 2, this._eyeHeight);
            renderer.setPixelRatio( 1 );
        }
    }
    else if (wasPresentingVR === true)
    {
        this._rendererSize = this._rendererSizeScreen;
        renderer.setPixelRatio( this._rendererPixelRatio );
    }

    // dispatch change event only when display is impacted
    if (this._presentingVR !== wasPresentingVR && this._onDisplayChange !== null)
    {
        this._onDisplayChange.dispatch();
    }
};

/**
 * Set fullscreen
 * @method FORGE.RenderDisplay#_setFullScreen
 * @param {boolean} status fullscreen status
 * @private
 */
FORGE.RenderDisplay.prototype._setFullScreen = function (status)
{
    var canvas = this._viewer.renderer.webGLRenderer.domElement;

    return new Promise(function (resolve, reject)
    {
        if (typeof this._vrDisplay === "undefined")
        {
            reject(new Error("No VR hardware found."));
            return;
        }

        if (this._presentingVR === status)
        {
            resolve();
            return;
        }

        if (this._webVR === true)
        {
            if (status)
            {
                resolve(this._vrDisplay.requestPresent([ { source: canvas } ] ));
            }
            else
            {
                resolve(this._vrDisplay.exitPresent());
            }
        }

    }.bind(this));
};

/**
 * VR controls reset routine
 * @method FORGE.RenderDisplay#_vrControlsReset
 * @private
 */
FORGE.RenderDisplay.prototype._vrControlsReset = function()
{
    if (this._vrDisplay !== null)
    {
        if (typeof this._vrDisplay.resetPose !== "undefined")
        {
            this._vrDisplay.resetPose();
        }
        else if (typeof this._vrDisplay.resetSensor !== "undefined")
        {
            // Deprecated API.
            this._vrDisplay.resetSensor();
        }
        else if (typeof this._vrDisplay.zeroSensor !== "undefined")
        {
            // Really deprecated API.
            this._vrDisplay.zeroSensor();
        }
    }
};

/**
 * Request presentation through VR display interface
 * @method FORGE.RenderDisplay#_requestPresent
 * @private
 */
FORGE.RenderDisplay.prototype._requestPresent = function()
{
    this._viewer.raf.stop();
    this._viewer.raf.start(this._vrDisplay);
    return this._setFullScreen(true);
};

/**
 * Exit presentation from VR display interface
 * @method FORGE.RenderDisplay#_exitPresent
 * @private
 */
FORGE.RenderDisplay.prototype._exitPresent = function()
{
    this._viewer.raf.stop();
    this._viewer.raf.start(window);
    return this._setFullScreen(false);
};

/**
 * Enable VR display
 * @method FORGE.RenderDisplay#enableVR
 */
FORGE.RenderDisplay.prototype.enableVR = function()
{
    this._requestPresent();
};

/**
 * Start or stop VR display
 * @method FORGE.RenderDisplay#disableVR
 */
FORGE.RenderDisplay.prototype.disableVR = function()
{
    this._exitPresent();
};

/**
 * Get camera orientation quaternion when presenting VR
 * @method FORGE.RenderDisplay#getQuaternionFromPose
 * @return {THREE.Quaternion} quaternion extracted from pose or null if vrDisplay is not available
 * @private
 */
FORGE.RenderDisplay.prototype.getQuaternionFromPose = function()
{
    if ( this._vrDisplay === null )
    {
        return null;
    }

    var pose = null;
    if (this._frameData !== null && typeof this._frameData.pose !== "undefined")
    {
        pose = this._frameData.pose;
    }
    else
    {
        pose = this._vrDisplay.getPose();
    }

    if (pose === null || pose.orientation === null)
    {
        return new THREE.Quaternion();
    }

    var o = pose.orientation;
    return new THREE.Quaternion(-o[1], -o[0], -o[2], o[3]);
};

/**
 * Get render parameters
 * @method FORGE.RenderDisplay#getRenderParams
 * @return {Array<FORGE.RenderParams>} Returns an array of render parameters.
 */
FORGE.RenderDisplay.prototype.getRenderParams = function()
{
    var renderer = this._viewer.renderer.webGLRenderer;
    var canvas = renderer.domElement;
    var camera = this._viewer.renderer.camera;
    var renderParams = [];

    if (typeof this._vrDisplay !== "undefined" && this._vrDisplay !== null && this._presentingVR === true)
    {
        // Setup render rectangle, that will be use as glViewport
        var rx = this._rendererSize.width * this._leftBounds[0],
        ry = this._rendererSize.height * this._leftBounds[1],
        rw = this._rendererSize.width * this._leftBounds[2],
        rh = this._rendererSize.height * this._leftBounds[3];

        var renderRectL = new FORGE.Rectangle(rx, ry, rw, rh);

        rx = this._rendererSize.width * this._rightBounds[0];
        ry = this._rendererSize.height * this._rightBounds[1];
        rw = this._rendererSize.width * this._rightBounds[2];
        rh = this._rendererSize.height * this._rightBounds[3];

        var renderRectR = new FORGE.Rectangle(rx, ry, rw, rh);

        renderParams.push(new FORGE.RenderParams(renderRectL, camera.left));
        renderParams.push(new FORGE.RenderParams(renderRectR, camera.right));
    }
    else
    {
        var rectangle = new FORGE.Rectangle(0, 0, this._rendererSizeScreen.width, this._rendererSizeScreen.height);
        var renderCamera = this._viewer.renderer.view.type === FORGE.ViewType.FLAT ? camera.flat : camera.main;
        renderParams.push(new FORGE.RenderParams(rectangle, renderCamera));
    }

    return renderParams;
};

/**
 * Set size of rendering objects
 * @method FORGE.RenderDisplay#setSize
 * @param {FORGE.Size} size - new renderer size
 */
FORGE.RenderDisplay.prototype.setSize = function(size)
{
    this._rendererSizeScreen = size;

    var renderer = this._viewer.renderer.webGLRenderer;
    renderer.setPixelRatio( 1 );
};

/**
 * Submit current frame to VR display interface
 * @method FORGE.RenderDisplay#submitFrame
 */
FORGE.RenderDisplay.prototype.submitFrame = function()
{
    if (this._webVR === true && typeof this._vrDisplay !== "undefined" && this._presentingVR === true)
    {
        if (this._vrDisplay.capabilities.hasExternalDisplay === true)
        {
            this._vrDisplay.submitFrame();
        }
    }
};

/**
 * Destroy sequence
 * @method FORGE.RenderDisplay#destroy
 */
FORGE.RenderDisplay.prototype.destroy = function()
{
    this._removeFullscreenListener();

    this._vrDisplay = null;
    this._frameData = null;
    this._leftBounds = null;
    this._rightBounds = null;
    this._rendererSize = null;
    this._viewer = null;
};

/**
 * Presenting in VR or not.
 * @name FORGE.RenderDisplay#presentingVR
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.RenderDisplay.prototype, "presentingVR",
{
    /** @this {FORGE.RenderDisplay} */
    get: function()
    {
        return this._presentingVR;
    }
});

/**
 * Get the VR display.
 * @name FORGE.RenderDisplay#vrDisplay
 * @type {VRDisplay}
 * @readonly
 */
Object.defineProperty(FORGE.RenderDisplay.prototype, "vrDisplay",
{
    /** @this {FORGE.RenderDisplay} */
    get: function()
    {
        return this._vrDisplay;
    }
});

/**
 * Get the VR display.
 * @name FORGE.RenderDisplay#vrFrameData
 * @type {VRFrameData}
 * @readonly
 */
Object.defineProperty(FORGE.RenderDisplay.prototype, "vrFrameData",
{
    /** @this {FORGE.RenderDisplay} */
    get: function()
    {
        if(this._vrDisplay !== null && typeof this._vrDisplay.getFrameData === "function" && this._frameData !== null)
        {
            this._vrDisplay.getFrameData(this._frameData);
            return this._frameData;
        }

        return null;
    }
});

/**
 * Render size.
 * @name FORGE.RenderDisplay#rendererSize
 * @readonly
 * @type {FORGE.Size}
 */
Object.defineProperty(FORGE.RenderDisplay.prototype, "rendererSize",
{
    /** @this {FORGE.RenderDisplay} */
    get: function()
    {
        return this._rendererSize;
    }
});

/**
 * Get the onDisplayChange {@link FORGE.EventDispatcher}.
 * @name FORGE.RenderDisplay#onDisplayChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.RenderDisplay.prototype, "onDisplayChange",
{
    /** @this {FORGE.RenderDisplay} */
    get: function()
    {
        if(this._onDisplayChange === null)
        {
            this._onDisplayChange = new FORGE.EventDispatcher(this);
        }

        return this._onDisplayChange;
    }
});

/**
 * Render parameters
 *
 * @constructor FORGE.RenderParams
 * @param {FORGE.Rectangle} rectangle render rectangle
 * @param {THREE.PerspectiveCamera} camera render camera
 */
FORGE.RenderParams = function(rectangle, camera)
{
    /**
     * Render rectangle.
     * @name FORGE.RenderParams#_rectangle
     * @type {FORGE.Rectangle}
     * @private
     */
    this._rectangle = rectangle || null;

    /**
     * Render camera.
     * @name FORGE.RenderParams#_camera
     * @type {THREE.PerspectiveCamera}
     * @private
     */
    this._camera = camera || null;
};

FORGE.RenderParams.prototype.constructor = FORGE.RenderParams;

/**
 * Get rectangle.
 * @name FORGE.RenderParams#rectangle
 * @type {FORGE.Rectangle}
 */
Object.defineProperty(FORGE.RenderParams.prototype, "rectangle",
{
    /** @this {FORGE.RenderParams} */
    get: function()
    {
        return this._rectangle;
    }
});

/**
 * Get camera.
 * @name FORGE.RenderParams#camera
 * @type {THREE.PerspectiveCamera}
 */
Object.defineProperty(FORGE.RenderParams.prototype, "camera",
{
    /** @this {FORGE.RenderParams} */
    get: function()
    {
        return this._camera;
    }
});

/**
 * FORGE.BackgroundRenderer
 * BackgroundRenderer class.
 *
 * @constructor FORGE.BackgroundRenderer
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {THREE.WebGLRenderTarget} target - render target
 * @param {SceneMediaOptionsConfig} options - the options for the cubemap
 * @param {string=} type - The type of the object as long as many other object inherits from this one.
 * @extends {FORGE.BaseObject}
 */
FORGE.BackgroundRenderer = function(viewer, target, options, type)
{
    /**
     * @name FORGE.BackgroundRenderer#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * @name FORGE.BackgroundRenderer#_canvas
     * @type {FORGE.Canvas}
     * @private
     */
    this._canvas = null;

    /**
     * The mesh (cube) the video is on.
     * @type {THREE.Mesh}
     * @private
     */
    this._mesh = null;

    /**
     * @name FORGE.BackgroundRenderer#_scene
     * @type {THREE.Scene}
     * @private
     */
    this._scene = null;

    /**
     * @name FORGE.BackgroundRenderer#_camera
     * @type {THREE.Camera}
     * @private
     */
    this._camera = null;

    /**
     * @name FORGE.BackgroundRenderer#_frustum
     * @type {THREE.Frustum}
     * @private
     */
    this._frustum = null;

    /**
     * Media format (cubemap, equi...)
     * @type {string}
     * @private
     */
    this._mediaFormat = options.mediaFormat || FORGE.MediaFormat.CUBE;

    /**
     * @name FORGE.BackgroundRenderer#_renderTarget
     * @type {THREE.WebGLRenderTarget}
     * @private
     */
    this._renderTarget = target || null;

    FORGE.BaseObject.call(this, type || "BackgroundRenderer");

    this._boot();
};

FORGE.BackgroundRenderer.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.BackgroundRenderer.prototype.constructor = FORGE.BackgroundRenderer;

/**
 * Init routine.
 * @method FORGE.BackgroundRenderer#_boot
 * @private
 */
FORGE.BackgroundRenderer.prototype._boot = function()
{
    this._scene = new THREE.Scene();
    this._scene.name = "Background scene";

    if (this._renderTarget === null)
    {
        var width = this._viewer.renderer.canvasResolution.width;
        var height = this._viewer.renderer.canvasResolution.height;

        var rtParams =
        {
            minFilter: THREE.LinearFilter,
            magFilter: THREE.LinearFilter,
            format: THREE.RGBAFormat,
            stencilBuffer: false
        };

        this._renderTarget = new THREE.WebGLRenderTarget(width, height, rtParams);
    }

    this._frustum = new THREE.Frustum();
};

/**
 * Abstract method that should be implemented by subclass.
 * @method FORGE.BackgroundRenderer#_setDisplayObject
 * @param {FORGE.DisplayObject} displayObject - The display object to set.
 * @private
 */
FORGE.BackgroundRenderer.prototype._setDisplayObject = function(displayObject)
{
    this.log(displayObject); //@closure
    throw "Please implement " + this._className + "::_setDisplayObject";
};

/**
 * Abstract method that should be implemented by subclass.
 * @method FORGE.BackgroundRenderer#_clear
 * @private
 */
FORGE.BackgroundRenderer.prototype._clear = function()
{
    throw "Please implement " + this._className + "::_clear";
};

/**
 * Update texture if needed (video only).
 * @method FORGE.BackgroundRenderer#_updateTexture
 * @private
 */
FORGE.BackgroundRenderer.prototype._updateTexture = function()
{
    // doesn't refresh when there is no texture or texture container and when a video as WebGL texture is paused
    if (this._texture === null || this._textureCanvas === null ||
        this._textureContext === null || typeof this._displayObject === "undefined" || this._displayObject === null ||
        this._displayObject.element === null || (FORGE.Utils.isTypeOf(this._displayObject, ["VideoHTML5", "VideoDash"]) === true && this._displayObject.playing === false))
    {
        return;
    }

    var video = this._displayObject.element;
    if (video instanceof HTMLVideoElement && video.readyState === HTMLMediaElement.HAVE_ENOUGH_DATA)
    {
        if (this._textureContext)
        {
            this._textureContext.drawImage(video,
                0, 0, video.videoWidth, video.videoHeight,
                0, 0, this._textureCanvas.width, this._textureCanvas.height);
            this._texture.needsUpdate = true;
            this.log("texture update done");
        }
    }
};

/**
 * Update after view change
 * @todo change name of this method to be more generic (used by init and )
 * Should be overriden by subclass
 * @method FORGE.BackgroundRenderer#updateAfterViewChange
 */
FORGE.BackgroundRenderer.prototype.updateAfterViewChange = function()
{
    throw new Error(this._className + "::updateAfterViewChange not implemented");
};

/**
 * Update size (resolution)
 * @method FORGE.BackgroundRenderer#setSize
 * @param {FORGE.Size} size - size [px]
 */
FORGE.BackgroundRenderer.prototype.setSize = function(size)
{
    if (this.renderTarget !== null)
    {
        this.renderTarget.setSize(size.width, size.height);
    }
};

/**
 * Render routine.
 * @method FORGE.BackgroundRenderer#render
 * @param {THREE.PerspectiveCamera} camera - perspective camera with mesh rendering, N/A with shader rendering (null)
 */
FORGE.BackgroundRenderer.prototype.render = function(camera)
{
    if (this._viewer.renderer === null || this._renderTarget === null)
    {
        return;
    }

    this._updateTexture();

    var renderCamera = (camera !== null) ? camera : this._camera;
    this._frustum.setFromMatrix( new THREE.Matrix4().multiplyMatrices( renderCamera.projectionMatrix, renderCamera.matrixWorldInverse ) );
    this._viewer.renderer.webGLRenderer.render ( this._scene, renderCamera, this._renderTarget, true );
};

/**
 * Check if some 3D object is interesecting the rendering frustum.
 * @method FORGE.BackgroundRenderer#isObjectInFrustum
 * @param {THREE.Object3D} object - 3D object
 */
FORGE.BackgroundRenderer.prototype.isObjectInFrustum = function(object)
{
    return this._frustum.intersectsObject(object);
};

/**
 * Check if some 3D object is in the scene
 * @method FORGE.BackgroundRenderer#isObjectInScene
 * @param {THREE.Object3D} object - 3D object
 */
FORGE.BackgroundRenderer.prototype.isObjectInScene = function(object)
{
    return this._scene.getObjectByName(object.name) !== undefined;
};

/**
 * Update routine.
 * @method FORGE.BackgroundRenderer#update
 */
FORGE.BackgroundRenderer.prototype.update = function()
{
    if (this._mesh === null || !(this._mesh.material instanceof THREE.ShaderMaterial))
    {
        this._viewer.renderer.view.current.updateUniforms();
        return;
    }

    var resolution = this._viewer.renderer.displayResolution;

    if (this._mesh.material.uniforms.hasOwnProperty("tViewportResolution"))
    {
        this._mesh.material.uniforms.tViewportResolution.value = new THREE.Vector2(resolution.width, resolution.height);
    }

    if (this._mesh.material.uniforms.hasOwnProperty("tViewportResolutionRatio"))
    {
        this._mesh.material.uniforms.tViewportResolutionRatio.value = resolution.ratio;
    }

    if (this._mesh.material.uniforms.hasOwnProperty("tModelViewMatrixInverse"))
    {
        this._mesh.material.uniforms.tModelViewMatrixInverse.value = this._viewer.renderer.camera.modelViewInverse;
    }

    this._viewer.renderer.view.current.updateUniforms(this._mesh.material.uniforms);
};

/**
 * Destroy sequence.
 * @method FORGE.BackgroundRenderer#destroy
 */
FORGE.BackgroundRenderer.prototype.destroy = function()
{
    this._camera = null;
    this._frustum = null;

    if (this._renderTarget !== null)
    {
        this._renderTarget.dispose();
        this._renderTarget = null;
    }

    while (this._scene.children.length > 0)
    {
        var mesh = this._scene.children.pop();

        if (mesh.geometry !== null)
        {
            mesh.geometry.dispose();
            mesh.geometry = null;
        }

        this._scene.remove(mesh);
    }

    this._scene = null;
    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get texture size.
 * @name FORGE.BackgroundRenderer#textureSize
 * @type {FORGE.Size}
 */
Object.defineProperty(FORGE.BackgroundRenderer.prototype, "textureSize",
{
    /** @this {FORGE.BackgroundRenderer} */
    get: function()
    {
        if (this._texture === null || typeof this._texture.image === "undefined")
        {
            return null;
        }

        return new FORGE.Size(this._texture.image.width, this._texture.image.height);
    }
});

/**
 * Get background render target.
 * @name FORGE.BackgroundRenderer#renderTarget
 * @type {THREE.WebGLRenderTarget}
 */
Object.defineProperty(FORGE.BackgroundRenderer.prototype, "renderTarget",
{
    /** @this {FORGE.BackgroundRenderer} */
    get: function()
    {
        return this._renderTarget;
    }
});

/**
 * Get/Set background renderer displayObject.
 * @name FORGE.BackgroundRenderer#displayObject
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundRenderer.prototype, "displayObject",
{
    /** @this {FORGE.BackgroundRenderer} */
    get: function()
    {
        return this._displayObject;
    },
    /** @this {FORGE.BackgroundRenderer} */
    set: function(value)
    {
        if (value === null)
        {
            this._clear();
        }
        else
        {
            this._setDisplayObject(value);
        }
    }
});

/**
 * Get background scene.
 * @name FORGE.BackgroundRenderer#scene
 * @type {THREE.Scene}
 */
Object.defineProperty(FORGE.BackgroundRenderer.prototype, "scene",
{
    /** @this {FORGE.BackgroundRenderer} */
    get: function()
    {
        return this._scene;
    }
});

/**
 * Get camera frustum.
 * @name FORGE.BackgroundRenderer#frustum
 * @type {THREE.Frustum}
 */
Object.defineProperty(FORGE.BackgroundRenderer.prototype, "frustum",
{
    /** @this {FORGE.BackgroundRenderer} */
    get: function()
    {
        return this._frustum;
    }
});

/**
 * @namespace {Object} FORGE.BackgroundType
 */
FORGE.BackgroundType = {};

/**
 * @name FORGE.BackgroundType.UNDEFINED
 * @type {string}
 * @const
 */
FORGE.BackgroundType.UNDEFINED = "undefined";

/**
 * @name FORGE.BackgroundType.SHADER
 * @type {string}
 * @const
 */
FORGE.BackgroundType.SHADER = "shader";

/**
 * @name FORGE.BackgroundType.MESH
 * @type {string}
 * @const
 */
FORGE.BackgroundType.MESH = "mesh";

/**
 * @name FORGE.BackgroundType.PYRAMID
 * @type {string}
 * @const
 */
FORGE.BackgroundType.PYRAMID = "pyramid";

/**
 * FORGE.BackgroundMeshRenderer
 * BackgroundMeshRenderer class.
 *
 * @constructor FORGE.BackgroundMeshRenderer
 * @extends {FORGE.BackgroundRenderer}
 *
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {THREE.WebGLRenderTarget} target - render target
 * @param {SceneMediaOptionsConfig} options - the options for the cubemap
 */
FORGE.BackgroundMeshRenderer = function(viewer, target, options)
{
    /**
     * Display object (image, canvas or video)
     * @type {FORGE.DisplayObject}
     * @private
     */
    this._displayObject = null;

    /**
     * Texture used for video rendering
     * @type {THREE.Texture}
     * @private
     */
    this._texture = null;

    /**
     * Texture canvas used for video rendering
     * @type {Element|HTMLCanvasElement}
     * @private
     */
    this._textureCanvas = null;

    /**
     * Texture context associated with texture canvas
     * @type {CanvasRenderingContext2D}
     * @private
     */
    this._textureContext = null;

    /**
     * Media type
     * @type {string}
     * @private
     */
    this._mediaType = options.type || FORGE.MediaType.GRID;

    /**
     * Media vertical fov (radians)
     * @type {number}
     * @private
     */
    this._mediaVFov = options.verticalFov || 90;

    /**
     * Grid color
     * @type {string}
     * @private
     */
    this._gridColor = options.color || "#ffffff";

    /**
     * The layout of the faces in the texture. There are six faces to specify:
     * Right (R), Left (L), Up (U), Down (D), Front (F), Back (B). The default
     * layout is the Facebook one, with RLUDFB.
     * @type {string}
     * @private
     */
    this._layout = options.order || "RLUDFB";

    /**
     * The number of horizontal faces and vertical ones in the media.
     * @type {THREE.Vector2}
     * @private
     */
    this._faces = new THREE.Vector2(0, 0);

    /**
     * The size of a tile (width = height)
     * @type {number}
     * @private
     */
    this._tile = options.tile || 512;

    /**
     * The size of the cube.
     * @type {number}
     * @private
     */
    this._size = 0;

    /**
     * The number of subdivision of a face, per direction. For example, if the
     * subdivision is 4, the cube would be composed of 4 * 4 quads per face (in
     * reality it is 4 * 4 * 2 triangles).
     * @type {number}
     * @private
     */
    this._subdivision = 0;

    /**
     * When source is a video, a reduction factor can be set to improve perf by lowering quality
     * @type {number}
     * @private
     */
    this._videoReductionFactor = 1;

    FORGE.BackgroundRenderer.call(this, viewer, target, options, "BackgroundMeshRenderer");
};

FORGE.BackgroundMeshRenderer.prototype = Object.create(FORGE.BackgroundRenderer.prototype);
FORGE.BackgroundMeshRenderer.prototype.constructor = FORGE.BackgroundMeshRenderer;

/**
 * Default texture name
 * @type {string}
 */
FORGE.BackgroundMeshRenderer.DEFAULT_TEXTURE_NAME = "Default Texture";

/**
 * Init routine.
 * @method FORGE.BackgroundMeshRenderer#_boot
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._boot = function()
{
    FORGE.BackgroundRenderer.prototype._boot.call(this);

    // Set perspective camera
    this._camera = this._viewer.renderer.camera.main;

    this._size = 2 * FORGE.RenderManager.DEPTH_FAR;

    this._subdivision = 32;
};

/**
 * Set display object.
 * @method FORGE.BackgroundMeshRenderer#_setDisplayObject
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._setDisplayObject = function(displayObject)
{
    if (this._mesh === null)
    {
        this._updateInternals();
    }

    this._displayObject = displayObject;

    if (FORGE.Utils.isTypeOf(displayObject, "Image"))
    {
        this._texture = new THREE.Texture();
        this._texture.image = displayObject.element;
    }
    else if (FORGE.Utils.isTypeOf(displayObject, "Canvas"))
    {
        this._texture = new THREE.Texture();
        this._texture.image = displayObject.element;
    }
    else if (FORGE.Utils.isTypeOf(displayObject, ["VideoHTML5", "VideoDash"]))
    {
        // Evil hack from Hell
        // Reduce texture size for big videos on safari
        if (FORGE.Device.browser.toLowerCase() === "safari" && displayObject.originalHeight > 1440)
        {
            this._videoReductionFactor = 2;
        }

        this._textureCanvas = document.createElement("canvas");
        this._textureCanvas.width = displayObject.originalWidth / this._videoReductionFactor;
        this._textureCanvas.height = displayObject.originalHeight / this._videoReductionFactor;
        this._textureContext = this._textureCanvas.getContext("2d");
        this._texture = new THREE.Texture(this._textureCanvas);
    }
    else
    {
        throw "Wrong type of display object " + displayObject.type;
    }

    this._texture.format = THREE.RGBAFormat;
    this._texture.mapping = THREE.Texture.DEFAULT_MAPPING;

    if (typeof this._mesh.material.uniforms.tTextureRatio !== "undefined")
    {
        this._mesh.material.uniforms.tTextureRatio.value = this._texture.image.width / this._texture.image.height;
    }

    this._texture.generateMipmaps = false;
    this._texture.minFilter = THREE.LinearFilter;

    if (this._mediaFormat === FORGE.MediaFormat.FLAT)
    {
        if (FORGE.Math.isPowerOfTwo(displayObject.width) && FORGE.Math.isPowerOfTwo(displayObject.height))
        {
            // Enable mipmaps for flat rendering to avoid aliasing
            this._texture.generateMipmaps = true;
            this._texture.minFilter = THREE.LinearMipMapLinearFilter;
        }

        // Replace geometry with a rectangle matching texture ratio
        // First release previous default geometry
        if (this._mesh.geometry != null)
        {
            this._mesh.geometry.dispose();
            this._mesh.geometry = null;
        }

        // Compute camera fov limits depending on geometry size and position and on display object size
        var canvasHeight = this._viewer.container.height;
        var canvasWidth = this._viewer.container.width;
        var canvasRatio = canvasWidth / canvasHeight;

        var texHeight = displayObject.height;
        var texRatio = displayObject.width / displayObject.height;

        var geomWidth = this._size;
        var geomHeight = Math.round(geomWidth / texRatio);
        var geomDepth = geomHeight / (2 * Math.tan(0.5 * this._mediaVFov));

        var geometry = new THREE.PlaneBufferGeometry(geomWidth, geomHeight);
        this._mesh.geometry = geometry;
        this._mesh.position.set(0, 0, -geomDepth);

        var fovMax = FORGE.Math.radToDeg(2 * Math.atan(0.5 * geomHeight / geomDepth));
        var fovMin = FORGE.Math.radToDeg(2 * Math.atan((0.5 * geomHeight / geomDepth) * (canvasHeight / texHeight)));

        this.log("Flat rendering boundaries [" + fovMin.toFixed() + ", " + fovMax.toFixed() + "]");
    }

    this._texture.needsUpdate = true;

    this._mesh.material.wireframe = false;

    if (this._texture.image !== null)
    {
        this._faces.x = this._texture.image.width / this._tile;
        this._faces.y = this._texture.image.height / this._tile;
    }

    this._mesh.onBeforeRender = this._onBeforeRender.bind(this);

    var uvMap = this._setUVMapping();
    if (uvMap !== null)
    {
        this._mesh.geometry.attributes.uv.set(uvMap);
    }
};

/**
 * Before render handler
 * @method FORGE.BackgroundMeshRenderer#_onBeforeRender
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._onBeforeRender = function()
{
    if (typeof this._mesh.material.uniforms === "undefined")
    {
        return;
    }

    if (this._mesh.material.uniforms.hasOwnProperty("tTexture"))
    {
        this._mesh.material.uniforms.tTexture.value = this._texture;
    }

    if (this._mesh.material.uniforms.hasOwnProperty("tOpacity"))
    {
        this._mesh.material.uniforms.tOpacity.value = 1.0;
    }
};

/**
 * Return an array containing each coord for the uv mapping of the cube geometry
 * @method FORGE.BackgroundMeshRenderer#_setUVMappingCube
 * @return {Float32Array} The array containing the UVs
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._setUVMappingCube = function()
{
    // the final array of uv coord for mapping
    var uvMap = new Float32Array((this._subdivision + 1) * (this._subdivision + 1) * 6 * 2);

    // iterator accross the uv coord
    var it = FORGE.Utils.arrayKeys(uvMap);

    // layout of the texture
    var layout = this._layout.split("");

    // the width/height of a division in uv coord
    // the 1.01 is the default value for expand_coef in the ffmpeg filter
    // see https://github.com/facebook/transform/blob/04ec220a5c066a75d87f9e463b219262f7527421/vf_transform.c#L961
    var u = (1 / 1.01) * ((1 / this._faces.x) / this._subdivision);
    var v = (1 / 1.01) * ((1 / this._faces.y) / this._subdivision);

    // tiny offsets are for compensating the expand_coef of the ffmpeg filter
    // u tiny offset
    var uto = 0.005 * (1 / this._faces.x);
    // v tiny offset
    var vto = 0.005 * (1 / this._faces.y);

    /**
     * Apply the correct UV to the uv map
     * @param  {number} idx
     * @param  {number} x
     * @param  {number} y
     * @param  {number} sub
     * @param  {boolean=} upOrDown
     */
    function applyUVMapForFace(idx, x, y, sub, upOrDown)
    {
        if (idx === -1)
        {
            throw "Unknown face for cube mapping.";
        }

        // iterator
        var i, j, ii, jj;

        // u offset, where is it in the layout, change for each face
        var uo = (idx % x) / x;
        // v offset, where is it in the layout, change for each face
        var vo = 1 - ((1 + parseInt(idx / x, 10)) / y);

        // not the same inversion if up or down
        if (upOrDown)
        {
            // vertical
            for (i = 0, ii = sub; i <= ii; i++)
            {
                // horizontal
                for (j = 0, jj = sub; j <= jj; j++)
                {
                    // u
                    uvMap[it.next().value] = uto + u * j + uo;
                    // v
                    uvMap[it.next().value] = vto + v * i + vo;
                }
            }
        }
        else
        {
            // vertical
            for (i = 0, ii = sub; i <= ii; ii--)
            {
                // horizontal
                for (j = 0, jj = sub; j <= jj; jj--)
                {
                    // u
                    uvMap[it.next().value] = uto + u * jj + uo;
                    // v
                    uvMap[it.next().value] = vto + v * ii + vo;
                }
            }
        }
    }

    applyUVMapForFace(layout.indexOf("R"), this._faces.x, this._faces.y, this._subdivision);
    applyUVMapForFace(layout.indexOf("L"), this._faces.x, this._faces.y, this._subdivision);
    applyUVMapForFace(layout.indexOf("U"), this._faces.x, this._faces.y, this._subdivision, true);
    applyUVMapForFace(layout.indexOf("D"), this._faces.x, this._faces.y, this._subdivision, true);
    applyUVMapForFace(layout.indexOf("B"), this._faces.x, this._faces.y, this._subdivision);
    applyUVMapForFace(layout.indexOf("F"), this._faces.x, this._faces.y, this._subdivision);

    return uvMap;
};

/**
 * Return an array containing each coord for the uv mapping of the sphere geometry
 * @method FORGE.BackgroundMeshRenderer#_setUVMappingSphere
 * @return {Float32Array} The array containing the UVs
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._setUVMappingSphere = function()
{
    // the final array of uv coord for mapping
    var uvMap = new Float32Array(this._mesh.geometry.attributes.uv.array.byteLength / 4);

    // iterator accross the uv coord
    var it = uvMap.keys();

    var div = this._subdivision;
    var d = 1 / div;

    var ix, iy;

    for (iy = 0; iy <= div; iy++)
    {
        var v = iy * d;

        for (ix = 0; ix <= div; ix++)
        {
            var u = ix * d;
            uvMap[it.next().value] = 1 - u;
            uvMap[it.next().value] = 1 - v;
        }
    }

    return uvMap;
};

/**
 * Return an array containing each coord for the uv mapping
 * @method FORGE.BackgroundMeshRenderer#_setUVMapping
 * @return {Float32Array} The array containing the UVs
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._setUVMapping = function()
{
    // BoxGeometry uv are not ok, compute them in our own way
    if (this._mesh.geometry instanceof THREE.BoxBufferGeometry)
    {
        return this._setUVMappingCube();
    }

    // Sphere uv are facing backward
    if (this._mesh.geometry instanceof THREE.SphereBufferGeometry)
    {
        return this._setUVMappingSphere();
    }

    return null;
};

/**
 * Clear background.
 * @method FORGE.BackgroundMeshRenderer#_clear
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._clear = function()
{
    // Draw to clear screen, then clear display object / texture
    this._viewer.renderer.webGLRenderer.clearColor();
};

/**
 * Add quadrilateral coordinates as geometry attribute
 * Used to draw wireframe
 * @method FORGE.BackgroundMeshRenderer#_addQuadrilateralCoordsAttribute
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._addQuadrilateralCoordsAttribute = function()
{
    if (this._mesh === null || typeof this._mesh.geometry === "undefined")
    {
        return;
    }

    // Quadrilateral is a 2 components system, reduce vertices array size from 3:2
    var size = this._mesh.geometry.attributes.position.array.length * 2 / 3;
    var quadri = new Int8Array(size);
    var it = FORGE.Utils.arrayKeys(quadri);

    var qa = new THREE.Vector2(1, 1);
    var qb = new THREE.Vector2(1, -1);
    var qc = new THREE.Vector2(-1, 1);
    var qd = new THREE.Vector2(-1, -1);

    var ipd = this._subdivision + 1; // indices per dimension
    for (var f = 0; f < 6; f++)
    {
        for (var r=0; r < ipd; r++)
        {
            var q0, q1;

            if (r & 1)
            {
                q0 = qa;
                q1 = qb;
            }
            else
            {
                q0 = qc;
                q1 = qd;
            }

            for (var c=0; c < ipd; c++)
            {
                if (c & 1)
                {
                    quadri[it.next().value] = q1.x;
                    quadri[it.next().value] = q1.y;
                }
                else
                {
                    quadri[it.next().value] = q0.x;
                    quadri[it.next().value] = q0.y;
                }
            }
        }
    }

    this._mesh.geometry.addAttribute("quadrilateralCoords", new THREE.BufferAttribute(quadri, 2));
};

/**
 * Update internals
 * @method FORGE.BackgroundMeshRenderer#_updateInternals
 * @private
 */
FORGE.BackgroundMeshRenderer.prototype._updateInternals = function()
{
    if (this._viewer.renderer.view.current === null)
    {
        this.log("Background renderer cannot update internals without a defined view");
        return;
    }

    var shader;
    if (this._mediaType === FORGE.MediaType.GRID)
    {
        shader = FORGE.Utils.clone(this._viewer.renderer.view.current.shaderWTS).wireframe;
        this.log("Media " + this._mediaType + ", use wireframe shader");
        this._subdivision = 8;
    }
    else
    {
        shader = FORGE.Utils.clone(this._viewer.renderer.view.current.shaderWTS).mapping;
        this.log("Media " + this._mediaType + ", use mapping shader");
    }

    var vertexShader = FORGE.ShaderLib.parseIncludes(shader.vertexShader);
    var fragmentShader = FORGE.ShaderLib.parseIncludes(shader.fragmentShader);

    var material = new THREE.RawShaderMaterial(
    {
        fragmentShader: fragmentShader,
        vertexShader: vertexShader,
        uniforms: /** @type {FORGEUniform} */ (shader.uniforms),
        name: "BackgroundMeshMaterial",
        transparent: true,
        side: THREE.BackSide
    });
    var geometry = null;

    if (this._texture !== null)
    {
        if (this._mesh.material.uniforms.hasOwnProperty("tTexture"))
        {
            material.uniforms.tTexture.value = this._texture;

        }

        if (this._mesh.material.uniforms.hasOwnProperty("tTextureRatio"))
        {
            material.uniforms.tTextureRatio.value = this._texture.image.width / this._texture.image.height;
        }
    }

    if (this._mediaType === FORGE.MediaType.GRID)
    {
        material.uniforms.tColor.value = new THREE.Color(this._gridColor);
        material.blending = THREE.CustomBlending;
        material.blendEquationAlpha = THREE.AddEquation;
        material.blendSrcAlpha = THREE.SrcAlphaFactor;
        material.blendDstAlpha = THREE.OneMinusSrcAlphaFactor;
    }

    if (this._scene.children.length === 0)
    {
        if (this._mediaFormat === FORGE.MediaFormat.EQUIRECTANGULAR)
        {
            // Sphere mapping of equirectangular texture becomes acceptable with subdivision greater or equal to 64
            this._subdivision = 64;
            geometry = new THREE.SphereBufferGeometry(this._size, this._subdivision, this._subdivision);
            this.log("Create sphere geometry");
        }
        else if (this._mediaFormat === FORGE.MediaFormat.FLAT)
        {
            geometry = new THREE.PlaneBufferGeometry(this._size, this._size);
            this.log("Create plane geometry");
        }
        else
        {
            geometry = new THREE.BoxBufferGeometry(this._size, this._size, this._size, this._subdivision, this._subdivision, this._subdivision);
            this.log("Create box geometry");
        }

        this._mesh = new THREE.Mesh(geometry, material);

        if (this._mediaType === FORGE.MediaType.GRID)
        {
            this._addQuadrilateralCoordsAttribute();
        }

        if (this._mediaFormat === FORGE.MediaFormat.FLAT)
        {
            this._mesh.position.set(0, 0, -this._size * 0.5);
            this._mesh.material.side = THREE.FrontSide;
        }

        // Equirectangular mapping on a sphere needs a yaw shift of PI/2 to set front at center of the texture
        if (this._mediaFormat === FORGE.MediaFormat.EQUIRECTANGULAR)
        {
            this._mesh.rotation.set(0, Math.PI / 2, 0, "YXZ");
        }

        this._scene.add(this._mesh);
    }
    else
    {
        this._mesh = /** @type {THREE.Mesh} */ (this._scene.children[0]);
        this._mesh.material.dispose();
    }

    this._mesh.material = material;
    material.needsUpdate = true;
};

/**
 * Update after view change
 * @method FORGE.BackgroundMeshRenderer#updateAfterViewChange
 */
FORGE.BackgroundMeshRenderer.prototype.updateAfterViewChange = function()
{
    this._updateInternals();

    if (typeof this._mesh.material.uniforms.tTextureRatio !== "undefined")
    {
        this._mesh.material.uniforms.tTextureRatio.value = this._texture.image.width / this._texture.image.height;
    }
};

/**
 * Render routine.
 * Do preliminary job of specific background renderer, then summon superclass method
 * @method FORGE.BackgroundMeshRenderer#render
 * @param {THREE.PerspectiveCamera} camera - perspective camera
 */
FORGE.BackgroundMeshRenderer.prototype.render = function(camera)
{
    if (this._viewer.renderer === null)
    {
        return;
    }

    FORGE.BackgroundRenderer.prototype.render.call(this, camera);
};

/**
 * Destroy sequence.
 * @method FORGE.BackgroundMeshRenderer#destroy
 */
FORGE.BackgroundMeshRenderer.prototype.destroy = function()
{
    this._clear();

    this._displayObject = null;
    this._textureCanvas = null;
    this._textureContext = null;

    if (this._mesh !== null)
    {
        this.log ("Destroy mesh (dispose geometry and material)");
        if (this._mesh.geometry !== null)
        {
            this._mesh.geometry.dispose();
            this._mesh.geometry = null;
        }

        if (this._mesh.material !== null)
        {
            this._mesh.material.dispose();
            this._mesh.material = null;
        }

        this._mesh = null;
    }

    if (this._texture !== null)
    {
        this._texture.dispose();
        this._texture = null;
    }

    FORGE.BackgroundRenderer.prototype.destroy.call(this);
};

/**
 * Get background renderer texture.
 * @name FORGE.BackgroundMeshRenderer#texture
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundMeshRenderer.prototype, "texture",
{
    /** @this {FORGE.BackgroundMeshRenderer} */
    get: function()
    {
        return this._texture;
    }
});


/**
 * FORGE.BackgroundPyramidRenderer
 * BackgroundPyramidRenderer class.
 *
 * @constructor FORGE.BackgroundPyramidRenderer
 * @extends {FORGE.BackgroundRenderer}
 *
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {THREE.WebGLRenderTarget} target - render target
 * @param {SceneMediaConfig} config - media config
 */
FORGE.BackgroundPyramidRenderer = function(viewer, target, config)
{
    /**
     * Input scene and media config
     * @name FORGE.BackgroundPyramidRenderer#_config
     * @type {?SceneMediaConfig}
     * @private
     */
    this._config = config;

    /**
     * Current level of the pyramid
     * @type {number}
     * @private
     */
    this._level = 0;

    /**
     * The size of the cube.
     * @type {number}
     * @private
     */
    this._size = 0;

    /**
     * Texture store
     * @type {?FORGE.MediaStore}
     * @private
     */
    this._textureStore = null;

    /**
     * Cache of tiles
     * @type {?Object}
     * @private
     */
    this._tileCache = null;

    /**
     * Number of pixels at current level
     * @type {number}
     * @private
     */
    this._levelPixels = 0;

    /**
     * Minimum fov computed with max level
     * @type {number}
     * @private
     */
    this._fovMin = 0;

    /**
     * Number of tiles for all levels
     * @type {Array<TilesOnLevel>}
     * @private
     */
    this._tilesLevel = null;

    /**
     * List of renderered tiles
     * @type {?Array<FORGE.Tile>}
     * @private
     */
    this._renderList = null;

    /**
     * List of tiles in renderered tiles neighborhood
     * @type {?Array<FORGE.Tile>}
     * @private
     */
    this._renderNeighborList = null;

    /**
     * Enable tile clearing policy
     * @type {boolean}
     * @private
     */
    this._clearTilesEnabled = true;

    /**
     * Number of pixels at current level presented in human readable format
     * @type {string}
     * @private
     */
    this._levelPixelsHumanReadable = "";

    FORGE.BackgroundRenderer.call(this, viewer, target, config, "BackgroundPyramidRenderer");
};

FORGE.BackgroundPyramidRenderer.prototype = Object.create(FORGE.BackgroundRenderer.prototype);
FORGE.BackgroundPyramidRenderer.prototype.constructor = FORGE.BackgroundPyramidRenderer;

/**
 * Max allowed time since a tile has been created before discarding it
 * @type {number}
 */
FORGE.BackgroundPyramidRenderer.MAX_ALLOWED_TIME_SINCE_CREATION_MS = 2000;

/**
 * Max allowed time since a tile has been displayed before discarding it
 * @type {number}
 */
FORGE.BackgroundPyramidRenderer.MAX_ALLOWED_TIME_SINCE_DISPLAY_MS = 3000;

/**
 * Boot routine.
 * @method FORGE.BackgroundPyramidRenderer#_boot
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._boot = function()
{
    FORGE.BackgroundRenderer.prototype._boot.call(this);

    this.log("boot");

    this._parseConfig(this._config);

    this._size = 2 * FORGE.RenderManager.DEPTH_FAR;

    this._tileCache = {};
    this._textureStore = this._viewer.story.scene.media.store;

    this._camera = this._viewer.renderer.camera.main;
    this._viewer.camera.onChange.add(this._onCameraChange, this);

    if (typeof this._config.preview !== "undefined")
    {
        this._createPreview();
    }

    for (var f = 0; f < 6; f++)
    {
        var face = Object.keys(FORGE.MediaStore.CUBE_FACE_CONFIG)[f];

        for (var y = 0, ty = this.nbTilesPerAxis(0, "y"); y < ty; y++)
        {
            for (var x = 0, tx = this.nbTilesPerAxis(0, "x"); x < tx; x++)
            {
                this.getTile(null, 0, face, x, y, "pyramid init");
            }
        }
    }
};

/**
 * Parse configuration.
 * @method FORGE.BackgroundPyramidRenderer#_parseConfig
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._parseConfig = function(config)
{
    // Store all tiles number per level for quick access
    this._tilesLevel = [];
    if (typeof this._config.source.levels !== "undefined")
    {
        var level;

        for (var l = 0, levels = this._config.source.levels.length; l < levels; l++)
        {
            level = this._config.source.levels[l];
            this._tilesLevel.push({
                x: level.width / level.tile,
                y: level.height / level.tile
            });
        }
    }

    // Set min fov to be used to reach max level of resolution
    this._fovMin = 1.01 * FORGE.Math.degToRad(this._pyramidLevelToCameraFov(config.source.levels.length - 1));
};

/**
 * Create the preview (sub zero level)
 * @method FORGE.BackgroundPyramidRenderer.prototype._createPreview
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._createPreview = function()
{
    for (var f = 0; f < 6; f++)
    {
        var face = Object.keys(FORGE.MediaStore.CUBE_FACE_CONFIG)[f];

        this.getTile(null, FORGE.Tile.PREVIEW, face, 0, 0, "pyramid preview");
    }

    if (typeof(this._tileCache[FORGE.Tile.PREVIEW]) !== "undefined")
    {
        this._tileCache[FORGE.Tile.PREVIEW].forEach(function(tile)
        {
            this._scene.add(tile);
        }.bind(this));
    }
};

/**
 * Clear routine.
 * @method FORGE.BackgroundPyramidRenderer#_clear
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._clear = function()
{
    // Draw to clear screen, then clear display object / texture
    this._viewer.renderer.webGLRenderer.clearColor();
};


/**
 * Update after view change
 * @method FORGE.BackgroundPyramidRenderer#updateAfterViewChange
 */
FORGE.BackgroundPyramidRenderer.prototype.updateAfterViewChange = function()
{
    if (this._viewer.view.type !== FORGE.ViewType.RECTILINEAR)
    {
        this.warn("Only Rectilinear view is supported for multi resolution scene");
        this._viewer.view.type = FORGE.ViewType.RECTILINEAR;
    }
};

/**
 * Get pyramid level for a given fov
 * @method FORGE.BackgroundPyramidRenderer#_cameraFovToPyramidLevel
 * @param {number} fov camera field of view [degrees]
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._cameraFovToPyramidLevel = function(fov)
{
    // compute the optimal number of tiles on the y axis for this fov
    var tiles = Math.floor(180 / fov);

    // check the nearest level to this optimal number
    var level = this._tilesLevel.findIndex(function(lvl)
    {
        return lvl.y >= tiles;
    });

    if (level === -1)
    {
        level = this._tilesLevel.length - 1;
    }

    return Math.max(0, level);
};

/**
 * Get fov threshold from a pyramid level
 * @method FORGE.BackgroundPyramidRenderer#_pyramidLevelToCameraFov
 * @param {number} level pyramid level
 * @return {number} fov camera field of view [degrees]
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._pyramidLevelToCameraFov = function(level)
{
    var lvl = this._tilesLevel[level];
    var fov = 90 / lvl.y;
    return fov;
};

/**
 * Camera change callback.
 * @method FORGE.BackgroundPyramidRenderer#_onCameraChange
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._onCameraChange = function(event)
{
    var camera = event.emitter;
    var fov = camera.fov;

    var level = this._cameraFovToPyramidLevel(fov);
    if (this._level !== level)
    {
        this.selectLevel(level);
    }
};

/**
 * Tile destroyed event handler
 * @param {FORGE.Event} event - event
 * @method FORGE.BackgroundPyramidRenderer#_onTileDestroyed
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._onTileDestroyed = function(event)
{
    var tile = event.emitter;
    tile.onDestroy.remove(this._onTileDestroyed, this);
    this._tileCache[tile.level].delete(tile.name);
    this._scene.remove(tile);
};

/**
 * Get XnYn normalized tile coords at a given level
 * @method FORGE.BackgroundPyramidRenderer#_levelXYToXnYn
 * @param {number} level - pyramid level
 * @param {THREE.Vector2} XY - coordinates
 * @return {THREE.Vector2} normalized tile coordinates vector
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._levelXYToXnYn = function(level, XY)
{
    var tx = this.nbTilesPerAxis(level, "x"),
        ty = this.nbTilesPerAxis(level, "y");

    return XY.divide(new THREE.Vector2(tx, ty));
};

/**
 * Get XY tile coords at a given level depending on normalized coordinates
 * @method FORGE.BackgroundPyramidRenderer#_levelXnYnToXY
 * @param {number} level - pyramid level
 * @param {THREE.Vector2} xnyn - normalized coordinates
 * @return {THREE.Vector2} tile coordinates vector
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._levelXnYnToXY = function(level, xnyn)
{
    var tx = this.nbTilesPerAxis(level, "x"),
        ty = this.nbTilesPerAxis(level, "y");

    var x = Math.min(Math.floor(tx * xnyn.x), Math.ceil(tx) - 1);
    var y = Math.min(Math.floor(ty * xnyn.y), Math.ceil(ty) - 1);

    return new THREE.Vector2(x, y);
};

/**
 * Tile clearing routine.
 * Clear tile policy
 * Only applies to tiles with non zero level
 * Delete all tiles with zero opacity
 * Delete all tiles that has been created more than 2s ago and have never been displayed
 * Delete all tiles that has not been displayed since 30s
 * @method FORGE.BackgroundPyramidRenderer#_clearTiles
 * @private
 */
FORGE.BackgroundPyramidRenderer.prototype._clearTiles = function()
{
    if (this._clearTilesEnabled === false)
    {
        return;
    }

    var clearList = [];

    var now = Date.now();

    this._scene.children.forEach(function(tile)
    {
        // Tile with level greater than renderer level -> always removed
        var tileLevelHigher = tile.level > this._level;

        // Tile with level different than renderer level
        var tileLevelEqualsCurrent = tile.level === this._level;

        // Tile with level different than renderer level
        var tileLevelPreview = tile.level === FORGE.Tile.PREVIEW;

        // Tile with level different than renderer level
        var tileLevelZero = tile.level === 0;

        // Only clear tiles from level different from current and higher than preview and zero
        var tileLevelClearable = !tileLevelEqualsCurrent && !tileLevelPreview && !tileLevelZero;

        // Tile has a texture set in its material map
        // Never clear a tile without a texture set to keep parents of the current displayed tile
        // available (with their texture) when the user quickly zooms out
        var tileHasATexture = tile.textureIsSet === true;

        // Tile is part of the neighbour list
        var isANeighbourTile = this._renderNeighborList.indexOf(tile) !== -1;

        // Tile has never been displayed and time to live delay has been exceeded
        var timeSinceCreate = now - tile.createTS;
        var tileNeverDisplayedAndTTLExceeded = tile.displayTS === null
            && timeSinceCreate > FORGE.BackgroundPyramidRenderer.MAX_ALLOWED_TIME_SINCE_CREATION_MS;

        // Tile has been displayed a too long time ago
        var timeSinceDisplay = now - tile.displayTS;
        var tileNotDisplayedRecentlyEnough = tile.displayTS !== null
            && timeSinceDisplay > FORGE.BackgroundPyramidRenderer.MAX_ALLOWED_TIME_SINCE_DISPLAY_MS;

        // Tile is out of delay and could be cleared
        // This flag will force clearing tiles with lower levels not displayed for a while
        // Tiles at lower level (parents) rendered but hidden by current level tiles should not
        // be cleared to keep them available immediatly when zooming out 
        var tileOutOfDelay = tileNeverDisplayedAndTTLExceeded || tileNotDisplayedRecentlyEnough;

        // Always clear all tiles from higher levels (performance and aliasing issues)
        // OR
        // Clear tiles from lower levels (except preview or zero) AND out of delay
        // AND with a texture set AND out of the neighbour list
        if (tileLevelHigher || (tileLevelClearable && tileOutOfDelay && tileHasATexture && !isANeighbourTile))
        {
            clearList.push(tile);
        }

    }.bind(this));

    clearList.forEach(function(tile)
    {
        this._scene.remove(tile);
    }.bind(this));
};

/**
 * Get parent tile reference and create it if does not exist
 * @method FORGE.BackgroundPyramidRenderer#getParentTile
 * @param {FORGE.Tile} tile - tile
 * @return {FORGE.Tile} parent tile or null if it has no parent (level 0)
 */
FORGE.BackgroundPyramidRenderer.prototype.getParentTile = function(tile)
{
    if (tile.level === FORGE.Tile.PREVIEW)
    {
        return null;
    }

    if (tile.level === 0)
    {
        return this.getTile(null, FORGE.Tile.PREVIEW, tile.face, 0, 0, "pyramid preview");
    }

    var xnyn = this._levelXYToXnYn(tile.level, new THREE.Vector2(tile.x, tile.y));

    var parentLevel = tile.level - 1;
    var parentCoords = this._levelXnYnToXY(parentLevel, xnyn);

    return this.getTile(null, parentLevel, tile.face, parentCoords.x, parentCoords.y, "parent of " + tile.name);
};

/**
 * Get tile
 * Lookup in cache first or create it if not already in cache
 * @method FORGE.BackgroundPyramidRenderer#getTile
 */
FORGE.BackgroundPyramidRenderer.prototype.getTile = function(parent, level, face, x, y, creator)
{
    if (typeof this._tileCache[level] === "undefined")
    {
        this._tileCache[level] = new Map();
    }

    var name = FORGE.Tile.createName(face, level, x, y);

    var tile = this._tileCache[level].get(name);

    if (typeof tile === "undefined")
    {
        tile = new FORGE.Tile(parent, this, x, y, level, face, creator);

        tile.onDestroy.add(this._onTileDestroyed, this);
        this.log("Create tile " + tile.name + " (" + creator + ")");
        this._tileCache[level].set(name, tile);
    }

    if (this._scene.children.indexOf(tile) === -1)
    {
        this._scene.add(tile);
    }

    return tile;
};

/**
 * Get number of tiles per axis for a given level
 * @param {number} level - pyramid level
 * @param {string} axis - axis ("x" or "y")
 * @method FORGE.BackgroundPyramidRenderer#nbTilesPerAxis
 */
FORGE.BackgroundPyramidRenderer.prototype.nbTilesPerAxis = function(level, axis)
{
    if (this._tilesLevel === null || level === FORGE.Tile.PREVIEW)
    {
        return 1;
    }

    var lvl = this._tilesLevel[level];

    if (typeof lvl !== "undefined")
    {
        if (axis === "x")
        {
            return lvl.x;
        }
        else if (axis === "y")
        {
            return lvl.y;
        }
    }

    return Math.pow(2, level);
};

/**
 * Get number of tiles for a given level
 * @method FORGE.BackgroundPyramidRenderer#nbTiles
 */
FORGE.BackgroundPyramidRenderer.prototype.nbTiles = function(level)
{
    return 6 * this.nbTilesPerAxis(level, "x") * this.nbTilesPerAxis(level, "y");
};

/**
 * Get tile size at a given level (world units)
 * @method FORGE.BackgroundPyramidRenderer#tileSize
 * @return {FORGE.Size} size of a tile in world units
 */
FORGE.BackgroundPyramidRenderer.prototype.tileSize = function(level)
{
    return new FORGE.Size(this._size / this.nbTilesPerAxis(level, "x"), this._size / this.nbTilesPerAxis(level, "y"));
};

/**
 * Select current level for the pyramid
 * @method FORGE.BackgroundPyramidRenderer#selectLevel
 * @param {number} level pyramid level
 */
FORGE.BackgroundPyramidRenderer.prototype.selectLevel = function(level)
{
    this._level = level;

    var tilesOnAxisX = this.nbTilesPerAxis(this._level, "x");
    var tilesOnAxisY = this.nbTilesPerAxis(this._level, "y");

    this.log("Tiles per axis - X: " + tilesOnAxisX + ", Y: " + tilesOnAxisY);

    if (typeof(this._tileCache[this._level]) !== "undefined")
    {
        this._tileCache[this._level].forEach(function(tile)
        {
            this._scene.add(tile);
        }.bind(this));
    }

    var levelConfig;
    if (level === FORGE.Tile.PREVIEW)
    {
        levelConfig = this._config.preview;
        levelConfig.width = levelConfig.tile;
        levelConfig.height = levelConfig.tile;
    }
    else
    {
        levelConfig = this._config.source.levels[level];
    }

    // Compute pixels count
    var tilePixels = levelConfig.width * levelConfig.height;

    this._levelPixels = this.nbTiles(this._level) * tilePixels;

    var prefixes = {
        3: "kilo",
        6: "mega",
        9: "giga",
        12: "tera",
        15: "peta",
        18: "exa",
        21: "zetta",
        24: "yotta"
    };

    var rank = Math.floor(Math.log(this._levelPixels) / (Math.LN10 * 3)) * 3;
    var humanPixels = this._levelPixels / Math.pow(10, rank);

    var prefix = "";
    if (rank >= 27)
    {
        prefix = " too many ";
    }
    else if (rank >= 3)
    {
        prefix = " " + prefixes[rank];
    }

    this._levelPixelsHumanReadable = humanPixels.toFixed(1) + prefix + "pixels";
    this.log("Select new level: " + level + ", " + this._levelPixelsHumanReadable + " (" + this._levelPixels + ")");
};

/**
 * Add tile to render list
 * @method FORGE.BackgroundPyramidRenderer#addToRenderList
 * @param {FORGE.Tile} tile - tile
 */
FORGE.BackgroundPyramidRenderer.prototype.addToRenderList = function(tile)
{
    this._renderList.push(tile);

    // Add tile neighbours to the list only if its level is the current one
    if (tile.level === this._level)
    {
        this._renderNeighborList = this._renderNeighborList.concat(tile.neighbours);
    }
};

/**
 * Render routine.
 * Do preliminary job of specific background renderer, then summon superclass method
 * @method FORGE.BackgroundPyramidRenderer#render
 * @param {THREE.PerspectiveCamera} camera - perspective camera
 */
FORGE.BackgroundPyramidRenderer.prototype.render = function(camera)
{
    // Renderer should find if some tile at current level are currently rendered
    this._renderList = [];
    this._renderNeighborList = [];

    FORGE.BackgroundRenderer.prototype.render.call(this, camera);

    this._renderNeighborList = FORGE.Utils.arrayUnique(this._renderNeighborList);

    this._clearTiles();
};

/**
 * Destroy sequence.
 * @method FORGE.BackgroundPyramidRenderer#destroy
 */
FORGE.BackgroundPyramidRenderer.prototype.destroy = function()
{
    this._viewer.camera.onChange.remove(this._onCameraChange, this);

    this._clear();

    var tile;
    for (var level in this._tileCache)
    {
        while (this._tileCache[level].length)
        {
            tile = this._tileCache[level].pop();
            tile.destroy();
        }

        this._tileCache[level] = null;
    }

    this._textureStore = null;
    this._tileCache = null;
    this._renderNeighborList.length = 0;
    this._renderNeighborList = null;
    this._renderList.length = 0;
    this._renderList = null;
    this._config = null;

    FORGE.BackgroundRenderer.prototype.destroy.call(this);
};

/**
 * Get cube size in world units.
 * @name FORGE.BackgroundPyramidRenderer#cubeSize
 * @type {number}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "cubeSize",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._size;
    }
});

/**
 * Get texture store.
 * @name FORGE.BackgroundPyramidRenderer#textureStore
 * @type {FORGE.MediaStore}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "textureStore",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._textureStore;
    }
});

/**
 * Get pyramid max level.
 * @name FORGE.BackgroundPyramidRenderer#levelMax
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "levelMax",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._config.source.levels.length - 1;
    }
});

/**
 * Get pyramid current level.
 * @name FORGE.BackgroundPyramidRenderer#level
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "level",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._level;
    }
});

/**
 * Get number of pixels at current level.
 * @name FORGE.BackgroundPyramidRenderer#pixelsAtCurrentLevel
 * @type {number}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "pixelsAtCurrentLevel",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._levelPixels;
    }
});

/**
 * Get number of pixels at current level presented as human readable string.
 * @name FORGE.BackgroundPyramidRenderer#pixelsAtCurrentLevelHumanReadable
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "pixelsAtCurrentLevelHumanReadable",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._levelPixelsHumanReadable;
    }
});

/**
 * Get fov min.
 * @name FORGE.BackgroundPyramidRenderer#fovMin
 * @type {number}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "fovMin",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this._fovMin;
    }
});

/**
 * Get number of tiles on axis X for current level.
 * @name FORGE.BackgroundPyramidRenderer#tilesOnAxisX
 * @type {number}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "tilesOnAxisX",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this.nbTilesPerAxis(this._level, "x");
    }
});

/**
 * Get number of tiles on axis Y for current level.
 * @name FORGE.BackgroundPyramidRenderer#tilesOnAxisY
 * @type {number}
 */
Object.defineProperty(FORGE.BackgroundPyramidRenderer.prototype, "tilesOnAxisY",
{
    /** @this {FORGE.BackgroundPyramidRenderer} */
    get: function()
    {
        return this.nbTilesPerAxis(this._level, "y");
    }
});

/**
 * FORGE.BackgroundShaderRenderer
 * BackgroundShaderRenderer class.
 *
 * @constructor FORGE.BackgroundShaderRenderer
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {THREE.WebGLRenderTarget} target - render target
 * @param {SceneMediaOptionsConfig} options - the options for the cubemap
 * @extends {FORGE.BackgroundRenderer}
 */
FORGE.BackgroundShaderRenderer = function(viewer, target, options)
{
    /**
     * Display object (image, canvas or video)
     * @name FORGE.BackgroundShaderRenderer#_displayObject
     * @type {FORGE.DisplayObject}
     * @private
     */
    this._displayObject = null;

    /**
     * Texture used for video rendering
     * @name FORGE.BackgroundShaderRenderer#_texture
     * @type {THREE.Texture}
     * @private
     */
    this._texture = null;

    /**
     * Texture canvas used for video rendering
     * @name FORGE.BackgroundShaderRenderer#_textureCanvas
     * @type {Element|HTMLCanvasElement}
     * @private
     */
    this._textureCanvas = null;

    /**
     * Texture context associated with texture canvas
     * @name FORGE.BackgroundShaderRenderer#_textureContext
     * @type {CanvasRenderingContext2D}
     * @private
     */
    this._textureContext = null;

    /**
     * When source is a video, a reduction factor can be set to improve perf by lowering quality
     * @name FORGE.BackgroundShaderRenderer#_videoReductionFactor
     * @type {number}
     * @private
     */
    this._videoReductionFactor = 1;

    FORGE.BackgroundRenderer.call(this, viewer, target, options, "BackgroundShaderRenderer");
};

FORGE.BackgroundShaderRenderer.prototype = Object.create(FORGE.BackgroundRenderer.prototype);
FORGE.BackgroundShaderRenderer.prototype.constructor = FORGE.BackgroundShaderRenderer;

/**
 * Init routine.
 * @method FORGE.BackgroundShaderRenderer#_boot
 * @private
 */
FORGE.BackgroundShaderRenderer.prototype._boot = function()
{
    FORGE.BackgroundRenderer.prototype._boot.call(this);

    // Finalize now
    this._updateInternals();

    // Set orthographic camera
    this._camera = new THREE.OrthographicCamera(-1, 1, 1, -1, -1, 1);

    // Debug: attach scene to window context to expose it into Three.js Inspector
    // window.scene = this._scene;
};

/**
 * Set display object.
 * @method FORGE.BackgroundShaderRenderer#_setDisplayObject
 * @private
 */
FORGE.BackgroundShaderRenderer.prototype._setDisplayObject = function(displayObject)
{
    this._displayObject = displayObject;

    if (FORGE.Utils.isTypeOf(displayObject, "Image") || FORGE.Utils.isTypeOf(displayObject, "Canvas"))
    {
        this._texture = new THREE.Texture();
        this._texture.image = displayObject.element;
    }
    else if (FORGE.Utils.isTypeOf(displayObject, ["VideoHTML5", "VideoDash"]))
    {
        // Evil hack from Hell
        // Reduce texture size for big videos on safari
        if (FORGE.Device.browser.toLowerCase() === "safari" && displayObject.originalHeight > 1440)
        {
            this._videoReductionFactor = 2;
        }

        if (this._displayObject.onQualityChange.has(this._mediaQualityChangeHandler, this))
        {
            this._displayObject.onQualityChange.remove(this._mediaQualityChangeHandler, this);
        }

        this._displayObject.onQualityChange.add(this._mediaQualityChangeHandler, this);

        this._textureCanvas = document.createElement("canvas");
        this._textureCanvas.width = displayObject.originalWidth / this._videoReductionFactor;
        this._textureCanvas.height = displayObject.originalHeight / this._videoReductionFactor;
        this._textureContext = this._textureCanvas.getContext("2d");
        this._texture = new THREE.Texture(this._textureCanvas);
    }
    else
    {
        throw "Wrong type of display object " + displayObject.className;
    }

    this._texture.format = THREE.RGBFormat;
    this._texture.mapping = THREE.Texture.DEFAULT_MAPPING;
    this._texture.magFilter = THREE.LinearFilter;
    this._texture.wrapS = THREE.ClampToEdgeWrapping;
    this._texture.wrapT = THREE.ClampToEdgeWrapping;

    this._texture.generateMipmaps = false;
    this._texture.minFilter = THREE.LinearFilter;

    if (this._mediaFormat === FORGE.MediaFormat.FLAT &&
        FORGE.Math.isPowerOfTwo(displayObject.width) && FORGE.Math.isPowerOfTwo(displayObject.height))
    {
        // Enable mipmaps for flat rendering to avoid aliasing
        this._texture.generateMipmaps = true;
        this._texture.minFilter = THREE.LinearMipMapLinearFilter;
    }

    this._texture.needsUpdate = true;

    this._mesh.material.uniforms.tTexture.value = this._texture;

    if (this._mesh.material.uniforms.hasOwnProperty("tTextureRatio"))
    {
        this._mesh.material.uniforms.tTextureRatio.value = this._texture.image.width / this._texture.image.height;
    }

    if (this._mesh.material.uniforms.hasOwnProperty("tTextureSize"))
    {
        this._mesh.material.uniforms.tTextureSize.value = new THREE.Vector2(this._texture.image.width, this._texture.image.height);
    }
};

/**
 * Handler of media quality change event
 * @method FORGE.BackgroundShaderRenderer#_mediaQualityChangeHandler
 * @private
 */
FORGE.BackgroundShaderRenderer.prototype._mediaQualityChangeHandler = function(event)
{
    this.log("Media quality has changed");

    var displayObject = event.emitter;
    this._textureCanvas.width = displayObject.originalWidth / this._videoReductionFactor;
    this._textureCanvas.height = displayObject.originalHeight / this._videoReductionFactor;
};

/**
 * Update internals
 * @method FORGE.BackgroundShaderRenderer#_updateInternals
 * @private
 */
FORGE.BackgroundShaderRenderer.prototype._updateInternals = function()
{
    var shaderSTW = this._viewer.renderer.view.current.shaderSTW;

    var vertexShader = FORGE.ShaderLib.parseIncludes(shaderSTW.vertexShader);
    var fragmentShader = FORGE.ShaderLib.parseIncludes(shaderSTW.fragmentShader);

    var material = new THREE.ShaderMaterial({
        uniforms: /** @type {FORGEUniform} */ (shaderSTW.uniforms),
        vertexShader: vertexShader,
        fragmentShader: fragmentShader,
        side: THREE.FrontSide
    });

    if (this._texture !== null)
    {
        material.uniforms.tTexture.value = this._texture;
    }

    if (this._scene.children.length === 0)
    {
        var geometry = new THREE.PlaneBufferGeometry( 2, 2 );
        this._mesh = new THREE.Mesh(geometry, material);
        this._scene.add(this._mesh);
    }
    else
    {
        this._mesh = /** @type {THREE.Mesh} */ (this._scene.children[0]);
        this._mesh.material = null;
    }

    this._mesh.material = material;
    material.needsUpdate = true;
};

/**
 * Clear background.
 * @method FORGE.BackgroundShaderRenderer#clear
 * @private
 */
FORGE.BackgroundShaderRenderer.prototype._clear = function()
{
    // Draw to clear screen, then clear display object / texture
    this._viewer.renderer.webGLRenderer.clearColor();
};

/**
 * Do preliminary job of specific background renderer, then summon superclass method
 * @method FORGE.BackgroundShaderRenderer#render
 * @param {?THREE.PerspectiveCamera} camera perspective camera
 */
FORGE.BackgroundShaderRenderer.prototype.render = function(camera)
{
    if (this._viewer.renderer === null)
    {
        return;
    }

    camera = null; //@closure

    FORGE.BackgroundRenderer.prototype.render.call(this, camera);
};

/**
 * Update after view change
 * @method FORGE.BackgroundShaderRenderer#updateAfterViewChange
 */
FORGE.BackgroundShaderRenderer.prototype.updateAfterViewChange = function()
{
    this._updateInternals();

    if (this._texture !== null && typeof this._texture.image !== "undefined")
    {
        if (this._mesh.material.uniforms.hasOwnProperty("tTextureRatio"))
        {
            this._mesh.material.uniforms.tTextureRatio.value = this._texture.image.width / this._texture.image.height;
        }

        if (this._mesh.material.uniforms.hasOwnProperty("tTextureSize"))
        {
            this._mesh.material.uniforms.tTextureSize.value = new THREE.Vector2(this._texture.image.width, this._texture.image.height);
        }
    }
};

/**
 * Destroy sequence.
 * @method FORGE.BackgroundShaderRenderer#destroy
 */
FORGE.BackgroundShaderRenderer.prototype.destroy = function()
{
    this._clear();

    if (typeof this._displayObject.onQualityChange !== "undefined" &&
        this._displayObject.onQualityChange.has(this._mediaQualityChangeHandler, this))
    {
        this._displayObject.onQualityChange.remove(this._mediaQualityChangeHandler, this);
    }

    this._displayObject = null;
    this._textureCanvas = null;
    this._textureContext = null;

    if (this._texture !== null)
    {
        this._texture.dispose();
        this._texture = null;
    }

    FORGE.BackgroundRenderer.prototype.destroy.call(this);
};

/**
 * Get background renderer texture.
 * @name FORGE.BackgroundShaderRenderer#texture
 * @type {string}
 */
Object.defineProperty(FORGE.BackgroundShaderRenderer.prototype, "texture",
{
    /** @this {FORGE.BackgroundShaderRenderer} */
    get: function()
    {
        return this._texture;
    }
});
/**
 * FORGE.PostProcessing
 * Post processing class.
 *
 * @constructor FORGE.PostProcessing
 * @param {FORGE.Viewer} viewer {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.PostProcessing = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.PostProcessing#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Input config.
     * @name FORGE.PostProcessing#_config
     * @type {?FXConfig}
     * @private
     */
    this._config = null;

    /**
     * Shader pass sets.
     * @name FORGE.PostProcessing#_sets
     * @type {Object<string, Array<FX>>}
     * @private
     */
    this._sets = null;

    FORGE.BaseObject.call(this, "PostProcessing");

    this._boot();
};

FORGE.PostProcessing.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PostProcessing.prototype.constructor = FORGE.PostProcessing;


/**
 * Init routine
 * @method FORGE.PostProcessing#_boot
 * @private
 */
FORGE.PostProcessing.prototype._boot = function()
{
    this._sets = {};
};

/**
 * Configuration parsing
 * @method FORGE.PostProcessing#_parseConfig
 * @param {FXConfig} config input media configuration
 * @private
 */
FORGE.PostProcessing.prototype._parseConfig = function(config)
{
    this._uid = config.uid;

    if (typeof config.fxSets !== "undefined" && config.fxSets.length > 0)
    {
        // Create sets of fx configuration
        for (var i = 0, ii = config.fxSets.length; i < ii; i++)
        {
            var fxConfig = config.fxSets[i];
            this._sets[fxConfig.uid] = fxConfig.set;
        }
    }
};

/**
 * Parse shader config
 * @method FORGE.PostProcessing#_parseShaderConfig
 * @param {FX} shaderConfig - a shader configuration
 * @return {THREE.Pass} pass to be added to the render pipeline
 */
FORGE.PostProcessing.prototype._parseShaderConfig = function(shaderConfig)
{
    var shader = THREE[shaderConfig.type];
    var pass = new FORGE.ShaderPass(shaderConfig.uid, shaderConfig.type, shader);

    if (typeof pass.uniforms !== "undefined")
    {
        for (var param in shaderConfig.params)
        {
            var paramValue = shaderConfig.params[param];
            var value = paramValue;

            // Check if param is an object to build
            if (typeof paramValue === "object" && paramValue.hasOwnProperty("type") && paramValue.hasOwnProperty("args"))
            {
                var args = paramValue.args;
                switch (paramValue.type)
                {
                    case "THREE.Color":
                        if (args.length === 3)
                        {
                            value = new THREE.Color(args[0], args[1], args[2]);
                        }
                        else if (typeof args === "string" || typeof args === "number")
                        {
                            value = new THREE.Color(args);
                        }
                        else
                        {
                            throw new Error("Cannot create THREE.Color with " + (typeof args) + " (length " + args.length + ")");
                        }
                        break;

                    case "THREE.Vector2":
                        value = new THREE.Vector2(args[0], args[1]);
                        break;

                    case "THREE.Vector3":
                        value = new THREE.Vector3(args[0], args[1], args[2]);
                        break;
                }
            }

            pass.uniforms[param].value = value;
        }
    }

    return pass;
};

/**
 * Parse shader config
 * @method FORGE.PostProcessing#_parsePassConfig
 * @param {FX} passConfig - a pass configuration
 * @return {THREE.Pass} pass to be added to the render pipeline
 */
FORGE.PostProcessing.prototype._parsePassConfig = function(passConfig)
{
    var constructor = THREE[passConfig.type];

    var args = [];
    if (typeof passConfig.args !== "undefined")
    {
        // If args is an array, feed constructor with it, else create array from values if it's an object
        if (passConfig.args instanceof Array)
        {
            args = [null].concat(passConfig.args);
        }
        else
        {
            args = [null];
            for (var prop in passConfig.args)
            {
                args.push(passConfig.args[prop]);
            }
        }
    }

    var pass = new (Function.prototype.bind.apply(constructor, args));
    pass.uid = passConfig.uid;

    if (typeof passConfig.params !== "undefined")
    {
        for (var param in passConfig.params)
        {
            pass[param] = passConfig.params[param];    
        }        
    }

    return pass;
};

/**
 * Parse shader passes from FX configuration object
 * @method FORGE.PostProcessing#parseShaderPasses
 * @param {Array<FX>} fxConfig - a set of FX
 * @return {Array<THREE.ShaderPass>} array of shader passes
 */
FORGE.PostProcessing.prototype.parseShaderPasses = function(fxConfig)
{
    if (typeof fxConfig === "undefined" ||
        fxConfig === null ||
        fxConfig.length === 0)
    {
        return [];
    }

    var passes = [];

    for (var j = 0, jj = fxConfig.length; j < jj; j++)
    {
        var fx = /** type {FX} */ (fxConfig[j]);

        if (!THREE.hasOwnProperty(fx.type))
        {
            this.warn("Unknown image effect (" + fx.type + ")");
            continue;
        }

        var pass = null;
        if (fx.type.indexOf("Shader") != -1)
        {
            pass = this._parseShaderConfig(fx);
        }
        else if (fx.type.indexOf("Pass") != -1)
        {
            pass = this._parsePassConfig(fx);
        }

        if (pass !== null)
        {
            passes.push(pass);
        }
    }

    return passes;
};

/**
 * Add a post processing config.
 * @method FORGE.PostProcessing#addConfig
 * @param {string} uid fx unique identifier
 * @return {Array<FX>} fx set
 */
FORGE.PostProcessing.prototype.getFxSetByUID = function(uid)
{
    return this._sets[uid];
};

/**
 * Add a post processing config.
 * @method FORGE.PostProcessing#addConfig
 * @param {FXConfig} config - The config you want to add.
 */
FORGE.PostProcessing.prototype.addConfig = function(config)
{
    this._config = config;
    this._parseConfig(this._config);
};

/**
 * Destroy sequence
 * @method FORGE.PostProcessing#destroy
 */
FORGE.PostProcessing.prototype.destroy = function()
{
    this._viewer = null;
    this._sets = null;
};

/**
 * @namespace {Object} FORGE.EffectComposerType
 */
FORGE.EffectComposerType = {};

/**
 * Main effect composer
 * @name FORGE.EffectComposerType.MAIN
 * @type {string}
 * @const
 */
FORGE.EffectComposerType.MAIN = "main";

/**
 * Render effect composer
 * @name FORGE.EffectComposerType.RENDER
 * @type {string}
 * @const
 */
FORGE.EffectComposerType.RENDER = "render";

/**
 * Picking effect composer
 * @name FORGE.EffectComposerType.PICKING
 * @type {string}
 * @const
 */
FORGE.EffectComposerType.PICKING = "picking";

/**
 * EffectComposer class.
 *
 * Used to feed time to shader when uniform is declared
 *
 * @constructor FORGE.EffectComposer
 * @param {string} type - effect composer type
 * @param {THREE.WebGLRenderer} renderer - WebGL renderer
 * @param {THREE.WebGLRenderTarget=} renderTarget - render target
 * @extends {THREE.EffectComposer}
 */
FORGE.EffectComposer = function(type, renderer, renderTarget)
{
    /**
     * Effect composer type
     * @name FORGE.EffectComposer#_type
     * @type {string}
     * @private
     */
    this._type = type || FORGE.EffectComposerType.RENDER;

    /**
     * Rendering target
     * @name FORGE.EffectComposer#_renderTarget
     * @type {?THREE.WebGLRenderTarget}
     * @private
     */
    this._renderTarget = renderTarget || null;

    /**
     * Name
     * @name FORGE.EffectComposer#_name
     * @type {string}
     * @private
     */
    this._name = "";

    /**
     * Enabled flag
     * @name FORGE.EffectComposer#_enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Sum of clock deltas.
     * @name FORGE.ShaderPass#_deltaSum
     * @type {number}
     * @private
     */
    this._deltaSum = 0;

    THREE.EffectComposer.call(this, renderer, renderTarget);

    this._boot();
};

FORGE.EffectComposer.prototype = Object.create(THREE.EffectComposer.prototype);
FORGE.EffectComposer.prototype.constructor = FORGE.EffectComposer;

/**
 * Boot sequence
 * @method FORGE.EffectComposer#_boot
 * @private
 */
FORGE.EffectComposer.prototype._boot = function()
{
    this._name = FORGE.EffectComposer.getUniqueName(this._type);
};

/**
 * Render
 * Set time for all passes
 * @method FORGE.EffectComposer#render
 */
FORGE.EffectComposer.prototype.render = function(delta)
{
    this._deltaSum += delta;
    var time = this._deltaSum;

    for (var i=0, ii=this.passes.length; i<ii; i++)
    {
        var pass = this.passes[i];

        if (typeof pass.uniforms !== "undefined")
        {
            if (typeof pass.uniforms.time !== "undefined")
            {
                pass.uniforms.time.value = time;
            }
        }

        if (typeof pass.time !== "undefined")
        {
            pass.time = time;
        }
    }
    
    THREE.EffectComposer.prototype.render.call(this, delta);
};

/**
 * Destroy sequence
 * @method FORGE.EffectComposer#destroy
 */
FORGE.EffectComposer.prototype.destroy = function()
{
    this._renderTarget = null;

    this.writeBuffer.dispose();
    this.writeBuffer = null;

    this.readBuffer.dispose();
    this.readBuffer = null;
};

/**
 * Static EffectComposer uid
 * @name FORGE.EffectComposer#uid
 * @type {number}
 */
FORGE.EffectComposer.uid = 0;

/**
 * Get EffectComposer unique name
 * @name FORGE.EffectComposer#getUniqueName
 * @param {string} type effect composer type
 * @return {string} unique name
 */
FORGE.EffectComposer.getUniqueName = function(type)
{
    return "FORGE.EffectComposer." + type + "." + (FORGE.EffectComposer.uid++);
};

/**
 * Get EffectComposer type
 * @name FORGE.EffectComposer#type
 * @type {string}
 */
Object.defineProperty(FORGE.EffectComposer.prototype, "type",
{
    /** @this {FORGE.EffectComposer} */
    get: function()
    {
        return this._type;
    }
});

/**
 * EffectComposer render target
 * @name FORGE.EffectComposer#renderTarget
 * @type {THREE.WebGLRenderTarget}
 */
Object.defineProperty(FORGE.EffectComposer.prototype, "renderTarget",
{
    /** @this {FORGE.EffectComposer} */
    get: function()
    {
        return this.readBuffer;
    },

    /** @this {FORGE.EffectComposer} */
    set: function(value)
    {
        if (this.readBuffer !== null)
        {
            this.readBuffer.dispose();
        }
        this.readBuffer = value;
    }
});

/**
 * EffectComposer enabled flag
 * @name FORGE.EffectComposer#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.EffectComposer.prototype, "enabled",
{
    /** @this {FORGE.EffectComposer} */
    get: function()
    {
        return this._enabled;
    },

    /** @this {FORGE.EffectComposer} */
    set: function(value)
    {
        this._enabled = value;
    }
});

/**
 * Get EffectComposer name
 * @name FORGE.EffectComposer#name
 * @type {string}
 */
Object.defineProperty(FORGE.EffectComposer.prototype, "name",
{
    /** @this {FORGE.EffectComposer} */
    get: function()
    {
        return this._name;
    }
});

/**
 * @namespace {Object} FORGE.PassPosition
 */
FORGE.PassPosition = {};

/**
 * @name FORGE.PassPosition.UNKNOWN
 * @type {string}
 * @const
 */
FORGE.PassPosition.UNKNOWN = "";

/**
 * @name FORGE.PassPosition.BACKGROUND
 * @type {string}
 * @const
 */
FORGE.PassPosition.BACKGROUND = "background";

/**
 * @name FORGE.PassPosition.RENDER
 * @type {string}
 * @const
 */
FORGE.PassPosition.RENDER = "render";

/**
 * @name FORGE.PassPosition.GLOBAL
 * @type {string}
 * @const
 */
FORGE.PassPosition.GLOBAL = "global";

/**
 * @constructor FORGE.TexturePass
 * @param {THREE.Texture} map - map texture
 * @param {number=} opacity - opacity
 * @extends {THREE.TexturePass}
 */
FORGE.TexturePass = function( map, opacity )
{
    /**
     * Rendering position.
     * @name FORGE.TexturePass#_position
     * @type {string}
     * @private
     */
    this._position = "";

    THREE.TexturePass.call(this, map, opacity);
};

FORGE.TexturePass.prototype = Object.create(THREE.TexturePass.prototype);
FORGE.TexturePass.prototype.constructor = FORGE.TexturePass;

/**
 * Get TexturePass position
 * @name FORGE.TexturePass#position
 * @type {string}
 */
Object.defineProperty(FORGE.TexturePass.prototype, "position",
{
    /** @this {FORGE.TexturePass} */
    get: function()
    {
        return this._position;
    },

    /** @this {FORGE.TexturePass} */
    set: function(position)
    {
        this._position = position;
    }
});

/**
 * Used to feed time to shader when uniform is declared
 *
 * @constructor FORGE.ShaderPass
 * @param {string} uid unique identifier
 * @param {string} type shader type
 * @param {Object} shader shader object
 * @param {string=} textureID uniform texture string identifier
 * @extends {THREE.ShaderPass}
 */
FORGE.ShaderPass = function(uid, type, shader, textureID)
{
    /**
     * Unique id
     * @name FORGE.ShaderPass#_uid
     * @type {string}
     * @private
     */
    this._uid = uid;

    /**
     * Shader pass type.
     * @name FORGE.ShaderPass#_type
     * @type {string}
     * @private
     */
    this._type = type || "shader";

    /**
     * Rendering position.
     * @name FORGE.ShaderPass#_position
     * @type {string}
     * @private
     */
    this._position = FORGE.PassPosition.UNKNOWN;

    /**
     * Sum of clock deltas.
     * @name FORGE.ShaderPass#_deltaSum
     * @type {number}
     * @private
     */
    this._deltaSum = 0;

    THREE.ShaderPass.call(this, shader, textureID || "tDiffuse");
};

FORGE.ShaderPass.prototype = Object.create(THREE.ShaderPass.prototype);
FORGE.ShaderPass.prototype.constructor = FORGE.ShaderPass;

/**
 * @method  FORGE.ShaderPass#render
 * @param  {THREE.WebGLRenderer} renderer
 * @param  {THREE.WebGLRenderTarget} writeBuffer
 * @param  {THREE.WebGLRenderTarget} readBuffer
 * @param  {number=} delta
 * @param  {boolean=} maskActive
 */
FORGE.ShaderPass.prototype.render = function(renderer, writeBuffer, readBuffer, delta, maskActive)
{
    this._deltaSum += delta;

    if (this.uniforms.hasOwnProperty("time"))
    {
        this.uniforms["time"].value = this._deltaSum;
    }

    THREE.ShaderPass.prototype.render.call(this, renderer, writeBuffer, readBuffer, delta, maskActive);
};

/**
 * Get ShaderPass uid
 * @name FORGE.ShaderPass#uid
 * @type {string}
 */
Object.defineProperty(FORGE.ShaderPass.prototype, "uid",
{
    /** @this {FORGE.ShaderPass} */
    get: function()
    {
        return this._uid;
    }
});

/**
 * Get ShaderPass type
 * @name FORGE.ShaderPass#type
 * @type {string}
 */
Object.defineProperty(FORGE.ShaderPass.prototype, "type",
{
    /** @this {FORGE.ShaderPass} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Get ShaderPass position
 * @name FORGE.ShaderPass#position
 * @type {string}
 */
Object.defineProperty(FORGE.ShaderPass.prototype, "position",
{
    /** @this {FORGE.ShaderPass} */
    get: function()
    {
        return this._position;
    },

    /** @this {FORGE.ShaderPass} */
    set: function(position)
    {
        this._position = position;
    }
});

/**
 * @constructor FORGE.AdditionPass
 * @param {FORGE.EffectComposer} composer - effect composer reference
 * @extends {FORGE.ShaderPass}
 */
FORGE.AdditionPass = function(composer)
{
    /**
     * Reference on composer rendering texture to be added by the pass
     * @name FORGE.AdditionPass#_enabled
     * @type {FORGE.EffectComposer}
     * @private
     */
    this._composer = composer;

    /**
     * Shader material
     * @name FORGE.AdditionPass#_material
     * @type {THREE.Material}
     * @private
     */
    this._material = null;

    var vertexShader = FORGE.ShaderLib.parseIncludes( FORGE.ShaderChunk.wts_vert_rectilinear );
    var fragmentShader = FORGE.ShaderLib.parseIncludes( FORGE.ShaderChunk.wts_frag_addition );

    /** @type (FORGEUniform) */
    var uniforms =
    {
        tTexture: { type: "t", value: null },
        tAdd: { type: "t", value: null }
    };

    this._material = new THREE.RawShaderMaterial({
        fragmentShader: fragmentShader,
        vertexShader: vertexShader,
        uniforms: uniforms,
        side: THREE.FrontSide,
        name: "FORGE.AdditionPassMaterial"
    });

    FORGE.ShaderPass.call(this, "", "Addition", this._material, "tTexture");
};

FORGE.AdditionPass.prototype = Object.create(FORGE.ShaderPass.prototype);
FORGE.AdditionPass.prototype.constructor = FORGE.AdditionPass;

/**
 * Boot sequence
 * @method FORGE.AdditionPass#_boot
 * @private
 */
FORGE.AdditionPass.prototype._boot = function()
{

};

/**
 * Destroy sequence
 * @method FORGE.AdditionPass#destroy
 */
FORGE.AdditionPass.prototype.destroy = function()
{
    if (this._material === null)
    {
        this._material.dispose();
        this._material = null;
    }

    this._composer = null;
};

/**
 * Get linked AdditionPass
 * @name FORGE.AdditionPass#composer
 * @type {FORGE.EffectComposer}
 */
Object.defineProperty(FORGE.AdditionPass.prototype, "composer",
{
    /** @this {FORGE.AdditionPass} */
    get: function()
    {
        return this._composer;
    }
});

/**
 * @constructor FORGE.RenderPass
 * @param {THREE.Scene} scene - scene to be rendered
 * @param {THREE.Camera} camera - rendering camera
 * @param {THREE.Material=} overrideMaterial - optional override material
 * @param {THREE.Color=} clearColor - optional clear color
 * @param {boolean=} clearAlpha - optional clear alpha
 * @extends {THREE.RenderPass}
 */
FORGE.RenderPass = function( scene, camera, overrideMaterial, clearColor, clearAlpha )
{
    /**
     * Rendering position.
     * @name FORGE.RenderPass#_position
     * @type {string}
     * @private
     */
    this._position = "";

    THREE.RenderPass.call(this, scene, camera, overrideMaterial, clearColor, clearAlpha);
};

FORGE.RenderPass.prototype = Object.create(THREE.RenderPass.prototype);
FORGE.RenderPass.prototype.constructor = FORGE.RenderPass;

/**
 * Get RenderPass position
 * @name FORGE.RenderPass#position
 * @type {string}
 */
Object.defineProperty(FORGE.RenderPass.prototype, "position",
{
    /** @this {FORGE.RenderPass} */
    get: function()
    {
        return this._position;
    },

    /** @this {FORGE.RenderPass} */
    set: function(position)
    {
        this._position = position;
    }
});

/**
 * @namespace {Object} FORGE.ShaderChunk
 */

FORGE.ShaderChunk = {};
var coordinates = "\nvec3 toCartesian(in vec3 rtp) {\n    float r = rtp.x;\n    float theta = rtp.y;\n    float phi = rtp.z;\n    float x = r * cos(phi) * sin(theta);\n    float y = r * sin(phi);\n    float z = r * cos(phi) * cos(theta);\n    return vec3(x,y,z);\n}\nvec3 toSpherical(in vec3 pt) {\n    float r = length(pt);\n    float theta = -atan(pt.x, pt.z);\n    float phi = asin(pt.y / r);\n    return vec3(r,theta,phi);\n}\n";

FORGE.ShaderChunk["coordinates"] = coordinates;

var defines = "precision highp float;\nprecision highp int;\n#define PI_2  1.5707963268\n#define PI  3.141592653589793\n#define PI2 6.28318530717959\n#define RECIPROCAL_PI2 0.1591549430919\n#define isTrue(a) ((a) != 0)\n";

FORGE.ShaderChunk["defines"] = defines;

var helpers = "\nfloat wrap(in float x, in float min, in float max) {\n    return x - ((max - min) * (1.0 - step(x, min) - step(x, max)));\n}\nvec2 getFragment() {\n    return (2.0 * gl_FragCoord.xy / tViewportResolution - 1.0) * vec2(tViewportResolutionRatio, 1.0);\n}\nvec2 smoothTexUV(vec2 uv, vec2 texSize) {\n    uv = uv * texSize + 0.5;\n    vec2 iuv = floor(uv);\n    vec2 fuv = uv - iuv;\n    uv = iuv + fuv * fuv * fuv * (fuv * (fuv * 6.0 - 15.0) + 10.0);\n    return (uv - 0.5) / texSize;\n}\n";

FORGE.ShaderChunk["helpers"] = helpers;

var texcoords = "\nvec2 toEquirectangularTexCoords(in vec2 thetaphi) {\n    thetaphi.x = wrap(thetaphi.x + PI, -PI, PI);\n    vec2 coords = 0.5 + thetaphi / vec2(PI2, PI);\n    return coords;\n}\n";

FORGE.ShaderChunk["texcoords"] = texcoords;

var uniforms = "uniform mat4 modelMatrix;\nuniform mat4 modelViewMatrix;\nuniform mat4 projectionMatrix;\nuniform mat4 viewMatrix;\nuniform mat3 normalMatrix;\nuniform vec3 cameraPosition;\n";

FORGE.ShaderChunk["uniforms"] = uniforms;

var vert_attributes = "attribute vec3 position;\nattribute vec3 normal;\nattribute vec2 uv;\n";

FORGE.ShaderChunk["vert_attributes"] = vert_attributes;

var vert_attributes_wireframe = "attribute vec3 position;\nattribute vec3 normal;\nattribute vec2 uv;\nattribute vec2 quadrilateralCoords;\n";

FORGE.ShaderChunk["vert_attributes_wireframe"] = vert_attributes_wireframe;

var stw_frag_proj_flat = "\n#include <defines>\nuniform sampler2D tTexture;\nuniform vec2 tViewportResolution;\nuniform float tViewportResolutionRatio;\nuniform float tTextureRatio;\nuniform float tFov;\nuniform float tYaw;\nuniform float tPitch;\nuniform int tRepeatX;\nuniform int tRepeatY;\n#include <helpers>\n#include <coordinates>\n#include <texcoords>\nvoid main() {\n    vec2 sTexResolution = PI * vec2(tTextureRatio, 1.0);\n    vec2 sReference = vec2(tYaw, tPitch) + sTexResolution / 2.0;\n    vec2 sTexel = sReference + (tFov * 0.5) * getFragment();\n    vec2 uv = sTexel / sTexResolution;\n    if (isTrue(tRepeatX)) {\n        uv.x = mod(uv.x, 1.0);\n    }\n    else if (uv.x > 1.0 || uv.x < 0.0) {\n        discard;\n    }\n    if (isTrue(tRepeatY)) {\n        uv.y = mod(uv.y, 1.0);\n    }\n    else if (uv.y > 1.0 || uv.y < 0.0) {\n        discard;\n    }\n    gl_FragColor = vec4(texture2D(tTexture, uv).rgb, 1.0);\n}\n";

FORGE.ShaderChunk["stw_frag_proj_flat"] = stw_frag_proj_flat;

var stw_frag_proj_gopro = "\n#include <defines>\nuniform sampler2D tTexture;\nuniform vec2 tViewportResolution;\nuniform float tViewportResolutionRatio;\nuniform mat4 tModelViewMatrixInverse;\nuniform float tProjectionDistance;\nuniform float tProjectionScale;\n#include <helpers>\n#include <coordinates>\n#include <texcoords>\nvec2 projection() {\n    vec2 frag = getFragment();\n    vec2 c = tProjectionScale * frag;\n    float zs = tProjectionDistance;\n    float xy2 = dot(c,c);\n    float zs12 = (zs + 1.0) * (zs + 1.0);\n    float delta = 4.0 * (zs * zs * xy2 * xy2 - (xy2 + zs12) * (xy2 * zs * zs - zs12));\n    if (delta < 0.0) { return vec2(-1.); }\n    float z = (2.0 * zs * xy2 - sqrt(delta)) / (2.0 * (zs12 + xy2));\n    float x = c.x * ((zs - z) / (zs + 1.0));\n    float y = c.y * ((zs - z) / (zs + 1.0));\n    vec3 vx = vec3(tModelViewMatrixInverse * vec4(x, y, z, 1.0));\n    return toSpherical(vx).yz;\n}\nvoid main() {\n    vec2 texCoords = toEquirectangularTexCoords(projection());\n    gl_FragColor = vec4(texture2D(tTexture, texCoords).rgb, 1.0);\n}\n";

FORGE.ShaderChunk["stw_frag_proj_gopro"] = stw_frag_proj_gopro;

var stw_frag_proj_rectilinear = "\n#include <defines>\nuniform sampler2D tTexture;\nuniform vec2 tViewportResolution;\nuniform float tViewportResolutionRatio;\nuniform mat4 tModelViewMatrixInverse;\nuniform float tProjectionScale;\n#include <helpers>\n#include <coordinates>\n#include <texcoords>\nvec2 projection() {\n    vec2 frag = tProjectionScale * getFragment();\n    vec4 screenPT = vec4(frag, -1.0, 1.0);\n    vec4 worldPT = tModelViewMatrixInverse * screenPT;\n    return toSpherical(worldPT.xyz).yz;\n}\nvoid main() {\n    vec2 texCoords = toEquirectangularTexCoords(projection());\n    gl_FragColor = vec4(texture2D(tTexture, texCoords).rgb, 1.0);\n}\n";

FORGE.ShaderChunk["stw_frag_proj_rectilinear"] = stw_frag_proj_rectilinear;

var stw_vert_proj = "\n#include <defines>\nvoid main() {\n   gl_Position = projectionMatrix * modelViewMatrix * vec4( position, 1.0 );\n}\n";

FORGE.ShaderChunk["stw_vert_proj"] = stw_vert_proj;

var wts_frag = "\n#include <defines>\nuniform sampler2D tTexture;\nuniform float tOpacity;\nvarying vec2 vUv;\nvoid main() {\n    vec4 texel = texture2D( tTexture, vUv );\n    gl_FragColor = vec4(texel.rgb, texel.a * tOpacity);\n}\n";

FORGE.ShaderChunk["wts_frag"] = wts_frag;

var wts_frag_addition = "\n#include <defines>\nuniform sampler2D tTexture;\nuniform sampler2D tAdd;\nvarying vec2 vUv;\nvoid main() {\n    vec4 texel = texture2D( tTexture, vUv );\n    vec4 texelAdd = texture2D( tAdd, vUv );\n    float alpha = max(texel.a, texelAdd.a);\n    gl_FragColor = vec4(vec3(texelAdd.a * texelAdd.rgb + (1.0 - texelAdd.a) * texel.rgb), alpha);\n}\n";

FORGE.ShaderChunk["wts_frag_addition"] = wts_frag_addition;

var wts_frag_color = "\n#include <defines>\nuniform vec3 tColor;\nuniform float tOpacity;\nvarying vec2 vUv;\nvoid main() {\n    vec2 texCoords = vUv;\n    gl_FragColor = vec4(tColor, tOpacity);\n}\n";

FORGE.ShaderChunk["wts_frag_color"] = wts_frag_color;

var wts_frag_wireframe = "\n#extension GL_OES_standard_derivatives : enable \n#include <defines>\n#define WIRE_THICKNESS 1.50\nuniform vec3 tColor;\nvarying vec2 vQuadrilateralCoords;\nvoid main() {\n    vec2 dfdx = dFdx(vQuadrilateralCoords);\n    vec2 dfdy = dFdy(vQuadrilateralCoords);\n    vec2 threshold = vec2(WIRE_THICKNESS * sqrt(dfdx * dfdx + dfdy * dfdy));\n    vec2 amount = smoothstep(vec2(0.0), WIRE_THICKNESS * threshold, vec2(1.0) - abs(vQuadrilateralCoords));\n    float alpha = 1.0 - amount.x * amount.y;\n    gl_FragColor = vec4(tColor.rgb, alpha);\n}\n";

FORGE.ShaderChunk["wts_frag_wireframe"] = wts_frag_wireframe;

var wts_vert_gopro = "\n#include <defines>\n#include <vert_attributes>\n#include <uniforms>\nuniform float tProjectionDistance;\nvarying vec2 vUv;\nvoid main() {\n    vec4 spherePt = normalize(modelViewMatrix * vec4( position, 1.0 ));\n    float radius = 1.0 - 0.5 * tProjectionDistance;\n    float offset = radius - 1.0;\n    spherePt.xyz *= radius;\n    spherePt.z += offset;\n    gl_Position = projectionMatrix * spherePt;\n    vUv = uv;\n}\n";

FORGE.ShaderChunk["wts_vert_gopro"] = wts_vert_gopro;

var wts_vert_gopro_wireframe = "\n#include <defines>\n#include <vert_attributes_wireframe>\n#include <uniforms>\nuniform float tProjectionDistance;\nvarying vec2 vQuadrilateralCoords;\nvoid main() {\n    vec4 spherePt = normalize(modelViewMatrix * vec4( position, 1.0 ));\n    float radius = 1.0 - 0.5 * tProjectionDistance;\n    float offset = radius - 1.0;\n    spherePt.xyz *= radius;\n    spherePt.z += offset;\n    gl_Position = projectionMatrix * spherePt;\n    vQuadrilateralCoords = quadrilateralCoords;\n}\n";

FORGE.ShaderChunk["wts_vert_gopro_wireframe"] = wts_vert_gopro_wireframe;

var wts_vert_rectilinear = "\n#include <defines>\n#include <vert_attributes>\n#include <uniforms>\nvarying vec2 vUv;\nvoid main() {\n    vec4 camPt = modelViewMatrix * vec4( position, 1.0 );\n    gl_Position = projectionMatrix * camPt;\n    vUv = uv;\n}\n";

FORGE.ShaderChunk["wts_vert_rectilinear"] = wts_vert_rectilinear;

var wts_vert_rectilinear_wireframe = "\n#include <defines>\n#include <vert_attributes_wireframe>\n#include <uniforms>\nvarying vec2 vQuadrilateralCoords;\nvoid main() {\n    vec4 camPt = modelViewMatrix * vec4( position, 1.0 );\n    gl_Position = projectionMatrix * camPt;\n    vQuadrilateralCoords = quadrilateralCoords;\n}\n";

FORGE.ShaderChunk["wts_vert_rectilinear_wireframe"] = wts_vert_rectilinear_wireframe;

/**
 * @namespace {FORGEProgram} FORGE.ShaderLib
 */

//jscs:disable

FORGE.ShaderLib = {

    screenToWorld:
    {
        rectilinear:
        {
            /** @type {FORGEUniform} */
            uniforms:
            {
                tTexture: { type: "t", value: null },
                tViewportResolution: { type: "v2", value: new THREE.Vector2() },
                tViewportResolutionRatio: { type: "f", value: 1.0 },
                tModelViewMatrixInverse: { type: "m4", value: new THREE.Matrix4() },
                tProjectionScale: { type: "f", value: 1.0 }
            },

            vertexShader: FORGE.ShaderChunk.stw_vert_proj,
            fragmentShader: FORGE.ShaderChunk.stw_frag_proj_rectilinear
        },

        flat:
        {
            /** @type {FORGEUniform} */
            uniforms:
            {
                tTexture: { type: "t", value: null },
                tTextureRatio: { type: "f", value: 1.0 },
                tViewportResolution: { type: "v2", value: new THREE.Vector2() },
                tViewportResolutionRatio: { type: "f", value: 1.0 },
                tFov: { type: "f", value: 0.0 },
                tYaw: { type: "f", value: 0.0 },
                tPitch: { type: "f", value: 0.0 },
                tRepeatX: { type: "i", value: 0 },
                tRepeatY: { type: "i", value: 0 }
            },

            vertexShader: FORGE.ShaderChunk.stw_vert_proj,
            fragmentShader: FORGE.ShaderChunk.stw_frag_proj_flat
        },

        gopro:
        {
            /** @type {FORGEUniform} */
            uniforms:
            {
                tTexture: { type: "t", value: null },
                tViewportResolution: { type: "v2", value: new THREE.Vector2() },
                tViewportResolutionRatio: { type: "f", value: 1.0 },
                tModelViewMatrixInverse: { type: "m4", value: new THREE.Matrix4() },
                tProjectionDistance: { type: "f", value: 0.0 },
                tProjectionScale: { type: "f", value: 1.0 }
            },

            vertexShader: FORGE.ShaderChunk.stw_vert_proj,
            fragmentShader: FORGE.ShaderChunk.stw_frag_proj_gopro
        }
    },

    worldToScreen:
    {
        rectilinear:
        {
            mapping:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tOpacity: { type: "f", value: 1.0 },
                    tTexture: { type: "t", value: null }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_rectilinear,
                fragmentShader: FORGE.ShaderChunk.wts_frag
            },

            wireframe:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tColor: { type: "c", value: null },
                    tModelViewMatrixInverse: { type: "m4", value: null }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_rectilinear_wireframe,
                fragmentShader: FORGE.ShaderChunk.wts_frag_wireframe
            }
        },

        flat:
        {
            mapping:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tOpacity: { type: "f", value: 1.0 },
                    tTexture: { type: "t", value: null }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_rectilinear,
                fragmentShader: FORGE.ShaderChunk.wts_frag
            },

             wireframe:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tColor: { type: "c", value: null },
                    tModelViewMatrixInverse: { type: "m4", value: null }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_rectilinear_wireframe,
                fragmentShader: FORGE.ShaderChunk.wts_frag_wireframe
            }
        },

        gopro:
        {
            mapping:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tOpacity: { type: "f", value: 1.0 },
                    tTexture: { type: "t", value: null },
                    tProjectionDistance: { type: "f", value: 1 }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_gopro,
                fragmentShader: FORGE.ShaderChunk.wts_frag
            },

            wireframe:
            {
                /** @type {FORGEUniform} */
                uniforms:
                {
                    tColor: { type: "c", value: null },
                    tProjectionDistance: { type: "f", value: 1 }
                },
                vertexShader: FORGE.ShaderChunk.wts_vert_gopro_wireframe,
                fragmentShader: FORGE.ShaderChunk.wts_frag_wireframe
            }
        }
    }
};

//jscs:enable

FORGE.ShaderLib.parseIncludes = function(string)
{
    var pattern = /#include +<([\w\d.]+)>/g;

    function replace( match, include )
    {
        var r = FORGE.ShaderChunk[include];

        if (typeof r === "undefined")
        {
            r = THREE.ShaderChunk[include];

            if (typeof r === "undefined")
            {
                throw new Error( "Can not resolve #include <" + include + ">" );
            }
        }

        return FORGE.ShaderLib.parseIncludes(r);
    }

    return string.replace( pattern, replace );
};

/**
 * View manager class
 *
 * @constructor FORGE.ViewManager
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.ViewManager = function(viewer)
{
    /**
     * The Viewer reference.
     * @name FORGE.ViewManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The current view reference
     * @name FORGE.ViewManager#_view
     * @type {FORGE.ViewBase}
     * @private
     */
    this._view = null;

    /**
     * Ready flag
     * @name FORGE.ViewManager#_ready
     * @type {boolean}
     */
    this._ready = false;

    /**
     * The view type to restore when the user quit VR mode
     * @name  FORGE.ViewManager#_viewTypeBackup
     * @type {string}
     * @private
     */
    this._viewTypeBackup = "";

    /**
     * The view options to restore when the user quit VR mode
     * @name  FORGE.ViewManager#_viewOptionsBackup
     * @type {?ViewOptionsConfig}
     * @private
     */
    this._viewOptionsBackup = null;

    /**
     * Event dispatcher for view change.
     * @name FORGE.ViewManager#_onChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onChange = null;


    FORGE.BaseObject.call(this, "ViewManager");
};

FORGE.ViewManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ViewManager.prototype.constructor = FORGE.ViewManager;

/**
 * Set the view type
 * @method FORGE.ViewManager#_setView
 * @param {string} type - The type of the view to set.
 * @param {?ViewOptionsConfig} options - The view options.
 * @private
 */
FORGE.ViewManager.prototype._setView = function(type, options)
{
    if (this._view !== null && this._view.type === type)
    {
        return;
    }

    this._clearView();

    this.log("set view "+type);

    options = options || null;

    switch (type)
    {
        case FORGE.ViewType.GOPRO:
            this._view = new FORGE.ViewGoPro(this._viewer, options);
            break;

        case FORGE.ViewType.FLAT:
            this._view = new FORGE.ViewFlat(this._viewer, options);
            break;

        case FORGE.ViewType.RECTILINEAR:
        default:
            this._view = new FORGE.ViewRectilinear(this._viewer, options);
            break;
    }

    this._ready = true;

    this.notifyChange();
};

/**
 * Clear the view
 * @method FORGE.ViewManager#_clearView
 * @private
 */
FORGE.ViewManager.prototype._clearView = function()
{
    this._ready = false;

    if (this._view !== null)
    {
        this.log("clear view");

        this._view.destroy();
        this._view = null;
    }
};

/**
 * Used by views to notify a change to the manager.
 * @method FORGE.ViewManager#notifyChange
 * @private
 */
FORGE.ViewManager.prototype.notifyChange = function()
{
    if(this._onChange !== null)
    {
        this._onChange.dispatch();
    }
};

/**
 * Load a view configuration
 * @method FORGE.ViewManager#load
 * @param  {ViewConfig} config - The configuration of the view to load
 * @private
 */
FORGE.ViewManager.prototype.load = function(config)
{
    var sceneViewConfig = /** @type {ViewConfig} */ (config);
    var globalViewConfig = /** @type {ViewConfig} */ (this._viewer.mainConfig.view);
    var extendedViewConfig = /** @type {ViewConfig} */ (FORGE.Utils.extendMultipleObjects(globalViewConfig, sceneViewConfig));

    var type = (typeof extendedViewConfig.type === "string") ? extendedViewConfig.type.toLowerCase() : FORGE.ViewType.RECTILINEAR;
    var options = /** @type {ViewOptionsConfig} */ (extendedViewConfig.options);

    this._setView(type, options);
};

/**
 * Enable VR backup the view type then force to rectilinear
 * @method FORGE.ViewManager#enableVR
 */
FORGE.ViewManager.prototype.enableVR = function()
{
    this._viewTypeBackup = this._view.type;
    this._viewOptionsBackup = this._view.options;
    this._setView(FORGE.ViewType.RECTILINEAR, null);
};

/**
 * Disable VR restore the view type.
 * @method FORGE.ViewManager#disableVR
 */
FORGE.ViewManager.prototype.disableVR = function()
{
    if(this._viewTypeBackup !== "")
    {
        this._setView(this._viewTypeBackup, this._viewOptionsBackup); // Restore the view as before the VR mode
        this._viewer.camera.roll = 0; // Reset the roll to 0
    }
};

/**
 * Convert a point from world space to screen space with the current view type.
 *
 * @method FORGE.ViewManager#worldToScreen
 * @param {THREE.Vector3} worldPt - Point in world space
 * @param {number} parallax - Parallax factor [0..1]
 * @return {THREE.Vector2} Point in screen coordinates
 * @todo Implement worldToScreen
 */
FORGE.ViewManager.prototype.worldToScreen = function(worldPt, parallax)
{
    return this._view.worldToScreen(worldPt, parallax);
};

/**
 * Convert a point from screen space to world space with the current view type.
 *
 * @method FORGE.ViewManager#screenToWorld
 * @param {THREE.Vector2} screenPt point in screen space
 * @return {THREE.Vector3}
 * @todo Implement screenToWorld
 */
FORGE.ViewManager.prototype.screenToWorld = function(screenPt)
{
    return this._view.screenToWorld(screenPt);
};

/**
 * Destroy sequence
 * @method FORGE.ViewManager#destroy
 */
FORGE.ViewManager.prototype.destroy = function()
{
    this._viewer = null;

    if (this._view !== null)
    {
        this._view.destroy();
        this._view = null;
    }

    if (this._onChange !== null)
    {
        this._onChange.destroy();
        this._onChange = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the current view object.
 * @name  FORGE.ViewManager#current
 * @type {FORGE.ViewBase}
 * @readonly
 */
Object.defineProperty(FORGE.ViewManager.prototype, "current",
{
    /** @this {FORGE.ViewManager} */
    get: function()
    {
        return this._view;
    }
});

/**
 * Get the view ready flag.
 * @name  FORGE.ViewManager#ready
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.ViewManager.prototype, "ready",
{
    /** @this {FORGE.ViewManager} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get and set the current view type.
 * @name  FORGE.ViewManager#type
 * @type {string}
 */
Object.defineProperty(FORGE.ViewManager.prototype, "type",
{
    /** @this {FORGE.ViewManager} */
    get: function()
    {
        return (this._view !== null) ? this._view.type : null;
    },

    /** @this {FORGE.ViewManager} */
    set: function(value)
    {
        this._setView(value, null);
    }
});

/**
 * Get the onChange {@link FORGE.EventDispatcher}.
 * @name  FORGE.ViewManager#onChange
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.ViewManager.prototype, "onChange",
{
    /** @this {FORGE.ViewManager} */
    get: function()
    {
        if (this._onChange === null)
        {
            this._onChange = new FORGE.EventDispatcher(this);
        }

        return this._onChange;
    }
});

/**
 * @namespace {Object} FORGE.ViewType
 */
FORGE.ViewType = {};

/**
 * @name FORGE.ViewType.UNDEFINED
 * @type {string}
 * @const
 */
FORGE.ViewType.UNDEFINED = "undefined";

/**
 * @name FORGE.ViewType.FLAT
 * @type {string}
 * @const
 */
FORGE.ViewType.FLAT = "flat";

/**
 * @name FORGE.ViewType.RECTILINEAR
 * @type {string}
 * @const
 */
FORGE.ViewType.RECTILINEAR = "rectilinear";

/**
 * @name FORGE.ViewType.GOPRO
 * @type {string}
 * @const
 */
FORGE.ViewType.GOPRO = "gopro";

/**
 * Abstract base class for projeted views. Should be subclassed for every supported projection / view type.
 *
 * @constructor FORGE.ViewBase
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {?ViewOptionsConfig} options - The view options.
 * @param {string} className - object className.
 * @param {string} type - object view type.
 * @extends {FORGE.BaseObject}
 */
FORGE.ViewBase = function(viewer, options, className, type)
{
    /**
     * The Viewer reference.
     * @name FORGE.ViewBase#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The view options
     * @name FORGE.ViewBase#_options
     * @type {?ViewOptionsConfig}
     * @private
     */
    this._options = options || null;

    /**
     * Projection scale.
     * @name FORGE.ViewBase#_projectionScale
     * @type {number}
     * @private
     */
    this._projectionScale = 1.0;

    /**
     * View type.
     * @name FORGE.ViewBase#_type
     * @type {string}
     * @private
     */
    this._type = type;

    /**
     * Yaw min angle for current view type [radians].
     * @name FORGE.ViewBase#_yawMin
     * @type {number}
     * @private
     */
    this._yawMin = -Infinity;

    /**
     * Yaw max angle for current view type [radians].
     * @name FORGE.ViewBase#_yawMax
     * @type {number}
     * @private
     */
    this._yawMax = Infinity;

    /**
     * Pitch min angle for current view type [radians].
     * @name FORGE.ViewBase#_pitchMin
     * @type {number}
     * @private
     */
    this._pitchMin = -Infinity;

    /**
     * Pitch min angle for current view type [radians].
     * @name FORGE.ViewBase#_pitchMax
     * @type {number}
     * @private
     */
    this._pitchMax = Infinity;

    /**
     * Roll min angle for current view type [radians].
     * @name FORGE.ViewBase#_rollMin
     * @type {number}
     * @private
     */
    this._rollMin = -Infinity;

    /**
     * Roll max angle for current view type [radians].
     * @name FORGE.ViewBase#_rollMax
     * @type {number}
     * @private
     */
    this._rollMax = Infinity;

    /**
     * Fov min angle for current view type [radians].
     * @name FORGE.ViewBase#_fovMin
     * @type {number}
     * @private
     */
    this._fovMin = 0;

    /**
     * Fov max angle for current view type [radians].
     * @name FORGE.ViewBase#_fovMax
     * @type {number}
     * @private
     */
    this._fovMax = Infinity;

    /**
     * Shader screen to world
     * @name FORGE.ViewBase#_shaderSTW
     * @type {?ScreenToWorldProgram}
     * @private
     */
    this._shaderSTW = null;

    /**
     * Shader world to screen
     * @name FORGE.ViewBase#_shaderWTS
     * @type {?WorldToScreenProgram}
     * @private
     */
    this._shaderWTS = null;

    FORGE.BaseObject.call(this, className || "ViewBase");
};

FORGE.ViewBase.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ViewBase.prototype.constructor = FORGE.ViewBase;

/**
 * Boot sequence.
 *
 * @method FORGE.ViewBase#_boot
 * @private
 */
FORGE.ViewBase.prototype._boot = function()
{
    this._viewer.story.onSceneLoadComplete.add(this.updateUniforms, this);
};

/**
 * Compute fragment from a screen point.
 *
 * @method FORGE.ViewBase#_screenToFragment
 * @param  {THREE.Vector2} screenPt - Point in screen space
 * @return {THREE.Vector2} Fragment in normalized space
 * @private
 */
FORGE.ViewBase.prototype._screenToFragment = function(screenPt)
{
    var resolution = this._viewer.renderer.displayResolution;
    var fx = (2.0 * screenPt.x / resolution.width) - 1.0;
    var fy = (2.0 * screenPt.y / resolution.height) - 1.0;
    return new THREE.Vector2(fx * resolution.ratio, fy);
};

/**
 * Compute screen point from a fragment.
 *
 * @method FORGE.ViewBase#_fragmentToScreen
 * @param {THREE.Vector2} fragment - Fragment in normalized space
 * @return  {THREE.Vector2} Point in screen space
 * @private
 */
FORGE.ViewBase.prototype._fragmentToScreen = function(fragment)
{
    var resolution = this._viewer.renderer.displayResolution;
    var sx = ((fragment.x / resolution.ratio) + 1.0) * (resolution.width / 2.0);
    var sy = (fragment.y + 1.0) * (resolution.height / 2.0);
    return new THREE.Vector2(Math.round(sx), resolution.height - Math.round(sy));
};

/**
 * Update uniforms.
 * Abstract method that should be implemented by subclass.
 *
 * @method FORGE.ViewBase#updateUniforms
 * @param {FORGEUniform} uniforms
 */
FORGE.ViewBase.prototype.updateUniforms = function(uniforms)
{
    this.log(uniforms); //@closure
    throw "Please implement " + this._className + "::updateUniforms";
};

/**
 * Convert a point from world space to screen space.
 * Abstract method that should be implemented by subclass.
 *
 * @method FORGE.ViewBase#worldToScreen
 * @param {THREE.Vector3} worldPt - Point in world space
 * @param {number} parallax - Parallax factor [0..1]
 * @return {?THREE.Vector2} Point in screen coordinates or null if the point is out of bounds.
 * @todo Implement worldToScreen
 */
FORGE.ViewBase.prototype.worldToScreen = function(worldPt, parallax)
{
    this.log(worldPt); //@closure
    this.log(parallax); //@closure
    throw "Please implement " + this._className + "::worldToScreen";
};

/**
 * Convert a point from screen space to world space.
 * Abstract method that should be implemented by subclass.
 * @method FORGE.ViewBase#screenToWorld
 * @param {THREE.Vector2} screenPt point in screen space
 * @return {?THREE.Vector3} Point in world space or null if the screenPt is out of bounds.
 * @todo Implement screenToWorld
 */
FORGE.ViewBase.prototype.screenToWorld = function(screenPt)
{
    this.log(screenPt); //@closure
    throw "Please implement " + this._className + "::screenToWorld";
};

/**
 * Get fov computed for projection.
 * @method FORGE.ViewBase#getProjectionFov
 */
FORGE.ViewBase.prototype.getProjectionFov = function()
{
    return FORGE.Math.degToRad(this._viewer.renderer.camera.fov);
};

/**
 * Destroy method.
 *
 * @method FORGE.ViewBase#destroy
 */
FORGE.ViewBase.prototype.destroy = function()
{
    this._viewer = null;
    this._camera = null;

    this._shaderSTW = null;
    this._shaderWTS = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the view type.
 * @name FORGE.ViewBase#type
 * @type {string}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "type",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Get the minimum yaw value in radians.
 * @name FORGE.ViewBase#yawMin
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "yawMin",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._yawMin;
    }
});

/**
 * Get the maximum yaw value in radians.
 * @name FORGE.ViewBase#yawMax
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "yawMax",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._yawMax;
    }
});

/**
 * Get the minimum pitch value in radians.
 * @name FORGE.ViewBase#pitchMin
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "pitchMin",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._pitchMin;
    }
});

/**
 * Get the maximum pitch value in radians.
 * @name FORGE.ViewBase#pitchMax
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "pitchMax",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._pitchMax;
    }
});

/**
 * Get the minimum roll value in radians.
 * @name FORGE.ViewBase#rollMin
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "rollMin",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._rollMin;
    }
});

/**
 * Get the maximum roll value in radians.
 * @name FORGE.ViewBase#rollMax
 * @type {number}
 */
Object.defineProperty(FORGE.ViewBase.prototype, "rollMax",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._rollMax;
    }
});

/**
 * Get minimum fov for current view in radians.
 * @name FORGE.ViewBase#fovMin
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "fovMin",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._fovMin;
    }
});

/**
 * Get maximum fov for current view in radians.
 * @name FORGE.ViewBase#fovMax
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "fovMax",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._fovMax;
    }
});

/**
 * Shader screen to world
 * @name FORGE.ViewBase#shaderSTW
 * @type {ScreenToWorldProgram}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "shaderSTW",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._shaderSTW;
    }
});

/**
 * Shader world to screen
 * @name FORGE.ViewBase#shaderWTS
 * @type {WorldToScreenProgram}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "shaderWTS",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._shaderWTS;
    }
});

/**
 * Options getter
 * @name FORGE.ViewBase#options
 * @type {ViewOptionsConfig}
 * @readonly
 */
Object.defineProperty(FORGE.ViewBase.prototype, "options",
{
    /** @this {FORGE.ViewBase} */
    get: function()
    {
        return this._options;
    }
});

/**
 * Flat view class.
 *
 * @constructor FORGE.ViewFlat
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {?ViewOptionsConfig} options - The view options.
 * @extends {FORGE.ViewBase}
 */
FORGE.ViewFlat = function(viewer, options)
{
    FORGE.ViewBase.call(this, viewer, options, "ViewFlat", FORGE.ViewType.FLAT);

    this._boot();
};

FORGE.ViewFlat.prototype = Object.create(FORGE.ViewBase.prototype);
FORGE.ViewFlat.prototype.constructor = FORGE.ViewFlat;

/**
 * Flat view default options
 * @name FORGE.ViewFlat.DEFAULT_OPTIONS
 * @type {ViewOptionsConfig}
 * @const
 */
FORGE.ViewFlat.DEFAULT_OPTIONS =
{
    repeatX: false,
    repeatY: false
};

/**
 * Boot sequence.
 *
 * @method FORGE.ViewFlat#_boot
 * @private
 */
FORGE.ViewFlat.prototype._boot = function()
{
    FORGE.ViewBase.prototype._boot.call(this);

    this._shaderSTW = /** @type {ScreenToWorldProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.screenToWorld.flat));
    this._shaderWTS = /** @type {WorldToScreenProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.worldToScreen.flat));

    this._yawMin = FORGE.Math.degToRad(-360);
    this._yawMax = FORGE.Math.degToRad(360);

    this._pitchMin = FORGE.Math.degToRad(-180);
    this._pitchMax = FORGE.Math.degToRad(180);

    this._fovMin = FORGE.Math.degToRad(20);
    this._fovMax = FORGE.Math.degToRad(180);

    this._options = FORGE.Utils.extendSimpleObject(FORGE.ViewFlat.DEFAULT_OPTIONS, this._options);
};

/**
 * Update view params.
 * @method FORGE.ViewFlat#_updateViewParams
 * @private
 */
FORGE.ViewFlat.prototype._updateViewParams = function()
{
    if (this._viewer !== null)
    {
        // When repeat is ON, set yaw and pitch min and max depending on
        // texture and screen ratios

        if (this._viewer.renderer.backgroundRenderer instanceof FORGE.BackgroundShaderRenderer)
        {
            var vfov = FORGE.Math.degToRad(this._viewer.camera.fov);

            if (this._options.repeatX === false)
            {
                var hfov = vfov * this._viewer.renderer.displayResolution.ratio;
                var texRatio = this._viewer.renderer.backgroundRenderer.textureSize.ratio;
                this._yawMax = Math.min(360, Math.max(0, (Math.PI * texRatio - hfov) * 0.5)); // image
                this._yawMin = -this._yawMax;
            }
            else
            {
                this._yawMin = FORGE.Math.degToRad(-360);
                this._yawMax = FORGE.Math.degToRad(360);
            }

            if (this._options.repeatY === false)
            {
                this._pitchMax = 0.5 * Math.max(0, Math.PI - vfov);
                this._pitchMin = -this._pitchMax;
            }
            else
            {
                this._pitchMin = FORGE.Math.degToRad(-180);
                this._pitchMax = FORGE.Math.degToRad(180);
            }
        }

        // Mesh rendering in flat view is limited around -+ 20 degrees
        else
        {
            this._yawMax = this._pitchMax = FORGE.Math.degToRad(20);
            this._yawMin = -this._yawMax;
            this._pitchMin = -this._pitchMax;
        }
    }
};

/**
 * Update uniforms.
 *
 * @method FORGE.ViewFlat#updateUniforms
 * @param {FORGEUniform} uniforms
 */
FORGE.ViewFlat.prototype.updateUniforms = function(uniforms)
{
    this._updateViewParams();

    if (typeof uniforms === "undefined")
    {
        return;
    }

    if (uniforms.hasOwnProperty("tRepeatX"))
    {
        uniforms.tRepeatX.value = this._options.repeatX ? 1 : 0;
    }

    if (uniforms.hasOwnProperty("tRepeatY"))
    {
        uniforms.tRepeatY.value = this._options.repeatY ? 1 : 0;
    }

    if (uniforms.hasOwnProperty("tYaw"))
    {
        uniforms.tYaw.value = FORGE.Math.degToRad(this._viewer.camera.yaw);
    }

    if (uniforms.hasOwnProperty("tPitch"))
    {
        uniforms.tPitch.value = FORGE.Math.degToRad(this._viewer.camera.pitch);
    }

    if (uniforms.hasOwnProperty("tFov"))
    {
        uniforms.tFov.value = FORGE.Math.degToRad(this._viewer.renderer.camera.fov);
    }
};

/**
 * Convert a point from world space to screen space.
 *
 * @method FORGE.ViewFlat#worldToScreen
 * @return {THREE.Vector2} point in screen coordinates
 */
FORGE.ViewFlat.prototype.worldToScreen = function()
{
    return new THREE.Vector2();
};

/**
 * Convert a point from screen space to world space.
 *
 * @method FORGE.ViewFlat#screenToWorld
 * @return {THREE.Vector3} world point
 */
FORGE.ViewFlat.prototype.screenToWorld = function()
{
    return new THREE.Vector3();
};

/**
 * Destroy method.
 *
 * @method FORGE.ViewFlat#destroy
 */
FORGE.ViewFlat.prototype.destroy = function()
{
    FORGE.ViewBase.prototype.destroy.call(this);
};

/**
 * Get and set the repeat X beahvior.
 * @name  FORGE.ViewFlat#repeatX
 * @type {string}
 */
Object.defineProperty(FORGE.ViewFlat.prototype, "repeatX",
{
    /** @this {FORGE.ViewFlat} */
    get: function()
    {
        return this._options.repeatX;
    },

    /** @this {FORGE.ViewFlat} */
    set: function(value)
    {
        this._options.repeatX = value;
        this._updateViewParams();

        // Notify the view manager of the change
        this._viewer.view.notifyChange();
    }
});

/**
 * Get and set the repeat Y beahvior.
 * @name  FORGE.ViewFlat#repeatY
 * @type {string}
 */
Object.defineProperty(FORGE.ViewFlat.prototype, "repeatY",
{
    /** @this {FORGE.ViewFlat} */
    get: function()
    {
        return this._options.repeatY;
    },

    /** @this {FORGE.ViewFlat} */
    set: function(value)
    {
        this._options.repeatY = value;
        this._updateViewParams();

        // Notify the view manager of the change
        this._viewer.view.notifyChange();
    }
});

/**
 * GoPro view class.
 *
 * @constructor FORGE.ViewGoPro
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {?ViewOptionsConfig} options - The view options.
 * @extends {FORGE.ViewBase}
 */
FORGE.ViewGoPro = function(viewer, options)
{
    /**
     * Projection distance.
     * @name FORGE.ViewGoPro#_projectionDistance
     * @type {number}
     * @private
     */
    this._projectionDistance = 0;

    FORGE.ViewBase.call(this, viewer, options, "ViewGoPro", FORGE.ViewType.GOPRO);

    this._boot();
};

FORGE.ViewGoPro.prototype = Object.create(FORGE.ViewBase.prototype);
FORGE.ViewGoPro.prototype.constructor = FORGE.ViewGoPro;

/**
 * Boot sequence.
 *
 * @method FORGE.ViewGoPro#_boot
 * @private
 */
FORGE.ViewGoPro.prototype._boot = function()
{
    FORGE.ViewBase.prototype._boot.call(this);

    this._shaderSTW = /** @type {ScreenToWorldProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.screenToWorld.gopro));
    this._shaderWTS = /** @type {WorldToScreenProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.worldToScreen.gopro));

    this._fovMin = FORGE.Math.degToRad(30);
    this._fovMax = FORGE.Math.degToRad(330);
};

/**
 * Update view params.
 * @method FORGE.ViewGoPro#_updateViewParams
 * @private
 */
FORGE.ViewGoPro.prototype._updateViewParams = function()
{
    if (this._viewer !== null)
    {
        var projFovLow = 90;
        var projFovHigh = 180;
        var distance = 0;

        var fov = this._viewer.camera.fov;

        var fn = 0;

        if (fov < projFovLow)
        {
            distance = 0;
            fn = 0;
        }
        else if (fov > projFovHigh)
        {
            distance = 1;
            fn = 1;
        }
        else
        {
            // Apply sinus in out interpolation to smooth the transition
            fn = (fov - projFovLow) / (projFovHigh - projFovLow);
            distance = 0.5 * (1.0 + Math.sin(Math.PI / 2.0 * (2.0 * fn - 1)));
        }

        this._projectionDistance = distance;

        var fovRad = 0.5 * FORGE.Math.degToRad(fov);
        this._projectionScale = (distance + 1) * Math.sin(fovRad) / (distance + Math.cos(fovRad));
    }
};

/**
 * Update uniforms.
 *
 * @method FORGE.ViewGoPro#updateUniforms
 * @param {FORGEUniform} uniforms
 */
FORGE.ViewGoPro.prototype.updateUniforms = function(uniforms)
{
    this._updateViewParams();

    if (typeof uniforms === "undefined")
    {
        return;
    }

    if (uniforms.hasOwnProperty("tProjectionDistance"))
    {
        uniforms.tProjectionDistance.value = this._projectionDistance;
    }

    if (uniforms.hasOwnProperty("tProjectionScale"))
    {
        uniforms.tProjectionScale.value = this._projectionScale;
    }
};

/**
 * Convert a point from world space to screen space.
 *
 * @method FORGE.ViewGoPro#worldToScreen
 * @param {THREE.Vector3} worldPt - 3D point in world space
 * @param {number} parallaxFactor - parallax factor [0..1]
 * @return {?THREE.Vector2} point in screen coordinates
 */
FORGE.ViewGoPro.prototype.worldToScreen = function(worldPt, parallaxFactor)
{
    worldPt = worldPt || new THREE.Vector3();
    worldPt.normalize();
    parallaxFactor = parallaxFactor || 0;

    var worldPt4 = new THREE.Vector4(-worldPt.x, -worldPt.y, worldPt.z, 1.0);
    var camEuler = FORGE.Math.rotationMatrixToEuler(this._viewer.camera.modelView);
    var rotation = FORGE.Math.eulerToRotationMatrix(camEuler.yaw, camEuler.pitch, -camEuler.roll, true);
    rotation = rotation.transpose();
    worldPt4.applyMatrix4(rotation);

    if (worldPt4.z > this._projectionDistance)
    {
        return null;
    }

    var alpha = (this._projectionDistance + 1) / (this._projectionDistance - worldPt4.z);
    var x = -worldPt4.x * alpha;
    var y = -worldPt4.y * alpha;

    x /= (1 + parallaxFactor) * this._projectionScale;
    y /= this._projectionScale;

    return this._fragmentToScreen(new THREE.Vector2(x, y));
};

/**
 * Convert a point from screen space to world space.
 *
 * @method FORGE.ViewGoPro#screenToWorld
 * @param {THREE.Vector2} screenPt - 2D point in screen space [0..w, 0..h]
 * @return {?THREE.Vector3} world point
 */
FORGE.ViewGoPro.prototype.screenToWorld = function(screenPt)
{
    var resolution = this._viewer.renderer.displayResolution;

    screenPt = screenPt || new THREE.Vector2(resolution.width / 2, resolution.height / 2);

    var widthMargin = FORGE.ViewRectilinear.OFF_SCREEN_MARGIN * resolution.width,
        heightMargin = FORGE.ViewRectilinear.OFF_SCREEN_MARGIN * resolution.height;
    if (screenPt.x < -widthMargin || screenPt.x > resolution.width + widthMargin
        || screenPt.y < -heightMargin || screenPt.y > resolution.height + heightMargin)
    {
        return null;
    }

    var fragment = this._screenToFragment(screenPt);
    fragment.multiplyScalar(this._projectionScale);

    var xy2 = fragment.dot(fragment);
    var zs12 = Math.pow(this._projectionDistance + 1, 2);
    var delta = 4 * (this._projectionDistance * this._projectionDistance * xy2 * xy2
                    - (xy2 + zs12) * (xy2 * this._projectionDistance * this._projectionDistance - zs12));

    if (delta < 0)
    {
        return null;
    }

    // world coordinates
    var worldPt = new THREE.Vector4();
    worldPt.z = (2 * this._projectionDistance * xy2 - Math.sqrt(delta)) / (2 * (zs12 + xy2));
    worldPt.x = fragment.x * ((this._projectionDistance - worldPt.z) / (this._projectionDistance + 1));
    worldPt.y = fragment.y * ((this._projectionDistance - worldPt.z) / (this._projectionDistance + 1));

    // move the point in the world system
    var camEuler = FORGE.Math.rotationMatrixToEuler(this._viewer.camera.modelView);
    var rotation = FORGE.Math.eulerToRotationMatrix(-camEuler.yaw, camEuler.pitch, -camEuler.roll, true);
    worldPt.applyMatrix4(rotation);

    return new THREE.Vector3(worldPt.x, -worldPt.y, worldPt.z).normalize();
};

/**
 * Get fov computed for projection.
 * @method FORGE.ViewGoPro#getProjectionFov
 */
FORGE.ViewGoPro.prototype.getProjectionFov = function()
{
    this._updateViewParams();

    var theta = 0.5 * FORGE.Math.degToRad(this._viewer.camera.fov);

    var radius = 1.0 - this._projectionDistance / 2.0;
    var offset = Math.abs(radius - 1);

    var fov = 2 * Math.atan2(radius * Math.sin(theta), offset + radius * Math.cos(theta));

    return fov;
};

/**
 * Destroy method.
 *
 * @method FORGE.ViewGoPro#destroy
 */
FORGE.ViewGoPro.prototype.destroy = function()
{
    FORGE.ViewBase.prototype.destroy.call(this);
};


/**
 * Rectilinear view class.
 *
 * @constructor FORGE.ViewRectilinear
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {?ViewOptionsConfig} options - The view options.
 * @extends {FORGE.ViewBase}
 */
FORGE.ViewRectilinear = function(viewer, options)
{
    FORGE.ViewBase.call(this, viewer, options, "ViewRectilinear", FORGE.ViewType.RECTILINEAR);

    this._boot();
};

FORGE.ViewRectilinear.prototype = Object.create(FORGE.ViewBase.prototype);
FORGE.ViewRectilinear.prototype.constructor = FORGE.ViewRectilinear;

/**
 * Screen margin allowed for semi off screen elements (percentage)
 * @type {number}
 */
FORGE.ViewRectilinear.OFF_SCREEN_MARGIN = 0.5;

/**
 * Boot sequence.
 *
 * @method FORGE.ViewRectilinear#_boot
 * @private
 */
FORGE.ViewRectilinear.prototype._boot = function()
{
    FORGE.ViewBase.prototype._boot.call(this);

    this._shaderSTW = /** @type {ScreenToWorldProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.screenToWorld.rectilinear));
    this._shaderWTS = /** @type {WorldToScreenProgram} */ (FORGE.Utils.extendSimpleObject({}, FORGE.ShaderLib.worldToScreen.rectilinear));

    this._pitchMin = -FORGE.Math.degToRad(90);
    this._pitchMax = FORGE.Math.degToRad(90);

    this._fovMin = FORGE.Math.degToRad(40);
    this._fovMax = FORGE.Math.degToRad(140);
};

/**
 * Update view params.
 * @method FORGE.ViewRectilinear#_updateViewParams
 * @private
 */
FORGE.ViewRectilinear.prototype._updateViewParams = function()
{
    if (this._viewer !== null)
    {
        this._projectionScale = Math.tan(FORGE.Math.degToRad(this._viewer.camera.fov / 2));
    }
};

/**
 * Update uniforms.
 *
 * @method FORGE.ViewRectilinear#updateUniforms
 * @param {FORGEUniform} uniforms
 */
FORGE.ViewRectilinear.prototype.updateUniforms = function(uniforms)
{
    this._updateViewParams();

    if (typeof uniforms === "undefined")
    {
        return;
    }

    if (uniforms.hasOwnProperty("tProjectionScale"))
    {
        uniforms.tProjectionScale.value = this._projectionScale;
    }
};

/**
 * Convert a point from world space to screen space.
 *
 * @method FORGE.ViewRectilinear#worldToScreen
 * @param {THREE.Vector3} worldPt - 3D point in world space
 * @param {number} parallaxFactor - parallax factor [0..1]
 * @return {?THREE.Vector2} point in screen coordinates
 */
FORGE.ViewRectilinear.prototype.worldToScreen = function(worldPt, parallaxFactor)
{
    worldPt = worldPt || new THREE.Vector3();
    worldPt.normalize();
    parallaxFactor = parallaxFactor || 0;

    // Get point projected on unit sphere and apply camera rotation
    var worldPt4 = new THREE.Vector4(worldPt.x, worldPt.y, -worldPt.z, 1.0);
    var camEuler = FORGE.Math.rotationMatrixToEuler(this._viewer.camera.modelView);
    var rotation = FORGE.Math.eulerToRotationMatrix(camEuler.yaw, camEuler.pitch, -camEuler.roll, true);
    rotation = rotation.transpose();
    worldPt4.applyMatrix4(rotation);

    if (worldPt4.z < 0)
    {
        return null;
    }

    // Project on zn plane by dividing x,y components by z
    var projScale = Math.max(Number.EPSILON, worldPt4.z);
    var znPt = new THREE.Vector2(worldPt4.x, worldPt4.y).divideScalar(projScale);

    // Apply fov scaling
    znPt.x /= (1 + parallaxFactor) * this._projectionScale;
    znPt.y /= this._projectionScale;

    // Return fragment
    return this._fragmentToScreen(znPt);
};

/**
 * Convert a point from screen space to world space.
 *
 * @method FORGE.ViewRectilinear#screenToWorld
 * @param {THREE.Vector2} screenPt - 2D point in screen space [0..w, 0..h]
 * @return {?THREE.Vector3} world point
 */
FORGE.ViewRectilinear.prototype.screenToWorld = function(screenPt)
{
    var resolution = this._viewer.renderer.displayResolution;

    screenPt = screenPt || new THREE.Vector2(resolution.width / 2, resolution.height / 2);

    var widthMargin = FORGE.ViewRectilinear.OFF_SCREEN_MARGIN * resolution.width,
        heightMargin = FORGE.ViewRectilinear.OFF_SCREEN_MARGIN * resolution.height;
    if (screenPt.x < -widthMargin || screenPt.x > resolution.width + widthMargin
        || screenPt.y < -heightMargin || screenPt.y > resolution.height + heightMargin)
    {
        return null;
    }

    // move the point in a -1..1 square
    var fragment = this._screenToFragment(screenPt);

    // scale it (see _updateViewParams above)
    fragment.multiplyScalar(this._projectionScale);

    var worldPt = new THREE.Vector4(fragment.x, fragment.y, -1, 0);

    // move the point in the world system
    var camEuler = FORGE.Math.rotationMatrixToEuler(this._viewer.camera.modelView);
    var rotation = FORGE.Math.eulerToRotationMatrix(-camEuler.yaw, camEuler.pitch, -camEuler.roll, true);
    worldPt.applyMatrix4(rotation);

    return new THREE.Vector3(worldPt.x, -worldPt.y, worldPt.z).normalize();
};

/**
 * Destroy method.
 *
 * @method FORGE.ViewRectilinear#destroy
 */
FORGE.ViewRectilinear.prototype.destroy = function()
{
    FORGE.ViewBase.prototype.destroy.call(this);
};


/**
 * Timeline
 *
 * @constructor FORGE.Timeline
 * @param {?Array<FORGE.Keyframe>=} keyframes - Array of keyframes for this timeline.
 *
 * @extends {FORGE.BaseObject}
 */
FORGE.Timeline = function(keyframes)
{
    /**
     * Array of {@link FORGE.Keyframe} for this timeline
     * @name  FORGE.Timeline#_keyframes
     * @type {Array<FORGE.Keyframe>}
     * @private
     */
    this._keyframes = keyframes || [];

    FORGE.BaseObject.call(this, "Timeline");

    this._boot();
};

FORGE.Timeline.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Timeline.prototype.constructor = FORGE.Timeline;

/**
 * Boot routine, sort keyframes if any
 *
 * @method FORGE.Timeline#_boot
 * @private
 */
FORGE.Timeline.prototype._boot = function()
{
    if (FORGE.Utils.isArrayOf(this._keyframes, "Keyframe"))
    {
        this._keyframes = FORGE.Utils.sortArrayByProperty(this._keyframes, "time");
    }
    else
    {
        this._keyframes = [];
    }
};

/**
 * Check if this Timeline object has keyframes.
 *
 * @method  FORGE.Timeline#hasKeyframes
 * @return {boolean}
 */
FORGE.Timeline.prototype.hasKeyframes = function()
{
    return (this._keyframes !== null && this._keyframes.length > 0);
};

/**
 * Add a keyframe to the timeline.
 *
 * @method FORGE.Timeline#addKeyframe
 * @param {FORGE.Keyframe} keyframe - Keyframe to add.
 */
FORGE.Timeline.prototype.addKeyframe = function(keyframe)
{
    if (FORGE.Utils.isTypeOf(keyframe, "Keyframe"))
    {
        this._keyframes.push(keyframe);
        this._keyframes = FORGE.Utils.sortArrayByProperty(this._keyframes, "time");
    }
};

/**
 * Get an object containing the previous and next keyframes given a specific time.
 *
 * @method FORGE.Timeline#getKeyframes
 * @param  {number} time - Time to get the keyframes at
 * @return {TimelinePrevNext} The two keyframes
 */
FORGE.Timeline.prototype.getKeyframes = function(time)
{
    var indexes = this.getKeyframesIndexes(time);

    return {
        previous: this._keyframes[indexes.previous],
        next: this._keyframes[indexes.next] || null
    };
};

/**
 * Get an object containing the indexes of the previous and next keyframes given
 * a specific time.
 *
 * @method FORGE.Timeline#getKeyframesIndexes
 * @param  {number} time - Time to get the keyframes at
 * @return {TimelinePrevNextIndexes} The two indexes
 */
FORGE.Timeline.prototype.getKeyframesIndexes = function(time)
{
    var previous = -1,
        next = -1;

    // Check on each keyframe if there is one with a time shorter than
    // this one. If it exists, it is the previous keyframe
    for (var i = this._keyframes.length - 1; i >= 0; i--)
    {
        if (time >= this._keyframes[i].time)
        {
            previous = i;
            break; // Break allowed as keyframes should be sorted by time
        }
    }

    // If the previous is the last keyframes, consider that there is no more next keyframe. So only allow the set of next if there is a next keyframe.
    if (previous !== this._keyframes.length - 1)
    {
        next = previous + 1;
    }

    return {
        previous: previous,
        next: next
    };
};

/**
 * Get the side keyframes of the two keyframes associated to the given time.
 *
 * @method FORGE.Timeline#getSideKeyframes
 * @param  {number} time - Time to get the keyframes at
 * @return {TimelinePrevNext} The two keyframes
 */
FORGE.Timeline.prototype.getSideKeyframes = function(time)
{
    var indexes = this.getSideKeyframesIndexes(time);

    return {
        previous: this._keyframes[indexes.previous] || null,
        next: this._keyframes[indexes.next] || null
    };
};

/**
 * Get the indexes of the side keyframes of the two keyframes associated to the given time.
 *
 * @method FORGE.Timeline#getSideKeyframesIndexes
 * @param  {number} time - Time to get the keyframes at
 * @return {TimelinePrevNextIndexes} The two keyframes
 */
FORGE.Timeline.prototype.getSideKeyframesIndexes = function(time)
{
    var indexes = this.getKeyframesIndexes(time);

    var previous = 0,
        next = indexes.next;

    if (indexes.previous > 0)
    {
        previous = indexes.previous - 1;
    }

    if (indexes.next < this._keyframes.length - 2)
    {
        next = indexes.next + 1;
    }

    return {
        previous: previous,
        next: next
    };
};

/**
 * Clear all keyframes.
 *
 * @method FORGE.Timeline#emptyKeyframes
 */
FORGE.Timeline.prototype.emptyKeyframes = function()
{

    while (this._keyframes.length > 0)
    {
        var kf = this._keyframes.pop();
        kf.data = null;
        kf = null;
    }

    this._keyframes = [];
};

/**
 * Destroy this object.
 */
FORGE.Timeline.prototype.destroy = function()
{
    while (this._keyframes.length > 0)
    {
        var kf = this._keyframes.pop();
        kf.data = null;
        kf = null;
    }

    this._keyframes = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the keyframes array.
 * @name FORGE.Timeline#keyframes
 * @readonly
 * @type {Array<FORGE.Keyframe>}
 */
Object.defineProperty(FORGE.Timeline.prototype, "keyframes",
{
    /** @this {FORGE.Timeline} */
    get: function()
    {
        return this._keyframes;
    },

    /** @this {FORGE.Timeline} */
    set: function(value)
    {
        if (FORGE.Utils.isArrayOf(value, "Keyframe"))
        {
            this._keyframes = FORGE.Utils.sortArrayByProperty(value, "time");
        }
    }
});
/**
 * A keyframe is a data associated to a time.
 * @constructor FORGE.Keyframe
 * @param {number} time - Time of the keyframe.
 * @param {Object} data - Data bound to the keyframe.
 * @extends {FORGE.BaseObject}
 */
FORGE.Keyframe = function(time, data)
{
    FORGE.BaseObject.call(this, "Keyframe");

    /**
     * The time associated to this keyframe.
     * @name  FORGE.Keyframe#_time
     * @type {number}
     * @private
     */
    this._time = time;

    /**
     * The data associated to this keyframe.
     * @name FORGE.Keyframe#_data
     * @type {Object}
     * @private
     */
    this._data = data;
};

FORGE.Keyframe.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Keyframe.prototype.constructor = FORGE.Keyframe;

/**
 * Get and set the time of keyframe.
 * @name FORGE.Keyframe#time
 * @type {number}
 */
Object.defineProperty(FORGE.Keyframe.prototype, "time",
{
    /** @this {FORGE.Keyframe} */
    get: function()
    {
        return this._time;
    },

    /** @this {FORGE.Keyframe} */
    set: function(value)
    {
        if (typeof value === "number")
        {
            this._time = value;
        }
    }
});

/**
 * Get and set the data of keyframe.
 * @name FORGE.Keyframe#data
 * @type {Object}
 */
Object.defineProperty(FORGE.Keyframe.prototype, "data",
{
    /** @this {FORGE.Keyframe} */
    get: function()
    {
        return this._data;
    },

    /** @this {FORGE.Keyframe} */
    set: function(value)
    {
        this._data = value;
    }
});
/**
 * Abstract class allowing the animation of an element, given instructions about it.
 *
 * @constructor FORGE.Animation
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference
 * @param {*} target - The element to animate
 * @extends {FORGE.BaseObject}
 */
FORGE.Animation = function(viewer, target)
{
    /**
     * Viewer reference.
     * @name FORGE.Animation#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The target of the animation
     * @name FORGE.Animation#_target
     * @type {*}
     * @private
     */
    this._target = target || null;

    /**
     * The instruction to apply on the target.
     * @name FORGE.Animation#_instruction
     * @type {?AnimationInstruction}
     * @private
     */
    this._instruction = null;

    /**
     * Timeline reference.
     * @name FORGE.Animation#_timeline
     * @type {FORGE.Timeline}
     * @private
     */
    this._timeline = null;

    /**
     * The index of the previous keyframe in the timeline.
     * @name FORGE.Animation#_kfp
     * @type {?FORGE.Keyframe}
     * @private
     */
    this._kfp = null;

    /**
     * The index of the next keyframe in the timeline.
     * @name FORGE.Animation#_kfn
     * @type {?FORGE.Keyframe}
     * @private
     */
    this._kfn = null;

    /**
     * Current animation time.
     * @name FORGE.Animation#_time
     * @type {number}
     * @private
     */
    this._time = 0;

    /**
     * Duration of the animation
     * @name FORGE.Animation#_duration
     * @type {number}
     * @private
     */
    this._duration = 0;

    /**
     * Running flag.
     * @name FORGE.Animation#_running
     * @type {boolean}
     * @private
     */
    this._running = false;

    /**
     * Tween object.
     * @name FORGE.Animation#_tween
     * @type {FORGE.Tween}
     * @private
     */
    this._tween = null;

    /**
     * Tween time reference.
     * @name FORGE.Animation#_tweenTime
     * @type {number}
     * @private
     */
    this._tweenTime = 0;

    /**
     * Smooth interpolation flag.
     * @name FORGE.Animation#_smooth
     * @type {boolean}
     * @private
     */
    this._smooth = false;

    /**
     * Left spline for SQUAD algo.
     * @name FORGE.Animation#_sp
     * @type {THREE.Quaternion}
     * @private
     */
    this._sp = null;

    /**
     * Righ spline for SQUAD algo.
     * @name FORGE.Animation#_sn
     * @type {THREE.Quaternion}
     * @private
     */
    this._sn = null;

    /**
     * On animation play event dispatcher.
     * @name FORGE.Animation#_onPlay
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onPlay = null;

    /**
     * On animation stop event dispatcher.
     * @name FORGE.Animation#_onStop
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onStop = null;

    /**
     * On animation progress event dispatcher.
     * @name FORGE.Animation#_onProgress
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onProgress = null;

    /**
     * On animation complete event dispatcher.
     * @name FORGE.Animation#_onComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onComplete = null;

    FORGE.BaseObject.call(this, "Animation");

    this._boot();
};

FORGE.Animation.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Animation.prototype.constructor = FORGE.Animation;

/**
 * Boot sequence
 *
 * @method FORGE.Animation#_boot
 * @private
 */
FORGE.Animation.prototype._boot = function()
{
    this._register();

    // Create the tween object
    this._tween = new FORGE.Tween(this._viewer, this);
    this._tween.onStart.add(this._onTweenStartHandler, this);
    this._tween.onStop.add(this._onTweenStopHandler, this);
    this._tween.onProgress.add(this._onTweenProgressHandler, this);
    this._tween.onComplete.add(this._onTweenCompleteHandler, this);
    this._viewer.tween.add(this._tween);

    this._timeline = new FORGE.Timeline();
};

/**
 * Get the property from a target and the associated instructions
 *
 * @method FORGE.Animation#_getProp
 * @param  {string} prop - The instruction describing the access of the property
 * @param  {*} target - The target
 * @return {Array} An array composed of the property without the last accessor + the last accessor
 * @private
 */
FORGE.Animation.prototype._getProp = function(prop, target)
{
    var acc = prop.split(".");
    var res = target;

    for (var i = 0, ii = acc.length; i < ii - 1; i++)
    {
        res = res[acc[i]];
    }

    return [res, acc[i]];
};

/**
 * Interpolation evaluation function
 *
 * @method FORGE.Animation#_at
 * @param  {number} t - Current time [0...1]
 * @private
 */
FORGE.Animation.prototype._at = function(t)
{
    // If there is no next keyframe, abort
    if (this._kfn === null)
    {
        return;
    }

    // Update the time, given the current keyframes
    this._time = (t * this._duration) + this._kfp.time;

    var props = [],
        prop, i, ii;

    if (!Array.isArray(this._instruction.prop))
    {
        this._instruction.prop = [this._instruction.prop];
    }

    ii = this._instruction.prop.length;

    // Get all properties to update
    for (i = 0; i < ii; i++)
    {
        props.push(this._getProp(this._instruction.prop[i], this._target));
    }

    // If smooth, interpolate all properties together
    if (this._instruction.smooth === true && this._instruction.prop[0] === "quaternion")
    {
        // SQUAD interpolation
        prop = props[0];
        prop[0].quaternion = FORGE.Quaternion.squadNoInvert(this._kfp.data.quaternion, this._sp, this._sn, this._kfn.data.quaternion, t);
    }
    // Else no smooth interpolation, interpolate properties one by one
    else if (this._instruction.prop[0] === "quaternion")
    {
        prop = props[0];
        prop[0].quaternion = FORGE.Quaternion.slerp(this._kfp.data.quaternion, this._kfn.data.quaternion, t);
    }
    else
    {
        var alpha = this._tween.easing(t);
        var pp, pn, offset = 0;

        // For each property of the instruction
        for (i = 0; i < ii; i++)
        {
            // Get the previous and the next properties related to the current one
            pp = this._getProp(this._instruction.prop[i], this._kfp.data);
            pn = this._getProp(this._instruction.prop[i], this._kfn.data);

            prop = props[i];

            pp = pp[0][pp[1]];
            pn = pn[0][pn[1]];

            // Wrap it if necessary
            if (this._instruction.wrap && this._instruction.wrap[i])
            {
                var ilow = this._instruction.wrap[i][0];
                var ihigh = this._instruction.wrap[i][1];

                offset = Math.abs(ilow - ihigh) / 2;

                // Change the direction
                if (Math.abs(pp - pn) > offset)
                {
                    pp = FORGE.Math.wrap(pp + offset, ilow, ihigh);
                    pn = FORGE.Math.wrap(pn + offset, ilow, ihigh);
                }
                else
                {
                    offset = 0;
                }
            }

            // Update it using a classic alpha
            prop[0][prop[1]] = (1 - alpha) * pp + alpha * pn - offset;
        }
    }

    if (typeof this._instruction.fun === "function")
    {
        this._instruction.fun.call(this, t);
    }
};

/**
 * Go from the current keyframe to the next one. If no time is specified, it
 * takes the two next keyframes if there are one, given the current time.
 *
 * @method FORGE.Animation#_goTo
 * @param {number=} time - Time to start the animation at
 * @private
 */
FORGE.Animation.prototype._goTo = function(time)
{
    time = time || this._time;

    this.log("going to the next keyframes at " + time);

    // By default, increase keyframes by 1
    var kfs = this._timeline.getKeyframesIndexes(time || this._time);
    this._kfn = this._timeline.keyframes[kfs.next];
    this._kfp = this._timeline.keyframes[kfs.previous];

    if (this._kfn === null || kfs.next === -1)
    {
        return false;
    };

    // If smooth, prepare SQUAD interpolation
    if (this._smooth)
    {
        this._prepareSQUAD(time);
    }

    // Configure the tween
    this._duration = this._kfn.time - this._kfp.time;

    this._tween.to(
    {
        "tweenTime": 1
    }, this._duration);

    this._tweenTime = 0;

    // Start the tween
    this._tween.start();

    return true;
};

/**
 * Prepare the SQUAD interpolation if wanted.
 * From Shoemake 85 at SIGGRAPH
 * http://run.usc.edu/cs520-s12/assign2/p245-shoemake.pdf
 *
 * @method FORGE.Animation#_prepareSQUAD
 * @param {number} time - Time to start the animation at
 * @private
 */
FORGE.Animation.prototype._prepareSQUAD = function(time)
{
    var sideKfs = this._timeline.getSideKeyframes(time);

    if (sideKfs.previous === null || sideKfs.next === null)
    {
        return;
    }

    // Quaternions to join with a path
    var qp = new THREE.Quaternion().copy(this._kfp.data.quaternion).normalize();
    var qn = new THREE.Quaternion().copy(this._kfn.data.quaternion).normalize();

    // Quaternions before and after the previous ones, used for interpolation
    var qpp = new THREE.Quaternion().copy(sideKfs.previous.data.quaternion).normalize();
    var qnn = new THREE.Quaternion().copy(sideKfs.next.data.quaternion).normalize();

    // Compute the splines used for describing the path of the camera
    this._sp = FORGE.Quaternion.spline(qpp, qp, qn);
    this._sn = FORGE.Quaternion.spline(qp, qn, qnn);
};

/**
 * On tween start event handler
 *
 * @method FORGE.Animation#_onTweenStartHandler
 * @private
 */
FORGE.Animation.prototype._onTweenStartHandler = function()
{
    // Don't dispatch the event if the animation is already running
    if (this._running !== true)
    {
        if (this._onPlay !== null)
        {
            this._onPlay.dispatch();
        }

        this._running = true;
    }
};

/**
 * On tween stop event handler
 *
 * @method FORGE.Animation#_onTweenStopHandler
 * @private
 */
FORGE.Animation.prototype._onTweenStopHandler = function()
{
    this._running = false;

    if (this._onStop !== null)
    {
        this._onStop.dispatch();
    }
};

/**
 * On tween progress event handler
 *
 * @method FORGE.Animation#_onTweenProgressHandler
 * @private
 */
FORGE.Animation.prototype._onTweenProgressHandler = function()
{
    // Interpolate
    this._at(this._tweenTime);

    if (this._onProgress !== null)
    {
        this._onProgress.dispatch(
        {
            progress: this._time / this._duration
        });
    }
};

/**
 * On tween complete event handler
 *
 * @method FORGE.Animation#_onTweenCompleteHandler
 * @private
 */
FORGE.Animation.prototype._onTweenCompleteHandler = function()
{
    // Don't dispatch the event until the whole animation is stopped
    if (!this._goTo())
    {
        if (this._onComplete !== null)
        {
            this._onComplete.dispatch();
        }
    }
};

/**
 * Play the animation.
 *
 * @method  FORGE.Animation#play
 * @param {number=} time - Time to start the animation at
 */
FORGE.Animation.prototype.play = function(time)
{
    // Reset the time
    this._time = time || 0;

    // Start the animation
    this._goTo(this._time);
};

/**
 * Stop the animation.
 *
 * @method  FORGE.Animation#stop
 */
FORGE.Animation.prototype.stop = function()
{
    this._tween.stop();
};

/**
 * Resume the animation.
 *
 * @method  FORGE.Animation#resume
 */
FORGE.Animation.prototype.resume = function()
{
    this._goTo();
};

/**
 * Destroy sequence.
 *
 * @method FORGE.Animation#destroy
 */
FORGE.Animation.prototype.destroy = function()
{
    if (this._timeline !== null)
    {
        this._timeline.destroy();
        this._timeline = null;
    }

    if (this._tween !== null)
    {
        this._tween.destroy();
        this._tween = null;
    }

    if (this._onPlay !== null)
    {
        this._onPlay.destroy();
        this._onPlay = null;
    }

    if (this._onStop !== null)
    {
        this._onStop.destroy();
        this._onStop = null;
    }

    if (this._onProgress !== null)
    {
        this._onProgress.destroy();
        this._onProgress = null;
    }

    if (this._onComplete !== null)
    {
        this._onComplete.destroy();
        this._onComplete = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Accessors to the running flag
 * @name FORGE.Animation#running
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Animation.prototype, "running",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._running;
    }
});

/**
 * Accessors to timeline
 * @name FORGE.Animation#timeline
 * @readonly
 * @type {FORGE.Timeline}
 */
Object.defineProperty(FORGE.Animation.prototype, "timeline",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._timeline;
    }
});

/**
 * Accessors to instruction.
 * @name FORGE.Animation#instruction
 * @type {AnimationInstruction}
 */
Object.defineProperty(FORGE.Animation.prototype, "instruction",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._instruction;
    },

    /** @this {FORGE.Animation} */
    set: function(value)
    {
        this._instruction = value;
    }
});

/**
 * Accessors to tween.
 * @name FORGE.Animation#tween
 * @type {number}
 */
Object.defineProperty(FORGE.Animation.prototype, "tween",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._tween;
    },

    /** @this {FORGE.Animation} */
    set: function(value)
    {
        this._tween = value;
    }
});

/**
 * Accessors to normalized tween time.
 * @name FORGE.Animation#tweenTime
 * @type {number}
 */
Object.defineProperty(FORGE.Animation.prototype, "tweenTime",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._tweenTime;
    },

    /** @this {FORGE.Animation} */
    set: function(value)
    {
        this._tweenTime = value;
    }
});

/**
 * Accessors to smooth.
 * @name FORGE.Animation#smooth
 * @type {number}
 */
Object.defineProperty(FORGE.Animation.prototype, "smooth",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        return this._smooth;
    },

    /** @this {FORGE.Animation} */
    set: function(value)
    {
        this._smooth = value;
    }
});

/**
 * Get the "onPlay" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.Animation#onPlay
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Animation.prototype, "onPlay",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        if (this._onPlay === null)
        {
            this._onPlay = new FORGE.EventDispatcher(this);
        }

        return this._onPlay;
    }
});

/**
 * Get the "onStop" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.Animation#onStop
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Animation.prototype, "onStop",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        if (this._onStop === null)
        {
            this._onStop = new FORGE.EventDispatcher(this);
        }

        return this._onStop;
    }
});

/**
 * Get the "onProgress" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.Animation#onProgress
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Animation.prototype, "onProgress",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        if (this._onProgress === null)
        {
            this._onProgress = new FORGE.EventDispatcher(this);
        }

        return this._onProgress;
    }
});

/**
 * Get the "onComplete" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.Animation#onComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Animation.prototype, "onComplete",
{
    /** @this {FORGE.Animation} */
    get: function()
    {
        if (this._onComplete === null)
        {
            this._onComplete = new FORGE.EventDispatcher(this);
        }

        return this._onComplete;
    }
});

/**
 * A meta-animation, used to provide basic functionnality for the interface
 * between an ObjectAnimation and FORGE.Animation.
 *
 * @constructor FORGE.MetaAnimation
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {*} target - Target reference.
 * @param {string} name - Name of the ObjectAnimation.
 * @extends {FORGE.BaseObject}
 */
FORGE.MetaAnimation = function(viewer, target, name)
{
    /**
     * Viewer reference.
     * @name FORGE.MetaAnimation#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The target of the animation
     * @name FORGE.MetaAnimation#_target
     * @type {*}
     * @private
     */
    this._target = target || null;

    /**
     * The array of animations related to the camera.
     * @name FORGE.MetaAnimation#_animations
     * @type {Array<FORGE.Animation>}
     * @private
     */
    this._animations = null;

    /**
     * The number of finished animations.
     * @name FORGE.MetaAnimation#_finished
     * @type {number}
     * @private
     */
    this._finished = 0;

    /**
     * The instructions to apply on the target.
     * @name FORGE.MetaAnimation#_instructions
     * @type {Array<AnimationInstruction>}
     * @private
     */
    this._instructions = null;

    /**
     * On animation complete event dispatcher.
     * @name FORGE.MetaAnimation#_onComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onComplete = null;

    FORGE.BaseObject.call(this, name);
};

FORGE.MetaAnimation.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.MetaAnimation.prototype.constructor = FORGE.MetaAnimation;

/**
 * Stop the animations.
 *
 * @method FORGE.MetaAnimation#stop
 */
FORGE.MetaAnimation.prototype.stop = function()
{
    if (this._animations === null)
    {
        return;
    }

    for (var i = 0, ii = this._animations.length; i < ii; i++)
    {
        this._animations[i].stop();
    }
};

/**
 * Resume the animations.
 *
 * @method FORGE.MetaAnimation#resume
 */
FORGE.MetaAnimation.prototype.resume = function()
{
    if (this._animations === null)
    {
        return;
    }

    for (var i = 0, ii = this._animations.length; i < ii; i++)
    {
        this._animations[i].resume();
    }
};

/**
 * On animation complete event handler.
 *
 * @method FORGE.Animation#_onTrackPartialCompleteHandler
 * @private
 */
FORGE.MetaAnimation.prototype._onTrackPartialCompleteHandler = function()
{
    this._finished++;

    if (this._finished === this._animations.length && this._onComplete !== null)
    {
        this._onComplete.dispatch();
    }
};

/**
 * Compute the intermediate value of a property, approximatly.
 *
 * @method FORGE.MetaAnimation#_computeIntermediateValue
 * @param  {number} time - the time when to interpolate
 * @param  {Array<FORGE.Keyframe>} kfs - The list of keyframes
 * @param  {string} prop - The name of the property
 * @param  {Function} easing - The name of the easing property
 * @param  {string=} ns - The namespace to add before the property
 * @return {number} the interpolated property
 * @private
 */
FORGE.MetaAnimation.prototype._computeIntermediateValue = function(time, kfs, prop, easing, ns)
{
    var kp = null,
        kn = null,
        res;

    // Sort by time
    kfs = FORGE.Utils.sortArrayByProperty(kfs, "time");

    // Search for the two nearest keyframes
    for (var i = 0; i < kfs.length; i++)
    {
        var p = (typeof ns === "string") ? kfs[i].data[ns][prop] : kfs[i].data[prop];

        if (typeof p !== "undefined" && p !== null)
        {
            // Update previous key if time is still inferior
            if (kfs[i].time < time)
            {
                kp = kfs[i];
            }

            // Update next key on first one and break the loop
            if (kfs[i].time > time)
            {
                kn = kfs[i];
                break;
            }
        }
    }

    // If no kp, let's consider it is the data of the current target
    if (kp === null)
    {
        kp = {
            time: 0,
            data: {}
        };

        if (typeof ns === "string")
        {
            kp.data[ns] = this._target[ns];
        }
        else
        {
            kp.data[prop] = this._target[prop];
        }
    }

    // If no kn, then nothing needs to be done
    if (kn === null)
    {
        return (typeof ns === "string") ? kp.data[ns][prop] : kp.data[prop];
    }

    // Now that we have the two, we can compute the proportion of the time
    var ratio = (time - kp.time) / (kn.time - kp.time);
    var alpha = easing(ratio);

    // Return the interpolated property
    var kpp = (typeof ns === "string") ? kp.data[ns][prop] : kp.data[prop];
    var knp = (typeof ns === "string") ? kn.data[ns][prop] : kn.data[prop];
    return (1 - alpha) * kpp + alpha * knp;
};

/**
 * Empty the array of animations.
 *
 * @method FORGE.MetaAnimation#_emptyAnimations
 * @private
 */
FORGE.MetaAnimation.prototype._emptyAnimations = function()
{
    var animation;

    while (this._animations && this._animations.length > 0)
    {
        animation = this._animations.pop();
        animation.onComplete.remove(this._onTrackPartialCompleteHandler, this);
        animation.destroy();
    }

    this._animations = [];
    this._finished = 0;
};

/**
 * Destroy sequence.
 * @method FORGE.MetaAnimation#destroy
 */
FORGE.MetaAnimation.prototype.destroy = function()
{
    this._emptyAnimations();

    this._animations = null;

    var instruction;

    while (this._instructions && this._instructions.length > 0)
    {
        instruction = this._instructions.pop();
        instruction = null;
    }

    this._instructions = null;

    if (this._onComplete !== null)
    {
        this._onComplete.destroy();
        this._onComplete = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Accessors to timeline
 * @name FORGE.MetaAnimation#running
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.MetaAnimation.prototype, "running",
{
    /** @this {FORGE.MetaAnimation} */
    get: function()
    {
        for(var i = 0, ii = this._animations.length; i < ii; i++)
        {
            if(this._animations[i].running === true)
            {
                return true;
            }
        }

        return false;
    }
});

/**
 * Get the "onComplete" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.MetaAnimation#onComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.MetaAnimation.prototype, "onComplete",
{
    /** @this {FORGE.MetaAnimation} */
    get: function()
    {
        if (this._onComplete === null)
        {
            this._onComplete = new FORGE.EventDispatcher(this);
        }

        return this._onComplete;
    }
});

/**
 * Abstract class for Track used in animations. Contains three basic elements: name, description and keyframe. Useful for i18n. It also contains the number of time the track has been played.
 *
 * @constructor FORGE.Track
 * @param {string} name - The name of the class
 * @extends {FORGE.BaseObject}
 */
FORGE.Track = function(name)
{
    /**
     * Name of the track
     * @name FORGE.Track#_name
     * @type {?string}
     * @private
     */
    this._name = null;

    /**
     * Description of the track
     * @name FORGE.Track#_description
     * @type {?string}
     * @private
     */
    this._description = null;

    /**
     * List of keyframes composing the track
     * @name FORGE.Track#_keyframes
     * @type {Array}
     * @private
     */
    this._keyframes = null;

    /**
     * This is the number of times this track has been played.
     * @name  FORGE.Track#_count
     * @type {number}
     * @private
     */
    this._count = 0;

    /**
     * Easing curve for the tween between each keyframe.
     * @name FORGE.Track#_easing
     * @type {?string}
     * @private
     */
    this._easing = null;

    /**
     * Time to take to reach the current position to the first one.
     * @name  FORGE.Track#_offset
     * @type {number}
     * @private
     */
    this._offset = 0;

    FORGE.BaseObject.call(this, name);
};

FORGE.Track.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Track.prototype.constructor = FORGE.Track;

/**
 * Boot sequence
 *
 * @method FORGE.Track#_boot
 * @param  {Object} config - The information on the track
 * @private
 */
FORGE.Track.prototype._boot = function(config)
{
    this._uid = config.uid;
    this._name = config.name;
    this._description = config.description;
    this._keyframes = config.keyframes;

    this._easing = "linear";
    this._offset = 0;

    if (typeof config.easing === "object" && config.easing !== null)
    {
        this._easing = config.easing.default || this._easing;
        this._offset = config.easing.offset || this._offset;
    }

    this._register();
};

FORGE.Track.prototype.destroy = function()
{
    this._unregister();
};

/**
 * Get the name of the track.
 * @name FORGE.Track#name
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Track.prototype, "name",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        return this._name;
    }
});

/**
 * Get the description of the track.
 * @name FORGE.Track#description
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Track.prototype, "description",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        return this._description;
    }
});

/**
 * Get the keyframes of the track.
 * @name FORGE.Track#keyframes
 * @readonly
 * @type {Array}
 */
Object.defineProperty(FORGE.Track.prototype, "keyframes",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        return this._keyframes;
    }
});

/**
 * Get the number of times this director track has been played.
 * @name FORGE.Track#count
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Track.prototype, "count",
{
    /** @this {FORGE.Track} */
    set: function(value)
    {
        if (typeof value === "number")
        {
            this._count = value;
        }
    },

    /** @this {FORGE.Track} */
    get: function()
    {
        return this._count;
    }
});

/**
 * Get the easing of the track.
 * @name FORGE.Track#easing
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Track.prototype, "easing",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        return this._easing;
    }
});

/**
 * Get the starting time of the track.
 * @name FORGE.Track#offset
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Track.prototype, "offset",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        return this._offset;
    }
});

/**
 * Get the total duration of the track.
 * @name FORGE.Track#duration
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Track.prototype, "duration",
{
    /** @this {FORGE.Track} */
    get: function()
    {
        var res = this._keyframes[0].time;

        for (var i = 1, ii = this._keyframes.length; i < ii; i++)
        {
            if (this._keyframes[i].time > res)
            {
                res = this._keyframes[i].time;
            }
        }

        return res;
    }
});


/**
 * ForgeJS Base 3d object with a mesh and events mechanisms.
 * @constructor FORGE.Object3D
 * @param {FORGE.Viewer} viewer - Viewer reference.
 * @param {string=} className - Class name for objects that extends Object3D
 * @extends {FORGE.BaseObject}
 */
FORGE.Object3D = function(viewer, className)
{
    /**
     * Viewer reference
     * @name  FORGE.Object3D#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * THREE Mesh
     * @name FORGE.Object3D#_mesh
     * @type {THREE.Mesh}
     * @private
     */
    this._mesh = null;

    /**
     * The fx set id applied to the object
     * @name  FORGE.Object3D#_fx
     * @type {string}
     * @private
     */
    this._fx = "";

    /**
     * Visibility flag
     * @name  FORGE.Object3D#_visible
     * @type {boolean}
     * @private
     */
    this._visible = true;

    /**
     * Is this object is interactive / raycastable
     * @name FORGE.Object3D#_interactive
     * @type {boolean}
     * @private
     */
    this._interactive = true;

    /**
     * Events object that will keep references of the ActionEventDispatcher
     * @name FORGE.Object3D#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = null;

    /**
     * The hovered flag is set to true when any pointer is over the object.
     * @name FORGE.Object3D#_hovered
     * @type {boolean}
     * @private
     */
    this._hovered = false;

    /**
     * Color based on 3D Object id used for picking.
     * @name FORGE.Object3D#_pickingColor
     * @type {THREE.Color}
     * @private
     */
    this._pickingColor = null;

    /**
     * Is ready?
     * @name  FORGE.Object3D#_ready
     * @type {boolean}
     * @private
     */
    this._ready = false;

    /**
     * Click event dispatcher
     * @name FORGE.Object3D#_onClick
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onClick = null;

    /**
     * Over event dispatcher
     * @name FORGE.Object3D#_onOver
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onOver = null;

    /**
     * Out event dispatcher
     * @name FORGE.Object3D#_onOut
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onOut = null;

    /**
     * Event dispatcher for ready state.
     * @name FORGE.Object3D#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    FORGE.BaseObject.call(this, className || "Object3D");

    this._boot();
};

FORGE.Object3D.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Object3D.prototype.constructor = FORGE.Object3D;

/**
 * Boot sequence.
 * @method FORGE.Object3D#_boot
 */
FORGE.Object3D.prototype._boot = function()
{
    this._events = {};
    this._mesh = new THREE.Mesh();
    this._pickingColor = FORGE.PickingDrawPass.colorFrom3DObject(this._mesh);
    this._viewer.renderer.objects.register(this);
};

/**
 * Create action events dispatchers.
 * @method FORGE.Object3D#_createEvents
 * @private
 * @param {Object} events - The events config of the 3d object.
 */
FORGE.Object3D.prototype._createEvents = function(events)
{
    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all object events.
 * @method FORGE.Object3D#_clearEvents
 * @private
 */
FORGE.Object3D.prototype._clearEvents = function()
{
    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Triggers actions for the click event
 * @method FORGE.Object3D#click
 */
FORGE.Object3D.prototype.click = function()
{
    this.log("click " + this._mesh.id);

    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onClick, "ActionEventDispatcher") === true)
    {
        this._events.onClick.dispatch();
    }

    // Classical event dispatcher
    if(this._onClick !== null)
    {
        this._onClick.dispatch();
    }
};

/**
 * Triggers actions for the over event
 * @method FORGE.Object3D#over
 */
FORGE.Object3D.prototype.over = function()
{
    this.log("over " + this._mesh.id);

    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onOver, "ActionEventDispatcher") === true)
    {
        this._events.onOver.dispatch();
    }

    // Classical event dispatcher
    if(this._onOver !== null)
    {
        this._onOver.dispatch();
    }
};

/**
 * Triggers actions for the out event
 * @method FORGE.Object3D#out
 */
FORGE.Object3D.prototype.out = function()
{
    this.log("out " + this._mesh.id);

    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onOut, "ActionEventDispatcher") === true)
    {
        this._events.onOut.dispatch();
    }

    // Classical event dispatcher
    if(this._onOut !== null)
    {
        this._onOut.dispatch();
    }
};

/**
 * Destroy sequence
 * @method FORGE.Object3D#destroy
 */
FORGE.Object3D.prototype.destroy = function()
{
    this._viewer.renderer.objects.unregister(this);

    if (this._mesh !== null)
    {
        this._mesh.userData = null;

        if (this._mesh.geometry !== null)
        {
            this._mesh.geometry.dispose();
            this._mesh.geometry = null;
        }

        this._mesh.material = null;

        this._mesh = null;
    }

    if(this._onClick !== null)
    {
        this._onClick.destroy();
        this._onClick = null;
    }

    if(this._onOver !== null)
    {
        this._onOver.destroy();
        this._onOver = null;
    }

    if(this._onOut !== null)
    {
        this._onOut.destroy();
        this._onOut = null;
    }

    if(this._onReady !== null)
    {
        this._onReady.destroy();
        this._onReady = null;
    }

    this._clearEvents();
    this._events = null;

    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * 3D object mesh
 * @name FORGE.Object3D#mesh
 * @readonly
 * @type {THREE.Mesh}
  */
Object.defineProperty(FORGE.Object3D.prototype, "mesh",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._mesh;
    }
});

/**
 * Get and set the visible flag
 * @name FORGE.Object3D#visible
 * @type {boolean}
 */
Object.defineProperty(FORGE.Object3D.prototype, "visible",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._visible;
    },
    /** @this {FORGE.Object3D} */
    set: function(value)
    {
        this._visible = Boolean(value);

        if(this._mesh !== null)
        {
            this._mesh.visible = this._visible;
        }
    }
});

/**
 * Get and set the hovered flag
 * @name FORGE.Object3D#hovered
 * @type {boolean}
 */
Object.defineProperty(FORGE.Object3D.prototype, "hovered",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._hovered;
    },
    /** @this {FORGE.Object3D} */
    set: function(value)
    {
        this._hovered = Boolean(value);
    }
});


/**
 * Get and set the interactive flag
 * @name FORGE.Object3D#interactive
 * @type {boolean}
 */
Object.defineProperty(FORGE.Object3D.prototype, "interactive",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._interactive;
    },
    /** @this {FORGE.Object3D} */
    set: function(value)
    {
        this._interactive = Boolean(value);
    }
});

/**
 * Get the FX applied to this object
 * @name FORGE.Object3D#fx
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Object3D.prototype, "fx",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._fx;
    }
});

/**
 * Get the events of this object
 * @name FORGE.Object3D#events
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Object3D.prototype, "events",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._events;
    }
});

/**
 * Object3D ready flag
 * @name FORGE.Object3D#ready
 * @readonly
 * @type boolean
  */
Object.defineProperty(FORGE.Object3D.prototype, "ready",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get the onClick {@link FORGE.EventDispatcher}.
 * @name FORGE.Object3D#onClick
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Object3D.prototype, "onClick",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        if (this._onClick === null)
        {
            this._onClick = new FORGE.EventDispatcher(this);
        }

        return this._onClick;
    }
});

/**
 * Get the onOver {@link FORGE.EventDispatcher}.
 * @name FORGE.Object3D#onOver
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Object3D.prototype, "onOver",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        if (this._onOver === null)
        {
            this._onOver = new FORGE.EventDispatcher(this);
        }

        return this._onOver;
    }
});

/**
 * Get the onOut {@link FORGE.EventDispatcher}.
 * @name FORGE.Object3D#onOut
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Object3D.prototype, "onOut",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        if (this._onOut === null)
        {
            this._onOut = new FORGE.EventDispatcher(this);
        }

        return this._onOut;
    }
});

/**
 * Get the onReady {@link FORGE.EventDispatcher}.
 * @name FORGE.Object3D#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Object3D.prototype, "onReady",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        if (this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});


/**
 * @constructor FORGE.ObjectRenderer
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.ObjectRenderer = function(viewer)
{
    /**
     * Viewer reference
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * List of object that have to be rendered
     * @name FORGE.ObjectRenderer#_objects
     * @type {Array<FORGE.Object3D>}
     * @private
     */
    this._objects = null;

    /**
     * Array of render passes
     * @type {Array<FORGE.RenderScene>}
     * @private
     */
    this._renderScenes = null;

    FORGE.BaseObject.call(this, "ObjectRenderer");

    this._boot();
};

FORGE.ObjectRenderer.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ObjectRenderer.prototype.constructor = FORGE.ObjectRenderer;

/**
 * Init routine
 * @method FORGE.ObjectRenderer#_boot
 * @private
 */
FORGE.ObjectRenderer.prototype._boot = function()
{
    this._objects = [];
    this._renderScenes = [];
};

/**
 * Get 3d objects by FX
 * @method  FORGE.ObjectRenderer#_getByFX
 * @param  {?string} fx - The fx name you want to use to filter 3d objects. If undefined or null, will return 3d objects without fx.
 * @return {Array<FORGE.Object3D>}
 * @private
 */
FORGE.ObjectRenderer.prototype._getByFX = function(fx)
{
    var result = [];

    if(typeof fx === "undefined" || fx === "" || fx === null)
    {
        result = this._objects.filter(function(hs)
        {
            return (typeof hs.fx === "undefined" || hs.fx === "" || hs.fx === null);
        });
    }
    else
    {
        result = this._objects.filter(function(hs)
        {
            return hs.fx === fx;
        });
    }

    return result;
};

/**
 * Get 3d objects that have no fx
 * @method  FORGE.ObjectRenderer#_getWithoutFX
 * @return {Array<FORGE.Object3D>}
 * @private
 */
FORGE.ObjectRenderer.prototype._getWithoutFX = function()
{
    var result = this._getByFX(null);
    return result;
};

/**
 * Get 3d objects that have fx
 * @method  FORGE.ObjectRenderer#_getWithFX
 * @return {Array<FORGE.Object3D>}
 * @private
 */
FORGE.ObjectRenderer.prototype._getWithFX = function()
{
    var withoutFX = this._getWithoutFX();
    var result = FORGE.Utils.arrayByDifference(this._objects, withoutFX);
    return result;
};

/**
 * Get single fx list used by all objects
 * @method  FORGE.ObjectRenderer#_getFX
 * @return {Array<string>}
 * @private
 */
FORGE.ObjectRenderer.prototype._getFX = function()
{
    var withFX = this._getWithFX();

    var result = withFX.reduce(function(list, spot)
    {
        if (list.indexOf(spot.fx) < 0)
        {
            list.push(spot.fx);
        }

        return list;

    }, []);

    return result;
};

/**
 * Register an object to the object renderer
 * @method FORGE.ObjectRenderer#register
 * @param  {FORGE.Object3D} object - The object to register
 */
FORGE.ObjectRenderer.prototype.register = function(object)
{
    this._objects.push(object);
};

/**
 * Unregister an object from the object renderer.
 * @method  FORGE.ObjectRenderer#unregister
 * @param {FORGE.Object3D} object - The object to unregister from the object renderer.
 */
FORGE.ObjectRenderer.prototype.unregister = function(object)
{
    this._objects.splice(this._objects.indexOf(object), 1);
};

/**
 * @method FORGE.ObjectRenderer#createRenderScenes
 */
FORGE.ObjectRenderer.prototype.createRenderScenes = function()
{
    // First get all 3d objects without any FX and create a render pass for them
    // Then get all other 3d objects (with some FX), extract FX list and create
    // as many render passes as needed (one for each FX set).

    var camera = this._viewer.renderer.camera.main;

    // Get list of objects without any fx
    var withoutFX = this._getWithoutFX();

    // Create a render pass for them
    if (withoutFX.length > 0)
    {
        var scene = new THREE.Scene();

        for (var i = 0, ii = withoutFX.length; i < ii; i++)
        {
            scene.add(withoutFX[i].mesh);
        }

        var renderScene = new FORGE.RenderScene(this._viewer, scene, camera, null);
        this._renderScenes.push(renderScene);
    }

    var fxList = this._getFX();

    // For each FX in the list, create a render scene and assign all 3d objects
    // with the FX to it
    for (var j = 0, jj = fxList.length; j < jj; j++)
    {
        var fx = fxList[j];
        var renderList = this._getByFX(fx);
        var sceneFx = new THREE.Scene();

        for (var k = 0, kk = renderList.length; k < kk; k++)
        {
            sceneFx.add(renderList[k].mesh);
        }

        var fxSet = this._viewer.postProcessing.getFxSetByUID(fx);

        var renderScene = new FORGE.RenderScene(this._viewer, sceneFx, camera, fxSet);
        this._renderScenes.push(renderScene);
    }
};

/**
 * Get 3d objects that are eligible to raycast (interactive)
 * @method  FORGE.ObjectRenderer#getRaycastable
 * @return {Array<FORGE.Object3D>}
 */
FORGE.ObjectRenderer.prototype.getRaycastable = function()
{
    var result = this._objects.filter(function(object)
    {
        return (object.ready === true && object.interactive === true);
    });

    return result;
};

/**
 * Get a 3d object from it's mesh
 * @method  FORGE.ObjectRenderer#getByMesh
 * @return {?FORGE.Object3D}
 */
FORGE.ObjectRenderer.prototype.getByMesh = function(mesh)
{
    for(var i = 0, ii = this._objects.length; i < ii; i++)
    {
        if(this._objects[i].mesh === mesh)
        {
            return this._objects[i];
        }
    }

    return null;
};

/**
 * Clear all render scenes
 * @method FORGE.ObjectRenderer#clear
 */
FORGE.ObjectRenderer.prototype.clear = function()
{
    var count = this._renderScenes.length;
    while(count--)
    {
        var renderScene = this._renderScenes.pop();
        renderScene.destroy();
    }
};

/**
 * Destroy sequence
 * @method FORGE.ObjectRenderer#destroy
 */
FORGE.ObjectRenderer.prototype.destroy = function()
{
    this.clear();
    this._renderScenes = null;
    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get all the objects
 * @name FORGE.ObjectRenderer#all
 * @type {Array<FORGE.Object3D>}
 */
Object.defineProperty(FORGE.ObjectRenderer.prototype, "all",
{
    /** @this {FORGE.ObjectRenderer} */
    get: function()
    {
        return this._objects;
    }
});


/**
 * Get background renderer render items array.
 * @name FORGE.ObjectRenderer#renderScenes
 * @type {Array<FORGE.RenderScene>}
 */
Object.defineProperty(FORGE.ObjectRenderer.prototype, "renderScenes",
{
    /** @this {FORGE.ObjectRenderer} */
    get: function()
    {
        return this._renderScenes;
    }
});



/**
 * Raycaster for mouse interaction with 3d objects of the scene
 * @constructor FORGE.Raycaster
 * @param {FORGE.Viewer} viewer - Viewer reference
 * @extends {FORGE.BaseObject}
 */
FORGE.Raycaster = function(viewer)
{
    /**
     * Viewer reference
     * @name FORGE.Raycaster#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * THREE raycaster instance.
     * @name FORGE.Raycaster#_raycaster
     * @type {THREE.Raycaster}
     * @private
     */
    this._raycaster = null;

    /**
     * The last hovered 3d object
     * @name FORGE.Raycaster#_hoveredObject
     * @type {?FORGE.Object3D}
     * @private
     */
    this._hoveredObject = null;

    FORGE.BaseObject.call(this, "Raycaster");

    this._boot();
};

FORGE.Raycaster.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Raycaster.prototype.constructor = FORGE.Raycaster;

/**
 * Boot sequence
 * @method FORGE.Raycaster#_boot
 * @private
 */
FORGE.Raycaster.prototype._boot = function()
{
    // this._viewer.story.onSceneLoadComplete.add(this._sceneLoadCompleteHandler, this);
};

/**
 * Start raycasting.
 * Add event on the main canvas for mouse interaction
 * @method FORGE.Raycaster#_start
 * @param {string} mode - picking mode
 */
FORGE.Raycaster.prototype.start = function(mode)
{
    this.log("Raycaster start");

    if (this._raycaster === null)
    {
        this._raycaster = new THREE.Raycaster();
    }

    if (mode === FORGE.PickingManager.modes.POINTER)
    {
        this._startPointer();
    }
    else if (mode === FORGE.PickingManager.modes.GAZE)
    {
        this._startGaze();
    }
};

/**
 * Add the listeners for the pointer mode.
 * @method FORGE.Raycaster#_startPointer
 * @private
 */
FORGE.Raycaster.prototype._startPointer = function()
{
    this.log("Raycating start pointer");

    if (this._viewer.canvas.pointer.onClick.has(this._canvasPointerClickHandler, this) === false)
    {
        this._viewer.canvas.pointer.onClick.add(this._canvasPointerClickHandler, this);
    }

    if (this._viewer.canvas.pointer.onMove.has(this._canvasPointerMoveHandler, this) === false)
    {
        this._viewer.canvas.pointer.onMove.add(this._canvasPointerMoveHandler, this);
    }
};

/**
 * Add the listeners for the gaze mode.
 * @method FORGE.Raycaster#_startGaze
 * @private
 */
FORGE.Raycaster.prototype._startGaze = function()
{
    this.log("Raycating start gaze");

    if (this._viewer.renderer.camera.onChange.has(this._cameraChangeHandler, this) === false)
    {
        this._viewer.renderer.camera.onChange.add(this._cameraChangeHandler, this);
    }
};

/**
 * Stop raycasting, removes all event listeners
 * @method FORGE.Raycaster#_stop
 */
FORGE.Raycaster.prototype.stop = function()
{
    this.log("Raycaster stop");

    // Remove pointer event listeners
    if (this._viewer.canvas.pointer.onClick.has(this._canvasPointerClickHandler, this))
    {
        this._viewer.canvas.pointer.onClick.remove(this._canvasPointerClickHandler, this);
    }

    if (this._viewer.canvas.pointer.onMove.has(this._canvasPointerMoveHandler, this))
    {
        this._viewer.canvas.pointer.onMove.remove(this._canvasPointerMoveHandler, this);
    }

    //Remove gaze handlers
    if (this._viewer.renderer.camera.onChange.has(this._cameraChangeHandler, this))
    {
        this._viewer.renderer.camera.onChange.remove(this._cameraChangeHandler, this);
    }

    this._out();

    this._raycaster = null;
};


/**
 * Pointer click handler, launch raycasting
 * @method FORGE.Raycaster#_canvasPointerClickHandler
 * @param {Object} event click event
 * @private
 */
FORGE.Raycaster.prototype._canvasPointerClickHandler = function(event)
{
    var position = FORGE.Pointer.getRelativeMousePosition(event.data);
    this._raycast("click", position);
};

/**
 * Pointer over handler, launch raycasting
 * @method FORGE.Raycaster#_canvasPointerMoveHandler
 * @param {Object} event move event
 * @private
 */
FORGE.Raycaster.prototype._canvasPointerMoveHandler = function(event)
{
    var position = FORGE.Pointer.getRelativeMousePosition(event.data);
    this._raycast("move", position);
};

/**
 * Camera change handler, launch raycasting
 * @method FORGE.Raycaster#_cameraChangeHandler
 * @private
 */
FORGE.Raycaster.prototype._cameraChangeHandler = function()
{
    this._raycast("move");
};

/**
 * Raycasting internal method
 * @method FORGE.Raycaster#_raycast
 * @param {string} event - triggering event
 * @param {THREE.Vector2=} screenPoint - raycasting point in screen coordinates, if no screen point defined, it will raycast in the center of the view.
 * @private
 */
FORGE.Raycaster.prototype._raycast = function(event, screenPoint)
{
    // If there is an hovered object but not ready (maybe the texture is updating), ignore the raycast
    if(this._hoveredObject !== null && this._hoveredObject.ready === false)
    {
        return;
    }

    var resolution = this._viewer.renderer.canvasResolution;

    screenPoint = screenPoint || new THREE.Vector2(resolution.width / 2, resolution.height / 2);

    var camera = this._viewer.renderer.camera;
    var ndc = new THREE.Vector2(screenPoint.x / resolution.width, 1.0 - screenPoint.y / resolution.height).multiplyScalar(2).addScalar(-1);

    // Get all objects of the object renderer that are ready and interactive
    var objects = this._viewer.renderer.objects.getRaycastable();

    //Reset the hovered flag to false for all the objects
    this._setHovered(objects, false);

    // Get the first intersected object
    var intersected = this._intersect(objects, ndc, camera);

    if (intersected !== null)
    {
        if (event === "click")
        {
            this._hoveredObject = intersected; // For mobile we have to set the last intersect as the hovered object.
            this.click();
        }
        else if (event === "move")
        {

            if (intersected !== this._hoveredObject)
            {
                this._out();

                this._hoveredObject = intersected;
                intersected.over();

                if (this._viewer.renderer.pickingManager.mode === FORGE.PickingManager.modes.GAZE)
                {
                    this._viewer.renderer.camera.gaze.start();
                }
            }
        }
    }
    else
    {
        this._out();
    }
};

/**
 * The out routine. Stops the timer, trigger the out method the hovered object and nullify its reference.
 * @method FORGE.Raycaster#_out
 * @private
 */
FORGE.Raycaster.prototype._out = function()
{
    if (this._hoveredObject !== null)
    {
        this._hoveredObject.out();
        this._hoveredObject = null;

        if (this._viewer.renderer.pickingManager.mode === FORGE.PickingManager.modes.GAZE)
        {
            this._viewer.renderer.camera.gaze.stop();
        }
    }
};

/**
 * Set the hovered flag to an array ok {@link FORGE.Object3D}
 * @method FORGE.Raycaster#_setHovered
 * @param {Array<FORGE.Object3D>} objects - The array of 3d objects who'll be affected
 * @param {boolean} hovered - The hovered flag you want to set to all the 3d objects of the array
 * @private
 */
FORGE.Raycaster.prototype._setHovered = function(objects, hovered)
{
    for (var i = 0, ii = objects.length; i < ii; i++)
    {
        objects[i].hovered = hovered;
    }
};

/**
 * Intrsect 3d objects from a ndc point and a camera
 * @param  {Array<FORGE.Object3D>} objects - Array of object to test
 * @param  {THREE.Vector2} ndc - Normalized device coordinate point
 * @param  {FORGE.Camera} camera - Camera to use
 * @return {?FORGE.Object3D} Return the first hitted 3d Object
 * @private
 */
FORGE.Raycaster.prototype._intersect = function(objects, ndc, camera)
{
    var result = null;

    // Set the three raycaster with the ndc coordinates and the camera.
    this._raycaster.setFromCamera(ndc, camera.main);

    // Make an array of meshes for the three raycaster
    var meshes = [];
    for (var i = 0, ii = objects.length; i < ii; i++)
    {
        meshes.push(objects[i].mesh);
    }

    // Get all objects intersected by the ray
    var intersects = this._raycaster.intersectObjects(meshes);

    while (result === null && intersects.length > 0)
    {
        var intersect = intersects.shift();
        var mesh = intersect.object;
        var uv = intersect.uv;
        var object = this._viewer.renderer.objects.getByMesh(mesh);

        if (mesh.material.transparent === false)
        {
            result = object;
        }
        else
        {
            var color = this._getObjectColorFromUVCoords(object, uv);

            // If color is null we consider that we are hitting the object but its texture is not ready
            if (color === null || (color != null && color.alpha > 10))
            {
                result = object;
            }
        }
    }

    return result;
};

/**
 * Get RGBA color at a given point on a texture
 * @method FORGE.Raycaster#_getObjectColorFromUVCoords
 * @param {THREE.Vector2} uv point with normalized coordinates
 * @return {FORGE.ColorRGBA} RGBA color at given point or null if image is not available
 * @private
 */
FORGE.Raycaster.prototype._getObjectColorFromUVCoords = function(object, uv)
{
    var color = null;

    if (object.mesh !== null && object.mesh.material !== null && typeof object.mesh.material.uniforms.tTexture !== "undefined")
    {
        var texture = object.mesh.material.uniforms.tTexture.value;
        var canvas = null;

        if (texture !== null && texture.image !== null)
        {
            if (texture.image.nodeName.toLowerCase() === "img")
            {
                canvas = document.createElement("canvas");
                canvas.width = texture.image.width;
                canvas.height = texture.image.height;
                canvas.getContext("2d").drawImage(texture.image, 0, 0);
            }
            else if (texture.image.nodeName.toLowerCase() === "canvas")
            {
                canvas = texture.image;
            }

            if (canvas !== null)
            {
                var w = canvas.width;
                var h = canvas.height;
                var x = Math.floor(uv.x * w);
                var y = Math.floor(h - uv.y * h);
                var idx = 4 * (y * w + x);

                var data = canvas.getContext("2d").getImageData(0, 0, w, h).data;
                color = new FORGE.ColorRGBA(data[idx + 0], data[idx + 1], data[idx + 2], data[idx + 3]);
            }
        }
    }
    else if (typeof object.mesh.material.color !== "undefined")
    {
        color = object.mesh.material.color;
    }

    return color;
};
/**
 * Triggers the click method of the hovered object if exists.
 * @method FORGE.Raycaster#click
 */
FORGE.Raycaster.prototype.click = function()
{
    if (this._hoveredObject !== null)
    {
        this._hoveredObject.click();
    }
};

/**
 * Clear raycaster.
 * Release all references related to the scene
 * @method FORGE.Raycaster#clear
 */
FORGE.Raycaster.prototype.clear = function()
{
    this._hoveredObject = null;
};

/**
 * Get the hovered object
 * @name FORGE.Raycaster#hoveredObject
 * @type {FORGE.Object3D}
 * @readonly
 */
Object.defineProperty(FORGE.Raycaster.prototype, "hoveredObject",
{
    /** @this {FORGE.Raycaster} */
    get: function()
    {
        return this._hoveredObject;
    }
});
/**
 * Picking draw pass
 * 3D object picker based on additional draw passes rendering objects with a single color based on object ID
 * Bottleneck: should be used carefully as it uses gl.readPixels to access render target texture syncing CPU and GPU
 *
 * @constructor FORGE.PickingDrawPass
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.PickingDrawPass = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.PickingDrawPass#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Picking render target reference.
     * @name FORGE.PickingDrawPass#_target
     * @type {THREE.WebGLRenderTarget}
     * @private
     */
    this._target = null;

    /**
     * Picking material reference
     * Should be used to override all objects materials in some scene
     * @name FORGE.PickingDrawPass#_material
     * @type {THREE.Material}
     * @private
     */
    this._material = null;

    /**
     * The last hovered 3d object
     * @name FORGE.PickingDrawPass#_hoveredObject
     * @type {?FORGE.Object3D}
     * @private
     */
    this._hoveredObject = null;

    /**
     * Target scaling factor to reduce texture memory footprint
     * @name FORGE.PickingDrawPass#_scaling
     * @type {number}
     * @private
     */
    this._scaling = 3;

    FORGE.BaseObject.call(this, "PickingDrawPass");

    this._boot();
};

FORGE.PickingDrawPass.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PickingDrawPass.prototype.constructor = FORGE.PickingDrawPass;


/**
 * Boot sequence
 * @method FORGE.PickingDrawPass#_boot
 * @private
 */
FORGE.PickingDrawPass.prototype._boot = function()
{
    // Create picking material reference
    // Replace fragment shader with picking shader only drawing objects with one flat colour
    var shader = FORGE.Utils.clone(this._viewer.renderer.view.current.shaderWTS.mapping);
    shader.uniforms.tColor = { type: "c", value: new THREE.Color( 0x000000 ) };

    this._material = new THREE.RawShaderMaterial({
        fragmentShader: FORGE.ShaderLib.parseIncludes(FORGE.ShaderChunk.wts_frag_color),
        vertexShader: FORGE.ShaderLib.parseIncludes(shader.vertexShader),
        uniforms: shader.uniforms,
        side: THREE.FrontSide,
        name: "PickingMaterial"
    });

    // Create picking render target
    // Should be RGBA to allow WebGLRenderer to read pixels
    var rtParams =
    {
        format: THREE.RGBAFormat
    };

    var width = this._viewer.renderer.canvasResolution.width / this._scaling;
    var height = this._viewer.renderer.canvasResolution.height / this._scaling;

    this._target = new THREE.WebGLRenderTarget(width, height, rtParams);
    this._target.name = "Picking RenderTarget";
};

/**
 * Get object by its mesh id
 * @method FORGE.PickingDrawPass#_getObjectByID
 * @param {number} id - The ID of the object to get
 * @return {?FORGE.Object3D}
 * @private
 */
FORGE.PickingDrawPass.prototype._getObjectByID = function(id)
{
    var objects = this._viewer.renderer.objects.all;

    for (var i = 0, ii = objects.length; i < ii; i++)
    {
        if (objects[i].mesh.id === id)
        {
            return objects[i];
        }
    }

    return null;
};

/**
 * Convert object into a color based on its unique identifier
 * @method FORGE.PickingDrawPass#_colorFrom3DObject
 * @return {THREE.Color}
 */
FORGE.PickingDrawPass.colorFrom3DObject = function(object)
{
    return new THREE.Color(object.id);
};

/**
 * Convert color into object ID
 * @method FORGE.PickingDrawPass#_colorTo3DObject
 * @private
 * @return {FORGE.Object3D}
 */
FORGE.PickingDrawPass.prototype._colorTo3DObject = function(color)
{
    var id = ((color.r & 0x0000ff) << 16) | ((color.g & 0x0000ff) << 8) + (color.b & 0x0000ff);
    return this._getObjectByID(id);
};

/**
 * Get object located at a given normalized set of coordinates
 * @method FORGE.PickingDrawPass#getObjectAtXnYn
 * @param {number} xn - x normalized coordinate
 * @param {number} yn - y normalized coordinate
 * @return {FORGE.Object3D}
 * @private
 */
FORGE.PickingDrawPass.prototype._getObjectAtXnYn = function(xn, yn)
{
    var renderer = this._viewer.renderer.webGLRenderer;

    var data = new Uint8Array(4);
    renderer.readRenderTargetPixels(this._target,
                                    xn * this._target.width,
                                    yn * this._target.height,
                                    1, 1, data );

    return this._colorTo3DObject(new THREE.Color(data[0], data[1], data[2]));
};

/**
 * Get object under pointer event
 * @method FORGE.PickingDrawPass#_getObjectUnderPointerEvent
 * @private
 * @return {FORGE.Object3D}
 */
FORGE.PickingDrawPass.prototype._getObjectUnderPointerEvent = function(event)
{
    var e = event.data;
    var position = FORGE.Pointer.getRelativeMousePosition(e);
    var xn = position.x / event.data.target.width;
    var yn = 1 - position.y / event.data.target.height;
    return this._getObjectAtXnYn(xn, yn);
};

/**
 * Pointer click handler
 * @method FORGE.PickingDrawPass#_canvasPointerClickHandler
 * @private
 */
FORGE.PickingDrawPass.prototype._canvasPointerClickHandler = function(event)
{
    var object = this._getObjectUnderPointerEvent(event);

    if (object === null)
    {
        return;
    }

    object.click();
};

/**
 * Pointer move handler
 * @method FORGE.PickingDrawPass#_canvasPointerMoveHandler
 * @private
 */
FORGE.PickingDrawPass.prototype._canvasPointerMoveHandler = function(event)
{
    var object = this._getObjectUnderPointerEvent(event);

    if (object === null)
    {
        if (this._hoveredObject !== null)
        {
            this._hoveredObject.out();
            this._hoveredObject = null;
        }

        return;
    }

    if (object !== this._hoveredObject)
    {
        // Pointer goes from one object to another one directly
        if (this._hoveredObject !== null)
        {
            this._hoveredObject.out();
        }

        // Invoke over method and store new hovered object
        object.over();
        this._hoveredObject = object;
    }
};


/**
 * Start picking
 * @method FORGE.PickingDrawPass#start
 */
FORGE.PickingDrawPass.prototype.start = function()
{
    this._viewer.canvas.pointer.enabled = true;

    if (this._viewer.canvas.pointer.onClick.has(this._canvasPointerClickHandler, this) === false)
    {
        this._viewer.canvas.pointer.onClick.add(this._canvasPointerClickHandler, this);
    }

    if (this._viewer.canvas.pointer.onMove.has(this._canvasPointerMoveHandler, this) === false)
    {
        this._viewer.canvas.pointer.onMove.add(this._canvasPointerMoveHandler, this);
    }
};

/**
 * Stop picking
 * @method FORGE.PickingDrawPass#start
 */
FORGE.PickingDrawPass.prototype.stop = function()
{
    if (this._viewer.canvas.pointer.onClick.has(this._canvasPointerClickHandler, this))
    {
        this._viewer.canvas.pointer.onClick.remove(this._canvasPointerClickHandler, this);
    }

    if (this._viewer.canvas.pointer.onMove.has(this._canvasPointerMoveHandler, this))
    {
        this._viewer.canvas.pointer.onMove.remove(this._canvasPointerMoveHandler, this);
    }
};

/**
 * Triggers the click method of the hovered object if exists.
 * @method FORGE.PickingDrawPass#click
 */
FORGE.PickingDrawPass.prototype.click = function()
{
    if(this._hoveredObject !== null && this._viewer.canvas.pointer.onClick.has(this._canvasPointerClickHandler, this))
    {
        this._hoveredObject.click();
    }
};

/**
 * Set size (resolution)
 * @method FORGE.PickingDrawPass#setSize
 * @param {FORGE.Size} size - size [px]
 */
FORGE.PickingDrawPass.prototype.setSize = function(size)
{
    if (this._target !== null)
    {
        this._target.setSize(size.width, size.height);
    }
};

/**
 * Clear draw pass.
 * Release all references related to the scene
 * @method FORGE.PickingDrawPass#clear
 */
FORGE.PickingDrawPass.prototype.clear = function()
{
    this._hoveredObject = null;
};

/**
 * Destroy routine
 * @method FORGE.PickingDrawPass#destroy
 */
FORGE.PickingDrawPass.prototype.destroy = function()
{
    if (this._material !== null)
    {
        this._material.dispose();
        this._material = null;
    }

    if (this._target !== null)
    {
        this._target.dispose();
        this._target = null;
    }

    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get material.
 * @name FORGE.PickingDrawPass#material
 * @type {THREE.RawShaderMaterial}
 */
Object.defineProperty(FORGE.PickingDrawPass.prototype, "material",
{
    /** @this {FORGE.PickingDrawPass} */
    get: function()
    {
        return this._material;
    }
});

/**
 * Get render target.
 * @name FORGE.PickingDrawPass#renderTarget
 * @type {THREE.WebGLRenderTarget}
 */
Object.defineProperty(FORGE.PickingDrawPass.prototype, "renderTarget",
{
    /** @this {FORGE.PickingDrawPass} */
    get: function()
    {
        return this._target;
    }
});

/**
 * Picking manager
 *
 * @constructor FORGE.PickingManager
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.PickingManager = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.RenderManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Picking draw pass.
     * @name FORGE.PickingManager#_pickingDrawPass
     * @type {FORGE.PickingDrawPass}
     * @private
     */
    this._pickingDrawPass = null;

    /**
     * Raycaster.
     * @name FORGE.PickingManager#_raycaster
     * @type {FORGE.Raycaster}
     * @private
     */
    this._raycaster = null;

    /**
     * The current picking mode (pointer or gaze)
     * @name {FORGE.PickingManager#_mode}
     * @type {string}
     * @private
     */
    this._mode = FORGE.PickingManager.modes.POINTER;

    FORGE.BaseObject.call(this, "PickingManager");

    this._boot();
};

FORGE.PickingManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PickingManager.prototype.constructor = FORGE.PickingManager;

/**
 * List diffrent modes for picking
 * @type {Object}
 * @const
 */
FORGE.PickingManager.modes = {};

/**
 * This mode is for picking the pointer position (like mouse / touch)
 * @type {string}
 * @const
 */
FORGE.PickingManager.modes.POINTER = "pointer";

/**
 * This mode is for gaze picking for VR, always pick in the middle of the view.
 * @type {string}
 * @const
 */
FORGE.PickingManager.modes.GAZE = "gaze";

/**
 * Boot sequence
 * @method FORGE.PickingManager#_boot
 * @private
 */
FORGE.PickingManager.prototype._boot = function()
{

};

/**
 * Setup and start raycast picking
 * Destroy draw pass picking if it exists
 * @method FORGE.PickingManager#_startRaycasting
 * @private
 */
FORGE.PickingManager.prototype._startRaycastPicking = function()
{
    if (this._raycaster !== null)
    {
        return;
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.stop();
        this._pickingDrawPass.destroy();
        this._pickingDrawPass = null;
    }

    this._raycaster = new FORGE.Raycaster(this._viewer);
    this._raycaster.start(this._mode);
};

/**
 * Setup and start drawpass picking
 * Destroy raycaster if it exists
 * @method FORGE.PickingManager#_startDrawpassPicking
 * @private
 */
FORGE.PickingManager.prototype._startDrawpassPicking = function()
{
    if (this._pickingDrawPass !== null)
    {
        return;
    }

    if (this._raycaster !== null)
    {
        this._raycaster.stop();
        this._raycaster.destroy();
        this._raycaster = null;
    }

    this._pickingDrawPass = new FORGE.PickingDrawPass(this._viewer);
    this._pickingDrawPass.start();
};

/**
 * Update internals for a given view type
 * @method FORGE.PickingManager#updateForViewType
 * @param {string} type - view type
 */
FORGE.PickingManager.prototype.updateForViewType = function(type)
{
    // Check view type: rectilinear will use raycasting, otherwise picking draw passes
    if (type === FORGE.ViewType.RECTILINEAR)
    {
        this._startRaycastPicking();
    }
    else
    {
        this._startDrawpassPicking();
    }
};

/**
 * Clear picking manager
 * @method FORGE.PickingManager#clear
 */
FORGE.PickingManager.prototype.clear = function()
{
    if (this._raycaster !== null)
    {
        this._raycaster.clear();
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.clear();
    }
};

/**
 * Update size (resolution)
 * @method FORGE.PickingManager#setSize
 * @param {FORGE.Size} size - size [px]
 */
FORGE.PickingManager.prototype.setSize = function(size)
{
    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.setSize(size);
    }
};

/**
 * Trigger a click on hovered oject (if it exists)
 * @method FORGE.PickingManager#click
 */
FORGE.PickingManager.prototype.click = function()
{
    if (this._raycaster !== null)
    {
        this._raycaster.click();
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.click();
    }
};

/**
 * Start picking
 * @method FORGE.PickingManager#start
 */
FORGE.PickingManager.prototype.start = function()
{
    if (this._raycaster !== null)
    {
        this._raycaster.start(this._mode);
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.start();
    }
};

/**
 * Stop picking
 * @method FORGE.PickingManager#stop
 */
FORGE.PickingManager.prototype.stop = function()
{
    if (this._raycaster !== null)
    {
        this._raycaster.stop();
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.stop();
    }
};

/**
 * Destroy routine
 * @method FORGE.PickingManager#destroy
 */
FORGE.PickingManager.prototype.destroy = function()
{
    if (this._raycaster !== null)
    {
        this._raycaster.stop();
        this._raycaster.destroy();
        this._raycaster = null;
    }

    if (this._pickingDrawPass !== null)
    {
        this._pickingDrawPass.stop();
        this._pickingDrawPass.destroy();
        this._pickingDrawPass = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get picking render target.
 * @name FORGE.PickingManager#renderTarget
 * @type {THREE.WebGLRenderTarget}
 */
Object.defineProperty(FORGE.PickingManager.prototype, "renderTarget",
{
    /** @this {FORGE.PickingManager} */
    get: function()
    {
        if (this._pickingDrawPass === null)
        {
            return null;
        }

        return this._pickingDrawPass.renderTarget;
    }
});

/**
 * Get picking material.
 * @name FORGE.PickingManager#material
 * @type {THREE.Material}
 */
Object.defineProperty(FORGE.PickingManager.prototype, "material",
{
    /** @this {FORGE.PickingManager} */
    get: function()
    {
        if (this._pickingDrawPass === null)
        {
            return null;
        }

        return this._pickingDrawPass.material;
    }
});

/**
 * Get and set the raycaster mode between pointer or gaze
 * @name FORGE.PickingManager#mode
 * @type {string}
 */
Object.defineProperty(FORGE.PickingManager.prototype, "mode",
{
    /** @this {FORGE.PickingManager} */
    get: function()
    {
        return this._mode;
    },

    /** @this {FORGE.PickingManager} */
    set: function(value)
    {
        if(value !== FORGE.PickingManager.modes.POINTER && value !== FORGE.PickingManager.modes.GAZE)
        {
            return;
        }

        if(value !== this._mode)
        {
            this._mode = value;
            this.start();
        }
    }
});


/**
 * Tile class.
 *
 * @constructor FORGE.Tile
 * @param {?FORGE.Tile} parent - parent tile reference
 * @param {FORGE.BackgroundPyramidRenderer} renderer - renderer reference
 * @param {number} x - x coordinate (column)
 * @param {number} y - y coordinate (row)
 * @param {number} level - pyramid level
 * @param {string} face - cube face
 * @param {string} creator - string describing what created the tile
 * @extends {THREE.Mesh}
 */
FORGE.Tile = function(parent, renderer, x, y, level, face, creator)
{
    /**
     * String describing what created the tile
     * @type {string}
     */
    this._creator = creator;

    /**
     * Reference on background renderer
     * @name FORGE.Tile#_renderer
     * @type {FORGE.BackgroundPyramidRenderer}
     * @private
     */
    this._renderer = renderer;

    /**
     * X axis value on face coordinate system
     * @name FORGE.Tile#_x
     * @type {number}
     * @private
     */
    this._x = x;

    /**
     * Y axis value on face coordinate system
     * @name FORGE.Tile#_y
     * @type {number}
     * @private
     */
    this._y = y;

    /**
     * Resolution level
     * @name FORGE.Tile#_level
     * @type {number}
     * @private
     */
    this._level = level;

    /**
     * Cube face
     * @name FORGE.Tile#_face
     * @type {string}
     * @private
     */
    this._face = face;

    /**
     * Creation timestamp
     * @name FORGE.Tile#_createTS
     * @type {?number}
     * @private
     */
    this._createTS = null;

    /**
     * Last display timestamp
     * @name FORGE.Tile#_displayTS
     * @type {?number}
     * @private
     */
    this._displayTS = null;

    /**
     * Reference on parent tile
     * @name FORGE.Tile#_parent
     * @type {FORGE.Tile}
     * @private
     */
    this._parent = parent;

    /**
     * Reference on children tiles
     * @name FORGE.Tile#_children
     * @type {Array<FORGE.Tile>}
     * @private
     */
    this._children = null;

    /**
     * Array of references on neighbour tiles
     * @name FORGE.Tile#_neighbours
     * @type {Array<FORGE.Tile>}
     * @private
     */
    this._neighbours = null;

    /**
     * Flag to know if parent has been checked
     * @name FORGE.MediaStore#_parentNeedsCheck
     * @type {boolean}
     * @private
     */
    this._parentNeedsCheck = true;

    /**
     * Flag to know if neighbourhood has been checked
     * @name FORGE.MediaStore#_neighborsNeedCheck
     * @type {boolean}
     * @private
     */
    this._neighborsNeedCheck = true;

    /**
     * Global opacity value
     * @name FORGE.Tile#_opacity
     * @type {number}
     * @private
     */
    this._opacity = 0;

    /**
     * Texture set flag
     * @name FORGE.Tile#_textureIsSet
     * @type {boolean}
     * @private
     */
    this._textureIsSet = false;

    /**
     * Event dispatcher for destroy.
     * @name FORGE.Tile#_onDestroy
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onDestroy = null;

    THREE.Mesh.call(this);

    this._boot();
};

FORGE.Tile.prototype = Object.create(THREE.Mesh.prototype);
FORGE.Tile.prototype.constructor = FORGE.Tile;

/**
 * Cube faces table
 * @type {Array}
 */
FORGE.Tile.FACES = ["front", "right", "back", "left", "up", "down"];

/**
 * Preview tile
 * @type {number}
 */
FORGE.Tile.PREVIEW = -Infinity;

/**
 * Opacity increment [unit per render cycle]
 * @type {number}
 */
FORGE.Tile.OPACITY_INCREMENT = 0.04;

/**
 * Opacity decrement [unit per render cycle]
 * @type {number}
 */
FORGE.Tile.OPACITY_DECREMENT = 0.01;

/**
 * Texture load predelay (time between creation and display)
 * @type {number}
 */
FORGE.Tile.TEXTURE_LOADING_PREDELAY_MS = 200;

/**
 * Table describing previous cube face
 * @type {CubeFaceObject}
 */
FORGE.Tile.FACE_PREVIOUS = {
    "front": "left",
    "right": "front",
    "back": "right",
    "left": "back",
    "up": "up",
    "down": "down"
};

/**
 * Table describing next cube face
 * @type {CubeFaceObject}
 */
FORGE.Tile.FACE_NEXT = {
    "front": "right",
    "right": "back",
    "back": "left",
    "left": "front",
    "up": "up",
    "down": "down"
};

/**
 * Create tile name
 * @method FORGE.Tile#createName
 * @param {string|number} face - cube face
 * @param {number} level - pyramid level
 * @param {number} x - x coordinate (column)
 * @param {number} y - y coordinate (row)
 */
FORGE.Tile.createName = function(face, level, x, y)
{
    face = typeof face === "number" ? FORGE.Tile.FACES[face] : face.toLowerCase();
    if (level === FORGE.Tile.PREVIEW)
    {
        return face.substring(0, 1).toUpperCase() + "-preview";
    }

    return face.substring(0, 1).toUpperCase() + "-" + level + "-" + y + "-" + x;
};

/**
 * Get the coordinates of the parent tile
 * @method FORGE.Tile#getParentTileCoordinates
 * @param {FORGE.Tile} tile - tile
 * @return {THREE.Vector2} parent tile x,y coordinates
 */
FORGE.Tile.getParentTileCoordinates = function(tile)
{
    var px = Math.floor(tile.x / 2),
        py = Math.floor(tile.y / 2);

    return new THREE.Vector2(px, py);
};

/**
 * Boot sequence
 * @method FORGE.Tile#_boot
 * @private
 */
FORGE.Tile.prototype._boot = function()
{
    this._neighbours = [];
    this._children = [];

    this.name = FORGE.Tile.createName(this._face, this._level, this._x, this._y);

    // Always ensure a new tile has a parent tile
    // This will prevent from zomming out into some empty area
    if (this._level > 0 && this._parent === null)
    {
        this._checkParent();
    }

    this.renderOrder = this._level === FORGE.Tile.PREVIEW ? 0 : 2 * (this._level + 1);
    this.onBeforeRender = this._onBeforeRender.bind(this);
    this.onAfterRender = this._onAfterRender.bind(this);

    this._setGeometry();

    // Level 0 objects are opaque to be rendered first
    var transparent = (this._level !== FORGE.Tile.PREVIEW);
    this._opacity = transparent ? 0 : 1;

    this.material = new THREE.MeshBasicMaterial(
    {
        color: new THREE.Color(0x000000),
        transparent: transparent,
        opacity: this._opacity,
        depthTest: false,
        side: THREE.FrontSide
    });

    this._queryTexture();

    if (FORGE.Tile.DEBUG === true)
    {
        this._addDebugLayer();
    }

    this._createTS = Date.now();
};

/**
 * Before render callback
 * This is called by THREE.js render routine before render is done
 * Add to background renderer list and set opacity
 * @method FORGE.Tile#_onBeforeRender
 * @private
 */
FORGE.Tile.prototype._onBeforeRender = function()
{
    // Add to renderer render list
    this._renderer.addToRenderList(this);

    if (this._textureIsSet === true)
    {
        this._setOpacity(1);
    }
    else
    {
        this._setOpacity(0);
    }
};

/**
 * After render callback
 * This is called by THREE.js render routine after render has been done
 * Here we update all links to other tiles, ask for subdivision or neighbours
 * Request texture if none
 * @method FORGE.Tile#_onAfterRender
 * @private
 */
FORGE.Tile.prototype._onAfterRender = function()
{
    // Update last display timestamp
    this.refreshDisplayTS();

    if (this._level !== FORGE.Tile.PREVIEW)
    {
        // Check if tile should be divided
        if (this._renderer.level > this._level)
        {
            this._subdivide();

            // Restoration process for required tiles previously removed from the scene
            // Check if children are intersecting the frustum and add them back to the
            // scene (with refreshed display timer)
            for (var i=0, ii=this._children.length; i<ii; i++)
            {
                var child = this._children[i];

                if (!this._renderer.isObjectInScene(child) && this._renderer.isObjectInFrustum(child))
                {
                    this._renderer.scene.add(child);
                    child.refreshDisplayTS();
                }
            }
        }

        // Get all neighbour tiles references
        this._checkNeighbours();
    }

    this._queryTexture();
};

/**
 * Query texture for the tile
 * @method FORGE.Tile#_queryTexture
 * @private
 */
FORGE.Tile.prototype._queryTexture = function()
{
    // Update texture mapping
    if (this.material !== null && this.material.map === null && this._textureIsSet === false)
    {
        // Check if predelay since creation has been respected (except for preview)
        if ((this._level !== FORGE.Tile.PREVIEW || this._level !== this._renderer.level) &&
            this._displayTS - this._createTS < FORGE.Tile.TEXTURE_LOADING_PREDELAY_MS)
        {
            return;
        }

        var texPromise = this._renderer.textureStore.get(this);

        if (texPromise !== null)
        {
            texPromise.then(function(texture)
            {
                // Check if tile has been destroyed in the meantime
                if (this.material === null)
                {
                    return;
                }

                if (texture !== null && texture instanceof THREE.Texture)
                {
                    this._textureIsSet = true;

                    texture.generateMipmaps = false;
                    texture.minFilter = THREE.LinearFilter;
                    texture.needsUpdate = true;

                    this.material.color = new THREE.Color(0xffffff);
                    this.material.map = texture;
                    this.material.needsUpdate = true;
                }
            }.bind(this),

            function(error)
            {
                console.warn("Tile texture loading error: " + error);
            }.bind(this));
        }
    }
};

/**
 * Add graphical debug layer
 * @method FORGE.Tile#_addDebugLayer
 * @private
 */
FORGE.Tile.prototype._addDebugLayer = function()
{
    var canvas = document.createElement("canvas");
    canvas.width = canvas.height = 512;
    var ctx = canvas.getContext("2d");

    var x = canvas.width / 2;
    var y = canvas.height / 2 - 25;

    ctx.fillStyle = "gray";
    ctx.strokeStyle = "white";
    ctx.strokeRect(20, 20, canvas.width - 40, canvas.height - 40);

    // General font style
    ctx.textAlign = "center";
    ctx.textAlign = "center";
    ctx.textBaseline = "middle";
    ctx.fillStyle = "white";

    ctx.font = "16px Courier";
    ctx.fillText("TILE " + this.name, x, y);
    y += 40;

    var fontSize = 12;
    ctx.font = fontSize + "px Courier";
    var ceiling = canvas.width - 20;

    ctx.textAlign = "left";
    ctx.font = "10px Courier";
    if (this._level === FORGE.Tile.PREVIEW)
    {
        ctx.fillText("Preview", 10, canvas.height - 10);
    }
    else
    {
        ctx.fillText("Level " + this._level, 10, canvas.height - 10);
    }

    ctx.textAlign = "right";
    ctx.fillText(this._renderer.pixelsAtCurrentLevelHumanReadable, canvas.width - 10, canvas.height - 10);

    var texture = new THREE.Texture(canvas);
    texture.needsUpdate = true;

    var material = new THREE.MeshBasicMaterial(
    {
        transparent: true,
        map: texture,
        depthTest: false
    });
    var mesh = new THREE.Mesh(this.geometry.clone(), material);
    mesh.name = this.name + "-debug-canvas";
    this.add(mesh);

    mesh.renderOrder = this.renderOrder + 1;
};

/**
 * Set opacity of the tile and its children (recursive)
 * @method FORGE.Tile#_setOpacity
 * @param {number} opacity - tile opacity
 * @private
 */
FORGE.Tile.prototype._setOpacity = function(opacity)
{
    opacity = FORGE.Math.clamp(opacity, 0, 1);

    this._opacity = opacity;

    var setNodeOpacity = function(node)
    {
        if (node !== null && node.material !== null &&
            typeof node.material.opacity !== "undefined")
        {
            if (node.material.transparent === true)
            {
                node.material.opacity = opacity;
            }
        }

        for (var i = 0, ii = node.children.length; i < ii; i++)
        {
            setNodeOpacity(node.children[i]);
        }
    };

    setNodeOpacity(this);
};

/**
 * Set geometry of the tile
 * This means rotation and position in world coordinates
 * @method FORGE.Tile#_setGeometry
 * @private
 */
FORGE.Tile.prototype._setGeometry = function()
{
    var tx = this._renderer.nbTilesPerAxis(this._level, "x");
    var ty = this._renderer.nbTilesPerAxis(this._level, "y");

    var fullTileWidth = this._renderer.cubeSize / tx;
    var fullTileHeight = this._renderer.cubeSize / ty;

    var scaleX = this._x < Math.floor(tx) ? 1 : tx - Math.floor(tx);
    var scaleY = this._y < Math.floor(ty) ? 1 : ty - Math.floor(ty);

    var tileSize = new FORGE.Size(scaleX * fullTileWidth, scaleY * fullTileHeight);

    this.geometry = new THREE.PlaneBufferGeometry(tileSize.width, tileSize.height);

    var baseOffset = -this._renderer.cubeSize * 0.5;

    // position = tile offset in XY plane - half cube size + half tile size
    var position = new THREE.Vector3(
        baseOffset + fullTileWidth * this._x + 0.5 * tileSize.width,
        -(baseOffset + fullTileHeight * this._y + 0.5 * tileSize.height),
        baseOffset);

    var rotation = this._getRotation();
    this.rotation.copy(rotation);

    position.applyEuler(rotation);
    this.position.copy(position);
};

/**
 * Get rotation of the tile
 * @method FORGE.Tile#_setRotation
 * @private
 */
FORGE.Tile.prototype._getRotation = function()
{
    if (this._parent !== null)
    {
        return this._parent.rotation;
    }

    switch (this._face)
    {
        case "front":
            return new THREE.Euler(0, 0, 0);

        case "back":
            return new THREE.Euler(0, Math.PI, 0);

        case "left":
            return new THREE.Euler(0, Math.PI / 2, 0);

        case "right":
            return new THREE.Euler(0, -Math.PI / 2, 0);

        case "up":
            return new THREE.Euler(Math.PI / 2, 0, 0);

        case "down":
            return new THREE.Euler(-Math.PI / 2, 0, 0);
    }
};

/**
 * Subdivide tile into n tiles
 * @method FORGE.Tile#subdivide
 * @private
 */
FORGE.Tile.prototype._subdivide = function()
{
    if (this._children.length > 0)
    {
        return;
    }

    var tile;
    var level = this._level + 1;

    if (this._level === FORGE.Tile.PREVIEW)
    {
        level = 0;
    }

    // tiles per axis on children level
    var tx = this._renderer.nbTilesPerAxis(level, "x");
    var ty = this._renderer.nbTilesPerAxis(level, "y");
    // ratio of tiles on this level compared to children level
    var rx = Math.round(tx / this._renderer.nbTilesPerAxis(this._level, "x"));
    var ry = Math.round(ty / this._renderer.nbTilesPerAxis(this._level, "y"));

    // load first tile
    tile = this._renderer.getTile(this, level, this._face, rx * this._x, ry * this._y, "top left quarter of " + this.name);
    if (this._children.indexOf(tile) === -1)
    {
        tile.onDestroy.add(this._onChildTileDestroyed, this);
        this._children.push(tile);
    }

    // get all other children in
    var xn, yn, xn_in, yn_in;
    for (var i = 0; i < rx; i++)
    {
        for (var j = 0; j < ry; j++)
        {
            xn = rx * this._x + i;
            yn = ry * this._y + j;

            xn_in = xn < tx;
            yn_in = yn < ty;

            if (xn_in === true && yn_in === true)
            {
                tile = this._renderer.getTile(this, level, this._face, xn, yn, "children of " + this.name);
                if (this._children.indexOf(tile) === -1)
                {
                    tile.onDestroy.add(this._onChildTileDestroyed, this);
                    this._children.push(tile);
                }
            }
        }
    }
};

/**
 * Child tile destroy event handler
 * @method FORGE.Tile#_onChildTileDestroyed
 */
FORGE.Tile.prototype._onChildTileDestroyed = function(event)
{
    var tile = event.emitter;
    tile.onDestroy.remove(this._onChildTileDestroyed, this);

    if (this._children === null)
    {
        return;
    }

    var idx = this._children.indexOf(tile);

    if (idx !== -1)
    {
        this._children.splice(idx, 1);
    }
};

/**
 * Neighbour tile destroy event handler
 * @method FORGE.Tile#_onNeighbourTileDestroyed
 */
FORGE.Tile.prototype._onNeighbourTileDestroyed = function(event)
{
    var tile = event.emitter;
    tile.onDestroy.remove(this._onNeighbourTileDestroyed, this);

    this._neighborsNeedCheck = true;

    if (this._neighbours === null)
    {
        return;
    }

    var idx = this._neighbours.indexOf(tile);

    if (idx !== -1)
    {
        this._neighbours.splice(idx, 1);
    }
};

/**
 * Parent tile destroy event handler
 * @method FORGE.Tile#_onParentTileDestroyed
 */
FORGE.Tile.prototype._onParentTileDestroyed = function(event)
{
    var tile = event.emitter;
    tile.onDestroy.remove(this._onParentTileDestroyed, this);

    this._parent = null;
    this._parentNeedsCheck = true;
};

/**
 * Get parent tile
 * @method FORGE.Tile#_checkParent
 * @private
 */
FORGE.Tile.prototype._checkParent = function()
{
    if (this._parent !== null ||
        this._parentNeedsCheck === false ||
        this._level <= 0)
    {
        return;
    }

    this._parentNeedsCheck = false;

    var sequence = Promise.resolve();
    sequence.then(function()
    {
        this._parent = this._renderer.getParentTile(this);
        this._parent.onDestroy.add(this._onParentTileDestroyed, this);
    }.bind(this));
};

/**
 * Lookup tiles around current one and create them if needed
 * @method FORGE.Tile#_checkNeighbours
 * @private
 */
FORGE.Tile.prototype._checkNeighbours = function()
{
    if (this._neighborsNeedCheck === false ||
        this._level === FORGE.Tile.PREVIEW ||
        this._level !== this._renderer.level)
    {
        return;
    }

    this._neighborsNeedCheck = false;

    var tx = Math.ceil(this._renderer.nbTilesPerAxis(this._level, "x"));
    var ty = Math.ceil(this._renderer.nbTilesPerAxis(this._level, "y"));

    var name = this.name;

    var sequence = Promise.resolve();
    var sequenceFn = function(prenderer, plevel, pface, px, py, neighbours, plog)
    {
        sequence.then(function()
        {
            var tile = prenderer.getTile(null, plevel, pface, px, py, plog);
            if (neighbours.indexOf(tile) === -1)
            {
                tile.onDestroy.add(tileDestroyedCallback);
                neighbours.push(tile);
            }
        });
    };

    // Check tile neighbors in current face
    var xmin = Math.max(0, this._x - 1);
    var xmax = Math.min(tx - 1, this._x + 1);

    var ymin = Math.max(0, this._y - 1);
    var ymax = Math.min(ty - 1, this._y + 1);

    var tileDestroyedCallback = this._onNeighbourTileDestroyed.bind(this);

    // and do the job for the current face
    for (var y = ymin; y <= ymax; y++)
    {
        for (var x = xmin; x <= xmax; x++)
        {
            if (x === this._x && y === this._y)
            {
                continue;
            }

            sequenceFn(this._renderer, this._level, this._face, x, y, this._neighbours, "foo");
        }
    }


    var tileX = this._x;
    var tileY = this._y;

    // Check if tile is on a left or right edge of the cube face
    if (tileX === 0 || tileX === tx - 1)
    {
        var edge = (tileX === 0); // true for left, false for right
        var log = "neighbour-" + (edge ? "left" : "right") + "-edge of " + name;
        var face = edge ? FORGE.Tile.FACE_PREVIOUS[this._face] : FORGE.Tile.FACE_NEXT[this._face];
        var x = edge ? tx - 1 : 0;

        sequenceFn(this._renderer, this._level, face, x, tileY, this._neighbours, log);
    }

    // Check if tile is on a bottom or top edge of the cube face
    if (tileY === ty - 1 || tileY === 0)
    {
        var edge = (tileY === 0); // true for top, false for bottom
        var fx, fy,
            face = edge ? "up" : "down";

        if (this._face === "front")
        {
            fx = tileX;
            fy = edge ? 0 : ty - 1;
        }
        else if (this._face === "back")
        {
            fx = tx - 1 - tileX;
            fy = edge ? ty - 1 : 0;
        }
        else if (this._face === "right")
        {
            fx = ty - 1;
            fy = edge ? tx - 1 - tileX : tileX;
        }
        else if (this._face === "left")
        {
            fx = 0;
            fy = edge ? tileX : tx - 1 - tileX;
        }
        else if (this._face === "up")
        {
            fx = edge ? tx - 1 - tileX : tileX;
            fy = 0;
            face = edge ? "back" : "front";
        }
        else if (this._face === "down")
        {
            fx = edge ? tileX : tx - 1 - tileX;
            fy = edge ? tx - 1 : ty - 1;
            face = "back";
        }

        var log = "neighbour-" + (edge ? "top" : "bottom") + "-edge of " + name;

        sequenceFn(this._renderer, this._level, face, fx, fy, this._neighbours, log);

        // if edge but ty = 1
        if (edge && ty === 1)
        {
            log = "neighbour-bottom-edge of " + name;
            sequenceFn(this._renderer, this._level, "down", fx, fy, this._neighbours, log);
        }
    }
};

/**
 * Get name of the parent tile
 * @method FORGE.Tile#getParentName
 * @return {?string} parent tile name
 */
FORGE.Tile.prototype.getParentName = function()
{
    if (this._level <= 0)
    {
        return null;
    }

    var coords = FORGE.Tile.getParentTileCoordinates(this);
    return FORGE.Tile.createName(this._face, this._level - 1, coords.x, coords.y);
};

/**
 * Refresh display timestamp with current date
 * @method FORGE.Tile#refreshDisplayTS
 */
FORGE.Tile.prototype.refreshDisplayTS = function()
{
    this._displayTS = Date.now();
};

/**
 * Destroy sequence
 * @method FORGE.Tile#destroy
 */
FORGE.Tile.prototype.destroy = function()
{
    if (this.material !== null)
    {
        if (typeof this.material.uniforms !== "undefined")
        {
            for (var u in this.material.uniforms)
            {
                this.material.uniforms[u].value = null;
            }
        }

        if (this.material.map !== null)
        {
            this.material.map.dispose();
            this.material.map = null;
        }

        this.material.dispose();
        this.material = null;
    }

    if (this.geometry !== null)
    {
        this.geometry.dispose();
        this.geometry = null;
    }

    this._parent = null;

    for (var i = 0, ii = this._children.length; i < ii; i++)
    {
        this._children[i].onDestroy.remove(this._onChildTileDestroyed, this);
    }
    this._children.length = 0;
    this._children = null;

    for (var i = 0, ii = this._neighbours.length; i < ii; i++)
    {
        this._neighbours[i].onDestroy.remove(this._onNeighbourTileDestroyed, this);
    }
    this._neighbours.length = 0;
    this._neighbours = null;

    this._position = null;

    this._renderer = null;

    if (this._onDestroy !== null)
    {
        this._onDestroy.dispatch();
        this._onDestroy.destroy();
        this._onDestroy = null;
    }
};

/**
 * Get the onDestroy {@link FORGE.EventDispatcher}.
 * @name FORGE.Tile#onDestroy
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.Tile.prototype, "onDestroy",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        if (this._onDestroy === null)
        {
            this._onDestroy = new FORGE.EventDispatcher(this, true);
        }

        return this._onDestroy;
    }
});

/**
 * Get face.
 * @name FORGE.Tile#face
 * @type {string}
 */
Object.defineProperty(FORGE.Tile.prototype, "face",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        // return FORGE.Tile.FACES.indexOf(this._face);
        return this._face;
    }
});

/**
 * Get level.
 * @name FORGE.Tile#level
 * @type {string}
 */
Object.defineProperty(FORGE.Tile.prototype, "level",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._level;
    }
});

/**
 * Get x coordinate.
 * @name FORGE.Tile#x
 * @type {number}
 */
Object.defineProperty(FORGE.Tile.prototype, "x",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._x;
    }
});

/**
 * Get y coordinate.
 * @name FORGE.Tile#y
 * @type {number}
 */
Object.defineProperty(FORGE.Tile.prototype, "y",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._y;
    }
});

/**
 * Get opacity.
 * @name FORGE.Tile#opacity
 * @type {number}
 */
Object.defineProperty(FORGE.Tile.prototype, "opacity",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._opacity;
    }
});

/**
 * Create timestamp.
 * @name FORGE.Tile#createTS
 * @type {number}
 */
Object.defineProperty(FORGE.Tile.prototype, "createTS",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._createTS;
    }
});

/**
 * Last display timestamp.
 * @name FORGE.Tile#displayTS
 * @type {number}
 */
Object.defineProperty(FORGE.Tile.prototype, "displayTS",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._displayTS;
    }
});

/**
 * Neighbour tiles.
 * @name FORGE.Tile#neighbours
 * @type {Array<FORGE.Tile>}
 */
Object.defineProperty(FORGE.Tile.prototype, "neighbours",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._neighbours;
    }
});

/**
 * Is the texture set
 * @name FORGE.Tile#textureIsSet
 * @type {boolean}
 */
Object.defineProperty(FORGE.Tile.prototype, "textureIsSet",
{
    /** @this {FORGE.Tile} */
    get: function()
    {
        return this._textureIsSet;
    }
});

/**
 * The FORGE.HotspotManager is an object that manages hotspots of the project.
 *
 * @constructor FORGE.HotspotManager
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.HotspotManager = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.HotspotManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The general config backup.
     * @name FORGE.HotspotManager#_config
     * @type {Object}
     * @private
     */
    this._config = null;

    /**
     * Hotspots array
     * @name  FORGE.HotspotManager#_hotspots
     * @type {Array<(FORGE.Hotspot3D|FORGE.HotspotDOM)>}
     * @private
     */
    this._hotspots = [];

    FORGE.BaseObject.call(this, "HotspotManager");

};

FORGE.HotspotManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotManager.prototype.constructor = FORGE.HotspotManager;

/**
 * Parse a hotspots config object.
 * @method FORGE.HotspotManager#_parseConfig
 * @private
 * @param {Array<HotspotConfig>} config - The array of hotspot config you want to parse.
 */
FORGE.HotspotManager.prototype._parseConfig = function(config)
{
    for (var i = 0, ii = config.length; i < ii; i++)
    {
        this.create(config[i]);
    }
};

/**
 * Create a hotspot from a hotpsot config object.
 * @method FORGE.HotspotManager#create
 * @param {HotspotConfig} config - The config of the hotspot you want to create.
 * @return {(FORGE.Hotspot3D|FORGE.HotspotDOM|boolean)} Returns the hotspot if the hotspot is created, false if not.
 */
FORGE.HotspotManager.prototype.create = function(config)
{
    var hotspot = null;
    var type = config.type || FORGE.HotspotType.THREE_DIMENSIONAL; //3d is the default type

    switch (type)
    {
        case FORGE.HotspotType.THREE_DIMENSIONAL:
            hotspot = new FORGE.Hotspot3D(this._viewer, config);
            break;

        case FORGE.HotspotType.DOM:
            hotspot = new FORGE.HotspotDOM(this._viewer, config);
            break;
    }

    if (hotspot !== null)
    {
        this._hotspots.push(hotspot);
        return hotspot;
    }

    return false;
};

/**
 * Remove a hotspot from the manager
 * @method FORGE.HotspotManager#remove
 * @param  {(string|FORGE.Hotspot3D)} hotspot - the hotspot or its uid to remove
 */
FORGE.HotspotManager.prototype.remove = function(hotspot)
{
    if(FORGE.Utils.isTypeOf(hotspot, "string") === true)
    {
        hotspot = FORGE.UID.get(hotspot);
    }

    if(FORGE.Utils.isTypeOf(hotspot, "Hotspot3D") === true)
    {
        var index = this._hotspots.indexOf(hotspot);
        this._hotspots.splice(index, 1);
        hotspot.destroy();
    }
};

/**
 * Parse a list of tracks for hotspots movement.
 * @method FORGE.HotspotManager#_parseTracks
 * @param {Array<HotspotTrackConfig>} tracks - The array of tracks to add.
 * @private
 */
FORGE.HotspotManager.prototype._parseTracks = function(tracks)
{
    for (var i = 0, ii = tracks.length; i < ii; i++)
    {
        new FORGE.HotspotAnimationTrack(tracks[i]);
    }
};

/**
 * Check if all hotspots are ready
 * @method FORGE.HotspotManager#_checkHotspotsReady
 * @return boolean true if all hotspots are ready, false otherwise
 * @private
 */
FORGE.HotspotManager.prototype._checkHotspotsReady = function()
{
    for (var i = 0, ii = this._hotspots.length; i < ii; i++)
    {
        var hotspot = this._hotspots[i];

        if (hotspot.ready === false)
        {
            return false;
        }
    }

    return true;
};

/**
 * Event handler for scene load start.
 * @method  FORGE.HotspotManager#_sceneLoadStartHandler
 * @private
 */
FORGE.HotspotManager.prototype._sceneLoadStartHandler = function()
{
    var scene = this._viewer.story.scene;
    scene.onUnloadStart.addOnce(this._sceneUnloadStartHandler, this);

    if (typeof scene.config.hotspots !== "undefined")
    {
        var hotspots = scene.config.hotspots;
        this.addConfig(hotspots);
    }
};

/**
 * Event handler for scene unload start.
 * @method  FORGE.HotspotManager#_sceneUnloadStartHandler
 * @private
 */
FORGE.HotspotManager.prototype._sceneUnloadStartHandler = function()
{
    this.clear();
};

/**
 * Boot sequence.
 * @method FORGE.HotspotManager#boot
 */
FORGE.HotspotManager.prototype.boot = function()
{
    this._config = [];

    this._viewer.story.onSceneLoadStart.add(this._sceneLoadStartHandler, this);
};

/**
 * Get hotspots by type
 * @method  FORGE.HotspotManager#getByType
 * @param  {string} type - The type of hotspots you want to get.
 * @return {Array<(FORGE.Hotspot3D|FORGE.HotspotDOM)>}
 */
FORGE.HotspotManager.prototype.getByType = function(type)
{
    var result = this._hotspots.filter(function(hotspot)
    {
        return hotspot.type === type;
    });

    return result;
};

/**
 * Add a hotspots config to the manager.
 * @method FORGE.HotspotManager#addConfig
 * @param {Array<HotspotConfig>} config - The config you want to add.
 */
FORGE.HotspotManager.prototype.addConfig = function(config)
{
    this._parseConfig(config);
};

/**
 * Add a list of tracks for the hotspots.
 * @method FORGE.HotspotManager#addTracks
 * @param {HotspotsConfig} config - The tracks you want to add.
 */
FORGE.HotspotManager.prototype.addTracks = function(config)
{
    if (config.tracks !== null && typeof config.tracks !== "undefined")
    {
        this._parseTracks(config.tracks);
    }
};

/**
 * Update loop
 * @method FORGE.HotspotManager#update
 */
FORGE.HotspotManager.prototype.update = function()
{
    for (var i = 0, ii = this._hotspots.length; i < ii; i++)
    {
        this._hotspots[i].update();
    }
};

/**
 * Clear all hotspots from the manager
 * @method FORGE.HotspotManager#clear
 * @param {string=} type - the type of hotspots to clear, nothing for all
 */
FORGE.HotspotManager.prototype.clear = function(type)
{
    this._hotspots = this._hotspots.filter(function(hs)
    {
        var keep = (typeof type === "string" && hs.type !== type);

        if (keep === false)
        {
            hs.destroy();
        }

        return keep;
    });
};

/**
 * Dump the array of hotspot configurations.
 * @method FORGE.HotspotManager#dump
 * @return {Array<HotspotConfig>} Return an array of hotspot configurations of the current scene.
 */
FORGE.HotspotManager.prototype.dump = function()
{
    var dump = [];

    for(var i = 0, ii = this._hotspots.length; i < ii; i++)
    {
        dump.push(this._hotspots[i].dump());
    }

    return dump;
};

/**
 * Destroy sequence
 * @method FORGE.HotspotManager#destroy
 */
FORGE.HotspotManager.prototype.destroy = function()
{
    this.clear();

    this._viewer = null;
    this._hotspots = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get all the hotspots.
 * @name FORGE.HotspotManager#all
 * @readonly
 * @type {Array<(FORGE.Hotspot3D|FORGE.HotspotDOM)>}
 */
Object.defineProperty(FORGE.HotspotManager.prototype, "all",
{
    /** @this {FORGE.HotspotManager} */
    get: function()
    {
        return this._hotspots;
    }
});

/**
 * Get all the hotspots uids.
 * @name FORGE.HotspotManager#uids
 * @readonly
 * @type {Array<string>}
 */
Object.defineProperty(FORGE.HotspotManager.prototype, "uids",
{
    /** @this {FORGE.HotspotManager} */
    get: function()
    {
        var uids = [];

        for(var i = 0, ii = this._hotspots.length; i < ii; i++)
        {
            uids.push(this._hotspots[i].uid);
        }

        return uids;
    }
});

/**
 * Get the hotspots count.
 * @name FORGE.HotspotManager#count
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.HotspotManager.prototype, "count",
{
    /** @this {FORGE.HotspotManager} */
    get: function()
    {
        return this._hotspots.length;
    }
});

/**
 * A HotspotDOM, to be displayed like a billboard. This hotspot provides a
 * single div positioned at the right position in the scene, without any
 * content in it and any deformation. It is up to the FORGE user to specify
 * those. Two things can be tweaked here: the displayObject property of this
 * hotspot, which is a {@link FORGE.DisplayObjectContainer}, and the DOM part
 * of this container, accessible through `displayObject.dom` or more directly
 * using the `dom` property on the object HotspotDOM.
 *
 * @constructor FORGE.HotspotDOM
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {HotspotConfig} config - hotspot configuration
 * @extends {FORGE.BaseObject}
 *
 * @todo facingCenter with CSS 3D, rotation values and scale values
 */
FORGE.HotspotDOM = function(viewer, config)
{
    /**
     * The viewer reference
     * @name FORGE.HotspotDOM#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Hotspot configuration
     * @name FORGE.HotspotDOM#_config
     * @type {HotspotConfig}
     * @private
     */
    this._config = config;

    /**
     * HotspotTransform object for the positioning and scaling (no rotation)
     * @name  FORGE.HotspotDOM#_transform
     * @type {FORGE.HotspotTransform}
     * @private
     */
    this._transform = null;

    /**
     * The offset applied to the DOM object from it's center.<br>
     * Is expressed in pixels (x, y).
     * @name FORGE.HotspotDOM#_offset
     * @type {HotspotDomOffset}
     * @private
     */
    this._offset = FORGE.HotspotDOM.DEFAULT_OFFSET;

    /**
     * The HTML element composing the hotspot
     * @name FORGE.HotspotDOM#_dom
     * @type {Element|HTMLElement}
     * @private
     */
    this._dom = null;

    /**
     * Events object that will keep references of the ActionEventDispatcher
     * @name FORGE.HotspotDOM#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = null;

    /**
     * Visibility flag
     * @name  FORGE.HotspotDOM#_visible
     * @type {boolean}
     * @private
     */
    this._visible = true;

    /**
     * Is this object is interactive / raycastable
     * @name FORGE.HotspotDOM#_interactive
     * @type {boolean}
     * @private
     */
    this._interactive = true;

    /**
     * Does the hotspot is facing the camera ? Useful for a flat hotspot we want
     * to always be facing to the camera.
     * @name FORGE.HotspotDOM#_facingCenter
     * @type {boolean}
     * @private
     */
    this._facingCenter = false;

    /**
     * The pointer cursor when pointer is over the hotspot zone
     * @name FORGE.HotspotDOM#_cursor
     * @type {string}
     * @private
     */
    this._cursor = "pointer";

    /**
     * Event handler for a click on the hotspot.
     * @name FORGE.HotspotDOM#_domClickHandlerBind
     * @type {Function}
     * @private
     */
    this._domClickHandlerBind = null;

    /**
     * Event handler for overing on the hotspot.
     * @name FORGE.HotspotDOM#_domOverHandlerBind
     * @type {Function}
     * @private
     */
    this._domOverHandlerBind = null;

    /**
     * Event handler for getting out on the hotspot.
     * @name FORGE.HotspotDOM#_domOutHandlerBind
     * @type {Function}
     * @private
     */
    this._domOutHandlerBind = null;

    FORGE.BaseObject.call(this, "HotspotDOM");

    this._boot();
};

FORGE.HotspotDOM.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotDOM.prototype.constructor = FORGE.HotspotDOM;

/**
 * @name FORGE.HotspotDOM.DEFAULT_CONFIG
 * @type {HotspotConfig}
 */
FORGE.HotspotDOM.DEFAULT_CONFIG =
{
    dom:
    {
        id: "hostpot-dom",
        width: 320,
        height: 240,
        color: "white",
        index: 10,
        offset: FORGE.HotspotDOM.DEFAULT_OFFSET
    }
};

/**
 * @name FORGE.HotspotDOM.DEFAULT_OFFSET
 * @type {HotspotDomOffset}
 */
FORGE.HotspotDOM.DEFAULT_OFFSET =
{
    x: 0,
    y: 0
};

/**
 * Boot sequence.
 * @method FORGE.HotspotDOM#_boot
 * @private
 */
FORGE.HotspotDOM.prototype._boot = function()
{
    this._transform = new FORGE.HotspotTransform();

    this._events = {};

    this._parseConfig(this._config);
    this._register();

    this._viewer.renderer.view.onChange.add(this._viewChangeHandler, this);
};

/**
 * Parse the config object
 * @method FORGE.HotspotDOM#_parseConfig
 * @param {HotspotConfig} config - the hotspot config to parse
 * @private
 */
FORGE.HotspotDOM.prototype._parseConfig = function(config)
{
    config = /** @type {HotspotConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.HotspotDOM.DEFAULT_CONFIG, config));

    this._uid = config.uid;

    if (typeof config.events === "object" && config.events !== null)
    {
        this._createEvents(config.events);
    }

    var dom = config.dom;

    if (dom !== null && typeof dom !== "undefined")
    {
        var id;

        if (typeof dom.id === "string")
        {
            id = dom.id;
        }
        else
        {
            id = this._uid;
        }

        // store the offset values
        this._offset = dom.offset || FORGE.HotspotDOM.DEFAULT_OFFSET;

        // get the already present hotspot in the dom, or create it
        var div = document.getElementById(id);

        if (div !== null)
        {
            this._dom = div;
        }
        else
        {
            this._dom = document.createElement("div");
            this._dom.id = id;
        }

        this._dom.classList.add("hotspot-dom");
        this._dom.style.position = "absolute";

        if (typeof dom.class === "string")
        {
            this._dom.classList.add(dom.class);
        }
        else if (Array.isArray(dom.class) === true)
        {
            for (var i = 0, ii = dom.class.length; i < ii; i++)
            {
                this._dom.classList.add(dom.class[i]);
            }
        }

        // basic CSS from json configuration
        var rule = "." + this._dom.id + "-basic-class {";

        if (typeof dom.width === "number")
        {
            rule += "width: " + dom.width + "px;";
        }
        else if (typeof dom.width === "string")
        {
            rule += "width: " + dom.width + ";";
        }

        if (typeof dom.height === "number")
        {
            rule += "height: " + dom.height + "px;";
        }
        else if (typeof dom.height === "string")
        {
            rule += "height: " + dom.height + ";";
        }

        if (typeof dom.color === "string")
        {
            rule += "background-color: " + dom.color + ";";
        }

        if (typeof dom.index === "number")
        {
            rule+= "z-index: " + dom.index + ";";
        }

        rule += "}";
        this._viewer.domHotspotStyle.sheet.insertRule(rule, 0);
        this._dom.classList.add(this._dom.id + "-basic-class");
    }

    this._dom.style.pointerEvents = "auto";
    this._domClickHandlerBind = this._domClickHandler.bind(this);
    this._domOverHandlerBind = this._domOverHandler.bind(this);
    this._domOutHandlerBind = this._domOutHandler.bind(this);
    this._dom.addEventListener("click", this._domClickHandlerBind);
    this._dom.addEventListener("mouseover", this._domOverHandlerBind);
    this._dom.addEventListener("mouseout", this._domOutHandlerBind);

    if (config.transform !== null && typeof config.transform !== "undefined")
    {
        this._transform.load(config.transform, false);
    }

    this._visible = (typeof config.visible === "boolean") ? config.visible : true;
    // this._facingCenter = (typeof config.facingCenter === "boolean") ? config.facingCenter : false;
    this._interactive = (typeof config.interactive === "boolean") ? config.interactive : true;
    this._cursor = (typeof config.cursor === "string") ? config.cursor : "pointer";

    this.show();
};

/**
 * DOM click handler
 * @method FORGE.HotspotDOM#_domClickHandler
 * @private
 */
FORGE.HotspotDOM.prototype._domClickHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onClick, "ActionEventDispatcher") === true)
    {
        this._events.onClick.dispatch();
    }
};

/**
 * DOM over handler
 * @method FORGE.HotspotDOM#_domOverHandler
 * @private
 */
FORGE.HotspotDOM.prototype._domOverHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onOver, "ActionEventDispatcher") === true)
    {
        this._events.onOver.dispatch();
    }

    this._dom.style.cursor = this._cursor;
};

/**
 * DOM out handler.
 * @method FORGE.HotspotDOM#_domOutHandler
 * @private
 */
FORGE.HotspotDOM.prototype._domOutHandler = function()
{
    // Actions defined from the json
    if(FORGE.Utils.isTypeOf(this._events.onOut, "ActionEventDispatcher") === true)
    {
        this._events.onOut.dispatch();
    }

    this._dom.style.cursor = "default";
};

/**
 * Create action events dispatchers.
 * @method FORGE.HotspotDOM#_createEvents
 * @private
 * @param {Object} events - The events config of the dom hotspot.
 */
FORGE.HotspotDOM.prototype._createEvents = function(events)
{
    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all object events.
 * @method FORGE.HotspotDOM#_clearEvents
 * @private
 */
FORGE.HotspotDOM.prototype._clearEvents = function()
{
    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Handles the changing view, as it can only be present in the Rectilinear and GoPro view.
 * @method FORGE.HotspotDOM#_viewChangeHandler
 * @private
 */
FORGE.HotspotDOM.prototype._viewChangeHandler = function()
{
    this._dom.style.display = "block";

    if ((this._viewer.view.type !== FORGE.ViewType.RECTILINEAR && this._viewer.view.type !== FORGE.ViewType.GOPRO) || this._visible === false)
    {
        this._dom.style.display = "none";
    }
};

/**
 * Show the hotspot by appending it to the DOM container.
 * @method FORGE.HotspotDOM#show
 */
FORGE.HotspotDOM.prototype.show = function()
{
    this._viewer.domHotspotContainer.dom.appendChild(this._dom);
};

/**
 * Hide the hotspot by removing it to the DOM container.
 * @method FORGE.HotspotDOM#hide
 */
FORGE.HotspotDOM.prototype.hide = function()
{
    this._viewer.domHotspotContainer.dom.removeChild(this._dom, false);
};

/**
 * Update routine.
 * @method FORGE.HotspotDOM#update
 */
FORGE.HotspotDOM.prototype.update = function()
{
    // get the screen position of the hotspots
    var position = this._viewer.view.worldToScreen(this._transform.position.values);

    if (position !== null)
    {
        var x = position.x + this._offset.x - this._dom.clientWidth / 2;
        var y = position.y + this._offset.y - this._dom.clientHeight / 2;
        this._dom.style.left = x + "px";
        this._dom.style.top = y + "px";
    }
    else
    {
        this._dom.style.left = "99999px";
        this._dom.style.top = "99999px";
    }
};

/**
 * Destroy routine.
 * @method FORGE.HotspotDOM#destroy
 */
FORGE.HotspotDOM.prototype.destroy = function()
{
    this._dom.removeEventListener("click", this._domClickHandlerBind);
    this._dom.removeEventListener("mouseover", this._domOverHandlerBind);
    this._dom.removeEventListener("mouseout", this._domOutHandlerBind);

    this._domClickHandlerBind = null;
    this._domOverHandlerBind = null;
    this._domOutHandlerBind = null;

    this._clearEvents();
    this._events = null;

    // Hide dom, don't destroy it, as it may be used later
    this._dom.style.left = "99999px";
    this._dom.style.top = "99999px";
    this._dom = null;

    this._transform.destroy();
    this._transform = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the DOM content of the hotspot.
 * @name FORGE.HotspotDOM#dom
 * @readonly
 * @type {?Element|HTMLElement}
 */
Object.defineProperty(FORGE.HotspotDOM.prototype, "dom",
{
    /** @this {FORGE.HotspotDOM} */
    get: function()
    {
        return this._dom;
    }
});

/**
 * Get and set the visible flag.
 * @name FORGE.HotspotDOM#visible
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotDOM.prototype, "visible",
{
    /** @this {FORGE.HotspotDOM} */
    get: function()
    {
        return this._visible;
    },
    /** @this {FORGE.HotspotDOM} */
    set: function(value)
    {
        this._visible = Boolean(value);

        if (this._visible === true)
        {
            this._viewChangeHandler();
        }
        else
        {
            this._dom.style.display = "none";
        }
    }
});

/**
 * Get and set the interactive flag for the main hotspot DOM container.
 * @name FORGE.HotspotDOM#interactive
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotDOM.prototype, "interactive",
{
    /** @this {FORGE.HotspotDOM} */
    get: function()
    {
        return this._interactive;
    },
    /** @this {FORGE.HotspotDOM} */
    set: function(value)
    {
        this._interactive = Boolean(value);

        if (this._interactive === true)
        {
            this._dom.style.pointerEvents = "auto";
        }
        else
        {
            this._dom.style.pointerEvents = "none";
        }
    }
});

/**
 * Get/set the offset of the DOM object.
 * @name FORGE.HotspotDOM#offset
 * @type {HotspotDomOffset}
 */
Object.defineProperty(FORGE.HotspotDOM.prototype, "offset",
{
    /** @this {FORGE.HotspotDOM} */
    get: function()
    {
        return this._offset;
    },

    /** @this {FORGE.HotspotDOM} */
    set: function(value)
    {
        if (typeof value !== "undefined" && (typeof value.x === "number" || typeof value.y === "number"))
        {
            this._offset = /** @type {HotspotDomOffset} */ (FORGE.Utils.extendSimpleObject(FORGE.HotspotDOM.DEFAULT_OFFSET, value));
        }
    }
});

/**
 * FORGE.Hotspot3D
 * Abstract base class for projeted views. Should be subclassed for every supported projection / view type.
 *
 * @constructor FORGE.Hotspot3D
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {HotspotConfig} config - hostspot configuration
 * @extends {FORGE.Object3D}
 */
FORGE.Hotspot3D = function(viewer, config)
{
    /**
     * Hotspot configuration
     * @name  FORGE.Hotspot3D#_config
     * @type {HotspotConfig}
     * @private
     */
    this._config = config;

    /**
     * Name
     * @name  FORGE.Hotspot3D#_name
     * @type {string}
     * @private
     */
    this._name = "";

    /**
     * HotspotTransform object for the 3D object.
     * @name  FORGE.Hotspot3D#_transform
     * @type {FORGE.HotspotTransform}
     * @private
     */
    this._transform = null;

    /**
     * HotspotGeometry object.
     * @name FORGE.Hotspot3D#_geometry
     * @type {FORGE.HotspotGeometry}
     * @private
     */
    this._geometry = null;

    /**
     * Material object for the 3D object.
     * @name  FORGE.Hotspot3D#_material
     * @type {FORGE.HotspotMaterial}
     * @private
     */
    this._material = null;

    /**
     * Sound object for the 3D object.
     * @name  FORGE.Hotspot3D#_sound
     * @type {FORGE.HotspotSound}
     * @private
     */
    this._sound = null;

    /**
     * Animation object for the 3D object.
     * @name FORGE.Hotspot3D#_animation
     * @type {FORGE.HotspotAnimation}
     * @private
     */
    this._animation = null;

    /**
     * Hotspots states manager
     * @name FORGE.Hotspot3D#_states
     * @type {FORGE.HotspotStates}
     * @private
     */
    this._states = null;

    /**
     * Does the hotspot is facing the camera ? Useful for a flat hotspot we want
     * to always be facing to the camera.
     * @name FORGE.Hotspot3D#_facingCenter
     * @type {boolean}
     * @private
     */
    this._facingCenter = false;

    /**
     * Does the hotspot changes its scale according to the fov ?
     * @name FORGE.Hotspot3D#_autoScale
     * @type {boolean}
     * @private
     */
    this._autoScale = false;

    /**
     * The pointer cursor when pointer is over the Object3D
     * @name FORGE.Hotspot3D#_cursor
     * @type {string}
     * @private
     */
    this._cursor = "pointer";

    /**
     * Before render bound callback.
     * @name FORGE.Hotspot3D#_onBeforeRenderBound
     * @type {?function(this:THREE.Object3D,?THREE.WebGLRenderer,?THREE.Scene,?THREE.Camera,?THREE.Geometry,?THREE.Material,?THREE.Group)}
     * @private
     */
    this._onBeforeRenderBound = null;

    /**
     * After render bound callback.
     * @name FORGE.Hotspot3D#_onAfterRenderBound
     * @type {?function(this:THREE.Object3D,?THREE.WebGLRenderer,?THREE.Scene,?THREE.Camera,?THREE.Geometry,?THREE.Material,?THREE.Group)}
     * @private
     */
    this._onAfterRenderBound = null;

    FORGE.Object3D.call(this, viewer, "Hotspot3D");
};

FORGE.Hotspot3D.prototype = Object.create(FORGE.Object3D.prototype);
FORGE.Hotspot3D.prototype.constructor = FORGE.Hotspot3D;

/**
 * Boot sequence.<br>
 * Call superclass boot when objects are created as it will trigger parse config
 * @private
 */
FORGE.Hotspot3D.prototype._boot = function()
{
    FORGE.Object3D.prototype._boot.call(this);

    this._transform = new FORGE.HotspotTransform();
    this._transform.onChange.add(this._onTransformChangeHandler, this);

    this._animation = new FORGE.HotspotAnimation(this._viewer, this._transform);

    this._onBeforeRenderBound = this._onBeforeRender.bind(this);
    this._onAfterRenderBound = this._onAfterRender.bind(this);

    this._mesh.visible = false;
    this._mesh.onBeforeRender = /** @type {function(this:THREE.Object3D,?THREE.WebGLRenderer,?THREE.Scene,?THREE.Camera,?THREE.Geometry,?THREE.Material,?THREE.Group)} */ (this._onBeforeRenderBound);
    this._mesh.onAfterRender = /** @type {function(this:THREE.Object3D,?THREE.WebGLRenderer,?THREE.Scene,?THREE.Camera,?THREE.Geometry,?THREE.Material,?THREE.Group)} */ (this._onAfterRenderBound);

    this._viewer.renderer.view.onChange.add(this._viewChangeHandler, this);

    if (typeof this._config !== "undefined" && this._config !== null)
    {
        this._parseConfig(this._config);
    }
};

/**
 * Parse the config object.
 * @method FORGE.Hotspot3D#_parseConfig
 * @param {HotspotConfig} config - The hotspot config to parse.
 * @private
 */
FORGE.Hotspot3D.prototype._parseConfig = function(config)
{
    this._uid = config.uid;
    this._tags = config.tags;
    this._register();

    // Set the mesh name
    this._mesh.name = "mesh-" + this._uid;
    this._mesh.userData = config;

    this._name = (typeof config.name === "string") ? config.name : "";
    this._visible = (typeof config.visible === "boolean") ? config.visible : true;
    this._facingCenter = (typeof config.facingCenter === "boolean") ? config.facingCenter : false;
    this._autoScale = (typeof config.autoScale === "boolean") ? config.autoScale : false;
    this._interactive = (typeof config.interactive === "boolean") ? config.interactive : true;
    this._cursor = (typeof config.cursor === "string") ? config.cursor : "pointer";

    this._geometry = new FORGE.HotspotGeometry();
    this._material = new FORGE.HotspotMaterial(this._viewer, this._uid);
    this._sound = new FORGE.HotspotSound(this._viewer, this._uid);
    this._states = new FORGE.HotspotStates(this._viewer, this._uid);

    if (typeof config.states === "object" && config.states !== null)
    {
        this._states.addConfig(config.states);
    }

    if (typeof config.fx === "string" && config.fx !== "")
    {
        this._fx = config.fx;
    }

    if (typeof config.events === "object" && config.events !== null)
    {
        this._createEvents(config.events);
    }

    this._states.onLoadComplete.add(this._stateLoadCompleteHandler, this);
    this._states.load();
};

/**
 * Before render handler
 * @method FORGE.Hotspot3D#_onBeforeRender
 * @private
 */
FORGE.Hotspot3D.prototype._onBeforeRender = function(renderer, scene, camera, geometry, material, group)
{
    var g = group; // Just to avoid the jscs warning about group parameter not used.

    this._viewer.renderer.view.current.updateUniforms(material.uniforms);

    // Check what is the current render pass looking at the material: Hotspot or Picking Material
    if (material.name === "HotspotMaterial")
    {
        this._material.update();
    }
    else if (material.name === "PickingMaterial")
    {
        // As picking material is the same for all spots renderer in this pass, material uniforms won't be refreshed
        // Setting material.uniforms.tColor value will be useless, set direct value by acceding program uniforms map
        // Call useProgram first to avoid WebGL warning if material.program is not the current program
        // Set also material uniform to avoid both settings will collide on first object
        if (material.program)
        {
            var gl = this._viewer.renderer.webGLRenderer.getContext();
            gl.useProgram(material.program.program);
            material.program.getUniforms().map.tColor.setValue(gl, this._pickingColor);
            material.uniforms.tColor.value = this._pickingColor;
        }
    }
};

/**
 * After render handler
 * @method FORGE.Hotspot3D#_onAfterRender
 * @private
 */
FORGE.Hotspot3D.prototype._onAfterRender = function()
{

};

/**
 * Event handler for material ready. Triggers the creation of the hotspot3D.
 * @method FORGE.Hotspot3D#_stateLoadCompleteHandler
 * @private
 */
FORGE.Hotspot3D.prototype._stateLoadCompleteHandler = function()
{
    this.log("state load complete handler");

    this._mesh.geometry = this._geometry.geometry;
    this._mesh.material = this._material.material;
    this._mesh.visible = this._visible;

    this._updatePosition();

    if (this._animation.autoPlay === true && document[FORGE.Device.visibilityState] === "visible")
    {
        this._animation.play();
    }

    if (this._onReady !== null)
    {
        this._onReady.dispatch();
    }
};

/**
 * transform change handler
 * @method FORGE.Hotspot3D#_onTransformChangeHandler
 * @private
 */
FORGE.Hotspot3D.prototype._onTransformChangeHandler = function()
{
    this.log("transform change handler");
    this._updatePosition();
};

/**
 * Setup hotspot spatial position.
 * @method FORGE.Hotspot3D#_setupPosition
 * @private
 */
FORGE.Hotspot3D.prototype._updatePosition = function()
{
    this.log("update position");

    this._mesh.position.x = this._transform.position.x;
    this._mesh.position.y = this._transform.position.y;
    this._mesh.position.z = this._transform.position.z;

    if (this._facingCenter === true)
    {
        var spherical = new THREE.Spherical().setFromVector3(new THREE.Vector3(this._transform.position.x, this._transform.position.y, this._transform.position.z));

        this._mesh.rotation.set(-spherical.phi + Math.PI / 2, spherical.theta + Math.PI, 0, "YXZ");

        // Apply rotation
        this._mesh.rotation.x += -FORGE.Math.degToRad(this._transform.rotation.x); // pitch
        this._mesh.rotation.y += FORGE.Math.degToRad(this._transform.rotation.y); // yaw
        this._mesh.rotation.z += FORGE.Math.degToRad(this._transform.rotation.z);
    }
    else
    {
        // Apply rotation
        var rx = -FORGE.Math.degToRad(this._transform.rotation.x); // pitch
        var ry = FORGE.Math.degToRad(this._transform.rotation.y); // yaw
        var rz = FORGE.Math.degToRad(this._transform.rotation.z);

        this._mesh.rotation.set(rx, ry, rz, "YXZ");
    }

    // Scale
    if(this._autoScale === true)
    {
        this._updateAutoScale();
    }
    else
    {
        this._mesh.scale.x = FORGE.Math.clamp(this._transform.scale.x, 0.000001, 100000);
        this._mesh.scale.y = FORGE.Math.clamp(this._transform.scale.y, 0.000001, 100000);
        this._mesh.scale.z = FORGE.Math.clamp(this._transform.scale.z, 0.000001, 100000);
    }
};

/**
 * Updates the mesh scale according to the fov to keep a constant screen size
 * @method FORGE.Hotspot3D#_updateAutoScale
 * @private
 */
FORGE.Hotspot3D.prototype._updateAutoScale = function()
{
    var factor = this._viewer.camera.fov / this._viewer.camera.config.fov.default;

    this._mesh.scale.x = FORGE.Math.clamp(this._transform.scale.x * factor, 0.000001, 100000);
    this._mesh.scale.y = FORGE.Math.clamp(this._transform.scale.y * factor, 0.000001, 100000);
    this._mesh.scale.z = FORGE.Math.clamp(this._transform.scale.z * factor, 0.000001, 100000);
};

/**
 * Check the ready flag of hotspot
 * @method FORGE.Hotspot3D#_checkReady
 * @return {boolean}
 * @private
 */
FORGE.Hotspot3D.prototype._checkReady = function()
{
    return (this._states.ready === true);
};

/**
 * View change handler
 * @method FORGE.Hotspot3D#_viewChangeHandler
 * @private
 */
FORGE.Hotspot3D.prototype._viewChangeHandler = function()
{
    // Only enable frustum culling when view is rectilinear and frustum makes sense
    this._mesh.frustumCulled = this._viewer.renderer.view.current instanceof FORGE.ViewRectilinear;

    this._material.updateShader();
    this._mesh.material = this._material.material;
};

/**
 * Override of the over method to trigger the state change
 * @method FORGE.Hotspot3D#over
 */
FORGE.Hotspot3D.prototype.over = function()
{
    FORGE.Object3D.prototype.over.call(this);

    if(this._states.auto === true)
    {
        this._states.load("over");
    }

    this._viewer.canvas.pointer.cursor = this._cursor;
};

/**
 * Override of the out method to trigger the state change
 * @method FORGE.Hotspot3D#out
 */
FORGE.Hotspot3D.prototype.out = function()
{
    FORGE.Object3D.prototype.out.call(this);

    if(this._states.auto === true)
    {
        this._states.load();
    }

    this._viewer.canvas.pointer.cursor = "default";
};

/**
 * Update hotspot content
 * @method FORGE.Hotspot3D#update
 */
FORGE.Hotspot3D.prototype.update = function()
{
    if (this._sound !== null)
    {
        this._sound.update();
    }

    if(this._autoScale === true)
    {
        this._updateAutoScale();
    }
};

/**
 * Dump the hotspot actual configuration
 * @method FORGE.Hotspot3D#dump
 * @return {HotspotConfig} Return the hotspot actual configuration object
 */
FORGE.Hotspot3D.prototype.dump = function()
{
    var dump =
    {
        uid: this._uid,
        name: this._name,
        tags: this._tags,
        visible: this._visible,
        interactive: this._interactive,
        cursor: this._cursor,
        fx: this._fx,
        facingCenter: this._facingCenter,
        geometry: this._geometry.dump(),
        transform: this._transform.dump(),
        material: this._material.dump()
    };

    return dump;
};

/**
 * Destroy routine
 * @method FORGE.Hotspot3D#destroy
 */
FORGE.Hotspot3D.prototype.destroy = function()
{
    this._viewer.renderer.view.onChange.remove(this._viewChangeHandler, this);

    this._onBeforeRenderBound = null;
    this._onAfterRenderBound = null;

    if(this._states !== null)
    {
        this._states.destroy();
        this._states = null;
    }

    if (this._transform !== null)
    {
        this._transform.destroy();
        this._transform = null;
    }

    if(this._geometry !== null)
    {
        this._geometry.destroy();
        this._geometry = null;
    }

    if (this._animation !== null)
    {
        this._animation.destroy();
        this._animation = null;
    }

    if (this._material !== null)
    {
        this._material.destroy();
        this._material = null;
    }

    if (this._sound !== null)
    {
        this._sound.destroy();
        this._sound = null;
    }

    FORGE.Object3D.prototype.destroy.call(this);
};

/**
 * Hotspot config accessor
 * @name FORGE.Hotspot3D#config
 * @readonly
 * @type {HotspotConfig}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "config",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._config;
    }
});

/**
 * Hotspot name accessor
 * @name FORGE.Hotspot3D#name
 * @type {string}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "name",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._name;
    },

    /** @this {FORGE.Hotspot3D} */
    set: function(value)
    {
        if(typeof value === "string")
        {
            this._name = value;
        }
    }
});

/**
 * Hotspot animation accessor
 * @name FORGE.Hotspot3D#animation
 * @readonly
 * @type {FORGE.HotspotAnimation}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "animation",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._animation;
    }
});

/**
 * Hotspot material accessor
 * @name FORGE.Hotspot3D#material
 * @readonly
 * @type {FORGE.HotspotMaterial}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "material",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._material;
    }
});

/**
 * Hotspot sound accessor
 * @name FORGE.Hotspot3D#sound
 * @readonly
 * @type {FORGE.HotspotSound}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "sound",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._sound;
    }
});

/**
 * Hotspot transform accessor
 * @name FORGE.Hotspot3D#transform
 * @readonly
 * @type {FORGE.HotspotTransform}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "transform",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._transform;
    }
});

/**
 * Hotspot geometry accessor
 * @name FORGE.Hotspot3D#geometry
 * @readonly
 * @type {FORGE.HotspotGeometry}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "geometry",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._geometry;
    }
});

/**
 * Hotspot3D ready flag
 * @name FORGE.Hotspot3D#ready
 * @readonly
 * @type boolean
  */
Object.defineProperty(FORGE.Hotspot3D.prototype, "ready",
{
    /** @this {FORGE.Object3D} */
    get: function()
    {
        this._ready = this._checkReady();
        return this._ready;
    }
});

/**
 * Hotspot states accessor
 * @name FORGE.Hotspot3D#states
 * @readonly
 * @type {FORGE.HotspotStates}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "states",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._states;
    }
});

/**
 * Hotspot states accessor
 * @name FORGE.Hotspot3D#states
 * @readonly
 * @type {FORGE.HotspotStates}
 */
Object.defineProperty(FORGE.Hotspot3D.prototype, "state",
{
    /** @this {FORGE.Hotspot3D} */
    get: function()
    {
        return this._states.state;
    },

    /** @this {FORGE.Hotspot3D} */
    set: function(value)
    {
        this._states.load(value);
    }
});

/**
 * Hotspot sound handles the parse of the sound config and the loading of the needed sound resource.
 * It also ajusts the volume of the sound depending of your camera position.
 *
 * @constructor FORGE.HotspotSound
 * @param {FORGE.Viewer} viewer - The viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.HotspotSound = function(viewer, hotspotUid)
{
    /**
     * Viewer reference.
     * @name  FORGE.HotspotSound#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The hotspot uid
     * @name FORGE.HotspotMaterial#_hotspotUid
     * @type {string}
     * @private
     */
    this._hotspotUid = hotspotUid;

    /**
     * The FORGE.Sound object
     * @name FORGE.HotspotSound#_sound
     * @type {FORGE.Sound}
     * @private
     */
    this._sound = null;

    /**
     * The minimum volume when you are out of range
     * @name FORGE.HotspotSound#_volumeMin
     * @type {number}
     * @private
     */
    this._volumeMin = 0;

    /**
     * The maximum volume when you are in the center of the range
     * @name FORGE.HotspotSound#_volumeMax
     * @type {number}
     * @private
     */
    this._volumeMax = 1;

    /**
     * Is sound looped?
     * @name FORGE.HotspotSound#_loop
     * @type {boolean}
     * @private
     */
    this._loop = false;

    /**
     * Start time for the sound.
     * @name  FORGE.HotspotSound#_startTime
     * @type {number}
     * @private
     */
    this._startTime = 0;

    /**
     * Is sound auto started?
     * @name  FORGE.HotspotSound#_autoPlay
     * @type {boolean}
     * @private
     */
    this._autoPlay = false;

    /**
     * The theta/phi position of the sound.
     * @name  FORGE.HotspotSound#_position
     * @type {?HotspotTransformPosition}
     * @private
     *
     * @todo Use the HotspotTransformPosition object of the HotspotTransform class?
     */
    this._position = { x: 0, y: 0, z: 200 };

    /**
     * The range in degrees where sound can be played from it's position.
     * @name  FORGE.HotspotSound#_range
     * @type {number}
     * @default
     * @private
     */
    this._range = 360;

    /**
     * The onReady event dispatcher.
     * @name  FORGE.HotspotSound#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    FORGE.BaseObject.call(this, "HotspotSound");
};

FORGE.HotspotSound.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotSound.prototype.constructor = FORGE.HotspotSound;

/**
 * Parse the configuration object.
 * @method FORGE.HotspotSound#_parseConfig
 * @param {SoundConfig} config - The configuration object of the sound.
 * @private
 */
FORGE.HotspotSound.prototype._parseConfig = function(config)
{
    if (typeof config.source === "undefined" || typeof config.source !== "object")
    {
        return;
    }

    // Warning : UID is not registered and applied to the FORGE.Sound for registration
    this._uid = config.uid;

    // Is it a source url
    if (typeof config.source.url !== "undefined" && typeof config.source.url === "string" && config.source.url !== "")
    {
        this._url = config.source.url;
    }
    // or a source UID?
    else if (typeof config.source.target !== "undefined" && FORGE.UID.exists(config.source.target) === true)
    {
        //@todo
        var object = FORGE.UID.get(config.source.target);
        //this._url = "";
        return;
    }
    else
    {
        return;
    }

    if (typeof config.options !== "undefined" && typeof config.options === "object")
    {
        if(typeof config.options.volume === "number")
        {
            this._volumeMax = config.options.volume;
        }
        else if (typeof config.options.volume === "object")
        {
            var volume = /** @type {SoundVolumeConfig} */ (config.options.volume);
            this._volumeMin = (typeof volume.min === "number") ? FORGE.Math.clamp(volume.min, 0, 1) : 0;
            this._volumeMax = (typeof volume.max === "number") ? FORGE.Math.clamp(volume.max, 0, 1) : 1;
        }

        this._loop = (typeof config.options.loop === "boolean") ? config.options.loop : false;
        this._startTime = (typeof config.options.startTime === "number") ? config.options.startTime : 0; //in ms
        this._autoPlay = (typeof config.options.autoPlay === "boolean") ? config.options.autoPlay : false;
        this._range = (typeof config.options.range === "number") ? FORGE.Math.clamp(config.options.range, 1, 360) : 360;
    }

    var hotspot = FORGE.UID.get(this._hotspotUid);
    var position = hotspot.config.transform.position;

    this._position.theta = (typeof position.theta === "number") ? FORGE.Math.clamp(/** @type {number} */ (position.theta), -180, 180) : 0;
    this._position.phi = (typeof position.phi === "number") ? FORGE.Math.clamp(/** @type {number} */ (position.phi), -90, 90) : 0;
    //@todo manage radius

    this._setupSound();
};

/**
 * Setup the sound and apply options.
 * @method FORGE.HotspotSound#_setupSound
 * @private
 */
FORGE.HotspotSound.prototype._setupSound = function()
{
    this._sound = new FORGE.Sound(this._viewer, this._uid, this._url || "");
    this._sound.onCanPlayThrough.addOnce(this._onCanPlayThroughHandler, this);

    // spatial sound options
    if(this._isSpatialized() === true)
    {
        // Create world position from inversed theta angle and phi angle
        var positionWorld = FORGE.Math.sphericalToCartesian(1, /** @type {number} */ (this._position.theta), /** @type {number} */ (this._position.phi));

        this._sound.spatialized = this._isSpatialized();
        this._sound.x = positionWorld.x;
        this._sound.y = positionWorld.y;
        this._sound.z = positionWorld.z;
    }

    // sound options
    this._sound.volume = this._volumeMax;
    this._sound.loop = this._loop;
    this._sound.startTime = this._startTime;

    if (this._autoPlay === true)
    {
        if (document[FORGE.Device.visibilityState] === "visible")
        {
            this._sound.play(this._startTime, this._loop, true);
        }
        else
        {
            this._sound.resumed = true;
        }
    }
};

/**
 * Is the sound spatialized?
 * @method  FORGE.HotspotSound#_isSpatialized
 * @return {boolean} Is spatialized?
 * @private
 */
FORGE.HotspotSound.prototype._isSpatialized = function()
{
    return this._range < 360;
};

/**
 * On can play through handler.
 * @method  FORGE.HotspotSound#_onCanPlayThroughHandler
 * @private
 */
FORGE.HotspotSound.prototype._onCanPlayThroughHandler = function()
{
    this.log("Sound load complete");
    this._setupComplete();
};

/**
 * Setup completed handler.
 * @method FORGE.HotspotSound#_setupComplete
 * @private
 */
FORGE.HotspotSound.prototype._setupComplete = function()
{
    if(this._onReady !== null)
    {
        this._onReady.dispatch();
    }
};

/**
 * Set an observer cone for the sound volume.
 * @method FORGE.HotspotSound#_applyRange
 * @private
 *
 * @todo  doesn't work if the camera is reversed on the y axis !
 */
FORGE.HotspotSound.prototype._applyRange = function()
{
    if(this._isSpatialized() === true && typeof this._range === "number")
    {
        var camera = this._viewer.renderer.camera;
        var qCamera = FORGE.Quaternion.fromEuler(FORGE.Math.degToRad(camera.yaw), FORGE.Math.degToRad(camera.pitch), 0);
        var qSound = FORGE.Quaternion.fromEuler(FORGE.Math.degToRad(/** @type {number} */ (this._position.theta)), FORGE.Math.degToRad(/** @type {number} */ (this._position.phi)), 0);
        var distance = FORGE.Quaternion.angularDistance(qSound, qCamera);
        var radius = FORGE.Math.degToRad(this._range / 2); //from range to radius in radians

        // reduce the volume or "mute" volume when out of the range
        if (distance < radius)
        {
            var amplitude = this._volumeMax - this._volumeMin;
            var volume = this._volumeMin + (amplitude * (1 - distance / radius));
            this.log("in range | volume: "+volume);

            this._sound.volume = volume;
        }
        else
        {
            this.log("out range");
            this._sound.volume = this._volumeMin;
        }
    }
};

/**
 * Load a sound configuration
 * @method FORGE.HotspotSound#load
 * @param  {SoundConfig} config - The hotspot sound configuration object.
 */
FORGE.HotspotSound.prototype.load = function(config)
{
    if(FORGE.Utils.compareObjects(this._config, config) === true)
    {
        if(this._onReady !== null)
        {
            this._onReady.dispatch();
        }

        if(this._autoPlay === true && this._sound !== null)
        {
            this._sound.play(this._startTime, this._loop, true);
        }

        return;
    }

    this._config = config;

    if (this._sound !== null)
    {
        this._sound.destroy();
        this._sound = null;
    }

    this._parseConfig(config);
};

/**
 * Render sound.
 * @method FORGE.HotspotSound#update
 */
FORGE.HotspotSound.prototype.update = function()
{
    if (this._viewer.audio.enabled === false)
    {
        if(this._sound !== null)
        {
            this._sound.stop();
        }

        return;
    }

    if(this._isSpatialized() === true)
    {
        this._applyRange();
    }
};

/**
 * Destroy routine.
 * @method FORGE.HotspotSound#destroy
 */
FORGE.HotspotSound.prototype.destroy = function()
{
    if (this._sound !== null)
    {
        this._sound.destroy();
        this._sound = null;
    }

    if(this._onReady !== null)
    {
        this._onReady.destroy();
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the {@link FORGE.Sound} object.
 * @name FORGE.HotspotSound#sound
 * @readonly
 * @type {FORGE.Sound}
 */
Object.defineProperty(FORGE.HotspotSound.prototype, "sound",
{
    /** @this {FORGE.HotspotSound} */
    get: function()
    {
        return this._sound;
    }
});

/**
 * Get the "onReady" {@link FORGE.EventDispatcher} of this hotspot sound.
 * @name FORGE.HotspotSound#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotSound.prototype, "onReady",
{
    /** @this {FORGE.HotspotSound} */
    get: function()
    {
        if(this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});

/**
 * Hotspot material handles the parse of the material config and the loading of the needed resource.<br>
 * In the end it provides a THREE.MeshBasicMaterial when the resources are loaded.
 *
 * @constructor FORGE.HotspotMaterial
 * @param {FORGE.Viewer} viewer - The viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.HotspotMaterial = function(viewer, hotspotUid)
{
    /**
     * Viewer reference.
     * @name  FORGE.HotspotMaterial#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The hotspot uid
     * @name FORGE.HotspotMaterial#_hotspotUid
     * @type {string}
     * @private
     */
    this._hotspotUid = hotspotUid;

    /**
     * Hotspot material config
     * @name FORGE.HotspotMaterial#_config
     * @type {?HotspotMaterialConfig}
     * @private
     */
    this._config = null;

    /**
     * Material input type.<br>
     * Can be one of the values listed in FORGE.HotspotMaterial.types
     * @name  FORGE.HotspotMaterial#_type
     * @type {string}
     * @private
     */
    this._type = "";

    /**
     * THREE texture.
     * @name  FORGE.HotspotMaterial#_texture
     * @type {THREE.Texture|THREE.CanvasTexture}
     * @private
     */
    this._texture = null;

    /**
     * THREE material.
     * @name  FORGE.HotspotMaterial#_material
     * @type {THREE.RawShaderMaterial}
     * @private
     */
    this._material = null;

    /**
     * The opacity of the material (between 0 and 1)
     * @name  FORGE.HotspotMaterial#_opacity
     * @type {number}
     * @default
     * @private
     */
    this._opacity = 1;

    /**
     * The transparent flag of the material.<br>
     * if you use a PNG file as texture with some transparency, you have to set this to true.
     * @name  FORGE.HotspotMaterial#_transparent
     * @type {boolean}
     * @default
     * @private
     */
    this._transparent = false;

    /**
     * The base color of the material.<br>
     * Can be a number 0xff0000 or a string: rgb(255, 0, 0), rgb(100%, 0%, 0%), hsl(0, 100%, 50%), #ff0000.
     * @name  FORGE.HotspotMaterial#_opacity
     * @type {(number|string)}
     * @default
     * @private
     */
    this._color = 0xffffff;

    /**
     * The display object used for the texture
     * @name  FORGE.HotspotMaterial#_displayObject
     * @type {FORGE.DisplayObject}
     * @private
     */
    this._displayObject = null;

    /**
     * Does this material needs update
     * @name FORGE.HotspotMaterial#_update
     * @type {boolean}
     * @private
     */
    this._update = false;

    /**
     * Ready flag
     * @name FORGE.HotspotMaterial#_ready
     * @type {boolean}
     * @private
     */
    this._ready = false;

    /**
     * The onReady event dispatcher.
     * @name  FORGE.HotspotMaterial#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    FORGE.BaseObject.call(this, "HotspotMaterial");
};

FORGE.HotspotMaterial.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotMaterial.prototype.constructor = FORGE.HotspotMaterial;

/**
 * Material input type list.
 * @name FORGE.HotspotMaterial.types
 * @type {Object}
 * @const
 */
FORGE.HotspotMaterial.types = {};

/**
 * @name FORGE.HotspotMaterial.types.IMAGE
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.types.IMAGE = "image";

/**
 * @name FORGE.HotspotMaterial.types.SPRITE
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.types.SPRITE = "sprite";

/**
 * @name FORGE.HotspotMaterial.types.VIDEO
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.types.VIDEO = "video";

/**
 * @name FORGE.HotspotMaterial.types.PLUGIN
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.types.PLUGIN = "plugin";

/**
 * @name FORGE.HotspotMaterial.types.GRAPHICS
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.types.GRAPHICS = "graphics";

/**
 * Material sides list.
 * @name FORGE.HotspotMaterial.sides
 * @type {Object}
 * @const
 */
FORGE.HotspotMaterial.sides = {};

/**
 * @name FORGE.HotspotMaterial.sides.FRONT
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.sides.FRONT = "front";

/**
 * @name FORGE.HotspotMaterial.sides.BACK
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.sides.BACK = "back";

/**
 * @name FORGE.HotspotMaterial.sides.DOUBLE
 * @type {string}
 * @const
 */
FORGE.HotspotMaterial.sides.DOUBLE = "double";


/**
 * Materials presets.
 * @name FORGE.HotspotMaterial.presets
 * @type {Object}
 * @const
 */
FORGE.HotspotMaterial.presets = {};

/**
 * @name FORGE.HotspotMaterial.presets.TRANSPARENT
 * @type {HotspotMaterialConfig}
 * @const
 */
FORGE.HotspotMaterial.presets.TRANSPARENT =
{
    color: "#ffffff",
    opacity: 0,
    transparent: false
};

/**
 * @name FORGE.HotspotMaterial.presets.DEBUG
 * @type {HotspotMaterialConfig}
 * @const
 */
FORGE.HotspotMaterial.presets.DEBUG =
{
    color: "#00ff00",
    opacity: 0.8,
    transparent: false
};

/**
 * Parse the configuration object.
 * @method FORGE.HotspotMaterial#_parseConfig
 * @param  {HotspotMaterialConfig} config - Configuration object of the material.
 * @private
 */
FORGE.HotspotMaterial.prototype._parseConfig = function(config)
{
    this._opacity = (typeof config.opacity === "number") ? FORGE.Math.clamp(config.opacity, 0, 1) : 1;
    this._transparent = (typeof config.transparent === "boolean") ? config.transparent : false;
    this._color = (typeof config.color === "string") ? config.color : 0xffffff;
    this._update = (typeof config.update === "boolean") ? config.update : false;
    this._side = (typeof config.side === "string") ? config.side : FORGE.HotspotMaterial.sides.DOUBLE;

    // Hotspot with image as background
    if (typeof config.image !== "undefined" && config.image !== null)
    {
        this._setupWithImage(config.image);
    }

    // Hotspot with animated sprite as background
    else if (typeof config.sprite !== "undefined" && config.sprite !== null)
    {
        this._setupWithSprite(config.sprite);
    }

    // Hotspot with video as background
    else if (typeof config.video !== "undefined" && config.video !== null)
    {
        this._setupWithVideo(config.video);
    }

    // Hotspot with plugin that provide a texture as background
    else if (typeof config.plugin !== "undefined" && config.plugin !== null)
    {
        this._setupWithPlugin(config.plugin);
    }

    // Hotspot with graphical options as background
    else
    {
        this._setupWithGraphics();
    }
};

/**
 * Setup hotspot material with an image as texture.
 * @method FORGE.HotspotMaterial#_setupWithImage
 * @param {(string|ImageConfig)} config - The image configuration you want to load and use as a texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._setupWithImage = function(config)
{
    this._type = FORGE.HotspotMaterial.types.IMAGE;

    // If the config is a string, we assume that this is the image url.
    // We convert the config into a image config object.
    if(typeof config === "string")
    {
        config = { url: config };
    }

    //Force the render mode to canvas
    config.renderMode = FORGE.Image.renderModes.CANVAS;

    this._displayObject = new FORGE.Image(this._viewer, config);
    this._displayObject.onLoadComplete.addOnce(this._imageLoadCompleteHandler, this);
};

/**
 * Image loaded event handler for the image setup.
 * @method FORGE.HotspotMaterial#_imageLoadCompleteHandler
 * @param {FORGE.Event} event - load event
 * @private
 */
FORGE.HotspotMaterial.prototype._imageLoadCompleteHandler = function(event)
{
    var image = /** @type {FORGE.Image} */ (event.emitter);

    this.log("image load complete : " + image.element.src);
    this._createTextureFromImage(image);
};

/**
 * Create a THREE.Texture from the loaded FORGE.Image
 * @method  FORGE.HotspotMaterial#_createTextureFromImage
 * @param  {FORGE.Image} image - The FORGE.Image used to create the texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._createTextureFromImage = function(image)
{
    this._displayObject = image;

    this._texture = new THREE.Texture();
    this._texture.generateMipmaps = false;
    this._texture.minFilter = THREE.LinearFilter;

    this.setTextureFrame(image.frame);

    this.log("create texture from image");

    this._setupComplete();
};

/**
 * Setup hotspot material with a sprite as texture.
 * @method FORGE.HotspotMaterial#_setupWithSprite
 * @param {(string|SpriteConfig)} config - The sprite configuration you want to load and use as a texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._setupWithSprite = function(config)
{
    this._type = FORGE.HotspotMaterial.types.SPRITE;
    this._update = true;

    this._displayObject = new FORGE.Sprite(this._viewer, config);
    this._displayObject.onLoadComplete.addOnce(this._spriteLoadCompleteHandler, this);
};

/**
 * Sprite loaded event handler for the sprite setup.
 * @method FORGE.HotspotMaterial#_spriteLoadCompleteHandler
 * @param {FORGE.Event} event - load event
 * @private
 */
FORGE.HotspotMaterial.prototype._spriteLoadCompleteHandler = function(event)
{
    var sprite = /** @type {FORGE.Sprite} */ (event.emitter);

    this.log("sprite load complete");
    this._createTextureFromSprite(sprite);
};

/**
 * Create a THREE.Texture from the loaded FORGE.Sprite
 * @method  FORGE.HotspotMaterial#_createTextureFromSprite
 * @param  {FORGE.Sprite} sprite - The FORGE.Sprite used to create the texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._createTextureFromSprite = function(sprite)
{
    this._texture = new THREE.Texture();
    this._texture.generateMipmaps = false;
    this._texture.minFilter = THREE.LinearFilter;

    this.setTextureFrame(sprite.frame);

    this.log("create texture from sprite");

    this._setupComplete();
};

/**
 * Setup hotspot material with a video as texture.
 * @method FORGE.HotspotMaterial#_setupWithVideo
 * @param {(string|VideoConfig)} config - The video configuration you want to load and use as a texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._setupWithVideo = function(config)
{
    this._type = FORGE.HotspotMaterial.types.VIDEO;
    this._update = true;

    this._displayObject = new FORGE.VideoHTML5(this._viewer, this._hotspotUid+"-material-video");
    this._displayObject.currentTime = 100000;
    this._displayObject.onLoadedMetaData.addOnce(this._videoLoadedMetaDataHandler, this);
    this._displayObject.load(config.url);
};

/**
 * Video meta data loaded event handler for the video setup.
 * @method FORGE.HotspotMaterial#_videoLoadedMetaDataHandler
 * @param {FORGE.Event} event - load event
 * @private
 */
FORGE.HotspotMaterial.prototype._videoLoadedMetaDataHandler = function(event)
{
    var video = /** @type {FORGE.VideoBase} */ (event.emitter);
    video.play();

    this.log("video load complete");
    this._createTextureFromVideo(video);
};

/**
 * Create a THREE.Texture from the loaded FORGE.Video
 * @method FORGE.HotspotMaterial#_createTextureFromVideo
 * @param {FORGE.VideoBase} video - The FORGE.Video used to create the texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._createTextureFromVideo = function(video)
{
    this.log("create texture from video");

    this._texture = new THREE.Texture();
    this._texture.generateMipmaps = false;
    this._texture.minFilter = THREE.LinearFilter;

    this._texture.image = video.element;

    this._setupComplete();
};

/**
 * Create a hotspot with a plugin that will provide a texture as input.
 * @method FORGE.HotspotMaterial#_setupWithPlugin
 * @param {string} config - The plugin instance UID you want to use as a texture provider.<br>
 * The plugin have to have a "texture" public property.
 * @private
 */
FORGE.HotspotMaterial.prototype._setupWithPlugin = function(config)
{
    this._type = FORGE.HotspotMaterial.types.PLUGIN;

    var plugin = this._viewer.plugins.get(config);

    if (typeof plugin === "undefined" || plugin === null)
    {
        this._viewer.plugins.onInstanceCreate.add(this._pluginInstanceCreateHandler, this);
    }
    else
    {
        this._setPlugin(plugin);
    }
};

/**
 * Plugin instance create event handler.
 * @method FORGE.HotspotMaterial#_pluginInstanceCreateHandler
 * @param {FORGE.Event} event - instance create event
 * @private
 */
FORGE.HotspotMaterial.prototype._pluginInstanceCreateHandler = function(event)
{
    var plugin = /** @type {FORGE.Plugin} */ (event.data);

    if (plugin.uid === this._config.plugin)
    {
        this._viewer.plugins.onInstanceCreate.remove(this._pluginInstanceCreateHandler, this);
        this._setPlugin(plugin);
    }
};

/**
 * Once the plugin is created by the manager we can set the plugin that we will provide the texture.
 * This method will check if the instance is ready, if not will setup a listener.
 * @method FORGE.HotspotMaterial#_setPlugin
 * @param {FORGE.Plugin} plugin - The plugin that will provides the texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._setPlugin = function(plugin)
{
    if (plugin.instanceReady === true)
    {
        this._createTextureFromPlugin(plugin);
    }
    else
    {
        plugin.onInstanceReady.addOnce(this._pluginInstanceReadyHandler, this);
    }
};

/**
 * Plugin instance ready event handler.
 * @method FORGE.HotspotMaterial#_pluginInstanceReadyHandler
 * @param {FORGE.Event} event - instance ready event
 * @private
 */
FORGE.HotspotMaterial.prototype._pluginInstanceReadyHandler = function(event)
{
    var plugin = /** @type {FORGE.Plugin} */ (event.emitter);

    if (plugin.instance === null || plugin.instanceReady === false)
    {
        throw new Error("Plugin instance not available");
    }

    this._createTextureFromPlugin(plugin);
};

/**
 * Create a texture from a plugin that provides a display object on a "texture" property.
 * @method FORGE.HotspotMaterial#_createTextureFromPlugin
 * @param {FORGE.Plugin} plugin - plugin that provides the texture.
 * @private
 */
FORGE.HotspotMaterial.prototype._createTextureFromPlugin = function(plugin)
{
    this._displayObject = plugin.instance.texture;

    this._texture = new THREE.Texture(this._displayObject.dom);
    this._texture.format = THREE.RGBAFormat;
    this._texture.minFilter = THREE.LinearFilter;
    this._texture.generateMipmaps = false;
    this._texture.needsUpdate = true;

    this.log("create texture from plugin");

    this._setupComplete();
};

/**
 * Create a hotspot material with graphical attributes.
 * @method FORGE.HotspotMaterial#_setupWithGraphics
 * @private
 * @todo Make a config to use only graphical properties
 */
FORGE.HotspotMaterial.prototype._setupWithGraphics = function()
{
    this._type = FORGE.HotspotMaterial.types.GRAPHICS;

    this._setupComplete();
};

/**
 * This is the final setup step when any of the loading or wainting instance ready process is complete.
 * @method FORGE.HotspotMaterial#_setupComplete
 * @private
 */
FORGE.HotspotMaterial.prototype._setupComplete = function()
{
    this._createShaderMaterial();

    this._ready = true;

    if (this._onReady !== null)
    {
        this._onReady.dispatch();
    }
};

/**
 * Create the THREE.MeshBasicMaterial that will be used on a THREE.Mesh
 * @method FORGE.HotspotMaterial#_createShaderMaterial
 * @private
 */
FORGE.HotspotMaterial.prototype._createShaderMaterial = function()
{
    this.log("create shader material");

    if(this._viewer.renderer.view.current === null)
    {
        return;
    }

    if (this._material !== null)
    {
        this._material.dispose();
        this._material = null;
    }

    var shader = FORGE.Utils.clone(this._viewer.renderer.view.current.shaderWTS.mapping);

    if (this._type === FORGE.HotspotMaterial.types.GRAPHICS)
    {
        shader.fragmentShader = FORGE.ShaderChunk.wts_frag_color;
        shader.uniforms.tColor = { type: "c", value: new THREE.Color(this._color) };
    }

    shader.uniforms.tOpacity = { type: "f", value: this._opacity };

    var vertexShader = FORGE.ShaderLib.parseIncludes(shader.vertexShader);
    var fragmentShader = FORGE.ShaderLib.parseIncludes(shader.fragmentShader);

    this._material = new THREE.RawShaderMaterial(
    {
        fragmentShader: fragmentShader,
        vertexShader: vertexShader,
        uniforms: /** @type {FORGEUniform} */ (shader.uniforms),
        side: this._getThreeSide(this._side),
        name: "HotspotMaterial"
    });

    this._material.transparent = this._transparent;
    this._material.needsUpdate = true;
};

/**
 * Converts the side string to the side number of Three
 * @method FORGE.HotspotMaterial#_getThreeSide
 * @param {string} [side] the string that represents the side of the material from the HotspotMaterial.sides
 * @return {number} [description]
 * @private
 */
FORGE.HotspotMaterial.prototype._getThreeSide = function(side)
{
    var result = THREE.DoubleSide; // Default is double

    switch(side)
    {
        case FORGE.HotspotMaterial.sides.FRONT:
            result = THREE.FrontSide;
            break;

        case FORGE.HotspotMaterial.sides.BACK:
            result = THREE.BackSide;
            break;

        case FORGE.HotspotMaterial.sides.DOUBLE:
            result = THREE.DoubleSide;
            break;
    }

    return result;
};

FORGE.HotspotMaterial.prototype.updateShader = function()
{
    this._createShaderMaterial();
};

/**
 * Load a material configuration
 * @method FORGE.HotspotMaterial#load
 * @param  {HotspotMaterialConfig} config - The hotspot material configuration object.
 */
FORGE.HotspotMaterial.prototype.load = function(config)
{
    this._config = config;
    this._ready = false;
    this._parseConfig(this._config);
};

/**
 * Update method taht will be called by the Hotspot.
 * @method FORGE.HotspotMaterial#update
 */
FORGE.HotspotMaterial.prototype.update = function()
{
    if (this._material && this._material.uniforms)
    {
        this._material.uniforms.tTexture.value = this._texture;
        this._material.uniforms.tOpacity.value = this._opacity;
    }

    if(this._texture !== null && this._update === true)
    {
        this._texture.needsUpdate = true;
    }
};


/**
 * Set texture frame
 * @method FORGE.HotspotMaterial#setTextureFrame
 * @param {FORGE.Rectangle=} frame - texture frame
 */
FORGE.HotspotMaterial.prototype.setTextureFrame = function(frame)
{
    // Only support type IMAGE and SPRITE
    if (this._displayObject === null || (this._type !== FORGE.HotspotMaterial.types.IMAGE && this._type !== FORGE.HotspotMaterial.types.SPRITE))
    {
        return;
    }

    var textureFrame = frame || new FORGE.Rectangle(0, 0, this._displayObject.element.naturalWidth, this._displayObject.element.naturalHeight);

    this._displayObject.frame = textureFrame;

    this._texture.image = this._displayObject.canvas;
    this._texture.image.crossOrigin = "anonymous";
    this._texture.needsUpdate = true;

    this.update();
};

/**
 * Dump the material configuration
 * @method FORGE.HotspotMaterial#dump
 * @return {HotspotMaterialConfig}
 */
FORGE.HotspotMaterial.prototype.dump = function()
{
    var dump =
    {
        color: this._color,
        opacity: this._opacity,
        transparent: this._transparent,
        side: this._side,
        update: this._update
    };

    switch(this._type)
    {
        case FORGE.HotspotMaterial.types.IMAGE:
            dump.image = this._config.image;
            break;

        case FORGE.HotspotMaterial.types.SPRITE:
            dump.sprite = this._config.sprite;
            break;

        case FORGE.HotspotMaterial.types.VIDEO:
            dump.video = this._config.video;
            break;

        case FORGE.HotspotMaterial.types.PLUGIN:
            dump.plugin = this._config.plugin;
            break;
    }

    return dump;
};

/**
 * Destroy sequence.
 * @method FORGE.HotspotMaterial#destroy
 */
FORGE.HotspotMaterial.prototype.destroy = function()
{
    if (this._texture !== null)
    {
        this._texture.dispose();
        this._texture = null;
    }

    if (this._material !== null)
    {
        this._material.dispose();
        this._material = null;
    }

    if(this._displayObject !== null)
    {
        // Don't destroy the texture from plugin.
        // Let the plugin erase its own content.
        if (this._type !== FORGE.HotspotMaterial.types.PLUGIN)
        {
            this._displayObject.destroy();
        }

        this._displayObject = null;
    }

    if(this._onReady !== null)
    {
        this._onReady.destroy();
        this._onReady = null;
    }

    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the type of texture provider of this hotspot material.
 * @name FORGE.HotspotMaterial#type
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "type",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Get the THREE.Texture used for this hotspot material.
 * @name FORGE.HotspotMaterial#texture
 * @readonly
 * @type {THREE.Texture}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "texture",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._texture;
    }
});

/**
 * Get the THREE.MeshBasicMaterial used for this hotspot material.
 * @name FORGE.HotspotMaterial#material
 * @readonly
 * @type {THREE.MeshBasicMaterial}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "material",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._material;
    }
});

/**
 * Get the opacity of this hotspot material.
 * @name FORGE.HotspotMaterial#opacity
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "opacity",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._opacity;
    }
});

/**
 * Get the transparent flag of this hotspot material.
 * @name FORGE.HotspotMaterial#transparent
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "transparent",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._transparent;
    }
});

/**
 * Get the color of this hotspot material.
 * @name FORGE.HotspotMaterial#color
 * @readonly
 * @type {(string|number)}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "color",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._color;
    }
});

/**
 * Get the side of this hotspot material.
 * @name FORGE.HotspotMaterial#side
 * @readonly
 * @type {(string)}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "side",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._side;
    }
});

/**
 * Get the displayObject of this hotspot material.
 * @name FORGE.HotspotMaterial#displayObject
 * @readonly
 * @type {(FORGE.Image|FORGE.DisplayObject)}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "displayObject",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._displayObject;
    }
});

/**
 * Get the ready flag of this hotspot material.
 * @name FORGE.HotspotMaterial#ready
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "ready",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get the "onReady" {@link FORGE.EventDispatcher} of this hotspot material.<br/>
 * Dispatched when the material texture is loaded and ready to be used by a THREE.Texture.
 * @name FORGE.HotspotMaterial#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotMaterial.prototype, "onReady",
{
    /** @this {FORGE.HotspotMaterial} */
    get: function()
    {
        if (this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});

/**
 * Forge Hotspot Geometry
 * @constructor FORGE.HotspotGeometry
 */
FORGE.HotspotGeometry = function()
{
    /**
     * Geometry configuration
     * @name FORGE.HotspotGeometry#_config
     * @type {HotspotGeometryConfig}
     * @private
     */
    this._config = FORGE.HotspotGeometry.DEFAULT_CONFIG;

    /**
     * The geometry type
     * @name FORGE.HotspotGeometry#_type
     * @type {string}
     * @private
     */
    this._type = "";

    /**
     * The THREE geometry object
     * @name FORGE.HotspotGeometry#_geometry
     * @type {(THREE.Geometry|THREE.PlaneBufferGeometry|THREE.BoxBufferGeometry|THREE.SphereBufferGeometry|THREE.CylinderBufferGeometry|THREE.ShapeBufferGeometry)}
     * @private
     */
    this._geometry = null;

    /**
     * The offset applied to the geometry object from it's center.<br>
     * Is expressed in world units (x, y, z).
     * @name FORGE.HotspotGeometry#_offset
     * @type {HotspotGeometryOffset}
     * @private
     */
    this._offset = FORGE.HotspotGeometry.DEFAULT_OFFSET;

    /**
     * Event dispatcher for the onLoadComplete
     * @name FORGE.HotspotGeometry#_onLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadComplete = null;
};

/**
 * @name FORGE.HotspotGeometry.DEFAULT_CONFIG
 * @type {HotspotGeometryConfig}
 * @const
 */
FORGE.HotspotGeometry.DEFAULT_CONFIG =
{
    type: "plane",

    options:
    {
        width: 20,
        height: 20,
        widthSegments: 8,
        heightSegments: 8
    },

    offset: FORGE.HotspotGeometry.DEFAULT_OFFSET
};

/**
 * @name FORGE.HotspotGeometry.DEFAULT_OFFSET
 * @type {HotspotGeometryOffset}
 * @const
 */
FORGE.HotspotGeometry.DEFAULT_OFFSET =
{
    x: 0,
    y: 0,
    z: 0
};

/**
 * Parse the hotspot geometry config
 * @method FORGE.HotspotGeometry#_parseConfig
 * @param {HotspotGeometryConfig} config
 */
FORGE.HotspotGeometry.prototype._parseConfig = function(config)
{
    this._type = config.type;

    var options = config.options;

    this._offset = (typeof config.offset !== "undefined") ? config.offset : { x: 0, y: 0, z: 0 };

    switch (this._type)
    {
        case FORGE.HotspotGeometryType.PLANE:
            this._geometry = this._createPlane(options);
            break;

        case FORGE.HotspotGeometryType.BOX:
            this._geometry = this._createBox(options);
            break;

        case FORGE.HotspotGeometryType.SPHERE:
            this._geometry = this._createSphere(options);
            break;

        case FORGE.HotspotGeometryType.CYLINDER:
            this._geometry = this._createCylinder(options);
            break;

        case FORGE.HotspotGeometryType.SHAPE:
            this._geometry = this._createShape(options);
            break;

        default:
            this._geometry = this._createPlane(options);
            break;
    }

    // apply init offset values
    this._geometry.applyMatrix( new THREE.Matrix4().makeTranslation( this._offset.x, this._offset.y, this._offset.z ) );
};

/**
 * @method FORGE.HotspotGeometry#_createPlane
 * @param {HotspotGeometryPlane=} options
 * @return {THREE.PlaneBufferGeometry}
 * @private
 */
FORGE.HotspotGeometry.prototype._createPlane = function(options)
{
    options = options || {};

    var width = options.width || 20;
    var height = options.height || 20;
    var widthSegments = options.widthSegments || 8;
    var heightSegments = options.heightSegments || 8;

    return new THREE.PlaneBufferGeometry(width, height, widthSegments, heightSegments);
};

/**
 * @method FORGE.HotspotGeometry#_createBox
 * @param {HotspotGeometryBox=} options
 * @return {THREE.BoxBufferGeometry}
 * @private
 */
FORGE.HotspotGeometry.prototype._createBox = function(options)
{
    options = options || {};

    var width = options.width || 20;
    var height = options.height || 20;
    var depth = options.depth || 20;
    var widthSegments = options.widthSegments || 8;
    var heightSegments = options.heightSegments || 8;
    var depthSegments = options.depthSegments || 8;

    return new THREE.BoxBufferGeometry(width, height, depth, widthSegments, heightSegments, depthSegments);
};

/**
 * @method FORGE.HotspotGeometry#_createSphere
 * @param {HotspotGeometrySphere=} options
 * @return {THREE.SphereBufferGeometry}
 * @private
 */
FORGE.HotspotGeometry.prototype._createSphere = function(options)
{
    options = options || {};

    var radius = options.radius || 10;
    var widthSegments = options.widthSegments || 64;
    var heightSegments = options.heightSegments || 64;
    var phiStart = options.phiStart || 0;
    var phiLength = options.phiLength || 2 * Math.PI;
    var thetaStart = options.thetaStart || 0;
    var thetaLength = options.thetaLength || 2 * Math.PI;

    return new THREE.SphereBufferGeometry(radius, widthSegments, heightSegments, phiStart, phiLength, thetaStart, thetaLength);
};

/**
 * @method FORGE.HotspotGeometry#_createCylinder
 * @param {HotspotGeometryCylinder=} options
 * @return {THREE.CylinderBufferGeometry}
 * @private
 */
FORGE.HotspotGeometry.prototype._createCylinder = function(options)
{
    options = options || {};

    var radiusTop = options.radiusTop || 10;
    var radiusBottom = options.radiusBottom || 10;
    var height = options.height || 20;
    var radiusSegments = options.radiusSegments || 32;
    var heightSegments = options.heightSegments || 1;
    var openEnded = options.openEnded || false;
    var thetaStart = options.thetaStart || 0;
    var thetaLength = options.thetaLength || 2 * Math.PI;

    return new THREE.CylinderBufferGeometry(radiusTop, radiusBottom, height, radiusSegments, heightSegments, openEnded, thetaStart, thetaLength);
};

/**
 * @method FORGE.HotspotGeometry#_createShape
 * @param {HotspotGeometryShape=} options
 * @return {THREE.ShapeBufferGeometry}
 * @private
 */
FORGE.HotspotGeometry.prototype._createShape = function(options)
{
    options = options || {};

    // clean the points given to remove any duplicate when points are following each other
    if (Array.isArray(options.points))
    {
        options.points.push(options.points[0]);

        var a, b, res = [];
        for (var i = 0, ii = options.points.length - 1; i < ii; i++)
        {
            a = options.points[i];
            b = options.points[i + 1];

            if (a[0] !== b[0] || a[1] !== b[1])
            {
                res.push(a);
            }
        }

        options.points = res;

        if (options.points.length < 3)
        {
            console.warn("FORGE.HotspotGeometry.SHAPE: the points given to draw the shape should be a least 3");
            options.points = null;
        }
    }

    //Default points array that is a square
    if (Array.isArray(options.points) === false)
    {
        //Arrow shape
        options.points =
        [
            [ -10, 3 ],
            [ 3, 3 ],
            [ 3, 8 ],
            [ 15, 0 ],
            [ 3, -8 ],
            [ 3, -3 ],
            [ -10, -3 ]
        ];
    }

    var points = [];
    for (var i = 0, ii = options.points.length; i < ii; i++)
    {
        var point = options.points[i];
        var x, y;

        if(Array.isArray(point) === true)
        {
            x = (typeof point[0] === "number" && isNaN(point[0]) === false) ? point[0] : 0;
            y = (typeof point[1] === "number" && isNaN(point[1]) === false) ? point[1] : 0;
            points.push(new THREE.Vector2(x, y));
        }
    }

    return new THREE.ShapeBufferGeometry(new THREE.Shape(points));
};

/**
 * Load a hotspot geometry config
 * @method FORGE.HotspotGeometry#load
 * @param {HotspotGeometryConfig} config
 */
FORGE.HotspotGeometry.prototype.load = function(config)
{
    if (typeof config !== "undefined" && typeof config.type === "string")
    {
        this._config = config;
    }
    else
    {
        this._config = /** @type {HotspotGeometryConfig} */ (FORGE.Utils.extendSimpleObject({}, FORGE.HotspotGeometry.DEFAULT_CONFIG));
    }

    this._parseConfig(this._config);

    if(this._onLoadComplete !== null)
    {
        this._onLoadComplete.dispatch();
    }
};

/**
 * Dump the geometry configuration
 * @method FORGE.HotspotGeometry#dump
 * @return {HotspotGeometryConfig} Return the actual hotspot geometry config
 */
FORGE.HotspotGeometry.prototype.dump = function()
{
    var dump =
    {
        type: this._type
    };

    var options = {};

    switch(this._type)
    {
        case FORGE.HotspotGeometryType.PLANE:
        case FORGE.HotspotGeometryType.BOX:
        case FORGE.HotspotGeometryType.SPHERE:
        case FORGE.HotspotGeometryType.CYLINDER:
            options = this._geometry.parameters;
            break;

        case FORGE.HotspotGeometryType.SHAPE:

            var points = [];
            if(Array.isArray(this._geometry.parameters.shapes.curves) === true)
            {
                var c = this._geometry.parameters.shapes.curves;

                points.push([c[0].v1.x, c[0].v1.y]);
                points.push([c[0].v2.x, c[0].v2.y]);

                for(var i = 1, ii = c.length; i < ii; i++)
                {
                    points.push([c[i].v2.x, c[i].v2.y]);
                }
            }
            else
            {
                points = this._config.options.points;
            }

            options = { points: points };

            break;
    }

    dump.options = options;

    return dump;
};

/**
 * Destroy sequence
 * @method FORGE.HotspotGeometry#destroy
 */
FORGE.HotspotGeometry.prototype.destroy = function()
{
    if(this._geometry !== null)
    {
        this._geometry.dispose();
        this._geometry = null;
    }

    if(this._onLoadComplete !== null)
    {
        this._onLoadComplete.destroy();
        this._onLoadComplete = null;
    }
};

/**
 * Geometry type accessor
 * @name FORGE.HotspotGeometry#type
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.HotspotGeometry.prototype, "type",
{
    /** @this {FORGE.HotspotGeometry} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Geometry accessor
 * @name FORGE.HotspotGeometry#geometry
 * @readonly
 * @type {THREE.Geometry}
 */
Object.defineProperty(FORGE.HotspotGeometry.prototype, "geometry",
{
    /** @this {FORGE.HotspotGeometry} */
    get: function()
    {
        return this._geometry;
    }
});

/**
 * Get the onLoadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.HotspotGeometry#onLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotGeometry.prototype, "onLoadComplete",
{
    /** @this {FORGE.HotspotGeometry} */
    get: function()
    {
        if (this._onLoadComplete === null)
        {
            this._onLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onLoadComplete;
    }
});

/**
 * Get/set the offset of the geomerty object.<br>
 * The current offset is updated with these values and don't erase the init offset.
 * @name FORGE.HotspotGeometry#offset
 * @type {HotspotGeometryOffset}
 */
Object.defineProperty(FORGE.HotspotGeometry.prototype, "offset",
{
    /** @this {FORGE.HotspotGeometry} */
    get: function()
    {
        return this._offset;
    },

    /** @this {FORGE.HotspotGeometry} */
    set: function(value)
    {
        if (typeof value !== "undefined" && (typeof value.x === "number" || typeof value.y === "number" || typeof value.z === "number"))
        {
            var offset = /** @type {HotspotGeometryOffset} */ (FORGE.Utils.extendSimpleObject(FORGE.HotspotGeometry.DEFAULT_OFFSET, value));
            this._geometry.applyMatrix( new THREE.Matrix4().makeTranslation( offset.x, offset.y, offset.z ) );

            // update current offset values
            this._offset.x += offset.x;
            this._offset.y += offset.y;
            this._offset.z += offset.z;
        }
    }
});
/**
 * @namespace {Object} FORGE.HotspotGeometryType
 */
FORGE.HotspotGeometryType = {};

/**
 * @name FORGE.HotspotGeometryType.PLANE
 * @type {string}
 * @const
 */
FORGE.HotspotGeometryType.PLANE = "plane";

/**
 * @name FORGE.HotspotGeometryType.BOX
 * @type {string}
 * @const
 */
FORGE.HotspotGeometryType.BOX = "box";

/**
 * @name FORGE.HotspotGeometryType.SPHERE
 * @type {string}
 * @const
 */
FORGE.HotspotGeometryType.SPHERE = "sphere";

/**
 * @name FORGE.HotspotGeometryType.CYLINDER
 * @type {string}
 * @const
 */
FORGE.HotspotGeometryType.CYLINDER = "cylinder";

/**
 * @name FORGE.HotspotGeometryType.SHAPE
 * @type {string}
 * @const
 */
FORGE.HotspotGeometryType.SHAPE = "shape";


/**
 * HotspotTransform handle the parsing of the position, rotation and scale of a 3d Hotspot.
 *
 * @constructor FORGE.HotspotTransform
 * @extends {FORGE.BaseObject}
 */
FORGE.HotspotTransform = function()
{
    /**
     * The cartesian coordinates of a 3D object (x, y, z).
     * @name FORGE.HotspotTransform#_position
     * @type {FORGE.HotspotTransformValues}
     * @private
     */
    this._position = null;

    /**
     * The rotation of a 3D object (x, y, z).
     * @name FORGE.HotspotTransform#_rotation
     * @type {FORGE.HotspotTransformValues}
     * @private
     */
    this._rotation = null;

    /**
     * The scale of a 3D object.<br>
     * Is expressed in world units (x, y, z).
     * @name FORGE.HotspotTransform#_scale
     * @type {FORGE.HotspotTransformValues}
     * @private
     */
    this._scale = null;

    /**
     * onChange event dispatcher for transform change.
     * @name  FORGE.HotspotTransform#_onChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onChange = null;

    FORGE.BaseObject.call(this, "HotspotTransform");

    this._boot();
};

FORGE.HotspotTransform.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotTransform.prototype.constructor = FORGE.HotspotTransform;

/**
 * Init HotspotTransform.
 * @method  FORGE.HotspotTransform#_boot
 * @private
 */
FORGE.HotspotTransform.prototype._boot = function()
{
    this._register();

    this._position = new FORGE.HotspotTransformValues(this._uid, 0, 0, -200);
    this._rotation = new FORGE.HotspotTransformValues(this._uid, 0, 0, 0);
    this._scale = new FORGE.HotspotTransformValues(this._uid, 1, 1, 1);
};

/**
 * Parse the config object, set default values where values are undefined.
 * @method FORGE.HotspotTransform#_parseConfig
 * @param {HotspotTransformConfig} config - The transform config to parse.
 * @return {boolean} return true if one of the values has changed
 * @private
 */
FORGE.HotspotTransform.prototype._parseConfig = function(config)
{
    var changed = false;

    if (typeof config.position !== "undefined")
    {
        var position = FORGE.Utils.extendSimpleObject(this._position.dump(), this._parsePosition(config.position));

        if (FORGE.Utils.compareObjects(this._position.dump(), position) === false)
        {
            this._position.load(/** @type {HotspotTransformValuesConfig} */ (position), false);
            changed = true;
        }
    }

    if (typeof config.rotation !== "undefined")
    {
        var rotation = FORGE.Utils.extendSimpleObject({}, this._rotation.dump());

        rotation.x = (typeof config.rotation.x === "number") ? config.rotation.x : 0;
        rotation.y = (typeof config.rotation.y === "number") ? config.rotation.y : 0;
        rotation.z = (typeof config.rotation.z === "number") ? config.rotation.z : 0;

        if (FORGE.Utils.compareObjects(this._rotation.dump(), rotation) === false)
        {
            this._rotation.load(/** @type {HotspotTransformValuesConfig} */ (rotation), false);
            changed = true;
        }
    }

    if (typeof config.scale !== "undefined")
    {
        var scale = FORGE.Utils.extendSimpleObject({}, this._scale.dump());

        scale.x = (typeof config.scale.x === "number") ? FORGE.Math.clamp(config.scale.x, 0.000001, 100000) : 1;
        scale.y = (typeof config.scale.y === "number") ? FORGE.Math.clamp(config.scale.y, 0.000001, 100000) : 1;
        scale.z = (typeof config.scale.z === "number") ? FORGE.Math.clamp(config.scale.z, 0.000001, 100000) : 1;

        if (FORGE.Utils.compareObjects(this._scale.dump(), scale) === false)
        {
            this._scale.load(/** @type {HotspotTransformValuesConfig} */ (scale), false);
            changed = true;
        }
    }

    return changed;
};

/**
 * Parse the position object.
 * @method FORGE.HotspotTransform#_parsePosition
 * @param {HotspotTransformPosition} config - The transform position config to parse.
 * @private
 */
FORGE.HotspotTransform.prototype._parsePosition = function(config)
{
    var position =
    {
        x: 0,
        y: 0,
        z: -200
    };

    if (typeof config !== "undefined" && config !== null)
    {
        if (typeof config.radius === "number" || typeof config.theta === "number" || typeof config.phi === "number")
        {
            var radius = (typeof config.radius === "number") ? config.radius : 200;
            var theta = (typeof config.theta === "number") ? FORGE.Math.degToRad(config.theta) : 0;
            var phi = (typeof config.phi === "number") ? FORGE.Math.degToRad(config.phi) : 0;

            position = FORGE.Math.sphericalToCartesian(radius, theta, phi);
        }
        else
        {
            position.x = (typeof config.x === "number") ? config.x : 0;
            position.y = (typeof config.y === "number") ? config.y : 0;
            position.z = (typeof config.z === "number") ? config.z : -200;
        }
    }

    return position;
};

/**
 * Update all the transform values from the mesh.
 * @method FORGE.HotspotTransform#updateFromObject3D
 * @param {THREE.Object3D} object - The 3D object to read data from.
 */
FORGE.HotspotTransform.prototype.updateFromObject3D = function(object)
{
    this._position.load(object.position);

    var rotation =
    {
        x: -FORGE.Math.radToDeg(object.rotation.x),
        y: FORGE.Math.radToDeg(object.rotation.y),
        z: FORGE.Math.radToDeg(object.rotation.z)
    };

    this._rotation.load(rotation);

    this._scale.load(object.scale);
};

/**
 * Notify the transform that a value has changed.
 * @method FORGE.HotspotTransform#notifyChange
 */
FORGE.HotspotTransform.prototype.notifyChange = function()
{
    if(this._onChange !== null)
    {
        this._onChange.dispatch();
    }
};


/**
 * Load a transform configuration.
 * @method FORGE.HotspotTransform#load
 * @param {HotspotTransformConfig} config - The transform config to load.
 * @param {boolean} [notify=true] - Do we have to notify the change of the transform after the config loading
 */
FORGE.HotspotTransform.prototype.load = function(config, notify)
{
    var changed = this._parseConfig(config);

    if (notify !== false && changed === true)
    {
        this.notifyChange();
    }
};

/**
 * Dump the transform actual configuration
 * @method FORGE.HotspotTransform#dump
 * @return {HotspotTransformConfig} Return the hotspot transform configuration
 */
FORGE.HotspotTransform.prototype.dump = function()
{
    var dump =
    {
        position: this._position.dump(),
        rotation: this._rotation.dump(),
        scale: this._scale.dump()
    };

    return dump;
};

/**
 * Destroy sequence.
 * @method FORGE.HotspotTransform#destroy
 */
FORGE.HotspotTransform.prototype.destroy = function()
{
    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get/set the spherical position of the transform object
 * @name FORGE.HotspotTransform#position
 * @type {HotspotTransformPosition}
 */
Object.defineProperty(FORGE.HotspotTransform.prototype, "position",
{
    /** @this {FORGE.HotspotTransform} */
    get: function()
    {
        return this._position;
    },

    /** @this {FORGE.HotspotTransform} */
    set: function(value)
    {
        var config = { position: value };
        this._parseConfig(config);
    }
});

/**
 * Get/set the rotation of the transform object
 * @name FORGE.HotspotTransform#rotation
 * @type {HotspotTransformRotation}
 */
Object.defineProperty(FORGE.HotspotTransform.prototype, "rotation",
{
    /** @this {FORGE.HotspotTransform} */
    get: function()
    {
        return this._rotation;
    },

    /** @this {FORGE.HotspotTransform} */
    set: function(value)
    {
        var config = { rotation: value };
        this._parseConfig(config);
    }
});

/**
 * Get/set the scale of the transform object
 * @name FORGE.HotspotTransform#scale
 * @type {HotspotTransformScale}
 */
Object.defineProperty(FORGE.HotspotTransform.prototype, "scale",
{
    /** @this {FORGE.HotspotTransform} */
    get: function()
    {
        return this._scale;
    },

    /** @this {FORGE.HotspotTransform} */
    set: function(value)
    {
        var config = { scale: value };
        this._parseConfig(config);
    }
});

/**
 * Get the onChange {@link FORGE.EventDispatcher}.
 * @name FORGE.HotspotTransform#onChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotTransform.prototype, "onChange",
{
    /** @this {FORGE.HotspotTransform} */
    get: function()
    {
        if (this._onChange === null)
        {
            this._onChange = new FORGE.EventDispatcher(this);
        }

        return this._onChange;
    }
});

/**
 * HotspotTransformValues handle the three values x, y and z.
 *
 * @constructor FORGE.HotspotTransformValues
 * @param {number} x - The spherical coordinates of a 3D object (radius, theta, phi)
 * @param {number} y - The rotation of a 3D object (x, y, z).
 * @param {number} z - The scale of a 3D object (x, y, z).
 */
FORGE.HotspotTransformValues = function(transformUid, x, y, z)
{
    /**
     * The transformation related to this values.
     * @name FORGE.HotspotTransformValues#_transformUid
     * @type {string}
     * @private
     */
    this._transformUid = transformUid;

    /**
     * The x.
     * @name FORGE.HotspotTransformValues#_x
     * @type {number}
     * @private
     */
    this._x = typeof(x) === "number" ? x : 0;

    /**
     * The y.
     * @name FORGE.HotspotTransformValues#_y
     * @type {number}
     * @private
     */
    this._y = typeof(y) === "number" ? y : 0;

    /**
     * The z.
     * @name FORGE.HotspotTransformValues#_z
     * @type {number}
     * @private
     */
    this._z = typeof(z) === "number" ? z : 0;

    /**
     * onChange event dispatcher for transform values change.
     * @name  FORGE.HotspotTransformValues#_onChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onChange = null;
};

/**
 * Load values.
 * @method FORGE.HotspotTransformValues#load
 * @param {HotspotTransformValuesConfig} values
 * @param {boolean} [notify=true] - Do we notify the transform object after the load ?
 * @return {boolean} Returns if any value has changed.
 */
FORGE.HotspotTransformValues.prototype.load = function(values, notify)
{
    var changed = false;

    if(typeof values.x === "number" && isNaN(values.x) === false && this._x !== values.x)
    {
        this._x = values.x;
        changed = true;
    }

    if(typeof values.y === "number" && isNaN(values.y) === false && this._y !== values.y)
    {
        this._y = values.y;
        changed = true;
    }

    if(typeof values.z === "number" && isNaN(values.z) === false && this._z !== values.z)
    {
        this._z = values.z;
        changed = true;
    }

    if(notify !== false && changed === true)
    {
        if(this._onChange !== null)
        {
            this._onChange.dispatch();
        }

        var transform = FORGE.UID.get(this._transformUid);
        transform.notifyChange();
    }

    return changed;
};

/**
 * Dump values.
 * @method FORGE.HotspotTransformValues#dump
 * @return {HotspotTransformValuesConfig}
 */
FORGE.HotspotTransformValues.prototype.dump = function()
{
    var dump =
    {
        x: this._x,
        y: this._y,
        z: this._z
    };

    return dump;
};

/**
 * Destroy sequence.
 * @method FORGE.HotspotTransformValues#destroy
 */
FORGE.HotspotTransformValues.prototype.destroy = function()
{
    if(this._onChange !== null)
    {
        this._onChange.destroy();
        this._onChange = null;
    }
};

/**
 * Get/set the x value
 * @name FORGE.HotspotTransformValues#x
 * @type {number}
 */
Object.defineProperty(FORGE.HotspotTransformValues.prototype, "x",
{
    /** @this {FORGE.HotspotTransformValues} */
    get: function()
    {
        return this._x;
    },

    /** @this {FORGE.HotspotTransformValues} */
    set: function(value)
    {
        if(typeof value !== "number" || isNaN(value) === true || this._x === value)
        {
            return;
        }

        this._x = value;

        if(this._onChange !== null)
        {
            this._onChange.dispatch();
        }

        var transform = FORGE.UID.get(this._transformUid);
        transform.notifyChange();
    }
});

/**
 * Get/set the y value
 * @name FORGE.HotspotTransformValues#y
 * @type {number}
 */
Object.defineProperty(FORGE.HotspotTransformValues.prototype, "y",
{
    /** @this {FORGE.HotspotTransformValues} */
    get: function()
    {
        return this._y;
    },

    /** @this {FORGE.HotspotTransformValues} */
    set: function(value)
    {
        if(typeof value !== "number" || isNaN(value) === true || this._y === value)
        {
            return;
        }

        this._y = value;

        if(this._onChange !== null)
        {
            this._onChange.dispatch();
        }

        var transform = FORGE.UID.get(this._transformUid);
        transform.notifyChange();
    }
});

/**
 * Get/set the x value
 * @name FORGE.HotspotTransformValues#x
 * @type {number}
 */
Object.defineProperty(FORGE.HotspotTransformValues.prototype, "z",
{
    /** @this {FORGE.HotspotTransformValues} */
    get: function()
    {
        return this._z;
    },

    /** @this {FORGE.HotspotTransformValues} */
    set: function(value)
    {
        if(typeof value !== "number" || isNaN(value) === true || this._z === value)
        {
            return;
        }

        this._z = value;

        if(this._onChange !== null)
        {
            this._onChange.dispatch();
        }

        var transform = FORGE.UID.get(this._transformUid);
        transform.notifyChange();
    }
});

/**
 * Get the vector representing the values
 * @name FORGE.HotspotTransformValues#values
 * @type {THREE.Vector3}
 */
Object.defineProperty(FORGE.HotspotTransformValues.prototype, "values",
{
    /** @this {FORGE.HotspotTransformValues} */
    get: function()
    {
        return new THREE.Vector3(this._x, this._y, this._z);
    }
});

/**
 * Get the onChange {@link FORGE.EventDispatcher}.
 * @name FORGE.HotspotTransformValues#onChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotTransformValues.prototype, "onChange",
{
    /** @this {FORGE.HotspotTransformValues} */
    get: function()
    {
        if (this._onChange === null)
        {
            this._onChange = new FORGE.EventDispatcher(this);
        }

        return this._onChange;
    }
});

/**
 * A FORGE.HotspotAnimation is used to animate a hotspot.
 *
 * @constructor FORGE.HotspotAnimation
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference
 * @param {FORGE.HotspotTransform} hotspotTransform - {@link FORGE.HotspotTransform} reference
 * @extends {FORGE.MetaAnimation}
 */
FORGE.HotspotAnimation = function(viewer, hotspotTransform)
{
    /**
     * The UID of the selected track.
     * @name FORGE.HotspotAnimation#_track
     * @type {string}
     * @private
     */
    this._track = "";

    /**
     * The list of the tracks composing the animation
     * @name FORGE.HotspotAnimation#_track
     * @type {?Array<string>}
     * @private
     */
    this._tracks = null;

    /**
     * Does the animation loop ?
     * @name FORGE.HotspotAnimation#_loop
     * @type {boolean}
     * @private
     */
    this._loop = false;

    /**
     * Does the animation randomized ?
     * @name FORGE.HotspotAnimation#_random
     * @type {boolean}
     * @private
     */
    this._random = false;

    /**
     * Does the animation auto play ?
     * @name FORGE.HotspotAnimation#_autoPlay
     * @type {boolean}
     * @private
     */
    this._autoPlay = false;

    /**
     * Enabled flag
     * @name FORGE.HotspotAnimation#_autoPlay
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * On animation progress event dispatcher.
     * @name FORGE.HotspotAnimation#_onProgress
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onProgress = null;

    FORGE.MetaAnimation.call(this, viewer, hotspotTransform, "HotspotAnimation");
    this._boot();
};

FORGE.HotspotAnimation.prototype = Object.create(FORGE.MetaAnimation.prototype);
FORGE.HotspotAnimation.prototype.constructor = FORGE.HotspotAnimation;

/**
 * Boot sequence
 *
 * @method FORGE.HotspotAnimation#_boot
 * @private
 */
FORGE.HotspotAnimation.prototype._boot = function()
{
    this._instructions =
    [
        {
            prop: [ "rotation.x", "rotation.y", "rotation.z" ],
            wrap: [ [-180, 180], [-180, 180], [-180, 180] ],
            smooth: false
        },
        {
            prop: [ "position.x", "position.y", "position.z" ],
            smooth: false
        },
        {
            prop: [ "scale.x", "scale.y", "scale.z" ],
            smooth: false
        }
    ];
};

/**
 * Load an animation configuration.
 *
 * @method FORGE.HotspotAnimation#load
 * @param {HotspotTrackConfig} config - The animation config to load.
 */
FORGE.HotspotAnimation.prototype.load = function(config)
{
    this.stop(); // Stop current aimation if any in all cases.

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;
    this._loop = (typeof config.loop === "boolean") ? config.loop : false;
    this._random = (typeof config.random === "boolean") ? config.random : false;
    this._autoPlay = (typeof config.autoPlay === "boolean") ? config.autoPlay : false;

    if (config.tracks !== null && FORGE.Utils.isArrayOf(config.tracks, "string"))
    {
        // If it is random, mix it here
        this._tracks = (this._random === true) ? FORGE.Utils.randomize(config.tracks) : config.tracks;
    }
};

/**
 * On track complete event handler
 *
 * @method FORGE.HotspotAnimation#_onTrackPartialCompleteHandler
 * @private
 */
FORGE.HotspotAnimation.prototype._onTrackPartialCompleteHandler = function()
{
    if (this._tracks.length > 1)
    {
        // Go to the next track if any
        var idx = this._tracks.indexOf( /** @type {string} */ (this._track)) + 1;

        // If the index is too high, play the track
        if (idx < this._tracks.length)
        {
            this.play(idx);
            return;
        }
    }

    // Loop only if it is the end of the last track
    if (this._loop === true)
    {
        // If it is random, change the entire order
        if (this._random === true)
        {
            this._tracks = FORGE.Utils.randomize(this._tracks);
        }

        this.play(0);
        return;
    }

    FORGE.MetaAnimation.prototype._onTrackPartialCompleteHandler.call(this);
};

/**
 * On tween progress event handler.
 *
 * @method FORGE.HotspotAnimation#_onTweenProgressHandler
 * @private
 */
FORGE.HotspotAnimation.prototype._onTweenProgressHandler = function()
{
    if (this._onProgress !== null)
    {
        this._onProgress.dispatch();
    }
};

/**
 * Play a set of tracks if specified, else the current one, from the start.
 *
 * @method FORGE.HotspotAnimation#play
 * @param {(string|number)=} track - A track
 */
FORGE.HotspotAnimation.prototype.play = function(track)
{
    if(this._enabled === false)
    {
        return;
    }

    if (typeof track === "number")
    {
        this._track = this._tracks[track];
    }
    else if (typeof track === "string")
    {
        this._track = track;
    }
    else
    {
        this._track = this._tracks[0];
    }

    track = FORGE.UID.get(this._track);

    this._emptyAnimations();

    var kf;
    var offset = track.offset;

    // First instruction - rotation
    var rotAnimation = new FORGE.Animation(this._viewer, this._target);
    rotAnimation.tween.easing = track.easing;
    rotAnimation.instruction = this._instructions[0];
    rotAnimation.instruction.smooth = track.smooth;
    rotAnimation.onComplete.add(this._onTrackPartialCompleteHandler, this);
    rotAnimation.onProgress.add(this._onTweenProgressHandler, this);

    var rot, x, y, z, time;

    for (var i = 0, ii = track.keyframes.length; i < ii; i++)
    {
        time = track.keyframes[i].time + offset;
        rot = track.keyframes[i].data.rotation;

        if (typeof rot !== "undefined" && rot !== null)
        {
            x = (typeof rot.x !== "undefined" && rot.x !== null) ? rot.x : this._computeIntermediateValue(time, track.keyframes, "x", rotAnimation.tween.easing, "rotation");
            y = (typeof rot.y !== "undefined" && rot.y !== null) ? rot.y : this._computeIntermediateValue(time, track.keyframes, "y", rotAnimation.tween.easing, "rotation");
            z = (typeof rot.z !== "undefined" && rot.z !== null) ? rot.z : this._computeIntermediateValue(time, track.keyframes, "z", rotAnimation.tween.easing, "rotation");

            kf = new FORGE.Keyframe(time + offset,
            {
                rotation:
                {
                    x: x,
                    y: y,
                    z: z
                }
            });

            rotAnimation.timeline.addKeyframe(kf);
        }
    }

    // If the first keyframe is not at time 0 or there is an offset, add a
    // virtual keyframe
    if (rotAnimation.timeline.keyframes.length > 0 && rotAnimation.timeline.keyframes[0].time > 0 || offset > 0)
    {
        var data = { rotation: FORGE.Utils.clone(this._target.rotation) };
        kf = new FORGE.Keyframe(0, data);
        rotAnimation.timeline.addKeyframe(kf);
    }

    this._animations.push(rotAnimation);

    // Second instruction - position
    var posAnimation = new FORGE.Animation(this._viewer, this._target);
    posAnimation.tween.easing = track.easing;
    posAnimation.instruction = this._instructions[1];
    posAnimation.instruction.smooth = track.smooth;
    posAnimation.onComplete.add(this._onTrackPartialCompleteHandler, this);
    posAnimation.onProgress.add(this._onTweenProgressHandler, this);

    var pos;

    for (var i = 0, ii = track.keyframes.length; i < ii; i++)
    {
        time = track.keyframes[i].time + offset;
        pos = track.keyframes[i].data.position;

        if (typeof pos !== "undefined" && pos !== null)
        {
            // If no x/y/z are defined, AND there is at least one radius/theta/phi, convert it to x/y/z
            if (typeof pos.x !== "number" && typeof pos.y !== "number" && typeof pos.z !== "number" && (typeof pos.radius === "number" || typeof pos.theta === "number" || typeof pos.phi === "number"))
            {
                var radius = (typeof pos.radius === "number") ? pos.radius : this._computeIntermediateValue(time, track.keyframes, "radius", posAnimation.tween.easing, "position");
                var theta = (typeof pos.theta === "number") ? FORGE.Math.degToRad(pos.theta) : FORGE.Math.degToRad(this._computeIntermediateValue(time, track.keyframes, "theta", posAnimation.tween.easing, "position"));
                var phi = (typeof pos.phi === "number") ? FORGE.Math.degToRad(pos.phi) : FORGE.Math.degToRad(this._computeIntermediateValue(time, track.keyframes, "phi", posAnimation.tween.easing, "position"));

                var cartesian = FORGE.Math.sphericalToCartesian(radius, theta, phi);

                x = cartesian.x;
                y = cartesian.y;
                z = cartesian.z;
            }
            else
            {

                x = (typeof pos.x !== "undefined" && pos.x !== null) ? pos.x : this._computeIntermediateValue(time, track.keyframes, "x", posAnimation.tween.easing, "position");
                y = (typeof pos.y !== "undefined" && pos.y !== null) ? pos.y : this._computeIntermediateValue(time, track.keyframes, "y", posAnimation.tween.easing, "position");
                z = (typeof pos.z !== "undefined" && pos.z !== null) ? pos.z : this._computeIntermediateValue(time, track.keyframes, "z", posAnimation.tween.easing, "position");
            }

            kf = new FORGE.Keyframe(time + offset,
            {
                position:
                {
                    x: x,
                    y: y,
                    z: z
                }
            });

            posAnimation.timeline.addKeyframe(kf);
        }
    }

    // If the first keyframe is not at time 0 or there is an offset, add a
    // virtual keyframe
    if (posAnimation.timeline.keyframes.length > 0 && posAnimation.timeline.keyframes[0].time > 0 || offset > 0)
    {
        var data = { position: FORGE.Utils.clone(this._target.position) };
        kf = new FORGE.Keyframe(0, data);
        posAnimation.timeline.addKeyframe(kf);
    }

    this._animations.push(posAnimation);

    // Third animation - scale
    var scaleAnimation = new FORGE.Animation(this._viewer, this._target);
    scaleAnimation.tween.easing = track.easing;
    scaleAnimation.instruction = this._instructions[2];
    scaleAnimation.instruction.smooth = track.smooth;
    scaleAnimation.onComplete.add(this._onTrackPartialCompleteHandler, this);
    scaleAnimation.onProgress.add(this._onTweenProgressHandler, this);

    var scale;

    for (var i = 0, ii = track.keyframes.length; i < ii; i++)
    {
        time = track.keyframes[i].time + offset;
        scale = track.keyframes[i].data.scale;

        if (typeof scale !== "undefined" && scale !== null)
        {
            x = (typeof scale.x !== "undefined" && scale.x !== null) ? scale.x : this._computeIntermediateValue(time, track.keyframes, "x", scaleAnimation.tween.easing, "scale");
            y = (typeof scale.y !== "undefined" && scale.y !== null) ? scale.y : this._computeIntermediateValue(time, track.keyframes, "y", scaleAnimation.tween.easing, "scale");
            z = (typeof scale.z !== "undefined" && scale.z !== null) ? scale.z : this._computeIntermediateValue(time, track.keyframes, "z", scaleAnimation.tween.easing, "scale");

            kf = new FORGE.Keyframe(time + offset,
            {
                scale:
                {
                    x: x,
                    y: y,
                    z: z
                }
            });

            scaleAnimation.timeline.addKeyframe(kf);
        }
    }

    // If the first keyframe is not at time 0 or there is an offset, add a
    // virtual keyframe
    if (scaleAnimation.timeline.keyframes.length > 0 && scaleAnimation.timeline.keyframes[0].time > 0 || offset > 0)
    {
        var data = { scale: FORGE.Utils.clone(this._target.scale) };
        kf = new FORGE.Keyframe(0, data);
        scaleAnimation.timeline.addKeyframe(kf);
    }

    this._animations.push(scaleAnimation);

    // Play it !
    this._animations[0].play();
    this._animations[1].play();
    this._animations[2].play();
};

/**
 * Destroy sequence.
 * @method FORGE.HotspotAnimation#destroy
 */
FORGE.HotspotAnimation.prototype.destroy = function()
{
    this._tracks = null;

    if (this._onProgress !== null)
    {
        this._onProgress.destroy();
        this._onProgress = null;
    }

    FORGE.MetaAnimation.prototype.destroy.call(this);
};

/**
 * Enabled flag.
 * @name FORGE.HotspotAnimation#enabled
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotAnimation.prototype, "enabled",
{
    /** @this {FORGE.HotspotAnimation} */
    get: function()
    {
        return this._enabled;
    }
});

/**
 * Does the animation auto play.
 * @name FORGE.HotspotAnimation#autoPlay
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.HotspotAnimation.prototype, "autoPlay",
{
    /** @this {FORGE.HotspotAnimation} */
    get: function()
    {
        return this._autoPlay;
    }
});

/**
 * Get the "onProgress" {@link FORGE.EventDispatcher} of the target.
 * @name FORGE.HotspotAnimation#onProgress
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotAnimation.prototype, "onProgress",
{
    /** @this {FORGE.HotspotAnimation} */
    get: function()
    {
        if (this._onProgress === null)
        {
            this._onProgress = new FORGE.EventDispatcher(this);
        }

        return this._onProgress;
    }
});

/**
 * A director track, that defines a camera animation.
 *
 * @constructor FORGE.HotspotAnimationTrack
 * @param {HotspotTrackConfig} config - Configuration of the track from the JSON file.
 * @extends {FORGE.Track}
 */
FORGE.HotspotAnimationTrack = function(config)
{
    /**
     * Does the track has a smooth interpolation between keyframes ?
     * @name FORGE.HotspotAnimationTrack#_smooth
     * @type {boolean}
     * @private
     */
    this._smooth = false;

    FORGE.Track.call(this, "HotspotAnimationTrack");

    this._boot(config);
};

FORGE.HotspotAnimationTrack.prototype = Object.create(FORGE.Track.prototype);
FORGE.HotspotAnimationTrack.prototype.constructor = FORGE.HotspotAnimationTrack;

/**
 * Boot sequence
 *
 * @method FORGE.HotspotAnimationTrack#_boot
 * @param  {Object} config - The information on the track
 * @private
 */
FORGE.HotspotAnimationTrack.prototype._boot = function(config)
{
    this._smooth = config.smooth;

    FORGE.Track.prototype._boot.call(this, config);
};
/**
 * Hotspot states
 *
 * @constructor FORGE.HotspotStates
 * @param {FORGE.Viewer} viewer - The viewer reference.
 * @param {string} hotspotUid - The hotspot uid
 * @extends {FORGE.BaseObject}
 */
FORGE.HotspotStates = function(viewer, hotspotUid)
{
    /**
     * Viewer reference.
     * @name  FORGE.HotspotStates#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The hotspot uid
     * @name FORGE.HotspotStates#_hotspotUid
     * @type {string}
     * @private
     */
    this._hotspotUid = hotspotUid;

    /**
     * Hotspot material config
     * @name FORGE.HotspotStates#_config
     * @type {?HotspotStatesConfig}
     * @private
     */
    this._config = null;

    /**
     * The current state name
     * @name FORGE.HotspotStates#_state
     * @type {string}
     * @private
     */
    this._state = "";

    /**
     * Default state
     * @name FORGE.HotspotStates#_default
     * @type {string}
     * @private
     */
    this._default = "default";

    /**
     * Does the states change automatically on interactive 3d objects?
     * @name FORGE.HotspotStates#_auto
     * @type {boolean}
     * @private
     */
    this._auto = true;

    /**
     * Ready flag
     * @name FORGE.HotspotStates#_ready
     * @type {boolean}
     * @private
     */
    this._ready = false;

    /**
     * Loading flags
     * @name FORGE.HotspotStates#_loading
     * @type {Object}
     * @private
     */
    this._loading =
    {
        geometry: false,
        transform: false,
        material: false,
        sound: false,
        animation: false
    };

    /**
     * Load complete event dispatcher for state change.
     * @name  FORGE.HotspotStates#_onLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadComplete = null;

    FORGE.BaseObject.call(this, "HotspotStates");

    this._boot();
};

FORGE.HotspotStates.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.HotspotStates.prototype.constructor = FORGE.HotspotStates;

/**
 * List of reserved keyword for states names.
 * @name FORGE.HotspotStates._RESERVED
 * @type {Array<string>}
 * @const
 * @private
 */
FORGE.HotspotStates._RESERVED = ["options"];

/**
 * Boot method.
 * @method FORGE.HotspotStates#_boot
 * @private
 */
FORGE.HotspotStates.prototype._boot = function()
{
    var hotspot = FORGE.UID.get(this._hotspotUid);

    // If no match, return
    if(typeof hotspot === "undefined")
    {
        throw("No hotspot match with uid: "+this._hotspotUid);
    }

    // Set the default config
    this._config =
    {
        options: {},

        default: hotspot.config
    };
};

/**
 * Parse the states configuration
 * @method FORGE.HotspotStates#_parseConfig
 * @param {?HotspotStatesConfig} config - The configuration to parse
 * @private
 */
FORGE.HotspotStates.prototype._parseConfig = function(config)
{
    this._default = (typeof config.options.default === "string") ? config.options.default : "default";
    this._auto = (typeof config.options.auto === "boolean") ? config.options.auto : true;

    this._config.default = FORGE.Utils.extendSimpleObject(this._config.default, config[this._default]);
};

/**
 * Get the state names (states config keys without reserved keys that are not states)
 * @method FORGE.HotspotStates#_getStatesNames
 * @return {Array<string>} Returns the names of the available states.
 * @private
 */
FORGE.HotspotStates.prototype._getStatesNames = function()
{
    if(this._config === null)
    {
        return [];
    }

    var config = /** @type {!Object} */ (this._config);
    var keys = Object.keys(config);

    // Remove the reserved keywords from state names
    for(var i = 0, ii = FORGE.HotspotStates._RESERVED.length; i < ii; i++)
    {
        var index = keys.indexOf(FORGE.HotspotStates._RESERVED[i]);

        if(index !== -1)
        {
            keys.splice(index, 1);
        }
    }

    // Remove the state named "default" if there is a default state in the options
    if(this._default !== "default" && typeof this._config[this._default] !== "undefined")
    {
        index = keys.indexOf("default");

        if(index !== -1)
        {
            keys.splice(index, 1);
        }
    }

    return keys;
};

/**
 * Update the geometry for the current state
 * @method FORGE.HotspotStates#_updateGeometry
 * @param  {HotspotGeometryConfig} config - The hotspot geometry configuration object.
 * @private
 */
FORGE.HotspotStates.prototype._updateGeometry = function(config)
{
    this.log("update geometry");

    var hotspot = FORGE.UID.get(this._hotspotUid);
    hotspot.geometry.onLoadComplete.addOnce(this._geometryLoadCompleteHandler, this);
    hotspot.geometry.load(config);
};

/**
 * Geometry load complete event handler
 * @method FORGE.HotspotStates#_geometryLoadCompleteHandler
 * @private
 */
FORGE.HotspotStates.prototype._geometryLoadCompleteHandler = function()
{
    this._loading.geometry = false;
    this._checkLoading();
};

/**
 * Update the material for the current state
 * @method FORGE.HotspotStates#_updateMaterial
 * @param  {HotspotMaterialConfig} config - The hotspot material configuration object.
 * @private
 */
FORGE.HotspotStates.prototype._updateMaterial = function(config)
{
    this.log("update material");

    var hotspot = FORGE.UID.get(this._hotspotUid);
    hotspot.material.onReady.addOnce(this._materialReadyHandler, this);
    hotspot.material.load(config);
};

/**
 * Material ready event handler
 * @method FORGE.HotspotStates#_materialReadyHandler
 * @private
 */
FORGE.HotspotStates.prototype._materialReadyHandler = function()
{
    this._loading.material = false;
    this._checkLoading();
};

/**
 * Update the sound for the current state
 * @method FORGE.HotspotStates#_updateSound
 * @param  {SoundConfig} config - The hotspot sound configuration object.
 * @private
 */
FORGE.HotspotStates.prototype._updateSound = function(config)
{
    this.log("update sound");

    var hotspot = FORGE.UID.get(this._hotspotUid);
    hotspot.sound.onReady.addOnce(this._soundReadyHandler, this);
    hotspot.sound.load(config);
};

/**
 * Sound ready event handler
 * @method FORGE.HotspotStates#_soundReadyHandler
 * @private
 */
FORGE.HotspotStates.prototype._soundReadyHandler = function()
{
    this._loading.sound = false;
    this._checkLoading();
};

/**
 * Update the animation for the current state
 * @method FORGE.HotspotStates#_updateAnimation
 * @param  {HotspotTrackConfig} config - The hotspot animation configuration object.
 * @private
 */
FORGE.HotspotStates.prototype._updateAnimation = function(config)
{
    this.log("update animation");

    var hotspot = FORGE.UID.get(this._hotspotUid);
    hotspot.animation.load(config, false);

    this._loading.animation = false;
    this._checkLoading();
};

/**
 * Update the transform for the current state
 * @method FORGE.HotspotStates#_updateTransform
 * @param  {HotspotTransformConfig} config - The hotspot transform configuration object.
 * @private
 */
FORGE.HotspotStates.prototype._updateTransform = function(config)
{
    this.log("update transform");

    var hotspot = FORGE.UID.get(this._hotspotUid);
    hotspot.transform.load(config, false);

    this._loading.transform = false;
    this._checkLoading();
};

/**
 * Check the loading status of the current state.
 * Dispatch the laod complete event if all check are ok.
 * @method FORGE.HotspotStates#_checkLoading
 * @private
 */
FORGE.HotspotStates.prototype._checkLoading = function()
{
    for(var prop in this._loading)
    {
        if(this._loading[prop] === true)
        {
            return true;
        }
    }

    this._ready = true;

    if(this._onLoadComplete !== null)
    {
        this._onLoadComplete.dispatch();
    }

    return false;
};

/**
 * Add a states configuration object.
 * @method FORGE.HotspotStates#addConfig
 * @param  {HotspotStatesConfig} config - Configuration object with hotspots states.
 */
FORGE.HotspotStates.prototype.addConfig = function(config)
{
    this._config = FORGE.Utils.extendSimpleObject(this._config, config);
    this._parseConfig(this._config);
};

/**
 * Load a state.
 * @method FORGE.HotspotStates#load
 * @param  {string=} name - the name of the state to load.
 */
FORGE.HotspotStates.prototype.load = function(name)
{
    name = (typeof name === "string") ? name : this._default;

    this.log("Hotspot load state: "+name);

    var hotspot = FORGE.UID.get(this._hotspotUid);

    // If no hotspot match OR no state name match OR already on this state THEN return
    if(typeof hotspot === "undefined" || typeof this._config[name] !== "object" || name === this._state)
    {
        return;
    }

    this._ready = false;

    // Set the state name
    this._state = name;

    if(typeof this._config[name].sound === "object")
    {
        this._loading.sound = true;
    }

    if(typeof this._config[name].animation === "object")
    {
        this._loading.animation = true;
    }

    this._loading.material = true;
    this._loading.geometry = true;
    this._loading.transform = true;

    if(this._loading.material === true)
    {
        var materialConfig = /** @type {!HotspotMaterialConfig} */ (FORGE.Utils.extendSimpleObject(hotspot.config.material, this._config[name].material));
        this._updateMaterial(materialConfig);
    }

    if(this._loading.sound === true)
    {
        var soundConfig = /** @type {!SoundConfig} */ (FORGE.Utils.extendSimpleObject(hotspot.config.sound, this._config[name].sound));
        this._updateSound(soundConfig);
    }

    if(this._loading.animation === true)
    {
        var animationConfig = /** @type {HotspotTrackConfig} */ (FORGE.Utils.extendSimpleObject(hotspot.config.animation, this._config[name].animation));
        this._updateAnimation(animationConfig);
    }

    if(this._loading.geometry === true)
    {
        var geometryConfig = /** @type {HotspotGeometryConfig} */ (FORGE.Utils.extendSimpleObject(hotspot.config.geometry, this._config[name].geometry));
        this._updateGeometry(geometryConfig);
    }

    if(this._loading.transform === true)
    {
        var transformConfig = /** @type {HotspotTransformConfig} */ (FORGE.Utils.extendSimpleObject(hotspot.config.transform, this._config[name].transform));
        this._updateTransform(transformConfig);
    }
};

/**
 * Toggle from a state to another.
 * @method FORGE.HotspotStates#toggle
 * @param  {Array<string>} names - the names of the states to loop through.
 */
FORGE.HotspotStates.prototype.toggle = function(names)
{
    if(names === null || typeof names === "undefined" || names.length < 2)
    {
        names = this._getStatesNames();
    }

    var current = names.indexOf(this._state);
    var next = 0;

    if(current < names.length - 1)
    {
        next = current + 1;
    }

    this.load(names[next]);
};

/**
 * Destroy sequence.
 * @method FORGE.HotspotStates#destroy
 */
FORGE.HotspotStates.prototype.destroy = function()
{
    this._viewer = null;

    if(this._onLoadComplete !== null)
    {
        this._onLoadComplete.destroy();
        this._onLoadComplete = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get and set the current state
 * @name FORGE.HotspotStates#state
 * @type {string}
 */
Object.defineProperty(FORGE.HotspotStates.prototype, "state",
{
    /** @this {FORGE.HotspotStates} */
    get: function()
    {
        return this._state;
    },

    /** @this {FORGE.HotspotStates} */
    set: function(value)
    {
        this.load(value);
    }
});

/**
 * Get the states names.
 * @name FORGE.HotspotStates#names
 * @type {Array<string>}
 * @readonly
 */
Object.defineProperty(FORGE.HotspotStates.prototype, "names",
{
    /** @this {FORGE.HotspotStates} */
    get: function()
    {
        return this._getStatesNames();
    }
});

/**
 * Get the ready flag.
 * @name FORGE.HotspotStates#ready
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.HotspotStates.prototype, "ready",
{
    /** @this {FORGE.HotspotStates} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get the auto flag.
 * @name FORGE.HotspotStates#auto
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.HotspotStates.prototype, "auto",
{
    /** @this {FORGE.HotspotStates} */
    get: function()
    {
        return this._auto;
    }
});

/**
 * Get the onConfigLoadComplete {@link FORGE.EventDispatcher}.
 * @name  FORGE.HotspotStates#onLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.HotspotStates.prototype, "onLoadComplete",
{
    /** @this {FORGE.HotspotStates} */
    get: function()
    {
        if (this._onLoadComplete === null)
        {
            this._onLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onLoadComplete;
    }
});


/**
 * @namespace {Object} FORGE.HotspotType
 */
FORGE.HotspotType = {};

/**
 * @name FORGE.HotspotType.THREE_DIMENSIONAL
 * @type {string}
 * @const
 */
FORGE.HotspotType.THREE_DIMENSIONAL = "3d";

/**
 * @name FORGE.HotspotType.DOM
 * @type {string}
 * @const
 */
FORGE.HotspotType.DOM = "dom";

/**
 * A FORGE.Camera tells the renderer wich part of the scene to render.
 *
 * @constructor FORGE.Camera
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.Camera = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.Camera#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Camera configuration that has been loaded.
     * @name  FORGE.Camera#_config
     * @type {?CameraConfig}
     * @private
     */
    this._config = null;

    /**
     * The yaw value in radians.
     * @name FORGE.Camera#_yaw
     * @type {number}
     * @private
     */
    this._yaw = 0;

    /**
     * The yaw minimum value in radians.
     * @name FORGE.Camera#_yawMin
     * @type {number}
     * @private
     */
    this._yawMin = -Infinity;

    /**
     * The yaw maximum value in radians.
     * @name FORGE.Camera#_yawMax
     * @type {number}
     * @private
     */
    this._yawMax = Infinity;

    /**
     * The pitch value in radians.
     * @name FORGE.Camera#_pitch
     * @type {number}
     * @private
     */
    this._pitch = 0;

    /**
     * The pitch minimum value in radians.
     * @name FORGE.Camera#_pitchMin
     * @type {number}
     * @private
     */
    this._pitchMin = -Infinity;

    /**
     * The pitch maximum value  in radians.
     * @name FORGE.Camera#_pitchMax
     * @type {number}
     * @private
     */
    this._pitchMax = Infinity;

    /**
     * The roll value in radians.
     * @name FORGE.Camera#_roll
     * @type {number}
     * @private
     */
    this._roll = 0;

    /**
     * The roll minimum value  in radians.
     * @name FORGE.Camera#_rollMin
     * @type {number}
     * @private
     */
    this._rollMin = -Infinity;

    /**
     * The roll maximum value in radians.
     * @name FORGE.Camera#_rollMax
     * @type {number}
     * @private
     */
    this._rollMax = Infinity;

    /**
     * The fov value in radians.
     * @name FORGE.Camera#_fov
     * @type {number}
     * @private
     */
    this._fov = 90;

    /**
     * The fov minimum value in radians.
     * @name FORGE.Camera#_fovMin
     * @type {number}
     * @private
     */
    this._fovMin = 0;

    /**
     * The fov maximum value in radians.
     * @name FORGE.Camera#_fovMax
     * @type {number}
     * @private
     */
    this._fovMax = Infinity;

    /**
     * Parallax setting
     * Value range is between 0 and 1
     * @name FORGE.Camera#_parallax
     * @type {number}
     * @private
     */
    this._parallax = 0;

    /**
     * Does the camera keep its orientation between scenes?
     * @name FORGE.Camera#_keep
     * @type {boolean}
     * @private
     */
    this._keep = false;

    /**
     * The modelview rotation matrix.
     * @name FORGE.Camera#_modelView
     * @type {THREE.Matrix4}
     * @private
     */
    this._modelView = null;

    /**
     * The inverse of the modelview rotation matrix.
     * @name FORGE.Camera#_modelViewInverse
     * @type {THREE.Matrix4}
     * @private
     */
    this._modelViewInverse = null;

    /**
     * Rotation quaternion of the camera
     * @name FORGE.Camera#_quaternion
     * @type {THREE.Quaternion}
     * @private
     */
    this._quaternion = null;

    /**
     * Three Perspective Camera object
     * @name FORGE.Camera#_main
     * @type {THREE.PerspectiveCamera}
     * @private
     */
    this._main = null;

    /**
     * Three Orthographic Camera object
     * @name FORGE.Camera#_flat
     * @type {THREE.OrthographicCamera}
     * @private
     */
    this._flat = null;

    /**
     * Left camera for VR rendering
     * @name  FORGE.Camera._left
     * @type {THREE.PerspectiveCamera}
     * @private
     */
    this._left = null;

    /**
     * Right camera for VR rendering
     * @name  FORGE.Camera._right
     * @type {THREE.PerspectiveCamera}
     * @private
     */
    this._right = null;

    /**
     * Three Perspective Camera radius (depends on parallax)
     * @name FORGE.Camera#_radius
     * @type {number}
     * @private
     */
    this._radius = 0;

    /**
     * Camera animation object
     * @name FORGE.Camera#_cameraAnimation
     * @type {FORGE.CameraAnimation}
     * @private
     */
    this._cameraAnimation = null;

    /**
     * Camera gaze cursor
     * @name FORGE.Camera#_gaze
     * @type {FORGE.CameraGaze}
     * @private
     */
    this._gaze = null;

    /**
     * Is the camera have load its configuration at least one time? For keep feature.
     * @name FORGE.Camera#_initialized
     * @type {boolean}
     * @private
     */
    this._initialized = false;

    /**
     * Log the camera changes between two updates.
     * @name FORGE.Camera#_changelog
     * @type {?CameraChangelog}
     * @private
     */
    this._changelog = null;

    /**
     * On camera change event dispatcher.
     * @name FORGE.Camera#_onChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onChange = null;

    /**
     * On camera orientation change event dispatcher.
     * @name FORGE.Camera#_onOrientationChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onOrientationChange = null;

    /**
     * On camera fov change event dispatcher.
     * @name FORGE.Camera#_onFovChange
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onFovChange = null;

    FORGE.BaseObject.call(this, "Camera");

    this._boot();
};

FORGE.Camera.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Camera.prototype.constructor = FORGE.Camera;

/**
 * Camera default radius for parallax.
 * @name FORGE.Camera.RADIUS
 * @type {number}
 * @const
 */
FORGE.Camera.RADIUS = 50;

/**
 * Camera default configuration in degrees like in the json configuration.
 * @name FORGE.Camera.DEFAULT_CONFIG
 * @type {CameraConfig}
 * @const
 */
FORGE.Camera.DEFAULT_CONFIG =
{
    keep: false,
    parallax: 0,
    yaw:
    {
        min: -Infinity,
        max: Infinity,
        default: 0
    },
    pitch:
    {
        min: -Infinity,
        max: Infinity,
        default: 0
    },
    roll:
    {
        min: -Infinity,
        max: Infinity,
        default: 0
    },
    fov:
    {
        min: 0,
        max: Infinity,
        default: 90
    },
    gaze:
    {
        delay: 2000,
        cursor:
        {
            innerRadius: 0.02,
            outerRadius: 0.04,
            color: 0xffffff,
            opacity: 0.5
        },
        progress:
        {
            innerRadius: 0.02,
            outerRadius: 0.04,
            color: 0xff0000,
            opacity: 0.5
        }
    }
};

/**
 * Init sequence.
 * @method FORGE.Camera#_boot
 * @private
 */
FORGE.Camera.prototype._boot = function()
{
    this._resetChangelog();

    this._modelView = new THREE.Matrix4();
    this._modelViewInverse = new THREE.Matrix4();
    this._quaternion = new THREE.Quaternion();

    this._gaze = new FORGE.CameraGaze(this._viewer, FORGE.Camera.DEFAULT_CONFIG.gaze);

    this._viewer.renderer.view.onChange.add(this._updateInternals, this);
    this._viewer.renderer.onBackgroundReady.add(this._updateInternals, this);

    this._createMainCamera();
    this._createFlatCamera();
    this._createVRCameras();

    // Check config to allow default to be set if they were depending
    // on some parameter external to the camera. For example: multiresolution fovMin set
    // by the background renderer
    if (this._config !== null)
    {
        this._parseConfig(this._config);
    }
};

/**
 * Parse a camera configuration.
 * @method FORGE.Camera#_parseConfig
 * @param {?CameraConfig} config - The camera configuration to parse.
 * @private
 */
FORGE.Camera.prototype._parseConfig = function(config)
{
    this._parallax = config.parallax;
    this._radius = this._parallax * FORGE.Camera.RADIUS;
    this._keep = config.keep;

    if(this._keep === true && this._initialized === true)
    {
        return;
    }

    if (typeof config.fov.min === "number")
    {
        this._fovMin = FORGE.Math.degToRad(config.fov.min);
    }

    if (typeof config.fov.max === "number")
    {
        this._fovMax = FORGE.Math.degToRad(config.fov.max);
    }

    if (typeof config.fov.default === "number")
    {
        this._setFov(config.fov.default, FORGE.Math.DEGREES);
    }

    if (typeof config.yaw.min === "number")
    {
        this._yawMin = FORGE.Math.degToRad(config.yaw.min);
    }

    if (typeof config.yaw.max === "number")
    {
        this._yawMax = FORGE.Math.degToRad(config.yaw.max);
    }

    if (typeof config.yaw.default === "number")
    {
        this._setYaw(config.yaw.default, FORGE.Math.DEGREES);
    }

    if (typeof config.pitch.min === "number")
    {
        this._pitchMin = FORGE.Math.degToRad(config.pitch.min);
    }

    if (typeof config.pitch.max === "number")
    {
        this._pitchMax = FORGE.Math.degToRad(config.pitch.max);
    }

    if (typeof config.pitch.default === "number")
    {
        this._setPitch(config.pitch.default, FORGE.Math.DEGREES);
    }

    if (typeof config.roll.min === "number")
    {
        this._rollMin = FORGE.Math.degToRad(config.roll.min);
    }

    if (typeof config.roll.max === "number")
    {
        this._rollMax = FORGE.Math.degToRad(config.roll.max);
    }

    if (typeof config.roll.default === "number")
    {
        this._setRoll(config.roll.default, FORGE.Math.DEGREES);
    }

    this._updateFromEuler();

    this._updateMainCamera();
    this._updateFlatCamera();

    this._gaze.load(/** @type {CameraGazeConfig} */ (config.gaze));
};

/**
 * Reset the camera changelog.
 * @method FORGE.Camera#_resetChangelog
 * @private
 */
FORGE.Camera.prototype._resetChangelog = function()
{
    this._changelog =
    {
        yaw: false,
        pitch: false,
        roll: false,
        fov: false
    };
};

/**
 * Init the THREE PerspectiveCamera.
 * @method FORGE.Camera#_createMainCamera
 * @private
 */
FORGE.Camera.prototype._createMainCamera = function()
{
    if (typeof this._viewer.renderer !== "undefined")
    {
        var aspect = this._viewer.renderer.displayResolution.ratio;
        this._main = new THREE.PerspectiveCamera(this._fov, aspect, FORGE.RenderManager.DEPTH_NEAR, 2 * FORGE.RenderManager.DEPTH_FAR);
        this._main.name = "CameraMain";
        this._main.matrixAutoUpdate = false;
    }
};

/**
 * Init the THREE OrthographicCamera.
 * @method FORGE.Camera#_createFlatCamera
 * @private
 */
FORGE.Camera.prototype._createFlatCamera = function()
{
    if (typeof this._viewer.renderer !== "undefined")
    {
        this._flat = new THREE.OrthographicCamera(
            -1000, 1000,
            1000, -1000,
            FORGE.RenderManager.DEPTH_NEAR,
            FORGE.RenderManager.DEPTH_FAR);

        this._flat.name = "CameraFlat";
        this._flat.matrixAutoUpdate = false;
    }
};

/**
 * Create the left and right THREE PerspectiveCamera for VR.
 * @method FORGE.Camera#_createVRCameras
 * @private
 */
FORGE.Camera.prototype._createVRCameras = function()
{
    this._left = this._main.clone();
    this._left.name = "CameraLeft";
    this._left.layers.enable(1);

    this._left.add(this._gaze.object);

    this._right = this._main.clone();
    this._right.name = "CameraRight";
    this._right.layers.enable(2);
};

/**
 * Update VR cameras.
 * @method FORGE.Camera#_updateVRCameras
 * @private
 */
FORGE.Camera.prototype._updateVRCameras = function()
{
    var display = this._viewer.renderer.display;

    // Get frame data before pose to ensure pose values are up to date
    var frameData = display.vrFrameData;
    var quat = display.getQuaternionFromPose();

    if (quat !== null)
    {
        this._quaternion = quat;
        this._updateFromQuaternion();
    }

    var eyeParamsL = display.vrDisplay.getEyeParameters("left");
    var eyeParamsR = display.vrDisplay.getEyeParameters("right");

    this._main.matrixWorld.decompose(this._left.position, this._left.quaternion, this._left.scale);
    this._left.matrixWorld = new THREE.Matrix4().makeRotationFromQuaternion(this._main.quaternion);

    this._main.matrixWorld.decompose(this._right.position, this._right.quaternion, this._right.scale);
    this._right.matrixWorld = new THREE.Matrix4().makeRotationFromQuaternion(this._main.quaternion);

    // Get translation from central camera matrix
    this._left.matrixWorld.elements[12] = this._main.matrixWorld.elements[12] + eyeParamsL.offset[0];
    this._left.matrixWorld.elements[13] = this._main.matrixWorld.elements[13] + eyeParamsL.offset[1];
    this._left.matrixWorld.elements[14] = this._main.matrixWorld.elements[14] + eyeParamsL.offset[2];

    // Get translation from central camera matrix
    this._right.matrixWorld.elements[12] = this._main.matrixWorld.elements[12] + eyeParamsR.offset[0];
    this._right.matrixWorld.elements[13] = this._main.matrixWorld.elements[13] + eyeParamsR.offset[1];
    this._right.matrixWorld.elements[14] = this._main.matrixWorld.elements[14] + eyeParamsR.offset[2];

    // Setup camera projection matrix
    if (frameData !== null)
    {
        this._left.projectionMatrix.elements = frameData.leftProjectionMatrix;
        this._right.projectionMatrix.elements = frameData.rightProjectionMatrix;
    }
    else
    {
        var eyeFOVL = {
            upDegrees: eyeParamsL.fieldOfView.upDegrees,
            downDegrees: eyeParamsL.fieldOfView.downDegrees,
            leftDegrees: eyeParamsL.fieldOfView.leftDegrees,
            rightDegrees: eyeParamsL.fieldOfView.rightDegrees
        };

        this._left.projectionMatrix = this._fovToProjectionMatrix(eyeFOVL, this._main);

        var eyeFOVR = {
            upDegrees: eyeParamsR.fieldOfView.upDegrees,
            downDegrees: eyeParamsR.fieldOfView.downDegrees,
            leftDegrees: eyeParamsR.fieldOfView.leftDegrees,
            rightDegrees: eyeParamsR.fieldOfView.rightDegrees
        };

        this._right.projectionMatrix = this._fovToProjectionMatrix(eyeFOVR, this._main);
    }

    this._updateComplete();
};

/**
 * Clone VR cameras objects.
 * @method FORGE.Camera#_cloneVRCamerasChildren
 * @private
 */
FORGE.Camera.prototype._cloneVRCamerasChildren = function()
{
    //First clear all children from camera right
    for (var i = 0, ii = this._right.children.length; i < ii; i++)
    {
        this._right.remove(this._right.children[i]);
    }

    //Then clone all children of camera left to camera right
    var clone = null;
    for (var j = 0, jj = this._left.children.length; j < jj; j++)
    {
        clone = this._left.children[j].clone();
        this._right.add(clone);
    }
};

/**
 * Get projection matrix from a VRFieldOfView
 * @method FORGE.Camera#_fovToProjectionMatrix
 * @param {VRFieldOfViewObject} fov - VRFieldOfView for an eye
 * @param {THREE.PerspectiveCamera} camera - reference camera
 * @return {THREE.Matrix4} projection matrix
 * @private
 */
FORGE.Camera.prototype._fovToProjectionMatrix = function(fov, camera)
{
    // Get projections of field of views on zn plane
    var fovUpTan = Math.tan(FORGE.Math.degToRad(fov.upDegrees));
    var fovDownTan = Math.tan(FORGE.Math.degToRad(fov.downDegrees));
    var fovLeftTan = Math.tan(FORGE.Math.degToRad(fov.leftDegrees));
    var fovRightTan = Math.tan(FORGE.Math.degToRad(fov.rightDegrees));

    // and with scale/offset info for normalized device coords
    var pxscale = 2.0 / (fovLeftTan + fovRightTan);
    var pxoffset = (fovLeftTan - fovRightTan) * pxscale * 0.5;
    var pyscale = 2.0 / (fovUpTan + fovDownTan);
    var pyoffset = (fovUpTan - fovDownTan) * pyscale * 0.5;

    // start with an identity matrix
    var matrix = new THREE.Matrix4();
    var m = matrix.elements;

    // X result, map clip edges to [-w,+w]
    m[0 * 4 + 0] = pxscale;
    m[0 * 4 + 1] = 0.0;
    m[0 * 4 + 2] = -pxoffset;
    m[0 * 4 + 3] = 0.0;

    // Y result, map clip edges to [-w,+w]
    // Y offset is negated because this proj matrix transforms from world coords with Y=up,
    // but the NDC scaling has Y=down (thanks D3D?)
    m[1 * 4 + 0] = 0.0;
    m[1 * 4 + 1] = pyscale;
    m[1 * 4 + 2] = pyoffset;
    m[1 * 4 + 3] = 0.0;

    // Z result (up to the app)
    m[2 * 4 + 0] = 0.0;
    m[2 * 4 + 1] = 0.0;
    m[2 * 4 + 2] = camera.far / (camera.near - camera.far);
    m[2 * 4 + 3] = (camera.far * camera.near) / (camera.near - camera.far);

    // W result (= Z in)
    m[3 * 4 + 0] = 0.0;
    m[3 * 4 + 1] = 0.0;
    m[3 * 4 + 2] = -1.0;
    m[3 * 4 + 3] = 0.0;

    matrix.transpose();

    return matrix;
};

/**
 * Apply Camera change internally.
 * @method FORGE.Camera#_updateFromEuler
 * @private
 */
FORGE.Camera.prototype._updateFromEuler = function()
{
    this._modelView = FORGE.Math.eulerToRotationMatrix(this._yaw, this._pitch, this._roll, false);

    this._modelViewInverse = this._modelView.clone().transpose();

    this._quaternion = FORGE.Quaternion.fromEuler(this._yaw, this._pitch, this._roll);

    // complete camera update
    this._updateComplete();
};

/**
 * Camera update internals after quaternion has been set
 * @method FORGE.Camera#_updateFromQuaternion
 * @private
 */
FORGE.Camera.prototype._updateFromQuaternion = function()
{
    this._modelView = FORGE.Quaternion.toRotationMatrix(this._quaternion);

    this._modelViewInverse = this._modelView.clone().transpose();

    var euler = FORGE.Quaternion.toEuler(this._quaternion);

    this._setAll(euler.yaw, euler.pitch, euler.roll, null, FORGE.Math.RADIANS);
};

/**
 * Camera update internals after modelview matrix has been set.
 * @method FORGE.Camera#_updateFromMatrix
 * @private
 */
FORGE.Camera.prototype._updateFromMatrix = function()
{
    this._modelViewInverse = this._modelView.clone().transpose();

    var euler = FORGE.Math.rotationMatrixToEuler(this._modelView);

    this._setAll(euler.yaw, euler.pitch, euler.roll, null, FORGE.Math.RADIANS);

    this._quaternion = FORGE.Quaternion.fromRotationMatrix(this._modelView);
};

/**
 * THREE Perspective camera update internals after modelview matrix has been set.
 * @method FORGE.Camera#_updateMainCamera
 * @private
 */
FORGE.Camera.prototype._updateMainCamera = function()
{
    if (this._main === null || this._viewer.renderer.view.current === null)
    {
        return;
    }

    var mat = new THREE.Matrix4().copy(this._modelViewInverse);

    if (this._parallax !== 0)
    {
        mat.multiply(new THREE.Matrix4().makeTranslation(0, 0, -this._radius));
    }

    // Now set the object quaternion (side effect: it will override the world matrix)
    this._main.quaternion.setFromRotationMatrix(mat);

    this._main.matrixWorld = mat;
    this._main.matrixWorldInverse.getInverse(mat);

    this._main.fov = FORGE.Math.radToDeg(this._viewer.renderer.view.current.getProjectionFov());
    this._main.aspect = this._viewer.renderer.displayResolution.ratio;
    this._main.updateProjectionMatrix();
};

/**
 * THREE Orthographic camera update internals.
 * @method FORGE.Camera#_updateFlatCamera
 * @private
 */
FORGE.Camera.prototype._updateFlatCamera = function()
{
    if (this._flat === null)
    {
        return;
    }

    var camW = this._flat.right - this._flat.left;
    var camH = this._flat.top - this._flat.bottom;

    this._flat.left = this._flat.position.x - camW / 2;
    this._flat.right = this._flat.position.x + camW / 2;

    this._flat.top = this._flat.position.y + camH / 2;
    this._flat.bottom = this._flat.position.y - camH / 2;

    var max = this._fovMax;
    var view = this._viewer.renderer.view.current;

    if (view !== null && view.fovMax !== null)
    {
        max = Math.min(view.fovMax, this._fovMax);
        this._flat.zoom = max / this._fov;
    }
    else
    {
        this._flat.zoom = 1;
    }

    this._flat.updateProjectionMatrix();
};

/**
 * Final method call to complete camera update, ensure main camera is up to date.
 * @method FORGE.Camera#_updateComplete
 * @private
 */
FORGE.Camera.prototype._updateComplete = function()
{
    var changed = false;

    if(this._changelog.yaw === true || this._changelog.pitch === true || this._changelog.roll === true)
    {
        changed = true;

        if (this._onOrientationChange !== null)
        {
            this._onOrientationChange.dispatch(null, true);
        }
    }

    if(this._changelog.fov === true)
    {
        changed = true;

        if (this._onFovChange !== null)
        {
            this._onFovChange.dispatch(null, true);
        }
    }


    if (changed === true && this._onChange !== null)
    {
        this._onChange.dispatch(null, true);
    }
};


/**
 * Internal setter for yaw, take a value and a unit. Default unit is radians.
 * @method FORGE.Camera#_setYaw
 * @param {?number=} value - The value you want to set for yaw.
 * @param {string=} [unit="radians"] - The unit you use to set the yaw value.
 * @return {boolean} Returns true if the value has changed.
 * @private
 */
FORGE.Camera.prototype._setYaw = function(value, unit)
{
    if (typeof value !== "number" || isNaN(value) === true)
    {
        return false;
    }

    // If unit is not well defined, default will be radians
    unit = (unit === FORGE.Math.DEGREES || unit === FORGE.Math.RADIANS) ? unit : FORGE.Math.RADIANS;

    // Convert value in radians for clamp if unit is in degrees.
    value = (unit === FORGE.Math.DEGREES) ? FORGE.Math.degToRad(value) : value;

    // Wrap the value between -PI and +PI, except for FLAT view where we apply texture ratio
    if (this._viewer.renderer.backgroundRenderer !== null &&
        this._viewer.renderer.backgroundRenderer.displayObject !== null &&
        this._viewer.renderer.view.type === FORGE.ViewType.FLAT)
    {
        var displayObject = this._viewer.renderer.backgroundRenderer.displayObject;
        var ratio = displayObject.pixelWidth / displayObject.pixelHeight;

        if (displayObject.element instanceof HTMLVideoElement)
        {
            ratio = displayObject.element.videoWidth / displayObject.element.videoHeight;
        }

        value = FORGE.Math.wrap(value, -Math.PI * ratio, Math.PI * ratio);
    }
    else
    {
        value = FORGE.Math.wrap(value, -Math.PI, Math.PI);
    }

    var boundaries = this._getYawBoundaries();

    var yaw = FORGE.Math.clamp(value, boundaries.min, boundaries.max);

    var changed = this._yaw !== yaw;

    this._changelog.yaw = changed;

    this._yaw = yaw;

    return changed;
};

/**
 * Compute the yaw boundaries with yaw min and yaw max.
 * @method FORGE.Camera#_getYawBoundaries
 * @param {boolean=} relative - do we need to get the yaw relative to the current fov (default true)
 * @param {number=} fov - specify a fov if we do not want to use the current one (useful for simulation)
 * @return {CameraBoundaries} Returns the min and max yaw computed from the camera configuration and the view limits.
 * @private
 */
FORGE.Camera.prototype._getYawBoundaries = function(relative, fov)
{
    var min = this._yawMin;
    var max = this._yawMax;

    fov = fov || this._fov;

    if (relative !== false && min !== max)
    {
        var halfHFov = 0.5 * fov * this._viewer.renderer.displayResolution.ratio;
        min += halfHFov;
        max -= halfHFov;
    }

    var view = this._viewer.renderer.view.current;

    if (view !== null)
    {
        min = Math.max(view.yawMin, min);
        max = Math.min(view.yawMax, max);
    }

    return { min: min, max: max };
};

/**
 * Internal setter for pitch, take a value and a unit. Default unit is radians.
 * @method FORGE.Camera#_setPitch
 * @param {?number=} value - The value you want to set for pitch.
 * @param {string=} [unit="radians"] - The unit you use to set the pitch value.
 * @return {boolean} Returns true if the value has changed.
 * @private
 */
FORGE.Camera.prototype._setPitch = function(value, unit)
{
    if (typeof value !== "number" || isNaN(value) === true)
    {
        return false;
    }

    var oldPitch = this._pitch;

    // If unit is not well defined, default will be radians
    unit = (unit === FORGE.Math.DEGREES || unit === FORGE.Math.RADIANS) ? unit : FORGE.Math.RADIANS;

    // Convert value in radians for clamp if unit is in degrees.
    value = (unit === FORGE.Math.DEGREES) ? FORGE.Math.degToRad(value) : value;

    // Wrap the value between -PI and +PI
    value = FORGE.Math.wrap(value, -Math.PI, Math.PI);

    var boundaries = this._getPitchBoundaries();

    var pitch = FORGE.Math.clamp(value, boundaries.min, boundaries.max);

    // If old view accepted pitch out of [-PI/2 , PI/2] and new one does not,
    // check if old pitch value was in authorized range and if not, set to zero
    if (Math.abs(oldPitch) > Math.PI / 2 && Math.abs(pitch) === Math.PI / 2)
    {
        pitch = 0;
    }

    var changed = this._pitch !== pitch;

    this._changelog.pitch = changed;

    this._pitch = pitch;

    return changed;
};

/**
 * Compute the pitch boundaries with pitch min and pitch max.
 * @method FORGE.Camera#_getPitchBoundaries
 * @param {boolean=} relative - do we need to get the pitch relative to the current fov (default true)
 * @param {number=} fov - specify a fov if we do not want to use the current one (useful for simulation)
 * @return {CameraBoundaries} Returns the min and max pitch computed from the camera configuration and the view limits.
 * @private
 */
FORGE.Camera.prototype._getPitchBoundaries = function(relative, fov)
{
    var min = this._pitchMin;
    var max = this._pitchMax;

    fov = fov || this._fov;

    if (relative !== false && min !== max)
    {
        var halfFov = 0.5 * fov;
        min += halfFov;
        max -= halfFov;
    }

    var view = this._viewer.renderer.view.current;

    if (view !== null)
    {
        min = Math.max(view.pitchMin, min);
        max = Math.min(view.pitchMax, max);
    }

    return { min: min, max: max };
};

/**
 * Internal setter for roll, take a value and a unit. Default unit is radians.
 * @method FORGE.Camera#_setRoll
 * @param {?number=} value - The value you want to set for roll.
 * @param {string=} [unit="radians"] - The unit you use to set the roll value.
 * @return {boolean} Returns true if the value has changed.
 * @private
 */
FORGE.Camera.prototype._setRoll = function(value, unit)
{
    if (typeof value !== "number" || isNaN(value) === true)
    {
        return false;
    }

    // If unit is not well defined, default will be radians
    unit = (unit === FORGE.Math.DEGREES || unit === FORGE.Math.RADIANS) ? unit : FORGE.Math.RADIANS;

    // Convert value in radians for clamp if unit is in degrees.
    value = (unit === FORGE.Math.DEGREES) ? FORGE.Math.degToRad(value) : value;

    // Wrap the value between -PI and +PI
    value = FORGE.Math.wrap(value, -Math.PI, Math.PI);

    var boundaries = this._getRollBoundaries();

    var roll = FORGE.Math.clamp(value, boundaries.min, boundaries.max);

    var changed = this._roll !== roll;

    this._changelog.roll = changed;

    this._roll = roll;

    return changed;
};

/**
 * Compute the roll boundaries with yaw min and yaw max.
 * @method FORGE.Camera#_getRollBoundaries
 * @return {CameraBoundaries} Returns the min and max roll computed from the camera configuration and the view limits.
 * @private
 */
FORGE.Camera.prototype._getRollBoundaries = function()
{
    var min = this._rollMin;
    var max = this._rollMax;
    var view = this._viewer.renderer.view.current;

    if (view !== null)
    {
        min = Math.max(view.rollMin, min);
        max = Math.min(view.rollMax, max);
    }

    return { min: min, max: max };
};

/**
 * Internal setter for fov (field of view), take a value and a unit. Default unit is radians.
 * @method FORGE.Camera#_setFov
 * @param {?number=} value - The value you want to set for fov.
 * @param {string=} [unit="radians"] - The unit you use to set the fov value.
 * @return {boolean} Returns true if the value has changed.
 * @private
 */
FORGE.Camera.prototype._setFov = function(value, unit)
{
    if (typeof value !== "number" || isNaN(value) === true)
    {
        return false;
    }

    // If unit is not well defined, default will be radians
    unit = (unit === FORGE.Math.DEGREES || unit === FORGE.Math.RADIANS) ? unit : FORGE.Math.RADIANS;

    // Convert value in radians for clamp if unit is in degrees.
    value = (unit === FORGE.Math.DEGREES) ? FORGE.Math.degToRad(value) : value;

    var boundaries = this._getFovBoundaries();

    var fov = FORGE.Math.clamp(value, boundaries.min, boundaries.max);

    var changed = this._fov !== fov;

    this._changelog.fov = changed;

    this._fov = fov;

    if (changed)
    {
        this._setYaw(this._yaw);
        this._setPitch(this._pitch);
    }

    return changed;
};

/**
 * Compute the fov boundaries with yaw min and yaw max.
 * @method FORGE.Camera#_getFovBoundaries
 * @return {CameraBoundaries} Returns the min and max fov computed from the camera configuration and the view limits.
 * @private
 */
FORGE.Camera.prototype._getFovBoundaries = function()
{
    var min = this._fovMin;
    var max = this._fovMax;
    var view = this._viewer.renderer.view.current;

    // if JSON specifies a fov min (not default 0 value), use it
    // useful for multiresolution where fov limit will be computed depending
    // on max level of resolution available and stored in JSON
    if (this._viewer.renderer.backgroundRenderer !== null && "fovMin" in this._viewer.renderer.backgroundRenderer)
    {
        min = Math.max(this._viewer.renderer.backgroundRenderer.fovMin, min);
    }
    else if (min === 0)
    {
        if (view !== null)
        {
            min = Math.max(view.fovMin, min);
            max = Math.min(view.fovMax, max);
        }
    }

    if (view !== null && view.type !== FORGE.ViewType.FLAT)
    {
        // if there are limits, we may need to limit the maximum fov
        var pitchBoundaries = this._getPitchBoundaries(false);
        var pitchRange = pitchBoundaries.max - pitchBoundaries.min;

        if (pitchRange > 0)
        {
            max = Math.min(pitchRange, max);
        }

        var yawBoundaries = this._getYawBoundaries(false);
        var yawRange = yawBoundaries.max - yawBoundaries.min;
        yawRange /= this._viewer.renderer.displayResolution.ratio;

        if (yawRange > 0)
        {
            max = Math.min(yawRange, max);
        }

        // get the tiniest
        if (max < min)
        {
            min = max;
        }
    }

    return { min: min, max: max };
};

/**
 * Set all camera angles in one call (yaw, pitch, roll, fov)
 * @method FORGE.Camera#_setAll
 * @param {?number=} yaw - The yaw value you want to set.
 * @param {?number=} pitch - The pitch value you want to set.
 * @param {?number=} roll - The roll value you want to set.
 * @param {?number=} fov - The fov value you want to set.
 * @param {string=} unit - The unit you use for all the previous arguments (FORGE.Math.DEGREES or FORGE.Math.RADIANS)
 * @return {boolean} Returns true if any values has changed.
 * @private
 */
FORGE.Camera.prototype._setAll = function(yaw, pitch, roll, fov, unit)
{
    var fovChanged = this._setFov(fov, unit);
    var yawChanged = this._setYaw(yaw, unit);
    var pitchChanged = this._setPitch(pitch, unit);
    var rollChanged = this._setRoll(roll, unit);

    return (yawChanged === true || pitchChanged === true || rollChanged === true || fovChanged === true);
};

/**
 * Update internals after a remote component has changed something
 * @method FORGE.Camera#_updateInternals
 * @private
 */
FORGE.Camera.prototype._updateInternals = function()
{
    // Force camera to update its values to bound it in new boundaries after view change
    var changed = this._setAll(this._yaw, this._pitch, this._roll, this._fov);

    if (changed === true)
    {
        this._updateFromEuler();
    }
};

/**
 * Load a camera configuration.
 * @method FORGE.Camera#load
 * @param {CameraConfig} config - The camera configuration to load.
 */
FORGE.Camera.prototype.load = function(config)
{
    this._config = /** @type {CameraConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.Camera.DEFAULT_CONFIG, config));

    this._parseConfig(this._config);

    this._initialized = true;
};

/**
 * Set the Camera to look at a specified point into the yaw/pitch/roll space.
 * @method FORGE.Camera#lookAt
 * @param {?number=} yaw Euler yaw angle (deg)
 * @param {?number=} pitch Euler pitch angle (deg)
 * @param {?number=} roll Euler roll angle (deg)
 * @param {?number=} fov Field of view (deg)
 * @param {number=} durationMS - Rotation animation duration ms (undefined or zero means immediat effect)
 * @param {boolean=} [cancelRoll=false] - If set to true, roll will be cancelled (always at 0).<br> If false an auto roll movement will be done by the camera for a more natural movement effect.
 * @param {string=} easing - Easing method name (default to {@link FORGE.EasingType.LINEAR}).
 */
FORGE.Camera.prototype.lookAt = function(yaw, pitch, roll, fov, durationMS, cancelRoll, easing)
{
    if (typeof durationMS !== "number" || durationMS === 0)
    {
        var changed = this._setAll(yaw, pitch, roll, fov, FORGE.Math.DEGREES);

        if (changed === true)
        {
            this._updateFromEuler();
        }
    }
    else
    {
        if (fov !== null && typeof fov !== "undefined")
        {
            var fovBoundaries = this._getFovBoundaries();

            fov = FORGE.Math.clamp(fov, FORGE.Math.radToDeg(fovBoundaries.min), FORGE.Math.radToDeg(fovBoundaries.max));

            if (yaw !== null && typeof yaw !== "undefined")
            {
                var yawBoundaries = this._getYawBoundaries(true, FORGE.Math.degToRad(fov));
                yaw = FORGE.Math.clamp(yaw, FORGE.Math.radToDeg(yawBoundaries.min), FORGE.Math.radToDeg(yawBoundaries.max));
            }

            if (pitch !== null && typeof pitch !== "undefined")
            {
                var pitchBoundaries = this._getPitchBoundaries(true, FORGE.Math.degToRad(fov));
                pitch = FORGE.Math.clamp(pitch, FORGE.Math.radToDeg(pitchBoundaries.min), FORGE.Math.radToDeg(pitchBoundaries.max));
            }
        }

        // before creating a track, set the goto point in future boundaries
        var track = new FORGE.DirectorTrack(
        {
            easing:
            {
                default: easing || "LINEAR",
                start: 0
            },

            cancelRoll: Boolean(cancelRoll),

            keyframes:
            [
                {
                    time: durationMS,
                    data:
                    {
                        yaw: yaw,
                        pitch: pitch,
                        roll: roll,
                        fov: fov
                    }
                }
            ]
        });

        this.animation.play(track.uid);
    }
};

/**
 * Update routine called by render manager before rendering a frame.
 * All internals should be up to date.
 * @method FORGE.Camera#update
 */
FORGE.Camera.prototype.update = function()
{
    if (this._viewer.renderer.display.presentingVR === true)
    {
        this._gaze.update();
        this._updateVRCameras();
        this._cloneVRCamerasChildren();
    }

    this._updateMainCamera();
    this._updateFlatCamera();

    this._resetChangelog();
};

/**
 * Destroy sequence.
 * @method FORGE.Camera#destroy
 */
FORGE.Camera.prototype.destroy = function()
{
    this._modelView = null;
    this._modelViewInverse = null;
    this._quaternion = null;
    this._main = null;
    this._flat = null;

    this._gaze.destroy();
    this._gaze = null;

    this._viewer.renderer.view.onChange.remove(this._updateInternals, this);
    this._viewer.renderer.onBackgroundReady.remove(this._updateInternals, this);

    if (this._onChange !== null)
    {
        this._onChange.destroy();
        this._onChange = null;
    }

    if (this._onOrientationChange !== null)
    {
        this._onOrientationChange.destroy();
        this._onOrientationChange = null;
    }

    if (this._onFovChange !== null)
    {
        this._onFovChange.destroy();
        this._onFovChange = null;
    }

    if (this._cameraAnimation !== null)
    {
        this._cameraAnimation.destroy();
        this._cameraAnimation = null;
    }

    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get and set the camera configuration (default min & max for all angles yaw, pitch, roll and fov).
 * @name FORGE.Camera#config
 * @type {CameraConfig}
 */
Object.defineProperty(FORGE.Camera.prototype, "config",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._config;
    },

    /** @this {FORGE.Camera} */
    set: function(config)
    {
        this.load(config);
    }
});

/**
 * Get the keep flag
 * @name FORGE.Camera#keep
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "keep",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._keep;
    }
});

/**
 * Get and set the yaw value in degree.
 * @name FORGE.Camera#yaw
 * @type {number}
 */
Object.defineProperty(FORGE.Camera.prototype, "yaw",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return FORGE.Math.radToDeg(this._yaw);
    },

    /** @this {FORGE.Camera} */
    set: function(value)
    {
        var yawChanged = this._setYaw(value, FORGE.Math.DEGREES);

        if (yawChanged === true)
        {
            this._updateFromEuler();
        }
    }
});

/**
 * Get the yaw min value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#yawMin
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "yawMin",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getYawBoundaries();
        return FORGE.Math.radToDeg(boundaries.min);
    }
});

/**
 * Get the yaw max value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#yawMax
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "yawMax",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getYawBoundaries();
        return FORGE.Math.radToDeg(boundaries.max);
    }
});

/**
 * Get and set the pitch value in degree.
 * @name FORGE.Camera#pitch
 * @type {number}
 */
Object.defineProperty(FORGE.Camera.prototype, "pitch",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return FORGE.Math.radToDeg(this._pitch);
    },

    /** @this {FORGE.Camera} */
    set: function(value)
    {
        var pitchChanged = this._setPitch(value, FORGE.Math.DEGREES);

        if (pitchChanged)
        {
            this._updateFromEuler();
        }
    }
});

/**
 * Get the pitch min value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#pitchMin
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "pitchMin",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getPitchBoundaries();
        return FORGE.Math.radToDeg(boundaries.min);
    }
});

/**
 * Get the pitch max value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#pitchMax
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "pitchMax",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getPitchBoundaries();
        return FORGE.Math.radToDeg(boundaries.max);
    }
});

/**
 * Get and set the roll value in degree.
 * @name FORGE.Camera#roll
 * @type {number}
 */
Object.defineProperty(FORGE.Camera.prototype, "roll",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return FORGE.Math.radToDeg(this._roll);
    },

    /** @this {FORGE.Camera} */
    set: function(value)
    {
        var rollChanged = this._setRoll(value, FORGE.Math.DEGREES);

        if (rollChanged === true)
        {
            this._updateFromEuler();
        }
    }
});

/**
 * Get the roll min value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#rollMin
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "rollMin",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getRollBoundaries();
        return FORGE.Math.radToDeg(boundaries.min);
    }
});

/**
 * Get the roll max value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#rollMax
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "rollMax",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getRollBoundaries();
        return FORGE.Math.radToDeg(boundaries.max);
    }
});

/**
 * Get and set the fov value in degree.
 * @name FORGE.Camera#fov
 * @type {number}
 */
Object.defineProperty(FORGE.Camera.prototype, "fov",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return FORGE.Math.radToDeg(this._fov);
    },

    /** @this {FORGE.Camera} */
    set: function(value)
    {
        var fovChanged = this._setFov(value, FORGE.Math.DEGREES);

        if (fovChanged === true)
        {
            this._updateFromEuler();
        }
    }
});

/**
 * Get the fov min value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#fovMin
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "fovMin",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getFovBoundaries();
        return FORGE.Math.radToDeg(boundaries.min);
    }
});

/**
 * Get the fov max value.
 * Return the most restrictive value between the camera value and the view value.
 * @name FORGE.Camera#fovMax
 * @type {number}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "fovMax",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        var boundaries = this._getFovBoundaries();
        return FORGE.Math.radToDeg(boundaries.max);
    }
});

/**
 * Get/set quaternion rotation object of the camera.
 * Setter will update internal quaternion object
 * @name FORGE.Camera#quaternion
 * @readonly
 * @type {THREE.Quaternion}
 */
Object.defineProperty(FORGE.Camera.prototype, "quaternion",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._quaternion;
    },
    /** @this {FORGE.Camera} */
    set: function(value)
    {
        this._quaternion = value;
        this._updateFromQuaternion();
        this._updateComplete();
    }
});

/**
 * Get camera animation manager.
 * @name FORGE.Camera#animation
 * @readonly
 * @type {FORGE.CameraAnimation}
 */
Object.defineProperty(FORGE.Camera.prototype, "animation",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._cameraAnimation === null)
        {
            this._cameraAnimation = new FORGE.CameraAnimation(this._viewer, this);
        }

        return this._cameraAnimation;
    }
});

/**
 * Get/Set parallax setting.
 * @name FORGE.Camera#parallax
 * @type number
 */
Object.defineProperty(FORGE.Camera.prototype, "parallax",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._parallax;
    },
    /** @this {FORGE.Camera} */
    set: function(value)
    {
        this._parallax = FORGE.Math.clamp(value, 0, 1);
        this._radius = this._parallax * FORGE.Camera.RADIUS;
        this._updateComplete();
    }
});

/**
 * Get the modelView of the camera.
 * @name FORGE.Camera#modelView
 * @type {THREE.Matrix4}
 */
Object.defineProperty(FORGE.Camera.prototype, "modelView",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._modelView;
    },
    /** @this {FORGE.Camera} */
    set: function(value)
    {
        this._modelView = value;
        this._updateFromMatrix();
        this._updateComplete();
    }
});

/**
 * Get the modelViewInverse of the camera.
 * @name FORGE.Camera#modelViewInverse
 * @readonly
 * @type {THREE.Matrix4}
 */
Object.defineProperty(FORGE.Camera.prototype, "modelViewInverse",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._modelViewInverse;
    }
});

/**
 * Get the main THREE.PerspectiveCamera of the camera.
 * @name FORGE.Camera#main
 * @readonly
 * @type {THREE.PerspectiveCamera}
 */
Object.defineProperty(FORGE.Camera.prototype, "main",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._main === null)
        {
            this._createMainCamera();
        }

        return this._main;
    }
});

/**
 * Get the flat THREE.OrthographicCamera of the camera.
 * @name FORGE.Camera#flat
 * @readonly
 * @type {THREE.OrthographicCamera}
 */
Object.defineProperty(FORGE.Camera.prototype, "flat",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._flat === null)
        {
            this._createFlatCamera();
        }

        return this._flat;
    }
});

/**
 * Get the THREE.PerspectiveCamera radius.
 * @name FORGE.Camera#perspectiveCameraRadius
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Camera.prototype, "perspectiveCameraRadius",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._radius;
    }
});

/**
 * Get the left camera.
 * @name FORGE.Camera#left
 * @type {THREE.PerspectiveCamera}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "left",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._left;
    }
});

/**
 * Get the right camera.
 * @name FORGE.Camera#right
 * @type {THREE.PerspectiveCamera}
 * @readonly
 */
Object.defineProperty(FORGE.Camera.prototype, "right",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._right;
    }
});

/**
 * Get the camera gaze.
 * @name FORGE.Camera#gaze
 * @readonly
 * @type {FORGE.CameraGaze}
 */
Object.defineProperty(FORGE.Camera.prototype, "gaze",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        return this._gaze;
    }
});

/**
 * Get the "onChange" {@link FORGE.EventDispatcher} of the camera.
 * @name FORGE.Camera#onChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Camera.prototype, "onChange",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._onChange === null)
        {
            this._onChange = new FORGE.EventDispatcher(this);
        }

        return this._onChange;
    }
});

/**
 * Get the "onOrientationChange" {@link FORGE.EventDispatcher} of the camera.
 * @name FORGE.Camera#onOrientationChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Camera.prototype, "onOrientationChange",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._onOrientationChange === null)
        {
            this._onOrientationChange = new FORGE.EventDispatcher(this);
        }

        return this._onOrientationChange;
    }
});

/**
 * Get the "onFovChange" {@link FORGE.EventDispatcher} of the camera.
 * @name FORGE.Camera#onFovChange
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Camera.prototype, "onFovChange",
{
    /** @this {FORGE.Camera} */
    get: function()
    {
        if (this._onFovChange === null)
        {
            this._onFovChange = new FORGE.EventDispatcher(this);
        }

        return this._onFovChange;
    }
});

/**
 * A FORGE.CameraAnimation is used by the camera to animate along keyframes.
 *
 * @constructor FORGE.CameraAnimation
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {FORGE.Camera} camera - {@link FORGE.Camera} reference.
 * @extends {FORGE.MetaAnimation}
 *
 */
FORGE.CameraAnimation = function(viewer, camera)
{
    FORGE.MetaAnimation.call(this, viewer, camera, "CameraAnimation");

    this._boot();
};

FORGE.CameraAnimation.prototype = Object.create(FORGE.MetaAnimation.prototype);
FORGE.CameraAnimation.prototype.constructor = FORGE.CameraAnimation;

/**
 * Init sequence
 * @method FORGE.CameraAnimation#_boot
 * @private
 */
FORGE.CameraAnimation.prototype._boot = function()
{
    this._register();

    // Add the cancel roll effect
    var cameraAt = function()
    {
        if (this._instruction.cancelRoll === true)
        {
            this._target.quaternion = FORGE.Quaternion.cancelRoll(this._target.quaternion);
        }
    };

    this._instructions =
    [
        {
            prop: "quaternion",
            smooth: true,
            fun: cameraAt,
            cancelRoll: true
        },

        {
            prop: "fov",
            smooth: false
        }
    ];
};

/**
 * Load a camera animation keyframes configuration.<br>
 * Convert keyframe into timeline keyframes.
 *
 * @method FORGE.CameraAnimation#_prepareKeyframes
 * @param {Array<FORGE.Keyframe>} tracks - the array containing the keyframes
 * @param {number} offset - the time to take between the current position of the camera to the first keyframe.
 * @param {Function} easing - The easing function for this animation
 * @private
 */
FORGE.CameraAnimation.prototype._prepareKeyframes = function(tracks, offset, easing)
{
    offset = offset || 0;

    var kf, data, yaw, pitch, roll, fov, quat;

    // Add a keyframe for each one in the configuration
    for (var i = 0, ii = tracks.length; i < ii; i++)
    {
        data = tracks[i].data;

        // Quaternion
        yaw = (typeof data.yaw !== "undefined" && data.yaw !== null) ? data.yaw : this._computeIntermediateValue(tracks[i].time + offset, tracks, "yaw", easing);
        pitch = (typeof data.pitch !== "undefined" && data.pitch !== null) ? data.pitch : this._computeIntermediateValue(tracks[i].time + offset, tracks, "pitch", easing);
        roll = (typeof data.roll !== "undefined" && data.roll !== null) ? data.roll : this._computeIntermediateValue(tracks[i].time + offset, tracks, "roll", easing);

        yaw = FORGE.Math.degToRad(yaw);
        pitch = FORGE.Math.degToRad(pitch);
        roll = FORGE.Math.degToRad(roll);

        if (typeof yaw !== "undefined" && yaw !== null && typeof pitch !== "undefined" && pitch !== null && typeof roll !== "undefined" && roll !== null)
        {
            quat = FORGE.Quaternion.fromEuler(yaw, pitch, roll);

            kf = new FORGE.Keyframe(tracks[i].time + offset, { quaternion: quat });
            this._animations[0].timeline.addKeyframe(kf);
        }

        // FOV
        if (typeof data.fov !== "undefined" && data.fov !== null)
        {
            fov = data.fov;
            kf = new FORGE.Keyframe(tracks[i].time + offset, { fov: fov });
            this._animations[1].timeline.addKeyframe(kf);
        }
    }

    // If the first keyframe is not at time 0 or there is an offset, add a
    // virtual keyframe
    if (tracks[0].time > 0 || offset > 0)
    {
        kf = new FORGE.Keyframe(0, { quaternion: FORGE.Utils.clone(this._target.quaternion) });
        this._animations[0].timeline.addKeyframe(kf);

        kf = new FORGE.Keyframe(0, { fov: this._target.fov });
        this._animations[1].timeline.addKeyframe(kf);
    }
};

/**
 * Start to move along the keyframes.
 *
 * @method FORGE.CameraAnimation#play
 * @param {string|FORGE.DirectorTrack} track - Track to play
 * @param {number=} time - Time to start the animation at
 */
FORGE.CameraAnimation.prototype.play = function(track, time)
{
    if (typeof track === "string")
    {
        track = FORGE.UID.get(track);
    }

    this._emptyAnimations();

    // First instruction - quaternion
    var quatAnimation = new FORGE.Animation(this._viewer, this._target);
    quatAnimation.tween.easing = track.easing;
    quatAnimation.instruction = this._instructions[0];
    quatAnimation.instruction.smooth = track.smooth;
    quatAnimation.smooth = track.smooth;
    quatAnimation.instruction.cancelRoll = track.cancelRoll;
    quatAnimation.onComplete.add(this._onTrackPartialCompleteHandler, this);
    this._animations.push(quatAnimation);

    // Second instruction - fov
    var fovAnimation = new FORGE.Animation(this._viewer, this._target);
    fovAnimation.tween.easing = track.easing;
    fovAnimation.instruction = this._instructions[1];
    fovAnimation.instruction.smooth = track.smooth;
    fovAnimation.onComplete.add(this._onTrackPartialCompleteHandler, this);
    this._animations.push(fovAnimation);

    // Prepare the keyframes
    this._prepareKeyframes(track.keyframes, track.offset, quatAnimation.tween.easing);

    // Play !
    quatAnimation.play(time);
    fovAnimation.play(time);
};

/**
 * Destroy sequence.
 * @method FORGE.CameraAnimation#destroy
 */
FORGE.CameraAnimation.prototype.destroy = function()
{
    FORGE.MetaAnimation.prototype.destroy.call(this);
};

/**
 * Gaze Cursor base class. Draw a geometry on the camera space for gaze selection.
 *
 * @constructor FORGE.CameraGaze
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.CameraGaze = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.CameraGaze#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The gaze configuration object
     * @name  FORGE.CameraGaze#_config
     * @type {CameraGazeConfig}
     * @private
     */
    this._config = config;

    /**
     * THREE object
     * @name FORGE.CameraGaze#_object
     * @type {THREE.Object3D}
     * @private
     */
    this._object = null;

    /**
     * Flag to know if the gaze cursor is hovering an object
     * @name FORGE.CameraGaze#_hovering
     * @type {boolean}
     * @private
     */
    this._hovering = false;

    /**
     * Timer for delayed click by gaze
     * @name FORGE.CameraGaze#_timer
     * @type {FORGE.Timer}
     * @private
     */
    this._timer = null;

    /**
     * Timer event used by a current countdown forn click by gaze.
     * @name FORGE.CameraGaze#_timerEvent
     * @type {FORGE.TimerEvent}
     * @private
     */
    this._timerEvent = null;

    /**
     * The percentage of progress to have a valid gaze interaction.
     * @name  FORGE.CameraGaze#_progress
     * @type {number}
     * @private
     */
    this._progress = 0;

    FORGE.BaseObject.call(this, "CameraGaze");

    this._boot();
};

FORGE.CameraGaze.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.CameraGaze.prototype.constructor = FORGE.CameraGaze;

/**
 * Boot sequence.
 * @method FORGE.CameraGaze#_boot
 * @private
 */
FORGE.CameraGaze.prototype._boot = function()
{
    this._timer = this._viewer.clock.create(false);

    this._object = new THREE.Object3D();
    this._object.name = "CameraGaze";
    this._object.position.z = -2;

    this._createCursor();
};

/**
 * Create ring geometry.
 * @method FORGE.CameraGaze#_createRingGeometry
 * @param {number=} [innerRadius=0.02] Inner radius of the ring.
 * @param {number=} [outerRadius=0.04] OuterRadius of the ring.
 * @param {number=} [thetaSegments=32] Number of theta segments on the ring.
 * @param {number=} [phiSegments=1] Number of phi segments on the ring.
 * @param {number=} [thetaStart=0] Thetha start.
 * @param {number=} [thetaLength=6.28] Thetha length, default is two PI.
 * @return {THREE.BufferGeometry} The ring geometry.
 * @private
 */
FORGE.CameraGaze.prototype._createRingGeometry = function(innerRadius, outerRadius, thetaSegments, phiSegments, thetaStart, thetaLength)
{
    innerRadius = innerRadius || 0.02;
    outerRadius = outerRadius || 0.04;
    thetaSegments = thetaSegments || 32;
    phiSegments = phiSegments || 1;
    thetaStart = thetaStart || 0;
    thetaLength = thetaLength || Math.PI * 2;

    var geometry = new THREE.RingBufferGeometry(innerRadius, outerRadius, thetaSegments, phiSegments, thetaStart, thetaLength);

    return geometry;
};

/**
 * Create the base cursor based on its configuration
 * @method FORGE.CameraGaze#_createCursor
 * @private
 */
FORGE.CameraGaze.prototype._createCursor = function()
{
    var material = new THREE.MeshBasicMaterial(
    {
        color: this._config.cursor.color,
        opacity: this._config.cursor.opacity,
        transparent: true
    });

    var ring = new THREE.Mesh(this._createRingGeometry(this._config.cursor.innerRadius, this._config.cursor.outerRadius), material);
    ring.name = "cursor";

    this._object.add(ring);
};

/**
 * Update the second ring according to the progress percentage of the timer event.
 * @method FORGE.CameraGaze#_updateProgressRing
 * @param  {number} progress - The progress percentage of the timer.
 * @private
 */
FORGE.CameraGaze.prototype._updateProgressRing = function(progress)
{
    this._progress = progress;

    this._destroyRing("progress");

    this._createProgress();
};

/**
 * Create the progress cursor based on its configuration and the percentage of progress
 * @method FORGE.CameraGaze#_createProgress
 * @private
 */
FORGE.CameraGaze.prototype._createProgress = function()
{
    var material = new THREE.MeshBasicMaterial(
    {
        color: this._config.progress.color,
        opacity: this._config.progress.opacity,
        transparent: true,
        side: THREE.DoubleSide
    });

    var thetaLength = (this._progress / 100) * FORGE.Math.TWOPI;

    var ring = new THREE.Mesh(this._createRingGeometry(this._config.progress.innerRadius, this._config.progress.outerRadius, 32, 1, (Math.PI / 2), thetaLength), material);
    ring.name = "progress";
    ring.rotateY(Math.PI);

    this._object.add(ring);
};

/**
 * Destroy a ring from its name
 * @method FORGE.CameraGaze#_destroyRing
 * @param {string} name - The name of the ring to destroy
 * @private
 */
FORGE.CameraGaze.prototype._destroyRing = function(name)
{
    var ring = null;

    for (var i = 0, ii = this._object.children.length; i < ii; i++)
    {
        if (this._object.children[i].name === name)
        {
            ring = this._object.children[i];
        }
    }

    if (ring !== null)
    {
        this._object.remove(ring);

        ring.geometry.dispose();
        ring.material.dispose();
    }
};

/**
 * Timer complete handler, triggers the click action!
 * @method FORGE.CameraGaze#_timerCompleteHandler
 * @private
 */
FORGE.CameraGaze.prototype._timerCompleteHandler = function()
{
    this.stop();

    this._viewer.renderer.pickingManager.click();
};

/**
 * Loads a configuration for gaze cursors.
 * @method FORGE.CameraGaze#load
 * @param {CameraGazeConfig} config -  The configuration to load.
 */
FORGE.CameraGaze.prototype.load = function(config)
{
    this._config = config;

    this._destroyRing("progress");
    this._destroyRing("cursor");

    this._createCursor();

    if (this._hovering === true && this._progress !== 0)
    {
        this._createProgress();
    }
};

/**
 * Starts the gaze timer, this is called from the raycaster.
 * @method FORGE.CameraGaze#start
 */
FORGE.CameraGaze.prototype.start = function()
{
    this.log("startGazeAnimation");

    this._hovering = true;

    this._updateProgressRing(0);

    if (this._timer !== null)
    {
        var delay = 2000 || this._config.delay;

        this._timer.stop();
        this._timerEvent = this._timer.add( /** @type {number} */ (delay), this._timerCompleteHandler, this);
        this._timer.start();
    }
};

/**
 * Stops the gaze timer, this is called from the raycaster.
 * @method FORGE.CameraGaze#stop
 */
FORGE.CameraGaze.prototype.stop = function()
{
    this.log("stopGazeAnimation");

    this._hovering = false;

    this._timer.stop();
    this._timerEvent = null;

    this._progress = 0;

    var ring = this._object.children[1];

    if (typeof ring !== "undefined" && ring !== null)
    {
        this._object.remove(ring);

        ring.geometry.dispose();
        ring.material.dispose();
    }
};

/**
 * Update loop of the gaze cursor, it updates the graphics according to the progress percentage of the timer.
 * @method FORGE.CameraGaze#update
 */
FORGE.CameraGaze.prototype.update = function()
{
    if (this._hovering === true && this._timerEvent !== null)
    {
        var percent = (this._timerEvent.delay - this._timer.duration) * 100 / this._timerEvent.delay;
        this.log("timer update : " + percent + "%");

        this._updateProgressRing(percent);
    }
};

/**
 * Destroy sequence.
 * @method FORGE.CameraGaze#destroy
 */
FORGE.CameraGaze.prototype.destroy = function()
{
    this.stop();

    if (this._object !== null)
    {
        if (typeof this._object.geometry !== "undefined" && this._object.geometry !== null)
        {
            this._object.geometry.dispose();
        }

        if (typeof this._object.material !== "undefined" && this._object.material !== null)
        {
            this._object.material.dispose();
        }

        this._object = null;
    }

    this._scene = null;
};

/**
 * Visibility flag.
 * @name FORGE.CameraGaze#visible
 * @type {boolean}
 */
Object.defineProperty(FORGE.CameraGaze.prototype, "visible",
{
    /** @this {FORGE.CameraGaze} */
    get: function()
    {
        return this._object.visible;
    },

    /** @this {FORGE.CameraGaze} */
    set: function(value)
    {
        this._object.visible = value;
    }
});

/**
 * 3D object
 * @name FORGE.CameraGaze#object
 * @readonly
 * @type {THREE.Mesh}
 */
Object.defineProperty(FORGE.CameraGaze.prototype, "object",
{
    /** @this {FORGE.CameraGaze} */
    get: function()
    {
        return this._object;
    }
});

/**
 * @namespace {Object} FORGE.ControllerType
 */
FORGE.ControllerType = {};

/**
 * @name FORGE.ControllerType.BASE
 * @type {string}
 * @const
 */
FORGE.ControllerType.BASE = "base";

/**
 * @name FORGE.ControllerType.POINTER
 * @type {string}
 * @const
 */
FORGE.ControllerType.POINTER = "pointer";

/**
 * @name FORGE.ControllerType.KEYBOARD
 * @type {string}
 * @const
 */
FORGE.ControllerType.KEYBOARD = "keyboard";

/**
 * @name FORGE.ControllerType.GYROSCOPE
 * @type {string}
 * @const
 */
FORGE.ControllerType.GYROSCOPE = "gyroscope";

/**
 * @name FORGE.ControllerType.GAMEPAD
 * @type {string}
 * @const
 */
FORGE.ControllerType.GAMEPAD = "gamepad";


/**
 * @constructor FORGE.ControllerManager
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.ControllerManager = function(viewer)
{
    /**
     * Viewer reference.
     * @name FORGE.ControllerManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Array of controllers.
     * @name FORGE.ControllerManager#_controllers
     * @type {Array<FORGE.ControllerBase>}
     * @private
     */
    this._controllers = [];

    /**
     * Enabled flag.
     * @name  FORGE.ControllerManager#_enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Event dispatcher for control start.
     * @name FORGE.ControllerManager#_onControlStart
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onControlStart = null;

    /**
     * Event dispatcher for control end.
     * @name FORGE.ControllerManager#_onControlEnd
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onControlEnd = null;

    FORGE.BaseObject.call(this, "ControllerManager");
};

FORGE.ControllerManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ControllerManager.prototype.constructor = FORGE.ControllerManager;

/**
 * Controllers default configuration
 * @name  FORGE.ControllerManager.DEFAULT_CONFIG
 * @const
 */
FORGE.ControllerManager.DEFAULT_CONFIG =
{
    enabled: true,

    instances:
    [
        {
            type: FORGE.ControllerType.POINTER,
            enabled: true
        },

        {
            type: FORGE.ControllerType.KEYBOARD,
            enabled: true
        },

        {
            type: FORGE.ControllerType.GYROSCOPE,
            enabled: true
        },

        {
            type: FORGE.ControllerType.GAMEPAD,
            enabled: true
        }
    ]
};

/**
 * Parse configuration object
 * @method FORGE.ControllerManager#_parseConfig
 * @param {ControllersConfig} config - The config you want to add.
 * @private
 */
FORGE.ControllerManager.prototype._parseConfig = function(config)
{
    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;

    if(typeof config.instances !== "undefined")
    {
        var controllerConfig;
        var controller;

        for(var i = 0, ii = config.instances.length; i < ii; i++)
        {
            controllerConfig = config.instances[i];

            switch(controllerConfig.type)
            {
                case FORGE.ControllerType.POINTER:
                    controller = new FORGE.ControllerPointer(this._viewer, controllerConfig);
                    break;

                case FORGE.ControllerType.KEYBOARD:
                    controller = new FORGE.ControllerKeyboard(this._viewer, controllerConfig);
                    break;

                case FORGE.ControllerType.GYROSCOPE:
                    controller = new FORGE.ControllerGyroscope(this._viewer, controllerConfig);
                    break;

                case FORGE.ControllerType.GAMEPAD:
                    controller = new FORGE.ControllerGamepad(this._viewer, controllerConfig);
                    break;

                default:
                    controller = null;
                    break;
            }

            if(controller !== null)
            {
                this._controllers.push(controller);
            }
        }
    }
};

/**
 * Controllers will call this method to notify the manager that a control start.
 * @method FORGE.ControllerManager#notifyControlStart
 * @param  {FORGE.ControllerBase} controller - The controller that notifies the manager.
 */
FORGE.ControllerManager.prototype.notifyControlStart = function(controller)
{
    if(this._onControlStart !== null)
    {
        this._onControlStart.dispatch({"controller": controller});
    }
};

/**
 * Controllers will call this method to notify the manager that a controller ends.
 * @method FORGE.ControllerManager#notifyControlEnd
 * @param  {FORGE.ControllerBase} controller - The controller that notifies the manager.
 */
FORGE.ControllerManager.prototype.notifyControlEnd = function(controller)
{
    if(this._onControlEnd !== null)
    {
        this._onControlEnd.dispatch({"controller": controller});
    }
};

/**
 * Controllers update routine
 * @method FORGE.ControllerManager#update
 */
FORGE.ControllerManager.prototype.update = function()
{
    if (this._controllers === null)
    {
        return;
    }

    for(var i = 0, ii = this._controllers.length; i < ii; i++)
    {
        if (typeof this._controllers[i].update === "function")
        {
            this._controllers[i].update();
        }
    }
};

/**
 * Add a config to the manager.
 * @method FORGE.ControllerManager#addConfig
 * @param {ControllersConfig} config - The config you want to add.
 */
FORGE.ControllerManager.prototype.addConfig = function(config)
{
    config = (typeof config !== "undefined") ? config : FORGE.ControllerManager.DEFAULT_CONFIG;

    this._parseConfig(config);
};

/**
 * Get a controller by its type.
 * @method FORGE.ControllerManager#getByType
 * @param {string} type - The type of the controller you want to get.
 * @return {?FORGE.ControllerBase} return the desired type controller, null if not found
 */
FORGE.ControllerManager.prototype.getByType = function(type)
{
    var controller;

    for(var i = 0, ii = this._controllers.length; i < ii; i++)
    {
        controller = this._controllers[i];

        if(controller.type === type)
        {
            return controller;
        }
    }

    return null;
};

/**
 * Destroy method.
 */
FORGE.ControllerManager.prototype.destroy = function()
{
    var ctrl;

    while(this._controllers.length)
    {
        ctrl = this._controllers.pop();
        if (ctrl !== null && typeof ctrl !== "undefined")
        {
            ctrl.destroy();
        }
    }

    this._controllers = null;

    if(this._onControlStart !== null)
    {
        this._onControlStart.destroy();
        this._onControlStart = null;
    }

    if(this._onControlEnd !== null)
    {
        this._onControlEnd.destroy();
        this._onControlEnd = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get all controllers.
 * @name FORGE.ControllerManager#all
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.ControllerManager.prototype, "all",
{
    /** @this {FORGE.ControllerManager} */
    get: function()
    {
        return this._controllers;
    }
});

/**
 * Get or set the global enabled flag.
 * @name FORGE.ControllerManager#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.ControllerManager.prototype, "enabled",
{
    /** @this {FORGE.ControllerManager} */
    get: function()
    {
        return this._enabled;
    },

    /** @this {FORGE.ControllerManager} */
    set: function(value)
    {
        this._enabled = Boolean(value);
    }
});

/**
 * know if any of the controllers is currently in use
 * @name FORGE.ControllerManager#active
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.ControllerManager.prototype, "active",
{
    /** @this {FORGE.ControllerManager} */
    get: function()
    {
        for(var i = 0, ii = this._controllers.length; i < ii; i++)
        {
            if(this._controllers[i].active === true)
            {
                return true;
            }
        }

        return false;
    }
});

/**
 * Get the "onControlStart" {@link FORGE.EventDispatcher} of the camera controller.
 * @name FORGE.ControllerManager#onControlStart
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.ControllerManager.prototype, "onControlStart",
{
    /** @this {FORGE.ControllerManager} */
    get: function()
    {
        if(this._onControlStart === null)
        {
            this._onControlStart = new FORGE.EventDispatcher(this);
        }

        return this._onControlStart;
    }
});

/**
 * Get the "onControlEnd" {@link FORGE.EventDispatcher} of the camera controller.
 * @name FORGE.ControllerManager#onControlEnd
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.ControllerManager.prototype, "onControlEnd",
{
    /** @this {FORGE.ControllerManager} */
    get: function()
    {
        if(this._onControlEnd === null)
        {
            this._onControlEnd = new FORGE.EventDispatcher(this);
        }

        return this._onControlEnd;
    }
});

/**
 * FORGE.ControllerBase
 * CameraBaseController class.
 *
 * Base controller class
 *
 * @constructor FORGE.ControllerBase
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @param {string=} className - subclass className
 * @extends {FORGE.BaseObject}
 */
FORGE.ControllerBase = function(viewer, className)
{
    /**
     * Viewer reference
     * @type {FORGE.Viewer}
     * @name FORGE.ControllerBase#_viewer
     * @private
     */
    this._viewer = viewer;

    /**
     * Type of the controller
     * @name FORGE.ControllerBase#_type
     * @type {string}
     * @private
     */
    this._type = FORGE.ControllerType.BASE;

    /**
     * Main camera reference.
     * @type {FORGE.Camera}
     * @name FORGE.ControllerBase#_camera
     * @private
     */
    this._camera = null;

    /**
     * Enabled state flag.
     * @type {boolean}
     * @name FORGE.ControllerBase#_enabled
     * @private
     */
    this._enabled = false;

    /**
     * Active state flag.
     * @type {boolean}
     * @name FORGE.ControllerBase#_active
     * @private
     */
    this._active = false;

    /**
     * Control start event handler.
     * @type {FORGE.EventDispatcher}
     * @name FORGE.ControllerBase#_onControlStart
     * @private
     */
    this._onControlStart = null;

    /**
     * Control end event handler.
     * @type {FORGE.EventDispatcher}
     * @name FORGE.ControllerBase#_onControlEnd
     * @private
     */
    this._onControlEnd = null;

    FORGE.BaseObject.call(this, className || "ControllerBase");

    this._boot();
};

FORGE.ControllerBase.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ControllerBase.prototype.constructor = FORGE.ControllerBase;

/**
 * Boot sequence.
 * @method FORGE.ControllerBase#_boot
 * @private
 */
FORGE.ControllerBase.prototype._boot = function()
{
    this._viewer.canvas.pointer.enabled = true;

    this._camera = this._viewer.renderer.camera;
};

/**
 * Enable control.
 * @method FORGE.ControllerBase#enable
 */
FORGE.ControllerBase.prototype.enable = function()
{
    if(this._enabled === true)
    {
        return;
    }

    this._enabled = true;
};

/**
 * Disable control.
 * @method FORGE.ControllerBase#disable
 */
FORGE.ControllerBase.prototype.disable = function()
{
    if(this._enabled === false)
    {
        return;
    }

    this._enabled = false;
};

/**
 * Destroy sequence.
 * @method FORGE.ControllerBase#destroy
 * @private
 */
FORGE.ControllerBase.prototype.destroy = function()
{
    this._camera = null;
    this._viewer = null;

    if(this._onControlStart !== null)
    {
        this._onControlStart.destroy();
        this._onControlStart = null;
    }

    if(this._onControlEnd !== null)
    {
        this._onControlEnd.destroy();
        this._onControlEnd = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get or set the enabled flag.
 * @name FORGE.ControllerBase#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.ControllerBase.prototype, "enabled",
{
    /** @this {FORGE.ControllerBase} */
    get: function()
    {
        return this._enabled;
    },

    /** @this {FORGE.ControllerBase} */
    set: function(value)
    {
        if(Boolean(value) === true)
        {
            this.enable();
        }
        else
        {
            this.disable();
        }
    }
});

/**
 * Get type of the controller.
 * @name FORGE.ControllerBase#type
 * @type {string}
 * @readonly
 */
Object.defineProperty(FORGE.ControllerBase.prototype, "type",
{
    /** @this {FORGE.ControllerBase} */
    get: function()
    {
        return this._type;
    }
});

/**
 * Get the "active" flag of the camera controller. A controller is active when it is currently in use.
 * @name FORGE.ControllerBase#active
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.ControllerBase.prototype, "active",
{
    /** @this {FORGE.ControllerBase} */
    get: function()
    {
        return this._active;
    }
});

/**
 * Get the "onControlStart" {@link FORGE.EventDispatcher} of the camera controller.
 * @name FORGE.ControllerBase#onControlStart
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.ControllerBase.prototype, "onControlStart",
{
    /** @this {FORGE.ControllerBase} */
    get: function()
    {
        if(this._onControlStart === null)
        {
            this._onControlStart = new FORGE.EventDispatcher(this);
        }

        return this._onControlStart;
    }
});

/**
 * Get the "onControlEnd" {@link FORGE.EventDispatcher} of the camera controller.
 * @name FORGE.ControllerBase#onControlEnd
 * @type {FORGE.EventDispatcher}
 * @readonly
 */
Object.defineProperty(FORGE.ControllerBase.prototype, "onControlEnd",
{
    /** @this {FORGE.ControllerBase} */
    get: function()
    {
        if(this._onControlEnd === null)
        {
            this._onControlEnd = new FORGE.EventDispatcher(this);
        }

        return this._onControlEnd;
    }
});

/**
 * This controller takes mouse and touch events on screen to animate camera moves.
 *
 * @constructor FORGE.ControllerPointer
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @param {ControllerInstanceConfig} config - the configuration of the controller
 * @extends {FORGE.ControllerBase}
 */
FORGE.ControllerPointer = function(viewer, config)
{
    /**
     * Configuration
     * @name FORGE.ControllerPointer#_config
     * @type {ControllerInstanceConfig}
     * @private
     */
    this._config = config;

    /**
     * Orientation controller configuration.
     * @name FORGE.ControllerPointer#_orientation
     * @type {ControllerOrientationConfig}
     * @private
     */
    this._orientation;

    /**
     * Zoom controller configuration.
     * @name FORGE.ControllerPointer#_zoom
     * @type {ControllerZoomConfig}
     * @private
     */
    this._zoom;

    /**
     * Fullscreen configuration.
     * @name FORGE.ControllerPointer#_fullscreen
     * @type {boolean}
     * @private
     */
    this._fullscreen = true;

    /**
     * Start position vector.
     * @name FORGE.ControllerPointer#_positionStart
     * @type {THREE.Vector2}
     * @private
     */
    this._positionStart = null;

    /**
     * Current position vector.
     * @name FORGE.ControllerPointer#_positionCurrent
     * @type {THREE.Vector2}
     * @private
     */
    this._positionCurrent = null;

    /**
     * Previous position vector.
     * @name FORGE.ControllerPointer#_positionPrevious
     * @type {THREE.Vector2}
     * @private
     */
    this._positionPrevious = null;

    /**
     * The fov value when you start to pinch in/out
     * @name FORGE.ControllerPointer#_pinchStartFov
     * @type {number}
     * @private
     */
    this._pinchStartFov = 0;

    /**
     * This is used when we have reached the minimum or maximum scale,
     * in order not to have a "no-effect" zone when zooming the other way
     * @name FORGE.ControllerPointer#_pinchScaleFactorCorrection
     * @type {number}
     * @private
     */
    this._pinchScaleFactorCorrection = 1;

    /**
     * Current velocity vector.
     * @name FORGE.ControllerPointer#_velocity
     * @type {THREE.Vector2}
     * @private
     */
    this._velocity = null;

    /**
     * Previous velocity vector.
     * @name FORGE.ControllerPointer#_inertia
     * @type {THREE.Vector2}
     * @private
     */
    this._inertia = null;

    /**
     * Panning flag.
     * @name FORGE.ControllerPointer#_panning
     * @type {boolean}
     * @private
     */
    this._panning = false;

    FORGE.ControllerBase.call(this, viewer, "ControllerPointer");
};

FORGE.ControllerPointer.prototype = Object.create(FORGE.ControllerBase.prototype);
FORGE.ControllerPointer.prototype.constructor = FORGE.ControllerPointer;

/**
 * Default configuration
 * @name {FORGE.ControllerPointer.DEFAULT_OPTIONS}
 * @type {ControllerPointerConfig}
 */
FORGE.ControllerPointer.DEFAULT_OPTIONS =
{
    fullscreen: true,

    orientation:
    {
        drag: false,
        hardness: 0.6, //Hardness factor impatcing controller response to some instant force.
        damping: 0.15, //Damping factor controlling inertia.
        velocityMax: 300,
        invert: {
            x: false,
            y: false
        }
    },

    zoom:
    {
        hardness: 5,
        invert: false,
        toPointer: false
    }
};

/**
 * Boot sequence.
 * @method FORGE.ControllerPointer#_boot
 * @private
 */
FORGE.ControllerPointer.prototype._boot = function()
{
    FORGE.ControllerBase.prototype._boot.call(this);

    this._type = FORGE.ControllerType.POINTER;

    this._inertia = new THREE.Vector2();
    this._velocity = new THREE.Vector2();
    this._positionStart = new THREE.Vector2();
    this._positionPrevious = new THREE.Vector2();
    this._positionCurrent = new THREE.Vector2();

    this._parseConfig(this._config);

    if(this._enabled === true)
    {
        this.enable();
    }
};

/**
 * Parse the configuration.
 * @method FORGE.ControllerPointer#_parseConfig
 * @param {ControllerInstanceConfig} config - configuration object to parse
 */
FORGE.ControllerPointer.prototype._parseConfig = function(config)
{
    this._uid = config.uid;
    this._register();

    var options = config.options || {};

    this._orientation = /** @type {ControllerOrientationConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerPointer.DEFAULT_OPTIONS.orientation, options.orientation));
    this._zoom = /** @type {ControllerZoomConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerPointer.DEFAULT_OPTIONS.zoom, options.zoom));
    this._fullscreen = (typeof options.fullscreen === "boolean") ? options.fullscreen : FORGE.ControllerPointer.DEFAULT_OPTIONS.fullscreen;

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;
};

/**
 * Pan start event handler.
 * @method FORGE.ControllerPointer#_panStartHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._panStartHandler = function(event)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    this._viewer.canvas.pointer.onPanMove.add(this._panMoveHandler, this);
    this.log("_panStartHandler (" + event.data.velocityX.toFixed(2) + ", " + event.data.velocityY.toFixed(2) + ")");

    this._active = true;
    this._panning = true;

    var position = FORGE.Pointer.getRelativeMousePosition(event.data);
    this._positionStart = new THREE.Vector2(position.x, position.y);
    this._positionPrevious.copy(this._positionStart);
    this._positionCurrent.copy(this._positionStart);
    this._velocity.set(0, 0);

    if (this._onControlStart !== null)
    {
        this._onControlStart.dispatch();
    }

    this._viewer.controllers.notifyControlStart(this);
};

/**
 * Pan move event handler.
 * @method FORGE.ControllerPointer#_panMoveHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._panMoveHandler = function(event)
{
    var position = FORGE.Pointer.getRelativeMousePosition(event.data);

    if(this._viewer.controllers.enabled === false || position === null)
    {
        return;
    }

    this._positionPrevious.copy(this._positionCurrent);
    this._positionCurrent.set(position.x, position.y);
    // this.log("Current position: " + this._positionCurrent.x + ", " + this._positionCurrent.y);

    if(this._orientation.drag === true && this._panning === true)
    {
        this._updateCameraWithDrag();
    }
};

/**
 * Pan end event handler.
 * @method FORGE.ControllerPointer#_panEndHandler
 * @private
 */
FORGE.ControllerPointer.prototype._panEndHandler = function()
{
    this._viewer.canvas.pointer.onPanMove.remove(this._panMoveHandler, this);
    this.log("_panEndHandler");

    this._active = false;
    this._panning = false;

    this._velocity.set(0, 0);
    this._positionStart.set(0, 0);
    this._positionCurrent.copy(this._positionStart);

    if (this._onControlEnd !== null)
    {
        this._onControlEnd.dispatch();
    }

    this._viewer.controllers.notifyControlEnd(this);
};

/**
 * Update the camera from the mouse position.
 * @method FORGE.ControllerPointer#_updateCameraWithDrag
 * @private
 */
FORGE.ControllerPointer.prototype._updateCameraWithDrag = function()
{
    var stw0 = this._viewer.view.screenToWorld(this._positionPrevious);
    var stw1 = this._viewer.view.screenToWorld(this._positionCurrent);

    // If the screen point do not match any world position, return
    if(stw0 === null || stw1 === null)
    {
        return;
    }

    var spherical0 = FORGE.Math.cartesianToSpherical(stw0.x, stw0.y, stw0.z);
    var quat0 = FORGE.Quaternion.fromEuler(spherical0.theta, spherical0.phi, 0);

    var spherical1 = FORGE.Math.cartesianToSpherical(stw1.x, stw1.y, stw1.z);
    var quat1 = FORGE.Quaternion.fromEuler(spherical1.theta, spherical1.phi, 0);

    this._camera.yaw += FORGE.Math.radToDeg(spherical0.theta - spherical1.theta);
    this._camera.pitch += FORGE.Math.radToDeg(spherical0.phi - spherical1.phi);
    this._camera.roll = 0;
};

/**
 * Update the camera from the velocity that results of the mouse movement.
 * @method FORGE.ControllerPointer#_updateCameraWithVelocity
 * @private
 */
FORGE.ControllerPointer.prototype._updateCameraWithVelocity = function()
{
    var size = this._viewer.renderer.displayResolution;
    var hardness = 1 / (this._orientation.hardness * Math.min(size.width, size.height));

    var logZoomFactor = Math.min(1, this._camera.fov / 90) / Math.LN2;

    this._velocity.subVectors(this._positionCurrent, this._positionStart);

    if (this._velocity.length() > this._orientation.velocityMax)
    {
        this._velocity.setLength(this._orientation.velocityMax);
    }

    this._velocity.multiplyScalar(hardness);
    this._velocity.multiplyScalar(logZoomFactor);

    // this.log("Current velocity: " + this._velocity.x + ", " + this._velocity.y);

    var invert = this._orientation.invert;
    var invertX = (invert === true) ? -1 : (typeof invert === "object" && invert.x === true) ? -1 : 1;
    var invertY = (invert === true) ? -1 : (typeof invert === "object" && invert.y === true) ? -1 : 1;

    var dx = this._velocity.x + this._inertia.x;
    var dy = this._velocity.y + this._inertia.y;

    if (dx === 0 && dy === 0)
    {
        return;
    }

    var yaw = invertX * dx;

    var threshold = logZoomFactor * 0.05;
    //Do not move the camera anymore if the modifier is too low, this prevent the camera onChange event to be fired too much times
    if(Math.abs(yaw) > threshold)
    {
        this._camera.yaw += yaw;
        this._camera.flat.position.x += dx;
    }

    var pitch = invertY * dy;
    //Do not move the camera anymore if the modifier is too low, this prevent the camera onChange to be fired too much times
    if(Math.abs(pitch) > threshold)
    {
        this._camera.pitch -= pitch;
        this._camera.flat.position.y -= dy;
    }

    // Damping 1 -> stops instantly, 0 infinite rebounds
    this._inertia.add(this._velocity).multiplyScalar(FORGE.Math.clamp(1 - this._orientation.damping, 0, 1));
};

/**
 * Pinch start event handler.
 * @method FORGE.ControllerPointer#_pinchStartHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._pinchStartHandler = function(event)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    this._pinchStartFov = this._camera.fov;
    this._pinchScaleFactorCorrection = 1;

    this._viewer.canvas.pointer.onPinchMove.add(this._pinchMoveHandler, this);
    this.log("_pinchStartHandler "+event);
};

/**
 * Pinch move event handler.
 * @method FORGE.ControllerPointer#_pinchMoveHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._pinchMoveHandler = function(event)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    event.data.preventDefault();

    var scale = this._zoom.invert ? event.data.scale : 1 / event.data.scale;
    var fovMin = this._camera.fovMin;
    var fovMax = this._camera.fovMax;

    var tmpFov = this._pinchStartFov * scale / this._pinchScaleFactorCorrection;

    if (tmpFov < fovMin)
    {
        this._pinchScaleFactorCorrection = this._pinchStartFov * scale / fovMin;
    }
    else if (tmpFov > fovMax)
    {
        this._pinchScaleFactorCorrection = this._pinchStartFov * scale / fovMax;
    }

    tmpFov = this._pinchStartFov * scale / this._pinchScaleFactorCorrection;

    this._camera.fov = tmpFov;
};

/**
 * Pinch end event handler.
 * @method FORGE.ControllerPointer#_pinchEndHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._pinchEndHandler = function(event)
{
    this._viewer.canvas.pointer.onPinchMove.remove(this._pinchMoveHandler, this);
    this.log("_pinchEndHandler "+event);
};

/**
 * Wheel event handler.
 * @method FORGE.ControllerPointer#_wheelHandler
 * @param {FORGE.Event} event - Event object
 * @private
 */
FORGE.ControllerPointer.prototype._wheelHandler = function(event)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    event.data.preventDefault();

    var invert = this._zoom.invert ? 1 : -1;
    var delta = invert / this._zoom.hardness;
    var factorDeltaY = 1;

    if (event.data.deltaMode)
    {
        switch(event.data.deltaMode)
        {
            case 1: //DOM_DELTA_LINE (3 lines === 100 pixels)
                factorDeltaY = 33.3;
                break;
            case 2: //DOM_DELTA_PAGE
                factorDeltaY = self.innerHeight;
                break;
            default: //DOM_DELTA_PIXEL
                factorDeltaY = 1;
        }
    }

    if (event.data.deltaY)
    {
        delta *= (event.data.deltaY * factorDeltaY) / 5;
    }

    var logZoomFactor = Math.min(1, this._camera.fov / 90) / Math.LN2;
    this._camera.fov -= delta * logZoomFactor;

    if(this._zoom.toPointer === true)
    {
        var screen = FORGE.Pointer.getRelativeMousePosition(event.data);
        var stw0 = this._viewer.view.screenToWorld(screen);
        var spherical0 = FORGE.Math.cartesianToSpherical(stw0.x, stw0.y, stw0.z);
        var quat0 = FORGE.Quaternion.fromEuler(spherical0.theta, spherical0.phi, 0);

        this._viewer.view.current.updateUniforms();

        var stw1 = this._viewer.view.screenToWorld(screen);
        var spherical1 = FORGE.Math.cartesianToSpherical(stw1.x, stw1.y, stw1.z);
        var quat1 = FORGE.Quaternion.fromEuler(spherical1.theta, spherical1.phi, 0);

        var quat = FORGE.Quaternion.diffBetweenQuaternions(quat1, quat0);
        var euler = FORGE.Quaternion.toEuler(quat);

        this._camera.yaw += FORGE.Math.radToDeg(euler.yaw);
        this._camera.pitch += FORGE.Math.radToDeg(euler.pitch);
    }

    this.log("_wheelHandler (fov:" + this._camera.fov + ")");
};

/**
 * Double tap event handler. Toggle the fullscreen.
 * @method FORGE.ControllerPointer#_doubleTapHandler
 * @private
 */
FORGE.ControllerPointer.prototype._doubleTapHandler = function()
{
    if (this._viewer.controllers.enabled === false || this._fullscreen === false)
    {
        return;
    }

    this._viewer.fullscreen = !this._viewer.fullscreen;
};

/**
 * Enable controller
 * @method FORGE.ControllerPointer#enable
 */
FORGE.ControllerPointer.prototype.enable = function()
{
    FORGE.ControllerBase.prototype.enable.call(this);

    this._viewer.canvas.pointer.onPanStart.add(this._panStartHandler, this);
    this._viewer.canvas.pointer.onPanEnd.add(this._panEndHandler, this);

    this._viewer.canvas.pointer.onPinchStart.add(this._pinchStartHandler, this);
    this._viewer.canvas.pointer.onPinchEnd.add(this._pinchEndHandler, this);
    this._viewer.canvas.pointer.onWheel.add(this._wheelHandler, this);

    this._viewer.canvas.pointer.onDoubleTap.add(this._doubleTapHandler, this);
};

/**
 * Disable controller
 * @method FORGE.ControllerPointer#disable
 */
FORGE.ControllerPointer.prototype.disable = function()
{
    FORGE.ControllerBase.prototype.disable.call(this);

    this._viewer.canvas.pointer.onPanStart.remove(this._panStartHandler, this);
    this._viewer.canvas.pointer.onPanMove.remove(this._panMoveHandler, this);
    this._viewer.canvas.pointer.onPanEnd.remove(this._panEndHandler, this);

    this._viewer.canvas.pointer.onPinchStart.remove(this._pinchStartHandler, this);
    this._viewer.canvas.pointer.onPinchMove.remove(this._pinchEndHandler, this);
    this._viewer.canvas.pointer.onPinchEnd.remove(this._pinchEndHandler, this);
    this._viewer.canvas.pointer.onWheel.remove(this._wheelHandler, this);

    this._viewer.canvas.pointer.onDoubleTap.remove(this._doubleTapHandler, this);
};

/**
 * Update routine.
 * @method FORGE.ControllerPointer#update
 */
FORGE.ControllerPointer.prototype.update = function()
{
    if(this._orientation.drag !== true)
    {
        this._updateCameraWithVelocity();
    }
};

/**
 * Destroy routine
 * @method FORGE.ControllerPointer#destroy
 */
FORGE.ControllerPointer.prototype.destroy = function()
{
    FORGE.ControllerBase.prototype.destroy.call(this);
};

/**
 * Get and set the orientation options
 * @name FORGE.ControllerPointer#orientation
 * @type {boolean}
 */
Object.defineProperty(FORGE.ControllerPointer.prototype, "orientation",
{
    /** @this {FORGE.ControllerPointer} */
    get: function()
    {
        return this._orientation;
    },
    /** @this {FORGE.ControllerPointer} */
    set: function(value)
    {
        this._orientation = value;
    }
});

/**
 * Get and set the zoom options
 * @name FORGE.ControllerPointer#zoom
 * @type {boolean}
 */
Object.defineProperty(FORGE.ControllerPointer.prototype, "zoom",
{
    /** @this {FORGE.ControllerPointer} */
    get: function()
    {
        return this._zoom;
    },
    /** @this {FORGE.ControllerPointer} */
    set: function(value)
    {
        this._zoom = value;
    }
});

/**
 * @constructor FORGE.ControllerKeyboard
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @param {ControllerInstanceConfig} config - the configuration of the controller
 * @extends {FORGE.ControllerBase}
 */
FORGE.ControllerKeyboard = function(viewer, config)
{
    /**
     * Configuration
     * @name FORGE.ControllerKeyboard#_config
     * @type {ControllerInstanceConfig}
     * @private
     */
    this._config = config;

    /**
     * Orientation controller configuration.
     * @name FORGE.ControllerKeyboard#_orientation
     * @type {ControllerOrientationConfig}
     * @private
     */
    this._orientation;

    /**
     * Zoom controller configuration.
     * @name FORGE.ControllerKeyboard#_zoom
     * @type {ControllerZoomConfig}
     * @private
     */
    this._zoom;

    /**
     * Previous position vector.
     * @name FORGE.ControllerKeyboard#_positionStart
     * @type {THREE.Vector2}
     * @private
     */
    this._positionStart = null;

    /**
     * Previous position vector.
     * @name FORGE.ControllerKeyboard#_positionCurrent
     * @type {THREE.Vector2}
     * @private
     */
    this._positionCurrent = null;

    /**
     * Current velocity vector.
     * @name FORGE.ControllerKeyboard#_velocity
     * @type {THREE.Vector2}
     * @private
     */
    this._velocity = null;

    /**
     * Previous velocity vector.
     * @name FORGE.ControllerKeyboard#_inertia
     * @type {THREE.Vector2}
     * @private
     */
    this._inertia = null;

    /**
     * Array of all key bindings
     * @name FORGE.ControllerKeyboard#_keyBindings
     * @type {Array<FORGE.KeyBinding>}
     * @private
     */
    this._keyBindings = null;

    FORGE.ControllerBase.call(this, viewer, "ControllerKeyboard");
};

FORGE.ControllerKeyboard.prototype = Object.create(FORGE.ControllerBase.prototype);
FORGE.ControllerKeyboard.prototype.constructor = FORGE.ControllerKeyboard;

/**
 * Default configuration
 * @name {FORGE.ControllerKeyboard.DEFAULT_OPTIONS}
 * @type {ControllerKeyboardConfig}
 */
FORGE.ControllerKeyboard.DEFAULT_OPTIONS =
{
    orientation:
    {
        hardness: 0.6, //Hardness factor impatcing controller response to some instant force.
        damping: 0.15, //Damping factor controlling inertia.
        velocityMax: 300,
        invert: false
    },

    zoom:
    {
        hardness: 5,
        invert: false
    }
};

/**
 * Boot sequence.
 * @method FORGE.ControllerKeyboard#_boot
 * @private
 */
FORGE.ControllerKeyboard.prototype._boot = function()
{
    FORGE.ControllerBase.prototype._boot.call(this);

    this._type = FORGE.ControllerType.KEYBOARD;

    this._keyBindings = [];

    this._inertia = new THREE.Vector2();
    this._velocity = new THREE.Vector2();
    this._positionStart = new THREE.Vector2();
    this._positionCurrent = new THREE.Vector2();

    this._parseConfig(this._config);

    if(this._enabled === true)
    {
        this.enable();
    }
};

/**
 * Parse the configuration.
 * @method FORGE.ControllerKeyboard#_parseConfig
 * @param {ControllerInstanceConfig} config - configuration object to parse.
 */
FORGE.ControllerKeyboard.prototype._parseConfig = function(config)
{
    this._uid = config.uid;
    this._register();

    var options = config.options || {};

    this._orientation = /** @type {ControllerOrientationConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerKeyboard.DEFAULT_OPTIONS.orientation, options.orientation));
    this._zoom = /** @type {ControllerZoomConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerKeyboard.DEFAULT_OPTIONS.zoom, options.zoom));

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;

    if(options.default !== false)
    {
        this._addDefaultBindings();
    }

    if(Array.isArray(options.bindings) === true)
    {
        for(var i = 0, ii = options.bindings.length; i < ii; i++)
        {
            this._addBinding(options.bindings[i]);
        }
    }
};

/**
 * Add the default bindings to the controller keyboard : orientation with arrows and zoom with plus (+) and minus (-).
 * @method FORGE.ControllerKeyboard#_addDefaultBinding.
 * @private
 */
FORGE.ControllerKeyboard.prototype._addDefaultBindings = function()
{
    var bindingLeft = new FORGE.KeyBinding(this._viewer,
        [81, 37],
        this._orientationDownHandler,
        this._orientationUpHandler,
        this._orientationHoldHandler,
        [68, 39],
        this,
        "left"
    );
    this._keyBindings.push(bindingLeft);

    var bindingRight = new FORGE.KeyBinding(this._viewer,
        [68, 39],
        this._orientationDownHandler,
        this._orientationUpHandler,
        this._orientationHoldHandler,
        [81, 37],
        this,
        "right"
    );
    this._keyBindings.push(bindingRight);

    var bindingUp = new FORGE.KeyBinding(this._viewer,
        [90, 38],
        this._orientationDownHandler,
        this._orientationUpHandler,
        this._orientationHoldHandler,
        [83, 40],
        this,
        "up"
    );
    this._keyBindings.push(bindingUp);

    var bindingDown = new FORGE.KeyBinding(this._viewer,
        [83, 40],
        this._orientationDownHandler,
        this._orientationUpHandler,
        this._orientationHoldHandler,
        [90, 38],
        this,
        "down"
    );
    this._keyBindings.push(bindingDown);

    var bindingPlus = new FORGE.KeyBinding(this._viewer,
        107,
        this._zoomDownHandler,
        this._zoomUpHandler,
        this._zoomHoldHandler,
        109,
        this,
        "plus"
    );
    this._keyBindings.push(bindingPlus);

    var bindingMinus = new FORGE.KeyBinding(this._viewer,
        109,
        this._zoomDownHandler,
        this._zoomUpHandler,
        this._zoomHoldHandler,
        107,
        this,
        "minus"
    );
    this._keyBindings.push(bindingMinus);
};

/**
 * Add a keyboard binding config to this controller.
 * @method FORGE.ControllerKeyboard#_addBinding
 * @param {ControllerKeyboardBindingConfig} binding - The binding config to add.
 * @private
 */
FORGE.ControllerKeyboard.prototype._addBinding = function(binding)
{
    var keysIn = binding.in;
    var keysOut = binding.out;
    var name = binding.name;
    var events = binding.events;

    if(FORGE.Utils.isTypeOf(keysIn, "number") === false && FORGE.Utils.isArrayOf(keysIn, "number") === false)
    {
        this.warn("Can't add custom keyboard binding, keys in are invalid!");
        return;
    }

    if(typeof events !== "object" && events === null)
    {
        this.warn("Can't add custom keyboard binding, events are invalid!");
        return;
    }

    var keyBinding = new FORGE.KeyBinding(this._viewer, keysIn, events.onDown, events.onUp, events.onHold, keysOut, this, name);

    this._keyBindings.push(keyBinding);
};

/**
 * Event handler for orientation (arrows) down handler.
 * @method FORGE.ControllerKeyboard#_orientationDownHandler
 * @param  {FORGE.KeyBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerKeyboard.prototype._orientationDownHandler = function(binding)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    this._active = true;

    switch(binding.name)
    {
        case "left":
        case "right":
            this._velocity.setX(0);
            break;

        case "up":
        case "down":
            this._velocity.setY(0);
            break;
    }

    if (this._onControlStart !== null)
    {
        this._onControlStart.dispatch();
    }

    this._viewer.controllers.notifyControlStart(this);
};

/**
 * Event handler for orientation (arrows) hold handler.
 * @method FORGE.ControllerKeyboard#_orientationHoldHandler
 * @param  {FORGE.KeyBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerKeyboard.prototype._orientationHoldHandler = function(binding)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    switch(binding.name)
    {
        case "left":
            this._positionCurrent.setX(this._positionCurrent.x - 5);
            break;

        case "right":
            this._positionCurrent.setX(this._positionCurrent.x + 5);
            break;

        case "up":
            this._positionCurrent.setY(this._positionCurrent.y - 5);
            break;

        case "down":
            this._positionCurrent.setY(this._positionCurrent.y + 5);
            break;
    }
};

/**
 * Event handler for orientation (arrows) up handler.
 * @method FORGE.ControllerKeyboard#_orientationUpHandler
 * @param  {FORGE.KeyBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerKeyboard.prototype._orientationUpHandler = function(binding)
{
    this._active = false;

    switch(binding.name)
    {
        case "left":
        case "right":
            this._velocity.setX(0);
            this._positionCurrent.setX(0);
            break;

        case "up":
        case "down":
            this._velocity.setY(0);
            this._positionCurrent.setY(0);
            break;
    }

    if (this._onControlEnd !== null)
    {
        this._onControlEnd.dispatch();
    }

    this._viewer.controllers.notifyControlEnd(this);
};

/**
 * Event handler for zoom (+ / -) down handler.
 * @method FORGE.ControllerKeyboard#_zoomDownHandler
 * @param  {FORGE.KeyBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerKeyboard.prototype._zoomDownHandler = function(binding)
{
    if(this._viewer.controllers.enabled === false)
    {
        return;
    }

    this._active = true;

    this._zoomProcessBinding(binding);

    if (this._onControlStart !== null)
    {
        this._onControlStart.dispatch();
    }

    this._viewer.controllers.notifyControlStart(this);
};

/**
 * Event handler for zoom (+ / -) hold handler.
 * @method FORGE.ControllerKeyboard#_zoomHoldHandler
 * @param  {FORGE.KeyBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerKeyboard.prototype._zoomHoldHandler = function(binding)
{
    this._zoomProcessBinding(binding);
};

/**
 * Event handler for zoom (+ / -) up handler.
 * @method FORGE.ControllerKeyboard#_zoomUpHandler
 * @private
 */
FORGE.ControllerKeyboard.prototype._zoomUpHandler = function()
{
    this._active = false;

    if (this._onControlEnd !== null)
    {
        this._onControlEnd.dispatch();
    }

    this._viewer.controllers.notifyControlEnd(this);
};

/**
 * Process a key binding related to zoom for down and hold zoom handlers.
 * @method FORGE.ControllerKeyboard#_zoomProcessBinding
 * @param  {FORGE.KeyBinding} binding - The key binding related to zoom
 * @private
 */
FORGE.ControllerKeyboard.prototype._zoomProcessBinding = function(binding)
{
    var invert = this._zoom.invert ? 1 : -1;
    var delta = invert / this._zoom.hardness;

    switch(binding.name)
    {
        case "minus":
            delta *= 10;
            break;

        case "plus":
            delta *= -10;
            break;
    }

    this._camera.fov = this._camera.fov - delta;
};

/**
 * Enable controller
 * @method FORGE.ControllerKeyboard#enable
 */
FORGE.ControllerKeyboard.prototype.enable = function()
{
    FORGE.ControllerBase.prototype.enable.call(this);

    var binding;

    for(var i = 0, ii = this._keyBindings.length; i < ii; i++)
    {
        binding = this._keyBindings[i];
        this._viewer.keyboard.addBinding(binding);
    }
};

/**
 * Disable controller
 * @method FORGE.ControllerKeyboard#disable
 */
FORGE.ControllerKeyboard.prototype.disable = function()
{
    FORGE.ControllerBase.prototype.disable.call(this);

    var binding;

    for(var i = 0, ii = this._keyBindings.length; i < ii; i++)
    {
        binding = this._keyBindings[i];
        this._viewer.keyboard.removeBinding(binding);
    }
};

/**
 * Update routine.
 * @method FORGE.ControllerKeyboard#update
 */
FORGE.ControllerKeyboard.prototype.update = function()
{
    var size = this._viewer.renderer.displayResolution;
    var hardness = 1 / (this._orientation.hardness * Math.min(size.width, size.height));

    this._velocity.subVectors(this._positionCurrent, this._positionStart);

    if (this._velocity.length() > this._orientation.velocityMax)
    {
        this._velocity.setLength(this._orientation.velocityMax);
    }

    this._velocity.multiplyScalar(hardness);

    // this.log("Current velocity: " + this._velocity.x + ", " + this._velocity.y);

    var dx = this._velocity.x + this._inertia.x;
    var dy = this._velocity.y + this._inertia.y;

    if (dx === 0 && dy === 0)
    {
        return;
    }

    var invert = this._orientation.invert ? -1 : 1;
    this._camera.yaw += invert * dx;
    this._camera.pitch -= invert * dy;

    // Damping 1 -> stops instantly, 0 infinite rebounds
    this._inertia.add(this._velocity).multiplyScalar(FORGE.Math.clamp(1 - this._orientation.damping, 0, 1));
};

/**
 * Destroy routine
 * @method FORGE.ControllerKeyboard#destroy
 */
FORGE.ControllerKeyboard.prototype.destroy = function()
{
    FORGE.ControllerBase.prototype.destroy.call(this);
};

/**
 * This controller takes gyroscope events on screen to animate camera moves.
 *
 * @constructor FORGE.ControllerGyroscope
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {ControllerInstanceConfig} config - the configuration of the controller
 * @extends {FORGE.ControllerBase}
 */
FORGE.ControllerGyroscope = function(viewer, config)
{
    /**
     * Configuration
     * @name FORGE.ControllerGyroscope#_config
     * @type {ControllerInstanceConfig}
     * @private
     */
    this._config = config;

    /**
     * Position euler.
     * @name FORGE.ControllerGyroscope#_posEuler
     * @type {THREE.Euler}
     * @private
     */
    this._posEuler = null;

    /**
     * The intermediate quaternion where the position is computed.
     * @name FORGE.ControllerGyroscope#_posQuatIndermediate
     * @type {THREE.Quaternion}
     * @private
     */
    this._posQuatIndermediate = null;

    /**
     * The offset quaternion of the camera.
     * @name FORGE.ControllerGyroscope#_posQuatOffset
     * @type {THREE.Quaternion}
     * @private
     */
    this._posQuatOffset = null;

    /**
     * The quaternion to add given the orientation of the screen
     * @name FORGE.ControllerGyroscope#_posQuatScreenOrientation
     * @type {THREE.Quaternion}
     * @private
     */
    this._posQuatScreenOrientation = null;

    /**
     * Quaternion result of the position.
     * @name FORGE.ControllerGyroscope#_posQuatFinal
     * @type {THREE.Quaternion}
     * @private
     */
    this._posQuatFinal = null;

    /**
     * Orientation of the screen (not the device !)
     * @name FORGE.ControllerGyroscope#_screenOrientation
     * @type {number}
     * @private
     */
    this._screenOrientation = 0;

    /**
     * Is the controller paused ?
     * @name FORGE.ControllerGyroscope#_paused
     * @type {boolean}
     * @private
     */
    this._paused = false;

    FORGE.ControllerBase.call(this, viewer, "ControllerGyroscope");
};

FORGE.ControllerGyroscope.prototype = Object.create(FORGE.ControllerBase.prototype);
FORGE.ControllerGyroscope.prototype.constructor = FORGE.ControllerGyroscope;

/**
 * Boot sequence
 * @method FORGE.ControllerGyroscope#_boot
 * @private
 */
FORGE.ControllerGyroscope.prototype._boot = function()
{
    FORGE.ControllerBase.prototype._boot.call(this);

    this._type = FORGE.ControllerType.GYROSCOPE;

    this._posEuler = new THREE.Euler();
    this._posQuatIndermediate = new THREE.Quaternion();
    this._posQuatOffset = new THREE.Quaternion(-Math.sqrt(0.5), 0, 0, Math.sqrt(0.5));
    this._posQuatScreenOrientation = new THREE.Quaternion();
    this._posQuatFinal = new THREE.Quaternion();

    this._parseConfig(this._config);

    FORGE.Device.onReady.addOnce(this._deviceReadyHandler, this);
};

/**
 * Device ready handler. Enables the gyro controller if gyro is available on the device.
 * @method FORGE.ControllerGyroscope#_deviceReadyHandler
 * @private
 */
FORGE.ControllerGyroscope.prototype._deviceReadyHandler = function()
{
    this._viewer.renderer.display.onDisplayChange.add(this._displayChangeHandler, this);
    this._viewer.renderer.view.onChange.add(this._viewChangeHandler, this);

    if (this._enabled === true && FORGE.Device.gyroscope === true)
    {
        this.enable();
    }
};

/**
 * Parse the configuration.
 * @method FORGE.ControllerGyroscope#_parseConfig
 * @param {ControllerInstanceConfig} config - configuration object to parse.
 */
FORGE.ControllerGyroscope.prototype._parseConfig = function(config)
{
    this._uid = config.uid;
    this._register();

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;
};

/**
 * Display change handler, pause the gyro in VR (WebVR handles the gyro antoher way)
 * @method FORGE.ControllerGyroscope#_displayChangeHandler
 * @private
 */
FORGE.ControllerGyroscope.prototype._displayChangeHandler = function()
{
    if(this._viewer.renderer.display.presentingVR === true)
    {
        this._paused = true;
    }
    else
    {
        this._paused = false;
    }
};

/**
 * View change handler, pause the gyro if the view is not rectilinear
 * @method FORGE.ControllerGyroscope#_viewChangeHandler
 * @private
 */
FORGE.ControllerGyroscope.prototype._viewChangeHandler = function()
{
    if(this._viewer.renderer.view.type !== FORGE.ViewType.RECTILINEAR)
    {
        this._paused = true;
    }
    else
    {
        this._paused = false;
    }
};

/**
 * Orientation change handler.
 * @method FORGE.ControllerGyroscope#_deviceOrientationChangeHandler
 * @param {FORGE.Event=} event - Event object
 * @private
 */
FORGE.ControllerGyroscope.prototype._deviceOrientationChangeHandler = function(event)
{
    if (this._viewer.controllers.enabled === false)
    {
        return;
    }

    /** @type {DeviceOrientation} */
    var position = {
        beta: 0,
        alpha: 0,
        gamma: 0
    };

    if (typeof event !== "undefined" && event !== null && typeof event.data !== "undefined" && event.data !== null)
    {
        position = /** @type {DeviceOrientation} */ event.data;
    }

    // Set the position in correct Euler coordinates
    this._posEuler.set(FORGE.Math.degToRad(position.beta), FORGE.Math.degToRad(position.alpha), -FORGE.Math.degToRad(position.gamma), "YXZ");
    this._posQuatIndermediate.setFromEuler(this._posEuler);

    // Add the offset provided by the camera
    this._posQuatIndermediate.multiply(this._posQuatOffset);

    // Adjust given the screen orientation
    this._posQuatIndermediate.multiply(this._posQuatScreenOrientation);

    // Final inversion, see FORGE.RenderDisplay#getQuaternionFromPose method
    this._posQuatFinal.set(-this._posQuatIndermediate.y, -this._posQuatIndermediate.x, -this._posQuatIndermediate.z, this._posQuatIndermediate.w);
};

/**
 * Screen orientation change event.
 * @method FORGE.ControllerGyroscope#_screenOrientationChangeHandler
 * @private
 */
FORGE.ControllerGyroscope.prototype._screenOrientationChangeHandler = function()
{
    if (typeof screen.orientation !== "undefined")
    {
        this._screenOrientation = FORGE.Math.degToRad(screen.orientation.angle);
    }
    else if (typeof window.orientation !== "undefined")
    {
        this._screenOrientation = FORGE.Math.degToRad(window.orientation);
    }

    this._posQuatScreenOrientation.setFromAxisAngle(new THREE.Vector3(0, 0, 1), -this._screenOrientation);
};

/**
 * Enable controller
 * @method FORGE.ControllerGyroscope#enable
 */
FORGE.ControllerGyroscope.prototype.enable = function()
{
    if (this._viewer.controllers.enabled === false || this._config.enabled === false)
    {
        return;
    }

    FORGE.ControllerBase.prototype.enable.call(this);

    this._viewer.gyroscope.onDeviceOrientationChange.add(this._deviceOrientationChangeHandler, this);
    this._viewer.gyroscope.onScreenOrientationChange.add(this._screenOrientationChangeHandler, this);

    this._screenOrientationChangeHandler();
    this._deviceOrientationChangeHandler();
};

/**
 * Disable controller
 * @method FORGE.ControllerGyroscope#disable
 */
FORGE.ControllerGyroscope.prototype.disable = function()
{
    if (this._viewer.controllers.enabled === false || this._config.enabled === false)
    {
        return;
    }

    FORGE.ControllerBase.prototype.disable.call(this);

    this._viewer.gyroscope.onDeviceOrientationChange.remove(this._deviceOrientationChangeHandler, this);
    this._viewer.gyroscope.onScreenOrientationChange.remove(this._screenOrientationChangeHandler, this);
};

/**
 * Update routine.
 * @method FORGE.ControllerGyroscope#update
 */
FORGE.ControllerGyroscope.prototype.update = function()
{
    if (this._enabled === true && FORGE.Device.gyroscope === true && this._paused === false)
    {
        this._viewer.camera.quaternion = this._posQuatFinal;
    }
};

/**
 * Destroy squence
 * @method FORGE.ControllerGyroscope#destroy
 * @private
 */
FORGE.ControllerGyroscope.prototype.destroy = function()
{
    this.disable();

    this._viewer.renderer.display.onDisplayChange.remove(this._displayChangeHandler, this);
    this._viewer.renderer.view.onChange.remove(this._viewChangeHandler, this);

    this._posEuler = null;
    this._posQuatIndermediate = null;
    this._posQuatOffset = null;
    this._posQuatScreenOrientation = null;
    this._posQuatFinal = null;

    FORGE.ControllerBase.prototype.destroy.call(this);
};

/**
 * A gamepad controller (type xbox or ps4 for example).
 *
 * @constructor FORGE.ControllerGamepad
 * @param {FORGE.Viewer} viewer - viewer reference
 * @param {ControllerInstanceConfig} config - the configuration of the controller
 * @extends {FORGE.ControllerBase}
 */
FORGE.ControllerGamepad = function(viewer, config)
{
    /**
     * A list of associated gamepad
     * @name FORGE.ControllerGamepad#_gamepads
     * @type {?Array<FORGE.Gamepad>}
     * @private
     */
    this._gamepads = null;

    /**
     * Configuration
     * @name FORGE.ControllerGamepad#_config
     * @type {ControllerInstanceConfig}
     * @private
     */
    this._config = config;

    /**
     * Orientation controller configuration.
     * @name FORGE.ControllerGamepad#_orientation
     * @type {ControllerOrientationConfig}
     * @private
     */
    this._orientation;

    /**
     * Zoom controller configuration.
     * @name FORGE.ControllerGamepad#_zoom
     * @type {ControllerZoomConfig}
     * @private
     */
    this._zoom;

    /**
     * Previous position vector.
     * @name FORGE.ControllerGamepad#_positionCurrent
     * @type {THREE.Vector2}
     * @private
     */
    this._positionCurrent = null;

    /**
     * Current velocity vector.
     * @name FORGE.ControllerGamepad#_velocity
     * @type {THREE.Vector2}
     * @private
     */
    this._velocity = null;

    /**
     * Previous velocity vector.
     * @name FORGE.ControllerGamepad#_inertia
     * @type {THREE.Vector2}
     * @private
     */
    this._inertia = null;

    /**
     * Array of all button bindings
     * @name FORGE.ControllerGamepad#_buttonBindings
     * @type {Array<FORGE.ButtonBinding>}
     * @private
     */
    this._buttonBindings = null;

    /**
     * Array of all axis bindings
     * @name FORGE.ControllerGamepad#_axisBindings
     * @type {Array<FORGE.AxisBinding>}
     * @private
     */
    this._axisBindings = null;

    FORGE.ControllerBase.call(this, viewer, "ControllerGamepad");
};

FORGE.ControllerGamepad.prototype = Object.create(FORGE.ControllerBase.prototype);
FORGE.ControllerGamepad.prototype.constructor = FORGE.ControllerGamepad;

/**
 * Default configuration
 * @name {FORGE.ControllerGamepad.DEFAULT_OPTIONS}
 * @type {ControllerKeyboardConfig}
 */
FORGE.ControllerGamepad.DEFAULT_OPTIONS = {
    orientation:
    {
        hardness: 0.6, //Hardness factor impatcing controller response to some instant force.
        damping: 0.15, //Damping factor controlling inertia.
        velocityMax: 300,
        invert: {
            x: false,
            y: false
        }
    },

    zoom:
    {
        hardness: 5,
        invert: false
    }
};

/**
 * Boot sequence.
 * @method FORGE.ControllerGamepad#_boot
 * @private
 */
FORGE.ControllerGamepad.prototype._boot = function()
{
    FORGE.ControllerBase.prototype._boot.call(this);

    this._type = FORGE.ControllerType.GAMEPAD;

    this._gamepads = [];

    this._buttonBindings = [];
    this._axisBindings = [];

    this._inertia = new THREE.Vector2();
    this._velocity = new THREE.Vector2();
    this._positionStart = new THREE.Vector2();
    this._positionCurrent = new THREE.Vector2();

    this._parseConfig(this._config);

    if (this._enabled === true)
    {
        this.enable();
    }

    this._viewer.gamepad.onGamepadConnected.add(this._onGamepadConnectedHandler, this);
    this._viewer.gamepad.onGamepadDisconnected.add(this._onGamepadDisconnectedHandler, this);
};

/**
 * Parse the configuration.
 * @method FORGE.ControllerGamepad#_parseConfig
 * @param {ControllerInstanceConfig} config - configuration object to parse.
 */
FORGE.ControllerGamepad.prototype._parseConfig = function(config)
{
    this._uid = config.uid;
    this._register();

    var options = config.options || {};

    this._orientation = /** @type {ControllerOrientationConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerGamepad.DEFAULT_OPTIONS.orientation, options.orientation));
    this._zoom = /** @type {ControllerZoomConfig} */ (FORGE.Utils.extendMultipleObjects(FORGE.ControllerGamepad.DEFAULT_OPTIONS.zoom, options.zoom));

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;

    if (options.default !== false)
    {
        this._addDefaultBindings();
    }

    if (typeof options.bindings !== "undefined")
    {
        if (typeof options.bindings.buttons !== "undefined" && Array.isArray(options.bindings.buttons) === true)
        {
            for (var i = 0, ii = options.bindings.buttons.length; i < ii; i++)
            {
                this._addButtonBinding(options.bindings.buttons[i]);
            }
        }

        if (typeof options.bindings.axes !== "undefined" && Array.isArray(options.bindings.axes) === true)
        {
            for (var i = 0, ii = options.bindings.axes.length; i < ii; i++)
            {
                this._addAxisBinding(options.bindings.axes[i]);
            }
        }
    }
};

/**
 * Add the default bindings ot the controller gamepad : direction with left stick, zoom with LT and
 * RT buttons (6 and 7 according to standard mapping).
 * @method FORGE.ControllerGamepad#_addDefaultBindings
 * @private
 */
FORGE.ControllerGamepad.prototype._addDefaultBindings = function()
{
    var bindingPlus = new FORGE.ButtonBinding(this._viewer,
        7,
        this._zoomDownHandler,
        this._zoomUpHandler,
        this._zoomHoldHandler,
        6,
        this,
        "plus"
    );
    this._buttonBindings.push(bindingPlus);

    var bindingMinus = new FORGE.ButtonBinding(this._viewer,
        6,
        this._zoomDownHandler,
        this._zoomUpHandler,
        this._zoomHoldHandler,
        7,
        this,
        "minus"
    );
    this._buttonBindings.push(bindingMinus);

    var bindingYaw = new FORGE.AxisBinding(this._viewer,
        0,
        this._yawChangeHandler,
        this,
        "yaw"
    );
    this._axisBindings.push(bindingYaw);

    var bindingPitch = new FORGE.AxisBinding(this._viewer,
        1,
        this._pitchChangeHandler,
        this,
        "pitch"
    );
    this._axisBindings.push(bindingPitch);
};

/**
 * Add a gamepad binding config to this controller.
 * @method FORGE.ControllerGamepad#_addButtonBinding
 * @param {ControllerGamepadButtonBindingConfig} binding - The binding config to add.
 * @private
 */
FORGE.ControllerGamepad.prototype._addButtonBinding = function(binding)
{
    var buttonsIn = binding.in;
    var buttonsOut = binding.out;
    var name = binding.name;
    var events = binding.events;

    if (FORGE.Utils.isTypeOf(buttonsIn, "number") === false && FORGE.Utils.isArrayOf(buttonsIn, "number") === false)
    {
        this.warn("Can't add custom gamepad binding, buttons in are invalid!");
        return;
    }

    if (typeof events !== "object" && events === null)
    {
        this.warn("Can't add custom gamepad binding, events are invalid!");
        return;
    }

    var buttonBinding = new FORGE.ButtonBinding(this._viewer, buttonsIn, events.onDown, events.onUp, events.onHold, buttonsOut, this, name);

    this._buttonBindings.push(buttonBinding);
};

/**
 * Add a gamepad axis binding config to this controller.
 * @method FORGE.ControllerGamepad#_addAxisBinding
 * @param {ControllerGamepadAxisBindingConfig} binding - the binding config to add
 * @private
 */
FORGE.ControllerGamepad.prototype._addAxisBinding = function(binding)
{
    var axis = binding.axis;
    var name = binding.name;
    var events = binding.events;

    if (FORGE.Utils.isTypeOf(axis, "number") === false && FORGE.Utils.isArrayOf(axis, "number") === false)
    {
        this.warn("Can't add custom gamepad binding, axis in is invalid!");
        return;
    }

    if (typeof events !== "object" && events === null)
    {
        this.warn("Can't add custom gamepad binding, events are invalid!");
        return;
    }

    var axisBinding = new FORGE.AxisBinding(this._viewer, axis, events.onChange, this, name);

    this._axisBindings.push(axisBinding);
};

/**
 * Event handler for yaw changement.
 * @method FORGE.ControllerGamepad#_yawChangeHandler
 * @param {FORGE.AxisBinding} binding - the reference to the binding
 * @param {number} value - the value of the change of the yaw
 * @private
 */
FORGE.ControllerGamepad.prototype._yawChangeHandler = function(binding, value)
{
    // Check the delta, as the value isn't exactly 0 at rest
    if (Math.abs(value) < 0.1)
    {
        this._velocity.setX(0);
        this._positionCurrent.setX(0);
        return;
    }

    this._positionCurrent.setX(180 * value);
};

/**
 * Event handler for pitch changement.
 * @method FORGE.ControllerGamepad#_pitchChangeHandler
 * @param {FORGE.AxisBinding} binding - the reference to the binding
 * @param {number} value - the value of the change of the pitch
 * @private
 */
FORGE.ControllerGamepad.prototype._pitchChangeHandler = function(binding, value)
{
    // Check the delta, as the value isn't exactly 0 at rest
    if (Math.abs(value) < 0.1)
    {
        this._velocity.setY(0);
        this._positionCurrent.setY(0);
        return;
    }

    this._positionCurrent.setY(180 * value);
};

/**
 * Event handler for zoom (+ / -) down handler.
 * @method FORGE.ControllerGamepad#_zoomDownHandler
 * @param  {FORGE.ButtonBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerGamepad.prototype._zoomDownHandler = function(binding)
{
    if (this._viewer.controllers.enabled === false)
    {
        return;
    }

    this._active = true;

    this._zoomProcessBinding(binding);

    if (this._onControlStart !== null)
    {
        this._onControlStart.dispatch();
    }

    this._viewer.controllers.notifyControlStart(this);
};

/**
 * Event handler for zoom (+ / -) hold handler.
 * @method FORGE.ControllerGamepad#_zoomHoldHandler
 * @param  {FORGE.ButtonBinding} binding - The binding associated to the event.
 * @private
 */
FORGE.ControllerGamepad.prototype._zoomHoldHandler = function(binding)
{
    this._zoomProcessBinding(binding);
};

/**
 * Event handler for zoom (+ / -) up handler.
 * @method FORGE.ControllerGamepad#_zoomUpHandler
 * @private
 */
FORGE.ControllerGamepad.prototype._zoomUpHandler = function()
{
    this._active = false;

    if (this._onControlEnd !== null)
    {
        this._onControlEnd.dispatch();
    }

    this._viewer.controllers.notifyControlEnd(this);
};

/**
 * Process a key binding related to zoom for down and hold zoom handlers.
 * @method FORGE.ControllerGamepad#_zoomProcessBinding
 * @param  {FORGE.ButtonBinding} binding - The key binding related to zoom
 * @private
 */
FORGE.ControllerGamepad.prototype._zoomProcessBinding = function(binding)
{
    var invert = this._zoom.invert ? 1 : -1;
    var delta = invert / this._zoom.hardness;

    switch (binding.name)
    {
        case "minus":
            delta *= 5;
            break;

        case "plus":
            delta *= -5;
            break;
    }

    this._camera.fov = this._camera.fov - delta;
};

/**
 * When a gamepad is connected, add bindings to it.
 * @method FORGE.ControllerGamepad#_onGamepadConnectedHandler
 * @param {FORGE.Gamepad} gamepad - the gamepad to add
 */
FORGE.ControllerGamepad.prototype._onGamepadConnectedHandler = function(gamepad)
{
    gamepad = gamepad.data;

    if (this._gamepads.indexOf(gamepad) === -1)
    {
        this._gamepads.push(gamepad);
    }

    var binding;

    for (var i = 0, ii = this._buttonBindings.length; i < ii; i++)
    {
        binding = this._buttonBindings[i];
        gamepad.addBinding(binding);
    }

    for (var i = 0, ii = this._axisBindings.length; i < ii; i++)
    {
        binding = this._axisBindings[i];
        gamepad.addBinding(binding);
    }
};

/**
 * When a gamepad is disconnected, remove bindings to it.
 * @method FORGE.ControllerGamepad#_onGamepadDisconnectedHandler
 * @param {string} name - the name of the gamepad to remove
 */
FORGE.ControllerGamepad.prototype._onGamepadDisconnectedHandler = function(name)
{
    name = name.data;

    var index;

    for (var i = 0, ii = this._gamepads.length; i < ii; i++)
    {
        if (this._gamepads[i].name === name)
        {
            index = i;
            break;
        }
    }

    if (index !== -1)
    {
        this._gamepads.splice(index, 1);
    }
};

/**
 * Enable controller
 * @method FORGE.ControllerGamepad#enable
 */
FORGE.ControllerGamepad.prototype.enable = function()
{
    FORGE.ControllerBase.prototype.enable.call(this);

    var binding;

    for (var i = 0, ii = this._buttonBindings.length; i < ii; i++)
    {
        binding = this._buttonBindings[i];

        for (var j = 0, jj = this._gamepads.length; j < jj; j++)
        {
            this._gamepads[j].addBinding(binding);
        }
    }

    for (var i = 0, ii = this._axisBindings.length; i < ii; i++)
    {
        binding = this._axisBindings[i];

        for (var j = 0, jj = this._gamepads.length; j < jj; j++)
        {
            this._gamepads[j].addBinding(binding);
        }
    }
};

/**
 * Disable controller
 * @method FORGE.ControllerGamepad#disable
 */
FORGE.ControllerGamepad.prototype.disable = function()
{
    FORGE.ControllerBase.prototype.disable.call(this);

    var binding;

    for (var i = 0, ii = this._buttonBindings.length; i < ii; i++)
    {
        binding = this._buttonBindings[i];

        for (var j = 0, jj = this._gamepads.length; j < jj; j++)
        {
            this._gamepads[j].removeBinding(binding);
        }
    }

    for (var i = 0, ii = this._axisBindings.length; i < ii; i++)
    {
        binding = this._axisBindings[i];

        for (var j = 0, jj = this._gamepads.length; j < jj; j++)
        {
            this._gamepads[j].removeBinding(binding);
        }
    }
};

/**
 * Update routine.
 * @method FORGE.ControllerGamepad#update
 */
FORGE.ControllerGamepad.prototype.update = function()
{
    var size = this._viewer.renderer.displayResolution;
    var hardness = 1 / (this._orientation.hardness * Math.min(size.width, size.height));

    this._velocity = this._positionCurrent.clone();

    if (this._velocity.length() > this._orientation.velocityMax)
    {
        this._velocity.setLength(this._orientation.velocityMax);
    }

    this._velocity.multiplyScalar(hardness);

    var dx = this._velocity.x + this._inertia.x;
    var dy = this._velocity.y + this._inertia.y;

    if (dx === 0 && dy === 0)
    {
        return;
    }

    var invert = this._orientation.invert;
    var invertX = (invert === true) ? -1 : (typeof invert === "object" && invert.x === true) ? -1 : 1;
    var invertY = (invert === true) ? -1 : (typeof invert === "object" && invert.y === true) ? -1 : 1;

    this._camera.yaw += invertX * dx;
    this._camera.pitch -= invertY * dy;

    // Damping 1 -> stops instantly, 0 infinite rebounds
    this._inertia.add(this._velocity).multiplyScalar(FORGE.Math.clamp(1 - this._orientation.damping, 0, 1));
};

/**
 * Destroy routine
 * @method FORGE.ControllerGamepad#destroy
 */
FORGE.ControllerGamepad.prototype.destroy = function()
{
    this._viewer.gamepad.onGamepadConnected.remove(this._onGamepadConnectedHandler, this);
    this._viewer.gamepad.onGamepadDisconnected.remove(this._onGamepadDisconnectedHandler, this);

    this._positionStart = null;
    this._positionCurrent = null;
    this._velocity = null;
    this._inertia = null;

    var binding;

    while (this._buttonBindings.length > 0)
    {
        binding = this._buttonBindings.pop();
        binding.destroy();
    }
    this._buttonBindings = null;

    while (this._axisBindings.length > 0)
    {
        binding = this._axisBindings.pop();
        binding.destroy();
    }
    this._axisBindings = null;

    FORGE.ControllerBase.prototype.destroy.call(this);
};


/**
 * A FORGE.SoundManager is an object to manage all sounds.
 *
 * @constructor FORGE.SoundManager
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 *
 * @todo Start/Stop sound to avoid autoplay
 */
FORGE.SoundManager = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.SoundManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The global sound config backup.
     * @name FORGE.SoundManager#_config
     * @type {?AudioConfig}
     * @private
     */
    this._config = null;

    /**
     * The global volume for sounds.
     * @name FORGE.SoundManager#_volume
     * @type {number}
     * @private
     */
    this._volume = 1;

    /**
     * The volume has been changed?
     * @name FORGE.SoundManager#_volumeChanged
     * @type {boolean}
     * @private
     */
    this._volumeChanged = false;

    /**
     * The default volume for sounds.
     * Can't be greater than the maximum volume.
     * @name FORGE.SoundManager#_defaultVolume
     * @type {number}
     * @private
     */
    this._defaultVolume = 1;

    /**
     * The maximum volume for sounds.
     * @name  FORGE.SoundManager#_maxVolume
     * @type {number}
     * @private
     */
    this._maxVolume = 1;

    /**
     * The save of the global volume for sounds before a mute.
     * @name FORGE.SoundManager#_mutedVolume
     * @type {number}
     * @private
     */
    this._mutedVolume = 1;

    /**
     * Are all sounds muted?
     * @name FORGE.SoundManager#_muted
     * @type {boolean}
     * @private
     */
    this._muted = false;

    /**
     * Is the sound manager enabled?
     * @name  FORGE.SoundManager#_enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Array of {@link FORGE.Sound}.
     * @name FORGE.SoundManager#_sounds
     * @type {?Array<FORGE.Sound>}
     * @private
     */
    this._sounds = null;

    /**
     * Is audio deactivated?
     * @name FORGE.SoundManager#_noAudio
     * @type {boolean}
     * @private
     */
    this._noAudio = false;

    /**
     * Is Audio tag activated?
     * @name FORGE.SoundManager#_useAudioTag
     * @type {boolean}
     * @private
     */
    this._useAudioTag = false;

    /**
     * Is WebAudio API activated?
     * @name FORGE.SoundManager#_useWebAudio
     * @type {boolean}
     * @private
     */
    this._useWebAudio = true;

    /**
     * Number of sound channels.
     * @name FORGE.SoundManager#_channels
     * @type {number}
     * @private
     */
    this._channels = 32;

    /**
     * The AudioContext interface.
     * @name FORGE.SoundManager#_context
     * @type {?(AudioContext|webkitAudioContext)}
     * @private
     */
    this._context = null;

    /**
     * The AudioContext state.
     * @name FORGE.SoundManager#_contextState
     * @type {string}
     * @private
     */
    this._contextState = "running";

    /**
     * AnalyserNode to expose audio time and frequency data and create data visualisations.
     * @name FORGE.SoundManager#_analyser
     * @type {AnalyserNode}
     * @private
     */
    this._analyser = null;

    /**
     * Master GainNode used to control the overall volume of the audio graph.
     * @name FORGE.SoundManager#_masterGain
     * @type {GainNode}
     * @private
     */
    this._masterGain = null;

    /**
     * On sounds muted event dispatcher.
     * @name FORGE.SoundManager#_onMute
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onMute = null;

     /**
     * On sounds unmuted event dispatcher.
     * @name FORGE.SoundManager#_onUnmute
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onUnmute = null;

    /**
     * On sounds volume change event dispatcher.
     * @name FORGE.SoundManager#_onVolumeChange
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onVolumeChange = null;

    /**
     * On sounds disabled event dispatcher.
     * @name FORGE.SoundManager#_onDisable
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onDisable = null;

    /**
     * On sounds enabled event dispatcher.
     * @name FORGE.SoundManager#_onEnable
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onEnable = null;

    FORGE.BaseObject.call(this, "SoundManager");
};

FORGE.SoundManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.SoundManager.prototype.constructor = FORGE.SoundManager;

/**
 * Boot sequence.
 * @method FORGE.SoundManager#_boot
 * @private
 * @suppress {deprecated}
 */
FORGE.SoundManager.prototype.boot = function()
{
    this._sounds = [];

    if (FORGE.Device.iOS && FORGE.Device.webAudio === false)
    {
        this._channels = 1; //only 1 channel in iOS with AudioTag support
    }

    if (FORGE.Device.webAudio === true)
    {
        try
        {
            if (typeof window.AudioContext === "function")
            {
                this._context = new window.AudioContext();
            }
            else
            {
                this._context = new window.webkitAudioContext();
            }
        }
        catch (error)
        {
            this._context = null;
            this._useWebAudio = false;
            this._noAudio = true;
        }
    }

    if (FORGE.Device.audioTag === true && this._context === null)
    {
        this._useWebAudio = false;
        this._useAudioTag = true;
        this._noAudio = false;
    }

    if (this._context !== null)
    {
        if (typeof this._context.createGain === "undefined")
        {
            this._masterGain = this._context.createGainNode();
        }
        else
        {
            this._masterGain = this._context.createGain();
        }

        this._analyser = this._context.createAnalyser();
        this._analyser.connect(this._masterGain);

        this._masterGain.gain.value = this._volume;
        this._masterGain.connect(this._context.destination);

        // The common coordinate system used with WebGL.
        // The listener is always facing down the negative Z axis, the
        // positive Y axis points up, the positive X axis points right.
        this._context.listener.setOrientation(0, 0, -1, 0, 1, 0);
    }

    this._viewer.story.onSceneLoadStart.add(this._sceneLoadStartHandler, this);
};

/**
 * Event handler for scene start.
 * @method FORGE.SoundManager#_sceneLoadStartHandler
 * @private
 */
FORGE.SoundManager.prototype._sceneLoadStartHandler = function()
{
    if(typeof this._viewer.story.scene.config.audio !== "undefined")
    {
        this._parseSceneConfig(this._viewer.story.scene.config.audio);
    }
    else
    {
        //restore global sounds config
        this._applyConfig(this._config);

        //Is the sound manager enabled?
        if(this._enabled === false)
        {
            if(this._onDisable !== null)
            {
                this._onDisable.dispatch();
            }
        }
        else
        {
            if(this._onEnable !== null)
            {
                this._onEnable.dispatch();
            }
        }
    }
};

/**
 * Parse the scene configuration part related to sounds.
 * @method  FORGE.SoundManager#_parseSceneConfig
 * @private
 * @param  {AudioConfig} config - The scene configuration part related to sounds.
 */
FORGE.SoundManager.prototype._parseSceneConfig = function(config)
{
    var extendedConfig = /** @type {AudioConfig} */ FORGE.Utils.extendMultipleObjects(this._config, config);
    this._applyConfig(extendedConfig);

    //Is the sound manager enabled?
    if(this._enabled === false)
    {
        if(this._onDisable !== null)
        {
            this._onDisable.dispatch();
        }
    }
    else
    {
        if(this._onEnable !== null)
        {
            this._onEnable.dispatch();
        }
    }
};

/**
 * Set values from configuration file.
 * @method  FORGE.SoundManager#_applyConfig
 * @param {?AudioConfig} config - The config file.
 * @private
 */
FORGE.SoundManager.prototype._applyConfig = function(config)
{
    if(config !== null)
    {
        this._enabled = typeof config.enabled !== "undefined" ? Boolean(config.enabled) : true;
        this._maxVolume = (typeof config.volume !== "undefined" && typeof config.volume.max === "number") ? FORGE.Math.clamp(config.volume.max, 0, 1) : 1;

        if(this._volumeChanged === false)
        {
            this._defaultVolume = (typeof config.volume !== "undefined" && typeof config.volume.default === "number") ? FORGE.Math.clamp(config.volume.default, 0, this._maxVolume) : 1;
        }
        else
        {
            this._defaultVolume = FORGE.Math.clamp(this._volume, 0, this._maxVolume);
        }
    }
};

/**
 * Add a sound config to the manager.
 * @method FORGE.SoundManager#addConfig
 * @param {AudioConfig} config - The config you want to add.
 */
FORGE.SoundManager.prototype.addConfig = function(config)
{
    this._parseConfig(config);

    this._initSounds();
};

/**
 * Parse a playlist config object.
 * @method FORGE.SoundManager#_parseConfig
 * @private
 * @param {AudioConfig} config - The config you want to parse.
 */
FORGE.SoundManager.prototype._parseConfig = function(config)
{
    this._config = config;

    this._applyConfig(config);
};

/**
 * Initialize the sounds manager.
 * @method FORGE.PlaylistManager#_initSounds
 * @private
 */
FORGE.SoundManager.prototype._initSounds = function()
{
    if(typeof this._defaultVolume === "number" && this._volumeChanged === false)
    {
        this._volume = FORGE.Math.clamp(this._defaultVolume, 0, 1);
    }

    this._updateVolume();
};

/**
 * Update method called by the viewer main loop.
 * @method FORGE.SoundManager#update
 */
FORGE.SoundManager.prototype.update = function ()
{
    for (var i = 0, ii = this._sounds.length; i < ii; i++)
    {
        this._sounds[i].update();
    }

    // update listener position
    this._setContextListenerOrientation();
};

/**
 * Add a {@link FORGE.Sound} into the _sounds Array.
 * @method FORGE.SoundManager#add
 * @param {FORGE.Sound} sound - The {@link FORGE.Sound} to add.
 */
FORGE.SoundManager.prototype.add = function (sound)
{
    var index = this._indexOfSound(sound);

    if(index === -1)
    {
        this._sounds.push(sound);
    }
};

/**
 * Remove a {@link FORGE.Sound} into the _sounds Array.
 * @method FORGE.SoundManager#remove
 * @param {FORGE.Sound} sound - The {@link FORGE.Sound} to remove.
 */
FORGE.SoundManager.prototype.remove = function (sound)
{
    var index = this._indexOfSound(sound);

    if(index > -1)
    {
        this._sounds.splice(index, 1);
    }
};

/**
 * Internal method to find a {@link FORGE.Sound} index in the _sounds Array.
 * @method FORGE.SoundManager#_indexOfSound
 * @private
 * @param {FORGE.Sound} sound - The {@link FORGE.Sound} itself.
 * @return {number} Returns the index of the searched {@link FORGE.Sound} if found, -1 if not.
 *
 * @todo Either the {@link FORGE.Sound} itself or its index or its uid. (FORGE.Sound|Number|String)
 */
FORGE.SoundManager.prototype._indexOfSound = function (sound)
{
    if(this._sounds === null)
    {
        return -1;
    }

    var _sound;

    for (var i = 0, ii = this._sounds.length; i < ii; i++)
    {
        _sound = this._sounds[i];

        if(_sound === sound)
        {
            return i;
        }
    }

    return -1;
};

/**
 * Suspend audio context if no sound are playing.
 * @method FORGE.SoundManager#suspend
 */
FORGE.SoundManager.prototype.suspend = function()
{
    if(this._context !== null && typeof this._context.suspend !== "undefined" && this._useWebAudio === true && FORGE.Device.safari === false)
    {
        if(this._contextState === "running")
        {
            var allStopped = true;
            for (var i = 0, ii = this._sounds.length; i < ii; i++)
            {
                if(this._sounds[i].playing === true || this._sounds[i].paused === true)
                {
                    allStopped = false;
                    break;
                }
            }

            if(allStopped === true)
            {
                this._contextState = "suspended";
                this._context.suspend();
            }
        }
    }
};

/**
 * Resume the audio context if at least one sound is playing.
 * @method FORGE.SoundManager#resume
 */
FORGE.SoundManager.prototype.resume = function()
{
    if(this._context !== null && typeof this._context.resume !== "undefined" && this._useWebAudio === true && FORGE.Device.safari === false)
    {
        if(this._contextState === "suspended")
        {
            for (var i = 0, ii = this._sounds.length; i < ii; i++)
            {
                if(this._sounds[i].playing === true || this._sounds[i].paused === true)
                {
                    this._contextState = "running";
                    this._context.resume();
                    break;
                }
            }
        }
    }
};

/**
 * Pause all playing sounds.
 * @method FORGE.SoundManager#pauseAll
 */
FORGE.SoundManager.prototype.pauseAll = function()
{
    for (var i = 0, ii = this._sounds.length; i < ii; i++)
    {
        if (this._sounds[i].playing === true)
        {
            this._sounds[i].pause();
            this._sounds[i].resumed = true;
        }
    }
};

/**
 * Play all sounds that have been paused with the pauseAll method.
 * @method FORGE.SoundManager#resumeAll
 */
FORGE.SoundManager.prototype.resumeAll = function()
{
    for (var i = 0, ii = this._sounds.length; i < ii; i++)
    {
        if (this._sounds[i].resumed === true)
        {
            this._sounds[i].resume();
            this._sounds[i].resumed = false;
        }
    }
};

/**
 * Mute method of the sounds.
 * @method FORGE.SoundManager#mute
 */
FORGE.SoundManager.prototype.mute = function()
{
    if(this._muted === true)
    {
        return;
    }

    this._muted = true;
    this._mutedVolume = this._volume;
    this._volume = 0;

    if (this._useWebAudio === true)
    {
        this._mutedVolume = this._masterGain.gain.value;
    }

    this._updateVolume();

    if(this._onMute !== null)
    {
        this._onMute.dispatch();
    }
};

/**
 * Unmute method of the sounds.
 * @method FORGE.SoundManager#unmute
 */
FORGE.SoundManager.prototype.unmute = function()
{
    if(this._muted === false)
    {
        return;
    }

    this._muted = false;
    this._volume = this._mutedVolume;

    this._updateVolume();

    if(this._onUnmute !== null)
    {
        this._onUnmute.dispatch();
    }
};

/**
 * Update volume method for the sounds.
 * @method FORGE.SoundManager#unmute
 * @private
 */
FORGE.SoundManager.prototype._updateVolume = function()
{
    if (this._useWebAudio === true)
    {
        this._masterGain.gain.value = this._volume;
    }
    else if (this._useAudioTag === true)
    {
        // Loop through the sound cache and change the volume of all html audio tags
        for (var i = 0; i < this._sounds.length; i++)
        {
            this._sounds[i]._sound.data.volume = FORGE.Math.clamp(this._sounds[i]._volume, 0, 1) * this._volume;
        }
    }
    this._volumeChanged = true;

    if(this._onVolumeChange !== null)
    {
        this._onVolumeChange.dispatch();
    }
};

/**
 * Change the sound manager orientation to follow the THREE perspective camera.
 * @method FORGE.SoundManager#_setContextListenerOrientation
 * @private
 */
FORGE.SoundManager.prototype._setContextListenerOrientation = function()
{
    if (this._useWebAudio === true && this._viewer.renderer.camera.main !== null)
    {
        var cameraDirection = new THREE.Vector3();
        var qCamera = this._viewer.renderer.camera.main.quaternion;

        // front vector indicating where the listener is facing to
        cameraDirection.set(0, 0, -1);
        cameraDirection.applyQuaternion(qCamera);
        var camera = cameraDirection.clone();

        // up vector repesenting the direction of the top of the listener head
        cameraDirection.set(0, 1, 0);
        cameraDirection.applyQuaternion(qCamera);
        var cameraUp = cameraDirection;

        // apply orientation values
        this._context.listener.setOrientation(camera.x, camera.y, camera.z, cameraUp.x, cameraUp.y, cameraUp.z);
    }
};

/**
 * Destroy sequence
 * @method FORGE.SoundManager#destroy
 */
FORGE.SoundManager.prototype.destroy = function()
{
    this._viewer.story.onSceneLoadStart.remove(this._sceneLoadStartHandler, this);

    this._viewer = null;
    this._config = null;

    var i = this._sounds.length;
    while(i--)
    {
        this._sounds[i].destroy();
    }

    this._sounds = null;

    this._context = null;
    this._analyser = null;
    this._masterGain = null;

    if(this._onMute !== null)
    {
        this._onMute.destroy();
        this._onMute = null;
    }

    if(this._onUnmute !== null)
    {
        this._onUnmute.destroy();
        this._onUnmute = null;
    }

    if(this._onVolumeChange !== null)
    {
        this._onVolumeChange.destroy();
        this._onVolumeChange = null;
    }

    if(this._onDisable !== null)
    {
        this._onDisable.destroy();
        this._onDisable = null;
    }

    if(this._onEnable !== null)
    {
        this._onEnable.destroy();
        this._onEnable = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};


/**
 * Get the audio context.
 * @name FORGE.SoundManager#context
 * @type {?AudioContext}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "context",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._context;
    }
});

/**
 * Get the analyser.
 * @name FORGE.SoundManager#analyser
 * @type {?AnalyserNode}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "analyser",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._analyser;
    }
});

/**
 * Get the analyser
 * @name  FORGE.SoundManager#inputNode
 * @type {?AudioDestinationNode}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "inputNode",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._analyser;
    }
});

/**
 * Get the masterGain.
 * @name FORGE.SoundManager#masterGain
 * @type {?GainNode}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "masterGain",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._masterGain;
    }
});

/**
 * The WebAudio API tag must be used?
 * @name FORGE.SoundManager#useWebAudio
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "useWebAudio",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._useWebAudio;
    },

    /** @this {FORGE.SoundManager} */
    set: function(value)
    {
        if(typeof value === "boolean")
        {
            this._useWebAudio = value;
            this._useAudioTag = !value;
        }
    }
});

/**
 * The Audio tag must be used?
 * @name FORGE.SoundManager#useAudioTag
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.SoundManager.prototype, "useAudioTag",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._useAudioTag;
    },

    /** @this {FORGE.SoundManager} */
    set: function(value)
    {
        if(typeof value === "boolean")
        {
            this._useAudioTag = value;
            this._useWebAudio = !value;
        }
    }
});

/**
 * Get the enabled state for sounds.
 * @name FORGE.SoundManager#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "enabled", {

    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._enabled;
    }

});

/**
 * Get or set the muted state for sounds.
 * @name FORGE.SoundManager#muted
 * @type {boolean}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "muted", {

    /** @this {FORGE.SoundManager} */
    get: function()
    {
        return this._muted;
    },

    /** @this {FORGE.SoundManager} */
    set: function(value)
    {
        if(typeof value === "boolean")
        {
            if (value === true)
            {
                this.mute();
            }
            else
            {
                this.unmute();
            }
        }
    }
});

/**
 * Get or set the global volume for sounds.
 * @name FORGE.SoundManager#volume
 * @type {number}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "volume", {

    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if (this._useWebAudio === true)
        {
            return this._masterGain.gain.value;
        }
        else
        {
            return this._volume;
        }
    },

    /** @this {FORGE.SoundManager} */
    set: function(value)
    {
        if(typeof value !== "number")
        {
            return;
        }

        value = FORGE.Math.clamp(value, 0, 1);
        if(this._maxVolume < value)
        {
            this._volume = this._maxVolume;
        }
        else
        {
            this._volume = value;
        }

        if (this._volume > 0)
        {
            this._muted = false;
        }

        this._updateVolume();
    }
});

/**
 * Get the sounds "onMute" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.SoundManager#onMute
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "onMute",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if(this._onMute === null)
        {
            this._onMute = new FORGE.EventDispatcher(this);
        }

        return this._onMute;
    }
});

/**
 * Get the sounds "onUnmute" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.SoundManager#onUnmute
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "onUnmute",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if(this._onUnmute === null)
        {
            this._onUnmute = new FORGE.EventDispatcher(this);
        }

        return this._onUnmute;
    }
});

/**
 * Get the sounds "onVolumeChange" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.SoundManager#onVolumeChange
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "onVolumeChange",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if(this._onVolumeChange === null)
        {
            this._onVolumeChange = new FORGE.EventDispatcher(this);
        }

        return this._onVolumeChange;
    }
});

/**
 * Get the sounds "onDisable" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.SoundManager#onDisable
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "onDisable",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if(this._onDisable === null)
        {
            this._onDisable = new FORGE.EventDispatcher(this);
        }

        return this._onDisable;
    }
});

/**
 * Get the sounds "onEnable" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.SoundManager#onDisable
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.SoundManager.prototype, "onEnable",
{
    /** @this {FORGE.SoundManager} */
    get: function()
    {
        if(this._onEnable === null)
        {
            this._onEnable = new FORGE.EventDispatcher(this);
        }

        return this._onEnable;
    }
});


/**
 * @namespace {Object} FORGE.SoundType
 */
FORGE.SoundType = {};

/**
 * @name FORGE.SoundType.STEREO
 * @type {string}
 * @const
 */
FORGE.SoundType.STEREO = "stereo";

/**
 * @name FORGE.SoundType.SPATIAL
 * @type {string}
 * @const
 */
FORGE.SoundType.SPATIAL = "3d";

/**
 * @name FORGE.SoundType.AMBISONIC
 * @type {string}
 * @const
 */
FORGE.SoundType.AMBISONIC = "ambisonic";
/**
 * A FORGE.Sound is an object that manages a sound.
 *
 * @constructor FORGE.Sound
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @param {string} key - The sound file id reference.
 * @param {string} url - The sound file url.
 * @param {boolean=} ambisonic - Is the sound ambisonic and need binaural rendering?
 * @extends {FORGE.BaseObject}
 *
 * @todo  Ability to force audio type into config
 * @todo  Make a test plugin that creates sound, add sound to the PluginObjectFactory
 * @todo  Loop during x steps (parameter) only if loop is true
 */
FORGE.Sound = function(viewer, key, url, ambisonic)
{
    /**
     * The viewer reference.
     * @name FORGE.Sound#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The sound identifier.
     * @name FORGE.Sound#_key
     * @type {string}
     * @private
     */
    this._key = key;

    /**
     * The sound file url.
     * @name FORGE.Sound#_url
     * @type {string}
     * @private
     */
    this._url = url;

    /**
     * The current volume of the sound.
     * @name FORGE.Sound#_volume
     * @type {number}
     * @private
     */
    this._volume = 1;

    /**
     * The volume level before a mute of the sound.
     * @name FORGE.Sound#_mutedVolume
     * @type {number}
     * @private
     */
    this._mutedVolume = 1;

    /**
     * The muted state of the sound.
     * @name FORGE.Sound#_muted
     * @type {boolean}
     * @private
     */
    this._muted = false;

    /**
     * Is the sound enabled?
     * @name  FORGE.Sound#enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Is the sound spatialized?
     * @name  FORGE.Sound#_spatialized
     * @type {boolean}
     * @private
     */
    this._spatialized = false;

    /**
     * The duration in seconds of the sound.
     * @name FORGE.Sound#_duration
     * @type {number}
     * @private
     */
    this._duration = 0;

    /**
     * The duration in milliseconds of the sound.
     * @name FORGE.Sound#_durationMS
     * @type {number}
     * @private
     */
    this._durationMS = 0;

    /**
     * The start time in milliseconds of the sound linked to the global clock.
     * @name FORGE.Sound#_startTime
     * @type {number}
     * @private
     */
    this._startTime = 0;

    /**
     * The pause time in milliseconds of the sound linked to the global clock.
     * @name FORGE.Sound#_pauseTime
     * @type {number}
     * @private
     */
    this._pauseTime = 0;

    /**
     * The current time in milliseconds of the sound.<br>
     * The current time value is based on duration minus time on the current clock because the AudioContext currentTime value is global to the audio context.
     * @name FORGE.Sound#_currentTime
     * @type {number}
     * @private
     */
    this._currentTime = 0;

    /**
     * The loop state of the sound.
     * @name FORGE.Sound#_loop
     * @type {boolean}
     * @private
     */
    this._loop = false;

    /**
     * The playing state of the sound.
     * @name FORGE.Sound#_playing
     * @type {boolean}
     * @private
     */
    this._playing = false;

    /**
     * The number of play of the sound.
     * @name FORGE.Sound#_playCount
     * @type {number}
     * @private
     */
    this._playCount = 0;

    /**
     * The paused state of the sound.
     * @name FORGE.Sound#_paused
     * @type {boolean}
     * @private
     */
    this._paused = false;

    /**
     * The resumed state of the sound.
     * @name FORGE.Sound#_resumed
     * @type {boolean}
     * @private
     */
    this._resumed = false;

    /**
     * The sound file with augmented properties.
     * @property {AudioBuffer} data The sound file data contained into an AudioBuffer.
     * @name FORGE.Sound#_soundFile
     * @type {?FORGE.File}
     * @private
     */
    this._soundFile = null;

    /**
     * The AudioBufferSourceNode instance used to play audio data contained within an AudioBuffer object.
     * @name FORGE.Sound#_sound
     * @type {?AudioBufferSourceNode}
     * @private
     */
    this._sound = null;

    /**
     * The sound file data contained into an AudioBuffer.
     * @name FORGE.Sound#_buffer
     * @type {?AudioBuffer}
     * @private
     */
    this._buffer = null;

    /**
     * The sound file position data contained into a PannerNode.
     * @name FORGE.Sound#_panner
     * @type {?AudioPannerNode}
     * @private
     */
    this._panner = null;

    /**
     * The AudioContext interface.
     * @name FORGE.Sound#_context
     * @type {?AudioContext}
     * @private
     */
    this._context = null;

    /**
     * The AudioDestinationNode representing the final destination of all audio in the context.
     * @name FORGE.Sound#_inputNode
     * @type {?AudioDestinationNode}
     * @private
     */
    this._inputNode = null;

    /**
     * The GainNode which can be used to control the overall volume of the audio graph.
     * @type {?GainNode}
     * @private
     */
    this._gainNode = null;

    /**
     * THREE Vector3 x coordinate.
     * @name  FORGE.Sound#_x
     * @type {number}
     * @private
     */
    this._x = 0;

    /**
     * THREE Vector3 y coordinate.
     * @name  FORGE.Sound#_y
     * @type {number}
     * @private
     */
    this._y = 0;

    /**
     * THREE Vector3 z coordinate.
     * @name  FORGE.Sound#_z
     * @type {number}
     * @private
     */
    this._z = 0;

    /**
     * FOADecoder is a ready-made FOA decoder and binaural renderer.
     * @name  FORGE.Sound#_decoder
     * @type {?FOADecoder}
     * @private
     */
    this._decoder = null;

    /**
     * Is it an ambisonical sound?
     * @name  FORGE.Sound#_ambisonic
     * @type {boolean}
     * @private
     */
    this._ambisonic = ambisonic || false;

    /**
     * Default channel map for ambisonic sound.
     * @name  FORGE.Sound#_defaultChannelMap
     * @type {Array<number>}
     * @private
     */
    this._defaultChannelMap = [0, 1, 2, 3]; //AMBIX
    // this._defaultChannelMap = [0, 3, 1, 2]; //FUMA

    /**
     * To save the pending state to be applied after the sound object will be ready.
     * @name FORGE.Sound#_pendingPlay
     * @type {boolean}
     * @private
     */
    this._pendingPlay = false;

    /**
     * Is this sound object is ready?
     * @name FORGE.Sound#_ready
     * @type {boolean}
     * @private
     */
    this._ready = false;

    /**
     * This is a reference to decodeComplete function but with a different this bind reference.
     * @name  FORGE.Sound#_decodeCompleteBind
     * @type {?Function}
     * @private
     */
    this._decodeCompleteBind = null;

    /**
     * This is a reference to decodeError function but with a different this bind reference.
     * @name  FORGE.Sound#_decodeErrorBind
     * @type {?Function}
     * @private
     */
    this._decodeErrorBind = null;

    /**
     * Is the sound decoded?
     * @name  FORGE.Sound#_decoded
     * @type {boolean}
     * @private
     */
    this._decoded = false;

    /**
     * On sound decoded event dispatcher.
     * @name FORGE.Sound#_onSoundDecode
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onSoundDecode = null;

    /**
     * On load start event dispatcher.
     * @name FORGE.Sound#_onLoadStart
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onLoadStart = null;

    /**
     * On loaded data event dispatcher.
     * @name FORGE.Sound#_onLoadedData
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onLoadedData = null;

    /**
     * On can play event dispatcher.
     * @name FORGE.Sound#_onCanPlay
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onCanPlay = null;

    /**
     * On can play through event dispatcher.
     * @name FORGE.Sound#_onCanPlayThrough
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onCanPlayThrough = null;

    /**
     * On sound muted event dispatcher.
     * @name FORGE.Sound#_onMute
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onMute = null;

    /**
     * On sound unmuted event dispatcher.
     * @name FORGE.Sound#_onUnmute
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onUnmute = null;

    /**
     * On sound volume change event dispatcher.
     * @name FORGE.Sound#_onVolumeChange
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onVolumeChange = null;

    /**
     * On sound play event dispatcher.
     * @name FORGE.Sound#_onPlay
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPlay = null;

    /**
     * On sound stop event dispatcher.
     * @name FORGE.Sound#_onStop
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onStop = null;

    /**
     * On sound pause event dispatcher.
     * @name FORGE.Sound#_onPause
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPause = null;

    /**
     * On sound resume event dispatcher.
     * @name FORGE.Sound#_onResume
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onResume = null;

    /**
     * On sound decoded event dispatcher.
     * @name FORGE.Sound#_onEnded
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onEnded = null;

    FORGE.BaseObject.call(this, "Sound");

    this._boot();
};

FORGE.Sound.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Sound.prototype.constructor = FORGE.Sound;

/**
 * Boot sequence.
 * @method FORGE.Sound#_boot
 * @private
 * @suppress {deprecated}
 */
FORGE.Sound.prototype._boot = function()
{
    if (this._ambisonic === true && this._isAmbisonic() === false)
    {
        this.log("FORGE.Sound: can't manage ambisonic sound without Google Chrome Omnitone library and WebAudio API.");
        this._ambisonic = false;
    }

    //register the uid
    this._uid = this._key;
    this._register();

    if (this._viewer.audio.useWebAudio === true)
    {
        this._context = this._viewer.audio.context;

        this._inputNode = this._viewer.audio.inputNode;

        if (typeof this._context.createGain === "undefined")
        {
            this._gainNode = this._context.createGainNode();
        }
        else
        {
            this._gainNode = this._context.createGain();
        }

        this._gainNode.gain.value = this._volume * this._viewer.audio.volume;
        this._gainNode.connect(this._inputNode);
    }
    if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
    {
        if (this._viewer.cache.has(FORGE.Cache.types.SOUND, this._key) === true)
        {
            this._loadComplete(this._viewer.cache.get(FORGE.Cache.types.SOUND, this._key));
        }

        //Listen to the main volume change to adapt the sound volume accordingly.
        this._viewer.audio.onVolumeChange.add(this._mainVolumeChangeHandler, this);
    }

    this._viewer.audio.onDisable.add(this._disableSoundHandler, this);

    if (this._url !== "")
    {
        this._viewer.load.sound(this._key, this._url, this._loadComplete, this, this._isAmbisonic());

        if (this._onLoadStart !== null)
        {
            this._onLoadStart.dispatch();
        }
    }

    this._viewer.audio.add(this);
};

/**
 * Event handler for load complete event, it launch the decoding of the sound file.
 * @method FORGE.Sound#_loadComplete
 * @private
 * @param {FORGE.File} file - The sound file.
 */
FORGE.Sound.prototype._loadComplete = function(file)
{
    // In some case, the sound is destroyed before the loading
    if(this._alive === false)
    {
        return;
    }

    this._soundFile = file;

    this._ready = true;

    // loaded events
    if (this._onLoadedData !== null)
    {
        this._onLoadedData.dispatch();
    }

    if (this._viewer.audio.useWebAudio === true)
    {
        this._decode(this._soundFile);
    }
    else
    {
        this._dispatchDecodedEvents();

        this._decodeComplete(null);
    }
};

/**
 * Decoding of the sound file.
 * @method FORGE.Sound#_decode
 * @private
 * @param {Object} file - The sound file
 */
FORGE.Sound.prototype._decode = function(file)
{
    if (file)
    {
        if (this._decoded === false)
        {
            this._decodeCompleteBind = this._decodeComplete.bind(this);
            this._decodeErrorBind = this._decodeError.bind(this);

            if (this._isAmbisonic() === true)
            {
                // FOA decoder and binaural renderer
                this._decoder = Omnitone.createFOADecoder(this._context, file.data,
                {
                    channelMap: this._defaultChannelMap
                    // HRTFSetUrl: 'YOUR_HRTF_SET_URL', //Base URL for the cube HRTF sets.
                    // postGainDB: 0, //Post-decoding gain compensation in dB.
                });

                // Initialize the decoder
                this._decoder.initialize().then(this._decodeCompleteBind, this._decodeErrorBind);
            }
            else
            {
                this._context.decodeAudioData(file.data, this._decodeCompleteBind, this._decodeErrorBind);
            }
        }
    }
};

/**
 * Event handler for decode error event, it stores decoding data into the sound file object.
 * @method FORGE.Sound#_decodeError
 * @private
 */
FORGE.Sound.prototype._decodeError = function()
{
    if (this._soundFile !== null)
    {
        this._soundFile.data = null;
        this._decoded = false;
    }
};

/**
 * Dispatcher for decoded events.
 * @method  FORGE.Sound#_dispatchDecodedEvents
 * @private
 */
FORGE.Sound.prototype._dispatchDecodedEvents = function()
{
    if (this._onSoundDecode !== null)
    {
        this._onSoundDecode.dispatch();
    }

    if (this._onCanPlay !== null)
    {
        this._onCanPlay.dispatch();
    }

    if (this._onCanPlayThrough !== null)
    {
        this._onCanPlayThrough.dispatch();
    }
};

/**
 * Event handler for decode complete event, it stores decoding data into the sound file object.
 * @method FORGE.Sound#_decodeComplete
 * @private
 * @param  {?AudioBuffer} buffer - The raw binary data buffer.
 */
FORGE.Sound.prototype._decodeComplete = function(buffer)
{
    if (this._soundFile === null)
    {
        this.log("FORGE.Sound._decodeComplete error, sound file is null");
        return;
    }

    if (buffer)
    {
        this._soundFile.data = buffer;
    }

    this._decoded = true;

    this._dispatchDecodedEvents();

    if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
    {
        this._sound = this._viewer.cache.get(FORGE.Cache.types.SOUND, this._key);

        this._updateVolume();

        this._duration = 0;
        if (this._sound.data.duration)
        {
            this._duration = this._sound.data.duration;
            this._durationMS = Math.ceil(this._duration * 1000);
        }
    }

    if (this._pendingPlay === true)
    {
        this.play(this._currentTime, this._loop, true);
    }
};

/**
 * Handles the main volume change, update the volume factor to the sound volume.
 * @method FORGE.Sound#_mainVolumeChangeHandler
 * @private
 */
FORGE.Sound.prototype._mainVolumeChangeHandler = function()
{
    this._updateVolume();
};

/**
 * Apply the main volume factor to the sound volume.
 * @method FORGE.Sound#_updateVolume
 * @private
 */
FORGE.Sound.prototype._updateVolume = function()
{
    if (this._sound !== null && (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true))
    {
        this._sound.data.volume = FORGE.Math.clamp(this._volume, 0, 1) * this._viewer.audio.volume;
    }
};

/**
 * Disable the sound.
 * @method FORGE.Sound#_disableSoundHandler
 * @private
 */
FORGE.Sound.prototype._disableSoundHandler = function()
{
    if (this._playing === true)
    {
        this.pause();
    }
    else if (this.paused === false)
    {
        this.stop();
    }
};

/**
 * Reset of the sound values.
 * @method FORGE.Sound#_reset
 * @private
 */
FORGE.Sound.prototype._reset = function()
{
    this._currentTime = 0;
    this._pendingPlay = false;
    this._playing = false;
    this._paused = false;
};

/**
 * Setup for sound panner.
 * @method FORGE.Sound#_setupPanner
 * @private
 */
FORGE.Sound.prototype._setupPanner = function()
{
    this._panner = this._context.createPanner();
    this._panner.panningModel = "HRTF";
    this._panner.distanceModel = "inverse";
    this._panner.refDistance = 1;
    this._panner.maxDistance = 10000;
    this._panner.rolloffFactor = 1;
    this._panner.coneInnerAngle = 360;
    this._panner.coneOuterAngle = 0;
    this._panner.coneOuterGain = 0;
    // look to listener position (x, y, z)
    this._panner.setOrientation(0, 0, 0);
    // init the 3D position of the panner (x, y, z)
    this._panner.setPosition(0, 0, 0);
};

/**
 * Apply sound panner orientation.
 * @method  FORGE.Sound#_applyPanner
 * @param {boolean=} connect - Panner must be connected to sound and gainNode?
 * @private
 */
FORGE.Sound.prototype._applyPanner = function(connect)
{
    if (this._panner === null)
    {
        this._setupPanner();
    }

    this._panner.setPosition(this._x, this._y, this._z);

    if (connect === true)
    {
        this._sound.connect(this._panner);
        // Connect the "panner" object to the "destination" object.
        this._panner.connect(this._gainNode);
    }
};

/**
 * Does the audio sound must be considered as ambisonic?
 * @method FORGE.Sound#_isAmbisonic
 * @return {boolean} Is ambisonic?
 * @private
 */
FORGE.Sound.prototype._isAmbisonic = function()
{
    return (this._ambisonic === true && this._viewer.audio.useWebAudio === true && typeof Omnitone !== "undefined");
};

/**
 * Update method called by the viewer main loop.
 * @method FORGE.Sound#update
 */
FORGE.Sound.prototype.update = function()
{
    if (this._playing === true && this._paused === false)
    {
        var time = this._viewer.clock.time - this._startTime;
        if (time >= this._durationMS)
        {
            this._currentTime = this._durationMS;

            if (this._viewer.audio.useWebAudio === true || this._viewer.audio.useAudioTag === true)
            {
                this.stop(true);

                if (this._onEnded !== null)
                {
                    this._onEnded.dispatch();
                }

                if (this._loop === true)
                {
                    this._currentTime = 0;
                    this.resume();
                }
            }
        }
        else
        {
            //also for this case when using streaming for data or bad headers : this._duration === Infinity
            this._currentTime = time;
        }
    }

    if (this._decoder !== null && this._playing === true)
    {
        // Rotate the binaural renderer based on a Three.js camera object.
        var m4 = this._viewer.renderer.camera.modelViewInverse;
        this._decoder.setRotationMatrixFromCamera(m4);
    }
};

/**
 * Play method of the sound.
 * @method FORGE.Sound#play
 * @param {number=} position - The start position to play the sound in milliseconds.
 * @param {?boolean=} loop - The loop state of the sound.
 * @param {?boolean=} forceRestart - If the sound is already playing you can set forceRestart to restart it from the beginning.
 * @suppress {deprecated}
 */
FORGE.Sound.prototype.play = function(position, loop, forceRestart)
{
    if (this._viewer.audio.enabled === false || this._enabled === false)
    {
        this._playing = false;
        this._paused = false;
        return;
    }

    this._loop = loop || this._loop;

    if (this._playing === true)
    {
        if (forceRestart === true)
        {
            this.stop();
        }
        else
        {
            return;
        }
    }

    if (this._ready === false)
    {
        this._pendingPlay = true;
        return;
    }

    if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
    {
        //  Does the sound need decoding?
        if (this._decoded === true)
        {
            //  Do we need to do this every time we play? How about just if the buffer is empty?
            if (this._buffer === null)
            {
                this._buffer = this._soundFile.data;
            }

            this._sound = this._context.createBufferSource();
            this._sound.buffer = this._buffer;

            if (this._spatialized === true)
            {
                this._applyPanner(true);
            }
            else
            {
                this._sound.connect(this._gainNode);
            }

            this._duration = this._sound.buffer.duration;
            this._durationMS = Math.ceil(this._duration * 1000);

            if (!isNaN(position) && position < this._durationMS)
            {
                this._startTime = this._viewer.clock.time - position;
            }
            else
            {
                position = 0;
                this._startTime = this._viewer.clock.time;
            }

            var time = FORGE.Math.round10(position / 1000);
            //  Useful to cache this somewhere perhaps?
            if (typeof this._sound.start === "undefined")
            {
                this._sound.noteGrainOn(0, time % this._duration, this._duration);
            }
            else
            {
                // Start playback, but make sure we stay in bound of the buffer.
                this._sound.start(0, time % this._duration, this._duration);
            }

            this._playing = true;
            this._currentTime = /** @type {number} */ (position);
            this._playCount++;

            if (this._onPlay !== null)
            {
                this._onPlay.dispatch();
            }
        }
        else
        {
            this.log("Sound is not decoded yet");
            this._pendingPlay = true;
            return;
        }
    }
    else if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
    {
        if (this._duration === 0)
        {
            this._duration = this._sound.data.duration;
            this._durationMS = Math.ceil(this._duration * 1000);
        }
        else if (this._duration === Infinity)
        {
            this._sound.data.loop = true;
        }

        if (!isNaN(position) && this._durationMS !== 0 && position < this._durationMS)
        {
            this._sound.data.currentTime = FORGE.Math.round10(position / 1000);
            this._startTime = this._viewer.clock.time - position;
        }
        else
        {
            position = 0;
            this._sound.data.currentTime = position;
            this._startTime = this._viewer.clock.time;
        }

        this._sound.data.play();

        this._playing = true;
        this._currentTime = /** @type {number} */ (position);
        this._playCount++;

        if (this._onPlay !== null)
        {
            this._onPlay.dispatch();
        }
    }

    this._viewer.audio.resume();
};

/**
 * Stop method of the sound.
 * @method FORGE.Sound#stop
 * @param  {boolean=} internal - Internal use: true prevents event firing.
 */
FORGE.Sound.prototype.stop = function(internal)
{
    if (this._sound !== null)
    {
        this._stop(true);

        this._pauseTime = this._viewer.clock.time;
        this._startTime = this._viewer.clock.time;
        this._pendingPlay = false;
        this._playing = false;
        this._paused = false;

        if (this._onStop !== null && internal === true)
        {
            this._onStop.dispatch();
        }

        this._viewer.audio.suspend();
    }
    else if (this._ready === false || this._decoded !== true)
    {
        this._reset();
    }
};

/**
 * Stop actions to apply to the sound.
 * @method FORGE.Sound#_stop
 * @private
 * @param {boolean} resetCurrentTime - To force a reset of the current time
 * @suppress {deprecated}
 */
FORGE.Sound.prototype._stop = function(resetCurrentTime)
{
    if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
    {
        if (typeof this._sound.stop === "undefined")
        {
            this._sound.noteOff(0);
        }
        else
        {
            try
            {
                this._sound.stop(0);
            }
            catch (e)
            {

            }
        }

        // Clean up the buffer source
        this._sound.disconnect(0);

        if (resetCurrentTime === true)
        {
            this._currentTime = 0;
        }
    }
    else if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
    {
        this._sound.data.pause();
        if (resetCurrentTime === true)
        {
            this._currentTime = 0;
            this._sound.data.currentTime = 0;
        }
    }
};

/**
 * Pause method of the sound.
 * @method FORGE.Sound#pause
 */
FORGE.Sound.prototype.pause = function()
{
    if (this._playing === true && this._sound !== null)
    {
        this._stop(false);

        this._paused = true;
        this._pauseTime = this._viewer.clock.time;
        this._pendingPlay = false;
        this._playing = false;

        if (this._onPause !== null)
        {
            this._onPause.dispatch();
        }

        this._viewer.audio.suspend();
    }
    else if (this._ready === false || this._decoded !== true)
    {
        this._reset();
    }
};

/**
 * Resume method of the sound.
 * @method FORGE.Sound#resume
 * @suppress {deprecated}
 */
FORGE.Sound.prototype.resume = function()
{
    if (this._viewer.audio.enabled === false || this._enabled === false)
    {
        return;
    }

    if (this._paused === true || this._playing === false || this._resumed === true)
    {
        if (this._sound === null)
        {
            this.play(this._currentTime, this._loop, true);
            return;
        }
        else
        {
            this._startTime = this._viewer.clock.time - (this._pauseTime - this._startTime);
            this._currentTime = this._viewer.clock.time - this._startTime; //force current time update
            var time = FORGE.Math.round10(this._currentTime / 1000);

            if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
            {
                this._sound = this._context.createBufferSource();
                this._sound.buffer = this._buffer;

                if (this._spatialized === true)
                {
                    this._applyPanner(true);
                }
                else
                {
                    this._sound.connect(this._gainNode);
                }

                var duration = Math.ceil((this._durationMS - this._currentTime) / 1000);

                if (typeof this._sound.start === "undefined")
                {
                    this._sound.noteGrainOn(0, time % this._duration, duration);
                }
                else
                {
                    this._sound.start(0, time % this._duration, duration);
                }
            }
            else if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
            {
                this._sound.data.currentTime = time;
                this._sound.data.play();
            }
        }

        this._playing = true;
        this._paused = false;

        if (this._onResume !== null)
        {
            this._onResume.dispatch();
        }

        this._viewer.audio.resume();
    }
    else if (this._ready === false || this._decoded === false)
    {
        this._reset();
        this._pendingPlay = true;
    }
};

/**
 * Mute method of the sound.
 * @method FORGE.Sound#mute
 */
FORGE.Sound.prototype.mute = function()
{
    if (this._muted === true || this._viewer.audio.enabled === false || this._enabled === false)
    {
        return;
    }

    this._muted = true;
    this._mutedVolume = this._volume;
    this._volume = 0;

    if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
    {
        this._gainNode.gain.value = this._volume;
    }
    else if ((this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true) && this._sound !== null)
    {
        this._sound.data.volume = this._volume;
    }

    if (this._onMute !== null)
    {
        this._onMute.dispatch();
    }
};

/**
 * Unmute method of the sound.
 * @method FORGE.Sound#unmute
 */
FORGE.Sound.prototype.unmute = function()
{
    if (this._muted === false || this._viewer.audio.enabled === false || this._enabled === false)
    {
        return;
    }

    this._muted = false;
    this._volume = this._mutedVolume;

    if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
    {
        this._gainNode.gain.value = this._mutedVolume;
    }
    else if ((this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true) && this._sound !== null)
    {
        this._sound.data.volume = this._mutedVolume;
    }

    if (this._onUnmute !== null)
    {
        this._onUnmute.dispatch();
    }
};

/**
 * Augmented destroy method.
 * @method FORGE.Sound#destroy
 */
FORGE.Sound.prototype.destroy = function()
{
    this.stop();

    if (this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true)
    {
        this._viewer.audio.onVolumeChange.remove(this._mainVolumeChangeHandler, this);
    }

    this._viewer.audio.onDisable.remove(this._disableSoundHandler, this);

    this._viewer.audio.remove(this);

    this._viewer = null;

    this._soundFile = null;
    this._sound = null;
    this._buffer = null;
    this._context = null;
    this._inputNode = null;
    this._gainNode = null;
    this._panner = null;

    this._decodeCompleteBind = null;
    this._decodeErrorBind = null;

    if (this._onSoundDecode !== null)
    {
        this._onSoundDecode.destroy();
        this._onSoundDecode = null;
    }

    if (this._onLoadStart !== null)
    {
        this._onLoadStart.destroy();
        this._onLoadStart = null;
    }

    if (this._onLoadedData !== null)
    {
        this._onLoadedData.destroy();
        this._onLoadedData = null;
    }

    if (this._onCanPlay !== null)
    {
        this._onCanPlay.destroy();
        this._onCanPlay = null;
    }

    if (this._onCanPlayThrough !== null)
    {
        this._onCanPlayThrough.destroy();
        this._onCanPlayThrough = null;
    }

    if (this._onMute !== null)
    {
        this._onMute.destroy();
        this._onMute = null;
    }

    if (this._onUnmute !== null)
    {
        this._onUnmute.destroy();
        this._onUnmute = null;
    }

    if (this._onVolumeChange !== null)
    {
        this._onVolumeChange.destroy();
        this._onVolumeChange = null;
    }

    if (this._onPlay !== null)
    {
        this._onPlay.destroy();
        this._onPlay = null;
    }

    if (this._onStop !== null)
    {
        this._onStop.destroy();
        this._onStop = null;
    }

    if (this._onPause !== null)
    {
        this._onPause.destroy();
        this._onPause = null;
    }

    if (this._onResume !== null)
    {
        this._onResume.destroy();
        this._onResume = null;
    }

    if (this._onEnded !== null)
    {
        this._onEnded.destroy();
        this._onEnded = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get and set the sound enabled status.
 * @name FORGE.Sound#enabled
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "enabled",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._enabled;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "boolean")
        {
            this._enabled = value;

            if (this._enabled === false && (this._playing === true || this._paused === true))
            {
                this.stop();
            }
        }
    }
});


/**
 * Get or set the current time in milliseconds of the sound.
 * @name FORGE.Sound#currentTime
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "currentTime",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._currentTime;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value !== "number")
        {
            return;
        }

        this.pause();
        this._currentTime = value;
        if (this._playing === true)
        {
            this.resume();
        }
    }
});

/**
 * Get or set the spatialized state of the sound.
 * @name FORGE.Sound#spatialized
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "spatialized",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._spatialized;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "boolean")
        {
            this._spatialized = value;
        }
    }
});

/**
 * Get the ambisonic state of the sound.
 * @name FORGE.Sound#ambisonic
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "ambisonic",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._ambisonic;
    }
});

/**
 * Get the duration in seconds of the sound.
 * @name FORGE.Sound#duration
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "duration",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._duration;
    }
});

/**
 * Get or set the muted state of the sound.
 * @name FORGE.Sound#muted
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "muted",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return (this._muted || this._viewer.audio.mute);
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "boolean")
        {
            if (value === true)
            {
                this.mute();
            }
            else
            {
                this.unmute();
            }
        }
    }
});

/**
 * Get or set the volume of the sound.
 * @name FORGE.Sound#volume
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "volume",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._volume;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value !== "number" || isNaN(value) === true)
        {
            return;
        }

        value = FORGE.Math.clamp(value, 0, 1);

        if (value === this._volume || this._viewer.audio.enabled === false || this._enabled === false)
        {
            return;
        }

        this._volume = value;

        if (this._volume > 0)
        {
            this._muted = false;
        }

        if (this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false)
        {
            this._gainNode.gain.value = value;
        }
        else
        {
            this._updateVolume();
        }

        if (this._onVolumeChange !== null)
        {
            this._onVolumeChange.dispatch();
        }
    }
});

/**
 * Get the decoded status of the sound.
 * @name FORGE.Sound#decoded
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "decoded",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._soundFile === null)
        {
            return false;
        }

        return this._decoded;
    }
});

/**
 * Get the number of play of the sound.
 * @name FORGE.Sound#playCount
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "playCount",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._playCount;
    }
});

/**
 * Get the playing status of the sound.
 * @name FORGE.Sound#playing
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "playing",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._playing;
    }
});

/**
 * Get the ready status of the sound.
 * @name FORGE.Sound#ready
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "ready",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._ready;
    }
});

/**
 * Get the paused status of the sound.
 * @name FORGE.Sound#paused
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "paused",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._paused;
    }
});

/**
 * Get/Set the resumed status of the sound.
 * @name FORGE.Sound#resumed
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "resumed",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._resumed;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "boolean")
        {
            this._resumed = value;
        }
    }
});

/**
 * Get and set the loop status of the sound.
 * @name FORGE.Sound#loop
 * @type {boolean}
 */
Object.defineProperty(FORGE.Sound.prototype, "loop",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._loop;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "boolean")
        {
            this._loop = value;

            if ((this._viewer.audio.useAudioTag === true || this._isAmbisonic() === true) && this._sound !== null && this._duration === Infinity)
            {
                this._sound.data.loop = this._loop;
            }
        }
    }
});

/**
 * Get and set the x axis position of the sound.
 * @name FORGE.Sound#x
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "x",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._x;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "number")
        {
            this._x = value;

            if ((this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false) && this._sound !== null && this._panner !== null && this._spatialized === true)
            {
                this._applyPanner();
            }
        }
    }
});

/**
 * Get and set the y axis position of the sound.
 * @name FORGE.Sound#y
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "y",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._y;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "number")
        {
            this._y = value;

            if ((this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false) && this._sound !== null && this._panner !== null && this._spatialized === true)
            {
                this._applyPanner();
            }
        }
    }
});

/**
 * Get and set the z axis position of the sound.
 * @name FORGE.Sound#z
 * @type {number}
 */
Object.defineProperty(FORGE.Sound.prototype, "z",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        return this._y;
    },

    /** @this {FORGE.Sound} */
    set: function(value)
    {
        if (typeof value === "number")
        {
            this._z = value;

            if ((this._viewer.audio.useWebAudio === true && this._isAmbisonic() === false) && this._sound !== null && this._panner !== null && this._spatialized === true)
            {
                this._applyPanner();
            }
        }
    }
});

/**
 * Get the sound "onSoundDecode" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onSoundDecode
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onSoundDecode",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onSoundDecode === null)
        {
            this._onSoundDecode = new FORGE.EventDispatcher(this);
        }

        return this._onSoundDecode;
    }
});

/**
 * Get the sound "onLoadStart" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onLoadStart
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onLoadStart",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onLoadStart === null)
        {
            this._onLoadStart = new FORGE.EventDispatcher(this);
        }

        return this._onLoadStart;
    }
});

/**
 * Get the sound "onLoadedData" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onLoadedData
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onLoadedData",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onLoadedData === null)
        {
            this._onLoadedData = new FORGE.EventDispatcher(this);
        }

        return this._onLoadedData;
    }
});

/**
 * Get the sound "onCanPlay" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onCanPlay
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onCanPlay",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onCanPlay === null)
        {
            this._onCanPlay = new FORGE.EventDispatcher(this);
        }

        return this._onCanPlay;
    }
});

/**
 * Get the sound "onCanPlayThrough" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onCanPlayThrough
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onCanPlayThrough",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onCanPlayThrough === null)
        {
            this._onCanPlayThrough = new FORGE.EventDispatcher(this);
        }

        return this._onCanPlayThrough;
    }
});

/**
 * Get the sound "onMute" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onMute
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onMute",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onMute === null)
        {
            this._onMute = new FORGE.EventDispatcher(this);
        }

        return this._onMute;
    }
});

/**
 * Get the sound "onUnmute" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onUnmute
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onUnmute",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onUnmute === null)
        {
            this._onUnmute = new FORGE.EventDispatcher(this);
        }

        return this._onUnmute;
    }
});

/**
 * Get the sound "onVolumeChange" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onVolumeChange
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onVolumeChange",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onVolumeChange === null)
        {
            this._onVolumeChange = new FORGE.EventDispatcher(this);
        }

        return this._onVolumeChange;
    }
});

/**
 * Get the sound "onPlay" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onPlay
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onPlay",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onPlay === null)
        {
            this._onPlay = new FORGE.EventDispatcher(this);
        }

        return this._onPlay;
    }
});

/**
 * Get the sound "onStop" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onStop
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onStop",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onStop === null)
        {
            this._onStop = new FORGE.EventDispatcher(this);
        }

        return this._onStop;
    }
});

/**
 * Get the sound "onPause" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onPause
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onPause",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onPause === null)
        {
            this._onPause = new FORGE.EventDispatcher(this);
        }

        return this._onPause;
    }
});

/**
 * Get the sound "onResume" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onResume
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onResume",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onResume === null)
        {
            this._onResume = new FORGE.EventDispatcher(this);
        }

        return this._onResume;
    }
});

/**
 * Get the sound "onEnded" event {@link FORGE.EventDispatcher}.
 * The {@link FORGE.EventDispatcher} is created only if you ask for it.
 * @name FORGE.Sound#onEnded
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Sound.prototype, "onEnded",
{
    /** @this {FORGE.Sound} */
    get: function()
    {
        if (this._onEnded === null)
        {
            this._onEnded = new FORGE.EventDispatcher(this);
        }

        return this._onEnded;
    }
});


/**
 * A FORGE.Playlist is an object that represents a list of media.
 *
 * @constructor FORGE.Playlist
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @param {AudioPlaylistConfig} config - The playlist config object.
 * @extends {FORGE.BaseObject}
 */
FORGE.Playlist = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.Playlist#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The config object.
     * @name FORGE.Playlist#_config
     * @type {?AudioPlaylistConfig}
     * @private
     */
    this._config = config;

    /**
     * Array of PlaylistTrack uid.
     * @name FORGE.Playlist#_tracks
     * @type {?Array<string>}
     * @private
     */
    this._tracks = null;

    /**
     * The name of the playlist.
     * @name FORGE.Playlist#_name
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._name = null;

    /**
     * The default track uid or index of the playlist.
     * @name FORGE.Playlist#_defaultTrack
     * @type {string}
     * @private
     */
    this._defaultTrack = "";

    /**
     * UID of the current {@link FORGE.PlaylistTrack}.
     * @name FORGE.Playlist#_trackUID
     * @type {string}
     * @private
     */
    this._trackUID = "";

    /**
     * The maximum volume for playlist.
     * @name  FORGE.Playlist#_maxVolume
     * @type {number}
     * @private
     */
    this._maxVolume = 1;

    /**
     * The current volume of the playlist.
     * Can't be greater than the maximum volume.
     * @name FORGE.Playlist#_volume
     * @type {number}
     * @private
     */
    this._volume = 1;

    /**
     * Does the playlist will auto play?
     * @name FORGE.Playlist#_autoPlay
     * @type {boolean}
     * @private
     */
    this._autoPlay = true;

    /**
     * Internal flag to know if the playlist must check the autoPlay property.
     * @name  FORGE.Playlist#_checkAutoPlay
     * @type {boolean}
     * @private
     */
    this._checkAutoPlay = true;

    /**
     * Does the playlist will loop?
     * @name FORGE.Playlist#_loop
     * @type {boolean}
     * @private
     */
    this._loop = true;

    /**
     * Playlist with tracks ready event dispatcher.
     * @name FORGE.Playlist#_onReady
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    /**
     * On playlist play event dispatcher.
     * @name FORGE.Playlist#_onPlay
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPlay = null;

    /**
     * On playlist stop event dispatcher.
     * @name FORGE.Playlist#_onStop
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onStop = null;

    /**
     * On playlist pause event dispatcher.
     * @name FORGE.Playlist#_onPause
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPause = null;

    /**
     * On playlist resume event dispatcher.
     * @name FORGE.Playlist#_onResume
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onResume = null;

    /**
     * On playlist ended event dispatcher.
     * @name FORGE.Playlist#_onEnded
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onEnded = null;

    FORGE.BaseObject.call(this, "Playlist");

    this._boot();
};

FORGE.Playlist.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Playlist.prototype.constructor = FORGE.Playlist;

/**
 * Boot sequence.
 * @method FORGE.Playlist#_boot
 * @private
 */
FORGE.Playlist.prototype._boot = function()
{
    this._uid = this._config.uid;
    this._register();
    this._name = new FORGE.LocaleString(this._viewer, this._config.name);

    this._tracks = [];

    this._parseConfig(this._config);
    this._parseTracks(this._config);
};

/**
 * Parse playlist config.
 * @method FORGE.Playlist#_parseConfig
 * @private
 * @param {AudioPlaylistConfig} config - The object that describes the playlist config.
 */
FORGE.Playlist.prototype._parseConfig = function(config)
{
    this._autoPlay = typeof config.autoPlay !== "undefined" ? Boolean(config.autoPlay) : true;
    this._loop = typeof config.loop !== "undefined" ? Boolean(config.loop) : true;

    this._maxVolume = (typeof config.volume !== "undefined" && typeof config.volume.max === "number") ? FORGE.Math.clamp(config.volume.max, 0, 1) : 1;
    this._volume = (typeof config.volume !== "undefined" && typeof config.volume.default === "number") ? FORGE.Math.clamp(config.volume.default, 0, this._maxVolume) : FORGE.Math.clamp(1, 0, this._maxVolume);
};

/**
 * Parse playlist tracks.
 * @method FORGE.Playlist#_parseTracks
 * @private
 * @param {AudioPlaylistConfig} config - The object that describes the playlist config.
 */
FORGE.Playlist.prototype._parseTracks = function(config)
{
    if(typeof config.tracks !== "undefined" && FORGE.Utils.isArrayOf(config.tracks, "string") === true)
    {
        this._tracks = config.tracks;
    }
    else
    {
        this.warn("A playlist has no track in its configuration, or configuration is not valid!");
    }

    //Parse the default track of the playlist
    if(typeof config.default === "string" && config.default !== "")
    {
        this._defaultTrack = config.default;
    }
    else if(typeof config.default === "number" && config.default >= 0 && config.default < this._tracks.length)
    {
        this._defaultTrack = this._tracks[config.default].uid;
    }
    else
    {
        this.warn("A playlist has a default track that is not in its tracks array!");
    }

    this._trackUID = this._defaultTrack;

    if(this._onReady !== null)
    {
        this._onReady.dispatch();
    }
};

/**
 * Event handler that triggers when the current track ends, set the next track to play.
 * @method FORGE.Playlist#_trackEndHandler
 * @private
 */
FORGE.Playlist.prototype._trackEndHandler = function()
{
    if(this._tracks.length > 0)
    {
        var index = this._tracks.indexOf(this._trackUID) + 1;

        // loop if playlist loop is activated and track doesn't loop.
        if(index === this._tracks.length && (this._loop === false || this.track.loop === true))
        {
            // reset to the first track of the playlist for next call to play
            this._trackUID = this._tracks[0];
            return;
        }
    }

    this.nextTrack();
};

/**
 * Dispatch play event to the track
 * @method  FORGE.Playlist#_notifyPlay
 * @private
 */
FORGE.Playlist.prototype._notifyPlay = function()
{
    if(this._onPlay !== null)
    {
        this._onPlay.dispatch();
    }
};

/**
 * Dispatch stop event to the track
 * @method  FORGE.Playlist#_notifyStop
 * @private
 */
FORGE.Playlist.prototype._notifyStop = function()
{
    if(this._onStop !== null)
    {
        this._onStop.dispatch();
    }
};

/**
 * Dispatch pause event to the track
 * @method  FORGE.Playlist#_notifyPause
 * @private
 */
FORGE.Playlist.prototype._notifyPause = function()
{
    if(this._onPause !== null)
    {
        this._onPause.dispatch();
    }
};

/**
 * Dispatch resume event to the track
 * @method  FORGE.Playlist#_notifyResume
 * @private
 */
FORGE.Playlist.prototype._notifyResume = function()
{
    if(this._onResume !== null)
    {
        this._onResume.dispatch();
    }
};

/**
 * Dispatch ended event to the track
 * @method  FORGE.Playlist#_notifyEnded
 * @private
 */
FORGE.Playlist.prototype._notifyEnded = function()
{
    if(this._onEnded !== null)
    {
        this._onEnded.dispatch();
    }
};

/**
 * Know if the playlist have any {@link FORGE.PlaylistTrack}.
 * @method FORGE.Playlist#hasTracks
 * @return {boolean} Returns true if the playlist has at least a {@link FORGE.PlaylistTrack}, false if not.
 */
FORGE.Playlist.prototype.hasTracks = function()
{
    return this._tracks.length !== 0;
};

/**
 * Play the current track or set a track to be the current one then play it.
 * @method FORGE.Playlist#play
 * @param {?FORGE.PlaylistTrack|string|number=} track - The track you want to play or its uid or its index, if undefined, play the current track.
 * @param {boolean=} checkAutoPlay - Does the autoPlay status must be checked?
 * @return {FORGE.PlaylistTrack} Returns the playing track.
 */
FORGE.Playlist.prototype.play = function(track, checkAutoPlay)
{
    this._checkAutoPlay = typeof checkAutoPlay !== "undefined" ? checkAutoPlay : false;

    var uid;

    if(typeof track === "string" && FORGE.UID.isTypeOf(track, "PlaylistTrack"))
    {
        uid = track;
    }
    else if(FORGE.Utils.isTypeOf(track, "PlaylistTrack"))
    {
        uid = track.uid;
    }
    else if (typeof track === "number" && track >= 0 && track < this._tracks.length)
    {
        uid = this._tracks[track];
    }
    else if(this._trackUID !== "")
    {
        uid = this._trackUID;
    }
    else if(typeof track === "undefined" || track === "")
    {
        uid = this._tracks[0];
    }

    if(typeof uid !== "undefined")
    {
        this.stop();

        this._trackUID = uid;

        if (this._checkAutoPlay === true && this._autoPlay === false)
        {
            this.warn("FORGE.Playlist.play(); autoPlay is disabled");
            return this.track;
        }

        this.track.play();

        if(this.track.onEnded.has(this._trackEndHandler, this) === false)
        {
            this.track.onEnded.add(this._trackEndHandler, this);
        }

        this.log("FORGE.Playlist.play(); [uid: "+this._trackUID+"]");

        return this.track;
    }

    return null;
};

/**
 * Stop the current track.
 * @method FORGE.Playlist#stop
 */
FORGE.Playlist.prototype.stop = function()
{
    if(this.track !== null)
    {
        this.track.stop();

        if(typeof this.track.onEnded !== "undefined" && this.track.onEnded.has(this._trackEndHandler, this) === false)
        {
            this.track.onEnded.remove(this._trackEndHandler, this);
        }
    }
};

/**
 * Pause the current track.
 * @method FORGE.Playlist#pause
 */
FORGE.Playlist.prototype.pause = function()
{
    if(this.track !== null)
    {
        this.track.pause();
    }
};

/**
 * Resume the current track if it's paused.
 * @method FORGE.Playlist#resume
 */
FORGE.Playlist.prototype.resume = function()
{
    if(this.track !== null)
    {
        this.track.resume();
    }
};

/**
 * Set the next {@link FORGE.PlaylistTrack} to be the current track.
 * If the playlist is paused, keep the pause status of the playlist.
 * @method FORGE.Playlist#nextTrack
 */
FORGE.Playlist.prototype.nextTrack = function()
{
    var index = -1;

    if(this._tracks.length > 0)
    {
        index = this._tracks.indexOf(this._trackUID) + 1;

        if(index === this._tracks.length)
        {
            index = 0;
        }
    }

    this.play(index, this._checkAutoPlay);

    if(this.paused === true)
    {
        this.pause();
    }
};

/**
 * Set the previous {@link FORGE.PlaylistTrack} to be the current track.
 * If the playlist is paused, keep the pause status of the playlist.
 * @method FORGE.Playlist#previousTrack
 */
FORGE.Playlist.prototype.previousTrack = function()
{
    var index = -1;

    if(this._tracks.length > 0)
    {
        index = this._tracks.indexOf(this._trackUID) - 1;

        if(index === -1)
        {
            index = this._tracks.length - 1;
        }
    }

    this.play(index, this._checkAutoPlay);

    if(this.paused === true)
    {
        this.pause();
    }
};

/**
 * Destroy sequence
 * @method FORGE.Playlist#destroy
 */
FORGE.Playlist.prototype.destroy = function()
{
    this.stop();

    this._viewer = null;

    this._name.destroy();
    this._name = null;

    if(this._tracks.length > 0)
    {
        this._tracks.length = 0;
        this._tracks = null;
    }

    this._config = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the list of tracks of this playlist.
 * @name FORGE.Playlist#tracks
 * @readonly
 * @type {Array<string>}
 */
Object.defineProperty(FORGE.Playlist.prototype, "tracks",
{
    /** @this {FORGE.Playlist} */
    get: function ()
    {
        return this._tracks;
    }
});

/**
 * Get the name of this playlist.
 * @name FORGE.Playlist#name
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Playlist.prototype, "name",
{
    /** @this {FORGE.Playlist} */
    get: function ()
    {
        return this._name.value;
    }
});

/**
 * Get the ready status of this playlist current track.
 * @name FORGE.Playlist#ready
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "ready",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this.track !== null)
        {
            return this.track.ready;
        }

        return false;
    }
});

/**
 * Get the decoded status of this playlist current track.
 * @name FORGE.Playlist#decoded
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "decoded",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this.track !== null)
        {
            return this.track.decoded;
        }

        return false;
    }
});

/**
 * Get the playing status of this playlist current track.
 * @name FORGE.Playlist#playing
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "playing",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this.track !== null)
        {
            return this.track.playing;
        }

        return false;
    }
});

/**
 * Get the paused status of this playlist current track.
 * @name FORGE.Playlist#paused
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "paused",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this.track !== null)
        {
            return this.track.paused;
        }

        return false;
    }
});

/**
 * Get the current {@link FORGE.PlaylistTrack} of this playlist.
 * @name FORGE.Playlist#track
 * @readonly
 * @type {FORGE.PlaylistTrack}
 */
Object.defineProperty(FORGE.Playlist.prototype, "track",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(FORGE.UID.isTypeOf(this._trackUID, "PlaylistTrack") === true)
        {
            return FORGE.UID.get(this._trackUID, "PlaylistTrack");
        }

        return null;
    }
});

/**
 * Get and set the loop state of the playlist.
 * @name FORGE.Playlist#loop
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "loop",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        return this._loop;
    },

    /** @this {FORGE.Playlist} */
    set: function(value)
    {
        this._loop = Boolean(value);
    }
});

/**
 * Get and set the auto play state of the playlist.
 * @name FORGE.Playlist#autoPlay
 * @type {boolean}
 */
Object.defineProperty(FORGE.Playlist.prototype, "autoPlay",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        return this._autoPlay;
    },

    /** @this {FORGE.Playlist} */
    set: function(value)
    {
        this._autoPlay = Boolean(value);
    }
});

/**
 * Get the playlist volume.
 * @name FORGE.Playlist#volume
 * @type {number}
 */
Object.defineProperty(FORGE.Playlist.prototype, "volume",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        return this._volume;
    },

    /** @this {FORGE.Playlist} */
    set: function(value)
    {
        if(typeof value !== "number")
        {
            return;
        }

        value = FORGE.Math.clamp(value, 0, 1);
        if(this._maxVolume < value)
        {
            this._volume = this._maxVolume;
        }
        else
        {
            this._volume = value;
        }

        if(FORGE.UID.isTypeOf(this._trackUID, "PlaylistTrack") === true)
        {
            var track = FORGE.UID.get(this._trackUID, "PlaylistTrack");

            if(track !== null)
            {
                track.volume = this._volume * this._viewer.playlists.volume;
            }
        }
    }
});

/**
 * Get the "onPlay" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onPlay
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onPlay",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onPlay === null)
        {
            this._onPlay = new FORGE.EventDispatcher(this);
        }

        return this._onPlay;
    }
});

/**
 * Get the "onStop" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onStop
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onStop",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onStop === null)
        {
            this._onStop = new FORGE.EventDispatcher(this);
        }

        return this._onStop;
    }
});

/**
 * Get the "onPause" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onPause
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onPause",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onPause === null)
        {
            this._onPause = new FORGE.EventDispatcher(this);
        }

        return this._onPause;
    }
});

/**
 * Get the "onResume" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onResume
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onResume",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onResume === null)
        {
            this._onResume = new FORGE.EventDispatcher(this);
        }

        return this._onResume;
    }
});

/**
 * Get the "onEnded" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onEnded
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onEnded",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onEnded === null)
        {
            this._onEnded = new FORGE.EventDispatcher(this);
        }

        return this._onEnded;
    }
});

/**
 * Get the "onReady" {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.Playlist#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Playlist.prototype, "onReady",
{
    /** @this {FORGE.Playlist} */
    get: function()
    {
        if(this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});


/**
 * The FORGE.PlaylistManager is an object that manages playlists of the project.
 *
 * @constructor FORGE.PlaylistManager
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 *
 * @todo  previous/next playlist
 * @todo  real keepAudio to resume a "lost sound"
 * @todo  preload of all sounds
 */
FORGE.PlaylistManager = function(viewer)
{
    /**
     * The viewer reference.
     * @name FORGE.PlaylistManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The general config backup.
     * @name FORGE.PalylistManager#_config
     * @type {?AudioPlaylistsConfig}
     * @private
     */
    this._config = null;

    /**
     * Array of {@link FORGE.Playlist}.
     * @name FORGE.PlaylistManager#_playlists
     * @type {?Array<FORGE.Playlist>}
     * @private
     */
    this._playlists = null;

    /**
     * The tracks list object.
     * @name  FORGE.PlaylistManager#_tracks
     * @type {?Array<FORGE.PlaylistTrack>}
     * @private
     */
    this._tracks = null;

    /**
     * Uid of the current {@link FORGE.Playlist}.
     * @name  FORGE.PlaylistManager#_playlistUID
     * @type {string}
     * @private
     */
    this._playlistUID = "";

    /**
     * The default playlist uid.
     * @name FORGE.PlaylistManager#_defaultList
     * @type {string}
     * @private
     */
    this._defaultList = "";

    /**
     * The maximum volume for all playlists.
     * @name  FORGE.PlaylistManager#_maxVolume
     * @type {number}
     * @private
     */
    this._maxVolume = 1;

    /**
     * The current volume for all the playlists.
     * @name FORGE.PlaylistManager#_volume
     * @type {number}
     * @private
     */
    this._volume = 1;

    /**
     * Is the playlist manager enabled?
     * @name  FORGE.PlaylistManager#_enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Tracks must be preloaded?
     * @name  FORGE.PlaylistManager#_preload
     * @type {boolean}
     * @private
     */
    this._preload = false;

    /**
     * Is the playlistManager paused?
     * @name  FORGE.PlaylistManager#_paused
     * @type {boolean}
     * @private
     */
    this._paused = false;

    /**
     * On playlist manager ready event dispatcher.
     * @name FORGE.Sound#_onReady
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    /**
     * On playlist manager play event dispatcher.
     * @name FORGE.PlaylistManager#_onPlay
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPlay = null;

    /**
     * On playlist manager stop event dispatcher.
     * @name FORGE.PlaylistManager#_onStop
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onStop = null;

    /**
     * On playlist manager pause event dispatcher.
     * @name FORGE.PlaylistManager#_onPause
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onPause = null;

    /**
     * On playlist manager resume event dispatcher.
     * @name FORGE.PlaylistManager#_onResume
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onResume = null;

    /**
     * On playlist manager ended event dispatcher.
     * @name FORGE.PlaylistManager#_onEnded
     * @type {?FORGE.EventDispatcher}
     * @private
     */
    this._onEnded = null;

    FORGE.BaseObject.call(this, "PlaylistManager");

};

FORGE.PlaylistManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PlaylistManager.prototype.constructor = FORGE.PlaylistManager;

/**
 * Boot sequence.
 * @method FORGE.PlaylistManager#boot
 */
FORGE.PlaylistManager.prototype.boot = function()
{
    this._playlists = [];
    this._tracks = [];

    this._viewer.audio.onEnable.add(this._enableSoundHandler, this);
    this._viewer.story.onSceneLoadStart.add(this._sceneLoadStartHandler, this);
};

/**
 * Enable handler the sound manager.
 * @method FORGE.PlaylistManager#_enableSoundHandler
 * @private
 */
FORGE.PlaylistManager.prototype._enableSoundHandler = function()
{
    if(this._enabled === true && this._playlists.length > 0 && this.ready === true && this.decoded === true && this._paused === false)
    {
        if(this.paused === true)
        {
            this.resume();
        }
        else if(this.playing === false)
        {
            this.play();
        }
    }
};

/**
 * Event handler for scene start.
 * @method FORGE.PlaylistManager#_sceneLoadStartHandler
 * @private
 */
FORGE.PlaylistManager.prototype._sceneLoadStartHandler = function()
{
    if(typeof this._viewer.story.scene.config.playlists !== "undefined")
    {
        this._parseSceneConfig(this._viewer.story.scene.config.playlists);
    }
    else
    {
        //restore global playlists config
        this._applyConfig(this._config);

        if(this._paused === false && this._defaultList !== this._playlistUID)
        {
            if(this._enabled === false || this.ready === false || this.decoded === false || this.playing === true)
            {
                this._stopScenePlaylist();
            }

            //load the default playlist of the global audio if exists
            this._startScenePlaylist(this._defaultList);
        }
    }
};

/**
 * Parse the scene configuration part related to playlist.
 * @method  FORGE.PlaylistManager#_parseSceneConfig
 * @private
 * @param  {AudioPlaylistsConfig} config - The scene configuration part related to playlist.
 */
FORGE.PlaylistManager.prototype._parseSceneConfig = function(config)
{
    var extendedConfig = /** @type {AudioPlaylistsConfig} */ FORGE.Utils.extendMultipleObjects(this._config, config);
    this._applyConfig(extendedConfig);

    if(this._paused === false)
    {
        if(this._enabled === false)
        {
            this._stopScenePlaylist();
        }
        else
        {
            if(!this._startScenePlaylist(this._defaultList))
            {
                if(this.playing === true)
                {
                    this._stopScenePlaylist();
                }
            }
        }
    }
};

/**
 * Set values from configuration file.
 * @method  FORGE.PlaylistManager#_applyConfig
 * @param {?AudioPlaylistsConfig} config - The config file.
 * @private
 */
FORGE.PlaylistManager.prototype._applyConfig = function(config)
{
    if(config !== null)
    {
        this._enabled = typeof config.enabled !== "undefined" ? Boolean(config.enabled) : true;

        if (typeof config.default !== "undefined")
        {
            if (typeof config.default === "string" && config.default !== "")
            {
                this._defaultList = config.default;
            }
            else if (typeof config.default === "number" && config.default >= 0 && config.default < this._playlists.length)
            {
                this._defaultList = this._playlists[config.default].uid;
            }
        }

        this._maxVolume = (typeof config.volume !== "undefined" && typeof config.volume.max === "number") ? FORGE.Math.clamp(config.volume.max, 0, 1) : 1;
        this._volume = (typeof config.volume !== "undefined" && typeof config.volume.default === "number") ? FORGE.Math.clamp(config.volume.default, 0, this._maxVolume) : FORGE.Math.clamp(1, 0, this._maxVolume);
    }
};

/**
 * Add a playlist config to the manager.
 * @method FORGE.PlaylistManager#addConfig
 * @param {AudioPlaylistsConfig} config - The config you want to add.
 */
FORGE.PlaylistManager.prototype.addConfig = function(config)
{
    this._parseConfig(config);

    this._initPlaylist();
};

/**
 * Parse a playlist config object.
 * @method FORGE.PlaylistManager#_parseConfig
 * @private
 * @param {AudioPlaylistsConfig} config - The config you want to parse.
 */
FORGE.PlaylistManager.prototype._parseConfig = function(config)
{
    var playlist, track;

    this._config = config;

    if(typeof config.lists !== "undefined")
    {
        for(var i = 0, ii = config.lists.length; i<ii; i++)
        {
            playlist = new FORGE.Playlist(this._viewer, config.lists[i]);
            this.add(playlist);
        }
    }

    if(typeof config.tracks !== "undefined")
    {
        for(var j = 0, jj = config.tracks.length; j<jj; j++)
        {
            track = new FORGE.PlaylistTrack(this._viewer, config.tracks[j]);
            this.addTrack(track);

            if(this._preload === true)
            {
                // @todo manage preload queue
                this.warn("Preload is not supported yet.");
            }
        }
    }

    this._applyConfig(config);

    if(typeof config.lists !== "undefined" && config.lists.length > 0)
    {
        if (this._defaultList === "")
        {
            this.warn("The playlist manager has a default playlist that is not in its playlists array!");
        }

        this._playlistUID = this._defaultList;
    }
};

/**
 * Initialize the default playlist.
 * @method FORGE.PlaylistManager#_initPlaylist
 * @private
 */
FORGE.PlaylistManager.prototype._initPlaylist = function()
{
    // If the page is not visible at init, report it later
    if (document[FORGE.Device.visibilityState] !== "visible")
    {
        this._viewer.onResume.addOnce(this._initPlaylist, this);
        return;
    }

    if(this._playlists.length <= 0)
    {
        return;
    }

    var uid;
    if(typeof this._defaultList === "string" && this._defaultList !== "")
    {
        uid = this._defaultList;
    }
    else
    {
        uid = this._playlists[0].uid;
    }

    if(FORGE.UID.get(uid) === undefined)
    {
        this.warn("PlaylistManager : uid \""+uid+"\" is not into playlists");
    }

    if(FORGE.UID.isTypeOf(uid, "Playlist") === true)
    {
        this.play(uid, true);
    }
    else
    {
        this.warn("Impossible to play the playlist with uid "+uid+", it doesn't seem to be a playlist!");
    }
};

/**
 * Start or resume a playlist for a specific scene.
 * @method  FORGE.PlaylistManager#_startScenePlaylist
 * @param {string} playlistUID - The default playlist uid.
 * @return {boolean} Returns true if the playlist is found.
 * @private
 */
FORGE.PlaylistManager.prototype._startScenePlaylist = function(playlistUID)
{
    if(playlistUID !== null && FORGE.UID.isTypeOf(playlistUID, "Playlist") === true)
    {
        //Maybe the new scene shares the same playlist so we do not reset the playback
        if(this.playlist !== null && this.playlist.uid === playlistUID)
        {
            if(this.playlist.track !== null && (typeof this.playlist.track.uid !== "undefined" && this.playlist.track.uid !== ""))
            {
                if(this.paused === true)
                {
                    this.resume();
                }
                else if(this.playing === false)
                {
                    this.play(playlistUID, true);
                }
            }
            else
            {
                this.stop();
                this.play(playlistUID, true);
            }
        }
        else
        {
            this.stop();
            this.play(playlistUID, true);
        }

        return true;
    }

    return false;
};

/**
 * Stop or pause a playlist.
 * @method  FORGE.PlaylistManager#_stopScenePlaylist
 * @private
 */
FORGE.PlaylistManager.prototype._stopScenePlaylist = function()
{
    if(this.playing === true)
    {
        this.pause();
    }
    else
    {
        this.stop();
    }
};

/**
 * Dispatch play event and notify it to the playlist
 * @method  FORGE.PlaylistManager#_notifyPlay
 * @private
 */
FORGE.PlaylistManager.prototype._notifyPlay = function()
{
    if(this._onPlay !== null)
    {
        this._onPlay.dispatch();
    }

    if(this.playlist !== null)
    {
        FORGE.Playlist.prototype._notifyPlay.call(this.playlist);
    }
};

/**
 * Dispatch stop event and notify it to the playlist
 * @method  FORGE.PlaylistManager#_notifyStop
 * @private
 */
FORGE.PlaylistManager.prototype._notifyStop = function()
{
    if(this._onStop !== null)
    {
        this._onStop.dispatch();
    }

    if(this.playlist !== null)
    {
        FORGE.Playlist.prototype._notifyStop.call(this.playlist);
    }
};

/**
 * Dispatch pause event and notify it to the playlist
 * @method  FORGE.PlaylistManager#_notifyPause
 * @private
 */
FORGE.PlaylistManager.prototype._notifyPause = function()
{
    if(this._onPause !== null)
    {
        this._onPause.dispatch();
    }

    if(this.playlist !== null)
    {
        FORGE.Playlist.prototype._notifyPause.call(this.playlist);
    }
};

/**
 * Dispatch resume event and notify it to the playlist
 * @method  FORGE.PlaylistManager#_notifyResume
 * @private
 */
FORGE.PlaylistManager.prototype._notifyResume = function()
{
    if(this._onResume !== null)
    {
        this._onResume.dispatch();
    }

    if(this.playlist !== null)
    {
        FORGE.Playlist.prototype._notifyResume.call(this.playlist);
    }
};

/**
 * Dispatch ended event and notify it to the playlist
 * @method  FORGE.PlaylistManager#_notifyEnded
 * @private
 */
FORGE.PlaylistManager.prototype._notifyEnded = function()
{
    if(this._onEnded !== null)
    {
        this._onEnded.dispatch();
    }

    if(this.playlist !== null)
    {
        FORGE.Playlist.prototype._notifyEnded.call(this.playlist);
    }
};

/**
 * Add a {@link FORGE.Playlist} to the playlist manager.
 * @method  FORGE.PlaylistManager#add
 * @param {FORGE.Playlist} playlist - The {@link FORGE.Playlist} you want to add.
 */
FORGE.PlaylistManager.prototype.add = function(playlist)
{
    this._playlists.push(playlist);
};

/**
 * Add a {@link FORGE.PlaylistTrack} to the playlist manager.
 * @method  FORGE.PlaylistManager#addTrack
 * @param {FORGE.PlaylistTrack} track - The {@link FORGE.PlaylistTrack} you want to add.
 */
FORGE.PlaylistManager.prototype.addTrack = function(track)
{
    this._tracks.push(track);
};

/**
 * Know if the project have any {@link FORGE.Playlist}.
 * @method FORGE.PlaylistManager#hasPlaylists
 * @return {boolean} Returns true if the project has at least a {@link FORGE.Playlist}, false if not.
 */
FORGE.PlaylistManager.prototype.hasPlaylists = function()
{
    return this._playlists.length !== 0;
};

/**
 * Know if the project have any {@link FORGE.PlaylistTrack}.
 * @method FORGE.PlaylistManager#hasTracks
 * @return {boolean} Returns true if the project has at least a {@link FORGE.PlaylistTrack}, false if not.
 */
FORGE.PlaylistManager.prototype.hasTracks = function()
{
    return this._tracks.length !== 0;
};

/**
 * Play the current playlist or a specific one at a specific track.
 * @method  FORGE.PlaylistManager#play
 * @param  {FORGE.Playlist|string|number=} playlist - The {@link FORGE.Playlist} you want to play or its uid or its index.
 * @param {boolean=} checkAutoPlay - Set to true if the autoPlay status of the playlist must be checked.
 */
FORGE.PlaylistManager.prototype.play = function(playlist, checkAutoPlay)
{
    var uid;

    if(typeof playlist === "string" && FORGE.UID.isTypeOf(playlist, "Playlist"))
    {
        uid = playlist;
    }
    else if(FORGE.Utils.isTypeOf(playlist, "Playlist"))
    {
        uid = playlist.uid;
    }
    else if (typeof playlist === "number" && playlist >= 0 && playlist < this._playlists.length)
    {
        uid = this._playlists[playlist].uid;
    }
    else if(this._playlistUID !== "")
    {
        uid = this._playlistUID;
    }
    else if(typeof playlist === "undefined" || playlist === "")
    {
        uid = this._playlists[0].uid;
    }

    if(typeof uid !== "undefined")
    {
        if(this.playing === true)
        {
            this.stop();
        }

        this._playlistUID = uid;

        if(this._enabled === true)
        {
            this._paused = false;

            this.playlist.play(null, checkAutoPlay);

            this.log("FORGE.PlaylistManager.play(); [uid: "+this._playlistUID+"]");
        }
    }
};

/**
 * Stop the current {@link FORGE.Playlist}.
 * @method  FORGE.PlaylistManager#play
 */
FORGE.PlaylistManager.prototype.stop = function()
{
    if(this.playlist !== null)
    {
        this.playlist.stop();
    }
};

/**
 * Pause the current {@link FORGE.Playlist}.
 * @method  FORGE.PlaylistManager#pause
 */
FORGE.PlaylistManager.prototype.pause = function()
{
    if(this.playlist !== null && this.playlist.playing === true)
    {
        this._paused = true;
        this.playlist.pause();
    }
};

/**
 * Resume the current {@link FORGE.Playlist}.
 * @method  FORGE.PlaylistManager#resume
 */
FORGE.PlaylistManager.prototype.resume = function()
{
    if(this.playlist !== null && this._enabled === true && this.playlist.paused === true)
    {
        var isPaused = this._paused;
        this._paused = false;
        if (isPaused === true && this._defaultList !== this._playlistUID)
        {
            this._startScenePlaylist(this._defaultList);
        }
        else
        {
            this.playlist.resume();
        }
    }
};

/**
 * Set the next {@link FORGE.PlaylistTrack} of the current {@link FORGE.Playlist} to be the current track.<br>
 * If the playlist is paused, keep the pause status of the playlist.
 * @method FORGE.PlaylistManager#nextTrack
 */
FORGE.PlaylistManager.prototype.nextTrack = function()
{
    if(this.playlist !== null && this._enabled === true)
    {
        this.playlist.nextTrack();
    }
};

/**
 * Set the previous {@link FORGE.PlaylistTrack} of the current {@link FORGE.Playlist} to be the current track.<br>
 * If the playlist is paused, keep the pause status of the playlist.
 * @method FORGE.PlaylistManager#previousTrack
 */
FORGE.PlaylistManager.prototype.previousTrack = function()
{
    if(this.playlist !== null && this._enabled === true)
    {
        this.playlist.previousTrack();
    }
};

/**
 * Destroy sequence
 * @method FORGE.PlaylistManager#destroy
 */
FORGE.PlaylistManager.prototype.destroy = function()
{
    this.stop();

    this._viewer.audio.onEnable.remove(this._enableSoundHandler, this);
    this._viewer.story.onSceneLoadStart.remove(this._sceneLoadStartHandler, this);

    this._viewer = null;
    this._config = null;

    var i = this._playlists.length;
    while(i--)
    {
        this._playlists[i].destroy();
    }
    this._playlists = null;

    var j = this._tracks.length;
    while(j--)
    {
        this._tracks[j].destroy();
    }
    this._tracks = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the playlists Array.
 * @name FORGE.PlaylistManager#playlists
 * @readonly
 * @type {Array<FORGE.Playlist>}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "playlists",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        return this._playlists;
    }
});

/**
 * Get the tracks Array.
 * @name FORGE.PlaylistManager#tracks
 * @readonly
 * @type {Array<FORGE.Playlist>}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "tracks",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        return this._tracks;
    }
});

/**
 * Get the current playlist.
 * @name FORGE.PlaylistManager#playlist
 * @readonly
 * @type {?FORGE.Playlist}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "playlist",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(FORGE.UID.isTypeOf(this._playlistUID, "Playlist") === true)
        {
            return FORGE.UID.get(this._playlistUID, "Playlist");
        }

        return null;
    }
});

/**
 * Get the loop status of the current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#loop
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "loop",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.loop;
        }

        return true; // default state is true
    }
});

/**
 * Get the auto play status of the current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#autoPlay
 * @type {boolean}
 * @readonly
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "autoPlay",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.autoPlay;
        }

        return true; // default state is true
    }
});

/**
 * Preload status of audio files.
 * @name FORGE.PlaylistManager#preload
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "preload",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        return this._preload;
    },

    /** @this {FORGE.PlaylistManager} */
    set: function(value)
    {
        this._preload = Boolean(value);
    }
});

/**
 * Get the enabled status of the playlists manager.
 * @name FORGE.PlaylistManager#enabled
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "enabled",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        return this._enabled;
    }
});

/**
 * Get the ready status of current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#ready
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "ready",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.ready;
        }

        return false;
    }
});

/**
 * Get the decoded status of current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#decoded
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "decoded",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.decoded;
        }

        return false;
    }
});

/**
 * Get the playing status of current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#playing
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "playing",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.playing;
        }

        return false;
    }
});

/**
 * Get the pause status of current {@link FORGE.Playlist}.
 * @name FORGE.PlaylistManager#paused
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "paused",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this.playlist !== null)
        {
            return this.playlist.paused;
        }

        return false;
    }
});

/**
 * Get the playlist manager main volume.
 * @name FORGE.PlaylistManager#volume
 * @type {number}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "volume",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        return this._volume;
    },

    /** @this {FORGE.PlaylistManager} */
    set: function(value)
    {
        if(typeof value !== "number")
        {
            return;
        }

        value = FORGE.Math.clamp(value, 0, 1);
        if(this._maxVolume < value)
        {
            this._volume = this._maxVolume;
        }
        else
        {
            this._volume = value;
        }

        if(FORGE.UID.isTypeOf(this._playlistUID, "Playlist") === true)
        {
            var playlist = FORGE.UID.get(this._playlistUID, "Playlist");

            if(playlist !== null)
            {
                var track = playlist.track;

                if(track !== null)
                {
                    track.volume = playlist.volume * this._volume;
                }
            }
        }
    }
});

/**
 * Get the "onReady" event {@link FORGE.EventDispatcher} of the playlist.
 * @name FORGE.PlaylistManager#onReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onReady",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onReady === null)
        {
            this._onReady = new FORGE.EventDispatcher(this);
        }

        return this._onReady;
    }
});

/**
 * Get the "onPlay" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistManager#onPlay
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onPlay",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onPlay === null)
        {
            this._onPlay = new FORGE.EventDispatcher(this);
        }

        return this._onPlay;
    }
});

/**
 * Get the "onStop" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistManager#onStop
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onStop",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onStop === null)
        {
            this._onStop = new FORGE.EventDispatcher(this);
        }

        return this._onStop;
    }
});

/**
 * Get the "onPause" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistManager#onPause
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onPause",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onPause === null)
        {
            this._onPause = new FORGE.EventDispatcher(this);
        }

        return this._onPause;
    }
});

/**
 * Get the "onResume" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistManager#onResume
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onResume",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onResume === null)
        {
            this._onResume = new FORGE.EventDispatcher(this);
        }

        return this._onResume;
    }
});

/**
 * Get the "onEnded" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistManager#onEndedd
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistManager.prototype, "onEnded",
{
    /** @this {FORGE.PlaylistManager} */
    get: function()
    {
        if(this._onEnded === null)
        {
            this._onEnded = new FORGE.EventDispatcher(this);
        }

        return this._onEnded;
    }
});


/**
 * A FORGE.PlaylistTrack is an object that manages the sound atached to a playlist track.
 *
 * @constructor FORGE.PlaylistTrack
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @param {AudioTrackConfig} config - The track config object.
 * @extends {FORGE.BaseObject}
 */
FORGE.PlaylistTrack = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.PlaylistTrack#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The config object.
     * @name FORGE.PlaylistTrack#_config
     * @type {?AudioTrackConfig}
     * @private
     */
    this._config = config;

    /**
     * The name of the track.
     * @name  FORGE.PlaylistTrack#_name
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._name = null;

    /**
     * The author of the track.
     * @name  FORGE.PlaylistTrack#_author
     * @type {?FORGE.LocaleString}
     * @private
     */
    this._author = null;

    /**
     * The url of the track.
     * @name  FORGE.PlaylistTrack#_url
     * @type {string}
     * @private
     */
    this._url = "";

    /**
     * Does the track must be looped?
     * @name  FORGE.PlaylistTrack#_loop
     * @type {boolean}
     * @private
     */
    this._loop = false;

    /**
     * The {@link FORGE.Sound} attached to this track.
     * @name  FORGE.PlaylistTrack#_sound
     * @type {?FORGE.Sound}
     * @private
     */
    this._sound = null;

    FORGE.BaseObject.call(this, "PlaylistTrack");

    this._boot();
};

FORGE.PlaylistTrack.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PlaylistTrack.prototype.constructor = FORGE.PlaylistTrack;

/**
 * Boot sequence.
 * @method FORGE.PlaylistTrack#_boot
 * @private
 */
FORGE.PlaylistTrack.prototype._boot = function()
{
    this._uid = this._config.uid;
    this._register();

    this._parseConfig(this._config);
};

/**
 * Parse the track config.
 * @method FORGE.PlaylistTrack#_parseConfig
 * @param {AudioTrackConfig} config - The config to parse.
 * @private
 */
FORGE.PlaylistTrack.prototype._parseConfig = function(config)
{
    this._unregister();
    this._uid = config.uid;
    this._register();
    this._name = new FORGE.LocaleString(this._viewer, config.name);
    this._author = new FORGE.LocaleString(this._viewer, config.author);
    this._url = config.url;
    this._loop = config.loop || false;
};

/**
 * Bind handlers on the sound
 * @method  FORGE.PlaylistTrack#_bindEvents
 * @private
 */
FORGE.PlaylistTrack.prototype._bindEvents = function()
{
    this._sound.onPlay.add(this._onPlayHandler, this);
    this._sound.onStop.add(this._onStopHandler, this);
    this._sound.onPause.add(this._onPauseHandler, this);
    this._sound.onResume.add(this._onResumeHandler, this);
    this._sound.onEnded.add(this._onEndedHandler, this);
};

/**
 * Unbind handlers on the sound
 * @method  FORGE.PlaylistTrack#_unbindEvents
 * @private
 */
FORGE.PlaylistTrack.prototype._unbindEvents = function()
{
    this._sound.onPlay.remove(this._onPlayHandler, this);
    this._sound.onStop.remove(this._onStopHandler, this);
    this._sound.onPause.remove(this._onPauseHandler, this);
    this._sound.onResume.remove(this._onResumeHandler, this);
    this._sound.onEnded.remove(this._onEndedHandler, this);
};

/**
 * Play handler to notify event to the playlist manager
 * @method  FORGE.PlaylistTrack#_onPlayHandler
 * @private
 */
FORGE.PlaylistTrack.prototype._onPlayHandler = function()
{
    FORGE.PlaylistManager.prototype._notifyPlay.call(this._viewer.playlists);
};

/**
 * Stop handler to notify event to the playlist manager
 * @method  FORGE.PlaylistTrack#_onStopHandler
 * @private
 */
FORGE.PlaylistTrack.prototype._onStopHandler = function()
{
    FORGE.PlaylistManager.prototype._notifyStop.call(this._viewer.playlists);
};

/**
 * Pause handler to notify event to the playlist manager
 * @method  FORGE.PlaylistTrack#_onPauseHandler
 * @private
 */
FORGE.PlaylistTrack.prototype._onPauseHandler = function()
{
    FORGE.PlaylistManager.prototype._notifyPause.call(this._viewer.playlists);
};

/**
 * Resume handler to notify event to the playlist manager
 * @method  FORGE.PlaylistTrack#_onResumeHandler
 * @private
 */
FORGE.PlaylistTrack.prototype._onResumeHandler = function()
{
    FORGE.PlaylistManager.prototype._notifyResume.call(this._viewer.playlists);
};

/**
 * Ended handler to notify event to the playlist manager
 * @method  FORGE.PlaylistTrack#_onEndedHandler
 * @private
 */
FORGE.PlaylistTrack.prototype._onEndedHandler = function()
{
    FORGE.PlaylistManager.prototype._notifyEnded.call(this._viewer.playlists);
};

/**
 * Play the track.
 * @method FORGE.PlaylistTrack#play
 */
FORGE.PlaylistTrack.prototype.play = function()
{
    if(this._sound === null)
    {
        this._sound = new FORGE.Sound(this._viewer, this._uid + "-sound", this._url);
        this._sound.volume = this._viewer.playlists.volume * this._viewer.playlists.playlist.volume;
        if (this._loop === true)
        {
            this._sound.loop = this._loop;
        }

        this._bindEvents();
    }

    this._sound.play();
};

/**
 * Stop the track.
 * @method FORGE.PlaylistTrack#stop
 */
FORGE.PlaylistTrack.prototype.stop = function()
{
    if(this._sound !== null)
    {
        this._sound.stop(true);
    }
};

/**
 * Pause the track.
 * @method FORGE.PlaylistTrack#pause
 */
FORGE.PlaylistTrack.prototype.pause = function()
{
    if(this._sound !== null)
    {
        this._sound.pause();
    }
};

/**
 * Resume the track.
 * @method FORGE.PlaylistTrack#resume
 */
FORGE.PlaylistTrack.prototype.resume = function()
{
    if(this._sound !== null)
    {
        this._sound.resume();
    }
};

/**
 * Destroy sequence
 * @method  FORGE.PlaylistTrack#destroy
 */
FORGE.PlaylistTrack.prototype.destroy = function()
{
    this._viewer = null;

    this._config = null;

    this._name.destroy();
    this._name = null;

    this._author.destroy();
    this._author = null;

    if(this._sound !== null)
    {
        this._unbindEvents();

        this._sound.destroy();
        this._sound = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the name of the track.
 * @name  FORGE.PlaylistTrack#name
 * @type {string}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "name",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        return this._name.value;
    }
});

/**
 * Get the author of the track.
 * @name  FORGE.PlaylistTrack#author
 * @type {string}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "author",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        return this._author.value;
    }
});

/**
 * Get and set the current time of the track.
 * @name  FORGE.PlaylistTrack#currentTime
 * @type {number}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "currentTime",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.currentTime;
        }

        return 0;
    },

    /** @this {FORGE.PlaylistTrack} */
    set: function(value)
    {
        if(this._sound === null)
        {
            return;
        }

        this._sound.currentTime = value;
    }
});

/**
 * Get and set the volume of the track.
 * @name  FORGE.PlaylistTrack#volume
 * @type {number}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "volume",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.volume;
        }

        return false;
    },

    /** @this {FORGE.PlaylistTrack} */
    set: function(value)
    {
        if(this._sound === null)
        {
            return false;
        }

        this._sound.volume = value;
    }
});

/**
 * Get the ready status of the track.
 * @name FORGE.PlaylistTrack#ready
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "ready",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.ready;
        }

        return false;
    }
});

/**
 * Get the decoded status of the track.
 * @name FORGE.PlaylistTrack#decoded
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "decoded",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.decoded;
        }

        return false;
    }
});

/**
 * Get the playing status of the track.
 * @name FORGE.PlaylistTrack#playing
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "playing",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.playing;
        }

        return false;
    }
});

/**
 * Get the paused status of the track.
 * @name FORGE.PlaylistTrack#paused
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "paused",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.paused;
        }

        return false;
    }
});

/**
 * Get ans set the loop state of the track.
 * @name FORGE.PlaylistTrack#loop
 * @type {boolean}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "loop",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        return this._loop;
    },

    /** @this {FORGE.PlaylistTrack} */
    set: function(value)
    {
        this._loop = Boolean(value);
    }
});

/**
 * Get the "onPlay" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistTrack#onPlay
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "onPlay",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.onPlay;
        }

        return;
    }
});

/**
 * Get the "onStop" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistTrack#onStop
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "onStop",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.onStop;
        }

        return;
    }
});

/**
 * Get the "onPause" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistTrack#onPause
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "onPause",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.onPause;
        }

        return;
    }
});

/**
 * Get the "onResume" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistTrack#onResume
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "onResume",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.onResume;
        }

        return;
    }
});

/**
 * Get the "onEnded" event {@link FORGE.EventDispatcher} of the track.
 * @name FORGE.PlaylistTrack#onEnded
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PlaylistTrack.prototype, "onEnded",
{
    /** @this {FORGE.PlaylistTrack} */
    get: function()
    {
        if(this._sound !== null)
        {
            return this._sound.onEnded;
        }

        return;
    }
});
/**
 * A FORGE.Director is used to animate a hotspot.
 *
 * @constructor FORGE.Director
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference
 * @extends {FORGE.Animation}
 */
FORGE.Director = function(viewer)
{
    /**
     * Viewer reference.
     * @name FORGE.Director#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The UID of the selected track.
     * @name FORGE.Director#_track
     * @type {?string}
     * @private
     */
    this._track = null;

    /**
     * The list of the tracks composing the director's cut
     * @name FORGE.Director#_track
     * @type {?Array<string>}
     * @private
     */
    this._tracks = null;

    /**
     * Does the director's cut loop ?
     * @name FORGE.Director#_loop
     * @type {boolean}
     * @private
     */
    this._loop = false;

    /**
     * Does the director's cut randomized ?
     * @name FORGE.Director#_random
     * @type {boolean}
     * @private
     */
    this._random = false;

    /**
     * Can the director's cut be stopped ?
     * @name FORGE.Director#_stoppable
     * @type {boolean}
     * @private
     */
    this._stoppable = false;

    /**
     * The idle time to resume the animation after it was stopped by the user.
     * @name FORGE.Director#_idleTime
     * @type {number}
     * @private
     */
    this._idleTime = -1;

    /**
     * Timer reference used to trigger an animation after idle time
     * @name FORGE.Director#_idleTimer
     * @type {FORGE.Timer}
     * @private
     */
    this._idleTimer = null;

    /**
     * Timer idle event reference
     * @name FORGE.Director#_idleEvent
     * @type {FORGE.TimerEvent}
     * @private
     */
    this._idleEvent = null;

    /**
     * Event handler for the synchronization of the video. Needed for the visibilitychange event.
     * See https://www.w3.org/TR/page-visibility/#sec-visibilitychange-event
     * @name FORGE.Director#_onVisibilityChangeBind
     * @type {Function}
     * @private
     */
    this._onVisibilityChangeBind = null;

    FORGE.BaseObject.call(this, "Director");
    this._boot();
};

FORGE.Director.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Director.prototype.constructor = FORGE.Director;

/**
 * Boot sequence
 *
 * @method FORGE.Director#_boot
 * @private
 */
FORGE.Director.prototype._boot = function()
{
    // Idle timer
    this._idleTimer = this._viewer.clock.create(false);

    // Start the cut once the scene is loaded
    this._viewer.story.onSceneLoadComplete.add(this._sceneLoadCompleteHandler, this);

    // Add controllers after viewer is ready
    this._viewer.onReady.add(this._onViewerReady, this);

    // Bind onVisibilityChange handler
    this._onVisibilityChangeBind = this._onVisibilityChange.bind(this);
};

/**
 * Load director's cut configuration.
 *
 * @method FORGE.Director#load
 * @param {HotspotTrackConfig} config - The animation config to load.
 */
FORGE.Director.prototype.load = function(config)
{
    // Register tracks, no need to store them here
    if (config.tracks !== null && Array.isArray(config.tracks))
    {
        for (var i = 0, ii = config.tracks.length; i < ii; i++)
        {
            new FORGE.DirectorTrack(config.tracks[i]);
        }
    }
};

/**
 * Viewer ready handler
 * @method FORGE.Director#_onViewerReady
 * @private
 */
FORGE.Director.prototype._onViewerReady = function()
{
    this._viewer.controllers.onControlStart.add(this._controlStartHandler, this);
    this._viewer.controllers.onControlEnd.add(this._controlEndHandler, this);
};

/**
 * Event handler for scene load complete.
 * @method  FORGE.Director#_sceneLoadCompleteHandler
 * @private
 */
FORGE.Director.prototype._sceneLoadCompleteHandler = function()
{
    var scene = this._viewer.story.scene;

    // Stop all
    this.stop();
    this._clearEvents();

    // Empty tracks
    this._track = null;
    this._tracks = [];

    if (typeof scene.config.director !== "undefined")
    {
        var animation = scene.config.director.animation;

        if (typeof animation === "undefined" || animation === null || animation.enabled === false)
        {
            return;
        }

        // General properties
        this._loop = (typeof animation.loop === "boolean") ? animation.loop : false;
        this._random = (typeof animation.random === "boolean") ? animation.random : false;
        this._stoppable = (typeof animation.stoppable === "boolean") ? animation.stoppable : false;
        this._idleTime = (typeof animation.idleTime === "number") ? animation.idleTime : 0;

        // Load tracks
        if (animation.tracks !== null && FORGE.Utils.isArrayOf(animation.tracks, "string"))
        {
            this._tracks = (this._random === true) ? FORGE.Utils.randomize(animation.tracks) : animation.tracks;
        }

        // Add on complete handler
        this._viewer.camera.animation.onComplete.add(this._onTrackCompleteHandler, this);

        // Add specific behavior if the current media is a video
        if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
        {
            this._viewer.story.scene.media.displayObject.onPause.add(this._pauseHandler, this);

            // React on loading/buffering event
            this._viewer.story.scene.media.displayObject.onWaiting.add(this._waitingHandler, this);
            this._viewer.story.scene.media.displayObject.onStalled.add(this._waitingHandler, this);
            this._viewer.story.scene.media.displayObject.onSeeking.add(this._waitingHandler, this);

            // The director's cut begin again if video is looping
            this._viewer.story.scene.media.displayObject.onEnded.add(this._endedHandler, this);

            // Synchronization !
            this._viewer.story.scene.media.displayObject.onCurrentTimeChange.add(this._synchronizeWithVideo, this);

            // Double synchronization when the window lose visibility
            // Needed because we rely on RAF which is only working on visibility
            document.addEventListener(FORGE.Device.visibilityChange, this._onVisibilityChangeBind);
        }

        // Start given the idle time
        if (typeof animation.delay === "number")
        {
            this._idleEvent = this._idleTimer.add(animation.delay, this._idleTimerCompleteHandler, this);
            this._idleTimer.start();
        }
    }
};

/**
 * Event handler when the video is paused. Not trigger only after the "pause" event.
 * @method FORGE.Director#_pauseHandler
 * @private
 */
FORGE.Director.prototype._pauseHandler = function()
{
    // Stopping the animation
    this.stop();

    // Event handling
    // for animation
    if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onPause.remove(this._pauseHandler, this);
        this._viewer.story.scene.media.displayObject.onPlay.add(this._playHandler, this);
    }

    // for controllers
    this._viewer.controllers.onControlStart.remove(this._controlStartHandler, this);
    this._viewer.controllers.onControlEnd.remove(this._controlEndHandler, this);
};

/**
 * Event handler when the video start playing again. Not trigger only after the "play" event.
 * @method FORGE.Director#_playHandler
 * @private
 */
FORGE.Director.prototype._playHandler = function()
{
    // Starting the animation
    if (this._track !== null)
    {
        this.play(this._track);

        if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
        {
            this._synchronizeWithVideo();
        }
    }

    // Event handling
    // for animation
    if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onPause.add(this._pauseHandler, this);
        this._viewer.story.scene.media.displayObject.onPlay.remove(this._playHandler, this);
    }
    // for controllers
    this._viewer.controllers.onControlStart.add(this._controlStartHandler, this);
    this._viewer.controllers.onControlEnd.add(this._controlEndHandler, this);
};

/**
 * Waiting and stalled handler
 * @method FORGE.Director#_waitingHandler
 * @private
 */
FORGE.Director.prototype._waitingHandler = function()
{
    // Stopping the animation
    this.stop();

    // Event handling
    // for waiting
    if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onWaiting.remove(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onStalled.remove(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onSeeking.remove(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onPlaying.add(this._playingHandler, this);
        this._viewer.story.scene.media.displayObject.onSeeked.add(this._playingHandler, this);
    }
    // for controllers
    this._viewer.controllers.onControlStart.remove(this._controlStartHandler, this);
    this._viewer.controllers.onControlEnd.remove(this._controlEndHandler, this);
};

/**
 * Playing handler
 * @method FORGE.Director#_playingHandler
 * @private
 */
FORGE.Director.prototype._playingHandler = function()
{
    // Starting the animation
    if (this._track !== null)
    {
        this.play(this._track);

        if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
        {
            this._synchronizeWithVideo();
        }
    }

    // Event handling
    // for waiting
    if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onPlaying.remove(this._playingHandler, this);
        this._viewer.story.scene.media.displayObject.onSeeked.remove(this._playingHandler, this);
        this._viewer.story.scene.media.displayObject.onWaiting.add(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onStalled.add(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onSeeking.add(this._waitingHandler, this);
    }
    // for controllers
    this._viewer.controllers.onControlStart.add(this._controlStartHandler, this);
    this._viewer.controllers.onControlEnd.add(this._controlEndHandler, this);
};

/**
 * If the video is looping, this handler will start the director's cut all over again.
 * @method FORGE.Director#_endedHandler
 * @private
 */
FORGE.Director.prototype._endedHandler = function()
{
    if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO && this._viewer.story.scene.media.displayObject.loop === true)
    {
        // Starting the animation by considering it is a new scene
        this._sceneLoadCompleteHandler();
    }
};

/**
 * Event handler for control start.
 * @method FORGE.Director#_controlStartHandler
 * @private
 */
FORGE.Director.prototype._controlStartHandler = function()
{
    if (this._track !== null && this._stoppable === true)
    {
        this.stop();
    }

    this._idleTimer.stop(true);
};

/**
 * Event handler for control end.
 * @method FORGE.Director#_controlEndHandler
 * @private
 */
FORGE.Director.prototype._controlEndHandler = function()
{
    if (this._idleEvent !== null)
    {
        this._idleTimer.remove(this._idleEvent);
    }

    if (this._track !== null && this._stoppable === true)
    {
        // If the track is null, we haven't began yet the director's cut
        if (this._track === null && typeof this._viewer.story.scene.config.director.animation.delay === "number")
        {
            this._idleEvent = this._idleTimer.add(this._viewer.story.scene.config.director.animation.delay, this._idleTimerCompleteHandler, this);
            this._idleTimer.start();
        }
        else if (typeof this._idleTime === "number" && this._idleTime > -1)
        {
            this._idleEvent = this._idleTimer.add(this._idleTime, this._idleTimerCompleteHandler, this);
            this._idleTimer.start();
        }
    }
};

/**
 * Event handler when a track is completed. Play the next track in the list if
 * any, or a random one.
 * @method FORGE.Director#_onTrackCompleteHandler
 * @private
 */
FORGE.Director.prototype._onTrackCompleteHandler = function()
{
    if (this._tracks.length > 1)
    {
        // Go to the next track if any
        var idx = this._tracks.indexOf( /** @type {string} */ (this._track)) + 1;

        // If the index is too high, play the track
        if (idx < this._tracks.length)
        {
            this.play(idx);
            return;
        }
    }

    // Loop only if it is the end of the last track
    if (this._loop === true)
    {
        // If it is random, change the entire order
        if (this._random === true)
        {
            this._tracks = FORGE.Utils.randomize(this._tracks);
        }

        this.play(0);
        return;
    }
};

/**
 * Event handler for idle timer complete.
 *
 * @method FORGE.Director#_idleTimerCompleteHandler
 * @private
 */
FORGE.Director.prototype._idleTimerCompleteHandler = function()
{
    // Resume the current track
    this.play(this._track);

    this._idleTimer.stop(true);
};

/**
 * Event handler for visibility change on the window.
 * @method FORGE.Director#_onVisibilityChange
 * @private
 */
FORGE.Director.prototype._onVisibilityChange = function()
{
    if (document[FORGE.Device.visibilityState] !== "hidden" && this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onCurrentTimeChange.dispatch(this._viewer.story.scene.media.displayObject.currentTime);
    }
};

/**
 * When the currentTime property of the video change, synchronize the director's cut on it.
 * @method FORGE.Director#_synchronizeWithVideo
 * @param  {(number|FORGE.Event)=} time - the emitted event, containing the time to synchronize to, or the time to synchronize to (in seconds).
 * @private
 */
FORGE.Director.prototype._synchronizeWithVideo = function(time)
{
    this.stop();

    // Remove the current track
    this._track = null;

    // Get the correct time to sync
    if (typeof time === "number")
    {
        time = time * 1000;
    }
    else if (typeof time === "object" && typeof time.data === "number")
    {
        time = time.data * 1000;
    }
    else if (this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        time = this._viewer.story.scene.media.displayObject.currentTimeMS;
    }
    else
    {
        time = 0;
    }

    var offset = 0;
    var trackA, trackB;
    trackB = FORGE.UID.get(this._tracks[0]);

    // If the time is lower than the duration of the first track, it is this one
    if (trackB.duration > time)
    {
        this._track = trackB.uid;
    }
    else
    {
        // Else check for each track and its following one if the time is between this two.
        for (var i = 1, ii = this._tracks.length - 1; i < ii; i++)
        {
            trackA = trackB;
            trackB = FORGE.UID.get(this._tracks[i]);

            if (trackA.duration + offset < time && trackB.duration + trackA.duration + offset > time)
            {
                this._track = trackB.uid;
                offset += trackA.duration;
                break;
            }

            offset += trackA.duration;
        }
    }

    // If no track found, it is after the last track, so change the camera
    if (this._track === null)
    {
        // Look the same as the last point of our last track
        var point = trackB.keyframes[trackB.keyframes.length - 1].data;
        this._viewer.camera.lookAt(point.yaw, point.pitch, point.roll, point.fov);
    }
    else
    {
        this._viewer.camera.animation.play(this._track, time);
    }
};

/**
 * Play a set of tracks if specified, else the current one, from the start.
 *
 * @method FORGE.Director#play
 * @param {?(string|number)=} track - A track
 */
FORGE.Director.prototype.play = function(track)
{
    this.stop();
    this._track = null;

    if (typeof track === "number")
    {
        this._track = this._tracks[track];
    }
    else if (typeof track === "string")
    {
        this._track = track;
    }
    else
    {
        this._track = this._tracks[0];
    }

    this._viewer.camera.animation.play(this._track);
};

/**
 * Stops the current animation.
 * @method FORGE.Director#stop
 */
FORGE.Director.prototype.stop = function()
{
    this.log("stopping");
    this._viewer.camera.animation.stop();
};

FORGE.Director.prototype._clearEvents = function()
{
    this._viewer.camera.animation.onComplete.remove(this._onTrackCompleteHandler, this);

    if (this._viewer.story.scene !== null && this._viewer.story.scene.media.type === FORGE.MediaType.VIDEO)
    {
        this._viewer.story.scene.media.displayObject.onPlay.remove(this._playHandler, this);
        this._viewer.story.scene.media.displayObject.onPause.remove(this._pauseHandler, this);
        this._viewer.story.scene.media.displayObject.onWaiting.remove(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onStalled.remove(this._waitingHandler, this);
        this._viewer.story.scene.media.displayObject.onPlaying.remove(this._playingHandler, this);
        this._viewer.story.scene.media.displayObject.onEnded.remove(this._endedHandler, this);
        this._viewer.story.scene.media.displayObject.onCurrentTimeChange.remove(this._synchronizeWithVideo, this);

        document.removeEventListener(FORGE.Device.visibilityChange, this._onVisibilityChangeBind);
    }
};

/**
 * Destroy method.
 * @method  FORGE.Director#destroy
 */
FORGE.Director.prototype.destroy = function()
{
    this._clearEvents();

    this._viewer.onReady.remove(this._onViewerReady, this);
    this._viewer = null;

    this._tracks = null;

    this._track = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * A director track, that defines a camera animation.
 *
 * @constructor FORGE.DirectorTrack
 * @param {DirectorTrackConfig} config - Configuration of the track from the JSON file.
 * @extends {FORGE.Track}
 */
FORGE.DirectorTrack = function(config)
{
    /**
     * Does the track has a smooth interpolation between keyframes ?
     * @name FORGE.DirectorTrack#_smooth
     * @type {boolean}
     * @private
     */
    this._smooth = false;

    /**
     * Is the roll cancelled ?
     * @name FORGE.DirectorTrack#_cancelRoll
     * @type {boolean}
     * @private
     */
    this._cancelRoll = false;

    FORGE.Track.call(this, "DirectorTrack");

    this._boot(config);
};

FORGE.DirectorTrack.prototype = Object.create(FORGE.Track.prototype);
FORGE.DirectorTrack.prototype.constructor = FORGE.DirectorTrack;

/**
 * Boot sequence
 *
 * @method FORGE.DirectorTrack#_boot
 * @param  {Object} config - The information on the track
 * @private
 */
FORGE.DirectorTrack.prototype._boot = function(config)
{
    this._smooth = config.smooth;
    this._cancelRoll = config.cancelRoll;

    FORGE.Track.prototype._boot.call(this, config);
};

/**
 * Accessors to smooth
 * @name FORGE.DirectorTrack#smooth
 * @readonly
 * @type {FORGE.DirectorTrack}
 */
Object.defineProperty(FORGE.DirectorTrack.prototype, "smooth",
{
    /** @this {FORGE.DirectorTrack} */
    get: function()
    {
        return this._smooth;
    }
});

/**
 * Accessors to cancelRoll
 * @name FORGE.DirectorTrack#cancelRoll
 * @readonly
 * @type {FORGE.DirectorTrack}
 */
Object.defineProperty(FORGE.DirectorTrack.prototype, "cancelRoll",
{
    /** @this {FORGE.DirectorTrack} */
    get: function()
    {
        return this._cancelRoll;
    }
});

/**
 * A action set event dispatcher is in charge to triggers actions binded on an event.
 *
 * @constructor FORGE.ActionEventDispatcher
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {string} name - The name of the event.
 * @extends {FORGE.BaseObject}
 */
FORGE.ActionEventDispatcher = function(viewer, name)
{
    /**
     * The viewer reference.
     * @name FORGE.ActionEventDispatcher#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Event name
     * @name FORGE.ActionEventDispatcher#_name
     * @type {string}
     * @private
     */
    this._name = name;

    /**
     * The actions uid list of the set.
     * @name  FORGE.ActionEventDispatcher#_actions
     * @type {Array<string>}
     * @private
     */
    this._actions = [];

    /**
     * Flag to know if the dispatcher is currently dispatching the events.
     * It happen that the dispatcher destroy method is called during the dispatching the events.
     * This allows us to re schedule the destroy call after all events have been executed.
     * @name  FORGE.ActionEventDispatcher#_dispatching
     * @type {boolean}
     * @private
     */
    this._dispatching = false;

    /**
     * Is the destroy method have been called during the dispatching?
     * @name  FORGE.ActionEventDispatcher#_pendingDestroy
     * @type {boolean}
     * @private
     */
    this._pendingDestroy = false;

    FORGE.BaseObject.call(this, "ActionEventDispatcher");
};

FORGE.ActionEventDispatcher.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ActionEventDispatcher.prototype.constructor = FORGE.ActionEventDispatcher;

/**
 * Boot sequence.
 * @method FORGE.ActionEventDispatcher#addActions
 * @param {(string|Array<string>)} actions - The actions uids you want to add
 */
FORGE.ActionEventDispatcher.prototype.addActions = function(actions)
{
    if(FORGE.Utils.isArrayOf(actions, "string") === true || FORGE.Utils.isTypeOf(actions, "string"))
    {
        this._actions = this._actions.concat(actions);
    }
};

/**
 * This method execute every actions binded on this event dispatcher.
 * @method FORGE.ActionEventDispatcher#dispatch
 */
FORGE.ActionEventDispatcher.prototype.dispatch = function()
{
    this._dispatching = true;

    for(var i = 0, ii = this._actions.length; i < ii; i++)
    {
        var action = this._viewer.actions.get(this._actions[i]);

        if(typeof action !== "undefined" && action !== null)
        {
            action.execute();
        }
    }

    this._dispatching = false;

    if(this._pendingDestroy === true)
    {
        this.destroy();
    }
};

/**
 * Destroy sequence.
 * @method  FORGE.ActionEventDispatcher#destroy
 */
FORGE.ActionEventDispatcher.prototype.destroy = function()
{
    //If dispatching, wait for the dispatch is complete to execute the destroy
    if(this._dispatching === true)
    {
        this._pendingDestroy = true;
        return;
    }

    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * An action is a description of a method to execute in reaction to an event.<br>
 * The method will be executed from a target.
 *
 * @constructor FORGE.Action
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @param {ActionConfig} config - The config of the action.
 * @extends {FORGE.BaseObject}
 */
FORGE.Action = function(viewer, config)
{
    /**
     * The viewer reference.
     * @name FORGE.Action#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The config of the action.
     * @name FORGE.Action#_config
     * @type {ActionConfig}
     * @private
     */
    this._config = config;

    /**
     * The object related to this action.
     * @name FORGE.Action#_object
     * @type {(string|ActionTargetConfig)}
     * @private
     */
    this._target = "";

    /**
     * The method name with args who will be executed.
     * @name FORGE.Action#_method
     * @type {?ActionMethodConfig}
     * @private
     */
    this._method = null;

    /**
     * The property name who will be changed with the value.
     * @name FORGE.Action#_property
     * @type {?ActionPropertyConfig}
     * @private
     */
    this._property = null;

    /**
     * The number of time this action has been executed
     * @name FORGE.Action._count
     * @type {number}
     * @private
     */
    this._count = 0;

    FORGE.BaseObject.call(this, "Action");

    this._boot();
};

FORGE.Action.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Action.prototype.constructor = FORGE.Action;

/**
 * List of possible operation when you affect a property.
 * @name  FORGE.Action.operation
 * @type {Object}
 * @const
 */
FORGE.Action.operation = {};

/**
 * Operation set is the default one, it will just set the value.
 * @name  FORGE.Action.operation.SET
 * @type {string}
 * @const
 */
FORGE.Action.operation.SET = "set";

/**
 * Operation addition.
 * @name  FORGE.Action.operation.ADD
 * @type {string}
 * @const
 */
FORGE.Action.operation.ADD = "add";

/**
 * Operation substract.
 * @name  FORGE.Action.operation.SUBSTRACT
 * @type {string}
 * @const
 */
FORGE.Action.operation.SUBSTRACT = "substract";

/**
 * Operation multiply.
 * @name  FORGE.Action.operation.MULTIPLY
 * @type {string}
 * @const
 */
FORGE.Action.operation.MULTIPLY = "multiply";

/**
 * Operation divide.
 * @name  FORGE.Action.operation.DIVIDE
 * @type {string}
 * @const
 */
FORGE.Action.operation.DIVIDE = "divide";

/**
 * Operation boolean toggle.
 * @name  FORGE.Action.operation.TOGGLE
 * @type {string}
 * @const
 */
FORGE.Action.operation.TOGGLE = "toggle";

/**
 * Boot sequence.
 * @method FORGE.Action#_boot
 * @private
 */
FORGE.Action.prototype._boot = function()
{
    this._parseConfig(this._config);
    this._register();
};

/**
 * Parse the action configuration object from the json
 * @method FORGE.Action#_parseConfig
 * @param  {ActionConfig} config - Action configuration to parse
 * @private
 */
FORGE.Action.prototype._parseConfig = function(config)
{
    this._uid = config.uid;

    this._target = config.target;

    this._method = (typeof config.method !== "undefined") ? config.method : null;

    this._property = (typeof config.property !== "undefined") ? config.property : null;
};

/**
 * Parse the target.<br>
 * If undefined action will be executed on "window".<br>
 * If target start by "this" keyword, method will be searched on the action set instance.<br>
 * Target can also be the id of a plugin.
 * @method FORGE.Action#_parseTarget
 * @private
 * @param  {(string|ActionTargetConfig)} target - The target to parse.
 * @return {*} The target object.
 */
FORGE.Action.prototype._parseTarget = function(target)
{
    var result = null;

    if (typeof target === "undefined")
    {
        result = window;
    }
    else
    {
        var targetUid = target;
        var path = target;

        if (typeof target === "object")
        {
            if (typeof target.identifier === "string")
            {
                targetUid = target.identifier;
            }
            if (typeof target.accessor === "string")
            {
                path = target.accessor;
            }
        }

        // Root target
        var targetByUid = FORGE.UID.get(targetUid);

        if (typeof targetByUid !== "undefined" && targetByUid !== null)
        {
            if (FORGE.Utils.isTypeOf(targetByUid, "Plugin"))
            {
                result = targetByUid.instance;
            }
            else
            {
                result = targetByUid;
            }
        }

        // Path
        path = path.split(".");

        var i = 0;
        // If it is the viewer, reset the result to it
        if (path[0].toLowerCase() === "viewer" && result === null)
        {
            result = this._viewer;
            i = 1;
        }
        else if (result === null)
        {
            result = window;
        }

        for (var ii = path.length; i < ii; i++)
        {
            if (typeof result[path[i]] === "object")
            {
                result = result[path[i]];
            }
        }
    }

    if (result === null)
    {
        this.warn("The target of the action " + this._uid + " is invalid.");
    }

    return result;
};

/**
 * Apply the property configuration to the target.
 * @method FORGE.Action#_applyProperty
 * @param  {*} target - The target that own the desired method.
 * @param  {ActionPropertyConfig} property - The property configuration with its name and its arguments array
 * @private
 */
FORGE.Action.prototype._applyProperty = function(target, property)
{
    if (typeof target[property.name] !== "undefined")
    {
        if (typeof property.operation === "undefined")
        {
            property.operation = FORGE.Action.operation.SET;
        }

        switch (property.operation)
        {
            case FORGE.Action.operation.SET:
                target[property.name] = property.value;
                break;

            case FORGE.Action.operation.ADD:
                target[property.name] += property.value;
                break;

            case FORGE.Action.operation.SUBSTRACT:
                target[property.name] -= /** @type {number} */ (property.value);
                break;

            case FORGE.Action.operation.MULTIPLY:
                target[property.name] *= /** @type {number} */ (property.value);
                break;

            case FORGE.Action.operation.DIVIDE:
                target[property.name] /= /** @type {number} */ (property.value);
                break;

            case FORGE.Action.operation.TOGGLE:
                target[property.name] = !target[property.name];
                break;
        }
    }
};

/**
 * Apply the method configuration to the target.
 * @method FORGE.Action#_applyMethod
 * @param  {*} target - The target that own the desired method.
 * @param  {ActionMethodConfig} method - The method configuration with its name and its arguments array
 * @private
 */
FORGE.Action.prototype._applyMethod = function(target, method)
{
    if (typeof target[method.name] === "function")
    {
        var args = (typeof method.args !== "undefined") ? method.args : null;

        if (!Array.isArray(args))
        {
            args = [args];
        }

        target[method.name].apply(target, args);
    }
};

/**
 * The execute method will trigger the action, parse the target if necessary.
 * @method FORGE.Action#execute
 */
FORGE.Action.prototype.execute = function()
{
    //Get the target at the last time and do NOT keep a reference !
    var target = this._parseTarget(this._target);

    if (target !== null)
    {
        if (this._property !== null)
        {
            this._applyProperty(target, this._property);
        }

        if (this._method !== null)
        {
            this._applyMethod(target, this._method);
        }
    }
};

/**
 * Destroy sequence.
 * @method  FORGE.Action#destroy
 */
FORGE.Action.prototype.destroy = function()
{
    this._viewer = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};


/**
 * Get the target of the action.
 * @name FORGE.Action#target
 * @readonly
 * @type {*}
 */
Object.defineProperty(FORGE.Action.prototype, "target",
{
    /** @this {FORGE.Action} */
    get: function()
    {
        return this._parseTarget(this._target);
    }
});

/**
 * Get the number of time this action has been executed.
 * @name FORGE.Action#count
 * @readonly
 * @type {number}
 */
Object.defineProperty(FORGE.Action.prototype, "count",
{
    /** @this {FORGE.Action} */
    get: function()
    {
        return this._count;
    }
});
/**
 * Action manager
 * @constructor FORGE.ActionManager
 * @param {FORGE.Viewer} viewer - viewer reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.ActionManager = function(viewer)
{
    /**
     * Viewer reference.
     * @name FORGE.ActionManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Configuration of actions from the JSON
     * @name  FORGE.ActionManager#_config
     * @type {Array<ActionConfig>}
     * @private
     */
    this._config = null;

    /**
     * Camera reference.
     * @name FORGE.ActionManager#_actions
     * @type {Array}
     * @private
     */
    this._actions = [];

    /**
     * Event dispatcher for ready status.
     * @name FORGE.ActionManager#_onReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onReady = null;

    FORGE.BaseObject.call(this, "ActionManager");
};

FORGE.ActionManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.ActionManager.prototype.constructor = FORGE.ActionManager;

/**
 * Parse an array of configuration
 * @param {(Array<ActionConfig>|ActionConfig)} config - Array of action configurations or a single action configuration.
 * @private
 */
FORGE.ActionManager.prototype._parseConfig = function(config)
{
    var action;

    // If it is an array of actions
    if (Array.isArray(config) === true)
    {
        for (var i = 0, ii = config.length; i < ii; i++)
        {
            action = new FORGE.Action(this._viewer, config[i]);
            this._actions.push(action);
        }
    }
    // If it is a single action
    else
    {
        action = new FORGE.Action(this._viewer, /** @type {ActionConfig} */ (config));
        this._actions.push(action);
    }
};

/**
 * Get an action by id.
 * @method FORGE.ActionManager#get
 */
FORGE.ActionManager.prototype.get = function(uid)
{
    return FORGE.UID.get(uid, "Action");
};

/**
 * Add actions configuration
 * @method FORGE.ActionManager#addConfig
 * @param {(Array<ActionConfig>|ActionConfig)} config - Array of action configurations or a single action configuration.
 */
FORGE.ActionManager.prototype.addConfig = function(config)
{
    this._parseConfig(config);
};

/**
 * Destroy routine
 * @method FORGE.ActionManager#destroy
 */
FORGE.ActionManager.prototype.destroy = function()
{
    if (FORGE.Utils.isArrayOf(this._actions, "Action"))
    {
        while (this._actions.length > 0)
        {
            var a = this._actions.pop();
            a.destroy();
            a = null;
        }
    }
};

/**
 * A plugin object is an instance of a {@link FORGE.PluginEngine} on top of the viewer.<br>
 * It can be visual (display a logo) or not (gyroscope, stats ...).
 *
 * @constructor FORGE.Plugin
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {FORGE.PluginEngine} engine - The engine used to instantiate tis plugin.
 * @param {PluginInstanceConfig} config - The config of the plugin instance.
 * @param {number} index - The index of the plugin, use for display order.
 * @extends {FORGE.BaseObject}
 *
 * @todo  Scene array upgrade : Have a scene array with inclusive and exclusive uids
 * @todo  Same filter array for groups ?
 */
FORGE.Plugin = function(viewer, engine, config, index)
{
    /**
     * The viewer reference.
     * @name FORGE.Plugin#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The engine this plugin is based on.
     * @name  FORGE.Plugin#_engine
     * @type {FORGE.PluginEngine}
     * @private
     */
    this._engine = engine;

    /**
     * The instance config of the plugin.
     * @name  FORGE.Plugin#_instanceConfig
     * @type {?PluginInstanceConfig}
     * @private
     */
    this._instanceConfig = config;

    /**
     * The contextual config is the plugin configuration specific to a scene or a group.<br/>
     * It doesn't last at scene change.
     * @name FORGE.Plugin#_contextualConfig
     * @type {?PluginConfigurationConfig}
     * @private
     */
    this._contextualConfig = null;

    /**
     * This config will be the final config object that merges the default engine config, the instance config and the contextual config.
     * @name  FORGE.Plugin#_config
     * @type {?PluginInstanceConfig}
     * @private
     */
    this._config = null;

    /**
     * The index of the plugin, used if the plugin is a visual one.
     * @name FORGE.Plugin#_index
     * @type {number}
     * @private
     */
    this._index = index;

    /**
     * The options of this Plugin.
     * @name FORGE.Plugin#_options
     * @type {Object}
     * @private
     */
    this._options = null;

    /**
     * The actions of this Plugin.
     * @name FORGE.Plugin#_actions
     * @type {Object}
     * @private
     */
    this._actions = null;

    /**
     * Object that stores {@link FORGE.ActionEventDispatcher}.
     * @name FORGE.Plugin#_events
     * @type {Object<FORGE.ActionEventDispatcher>}
     * @private
     */
    this._events = null;

    /**
     * Instance of the plugin engine constructor.
     * @name  FORGE.Plugin#_instance
     * @type {?PluginStructure}
     * @private
     */
    this._instance = null;

    /**
     * Flag to tell if the instance is ready.<br>
     * It is the plugin developper that set the ready flag by calling the notifyInstanceReady method from its instance.
     * @name  FORGE.Plugin#_instanceReady
     * @type {boolean}
     * @private
     */
    this._instanceReady = false;

    /**
     * A DOM id or a DOM element.<br>
     * Used to inject graphical plugins anywhere in the document (outside the viewer container).
     * @name  FORGE.Plugin#_parent
     * @type {Element|HTMLElement|string}
     * @private
     */
    this._parent = null;

    /**
     * The display object container for this plugin if it is a graphical one.
     * @name FORGE.Plugin#_container
     * @type {?FORGE.DisplayObjectContainer}
     * @private
     */
    this._container = null;

    /**
     * Reference to the plugin object factory, every plugin have its own factory.
     * @name FORGE.Plugin#_create
     * @type {FORGE.PluginObjectFactory}
     * @private
     */
    this._create = null;

    /**
     * Scenes UID array, this plugin is allow to instantiate only on these scene if not null.
     * @name  FORGE.Plugin#_scenes
     * @type {Array<string>}
     * @private
     */
    this._scenes = null;

    /**
     * Object to handle persistent data.
     * @name  FORGE.Plugin#_persistentData
     * @type {Object}
     * @private
     */
    this._persistentData = null;

    /**
     * Is this plugin stay at the scene change event?
     * @name  FORGE.Plugin#_keep
     * @type {boolean}
     * @private
     */
    this._keep = true;

    /**
     * Is this plugin have to reset between each scene?
     * @name  FORGE.Plugin#_reset
     * @type {boolean}
     * @private
     */
    this._reset = true;

    /**
     * Event dispatcher for instance creation
     * @name  FORGE.Plugin#_onInstanceCreate
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onInstanceCreate = null;

     /**
     * Event dispatcher for instance ready
     * @name  FORGE.Plugin#_onInstanceReady
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onInstanceReady = null;

    FORGE.BaseObject.call(this, "Plugin");

    this._boot();
};

FORGE.Plugin.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.Plugin.prototype.constructor = FORGE.Plugin;

/**
 * Boot sequence
 * @method  FORGE.Plugin#_boot
 * @private
 */
FORGE.Plugin.prototype._boot = function()
{
    //First thing, very important is to register the plugin object!
    //We can't parse the rest of the config here, maybe the engine is not loaded yet.
    this._uid = this._instanceConfig.uid;
    this._tags = this._instanceConfig.tags;
    this._register();

    this._persistentData = {};
};

/**
 * Internal method to parse the config.
 * @method FORGE.Plugin#_parseConfig
 * @private
 * @param {Object} config - The config object to parse.
 */
FORGE.Plugin.prototype._parseConfig = function(config)
{
    //If there is a parent string in config, this should be an dom element id where to inject the plugin.
    if(typeof config.parent === "string")
    {
        this._parent = config.parent;
    }

    this._keep = (typeof config.keep === "boolean") ? config.keep : true;
    this._reset = (typeof config.reset === "boolean") ? config.reset : true;

    //If there is no scenes array alerady set and if there is a scene array in the instance config.
    if(this._scenes === null && FORGE.Utils.isArrayOf(config.scenes, "string") === true)
    {
        this._scenes = config.scenes;
    }

    this._mergeConfigurations();
};

/**
 * Merge the plugin configurations
 * @method FORGE.Plugin#_mergeConfigurations
 * @private
 */
FORGE.Plugin.prototype._mergeConfigurations = function()
{
    this._config = /** @type {PluginInstanceConfig} */ (FORGE.Utils.extendMultipleObjects(this._engine.defaultConfig, this._instanceConfig, this._contextualConfig));
    this._options = this._config.options || {};
    this._data = this._config.data || {};
    this._actions = this._config.actions || {};

    this._clearEvents();
    this._createEvents(this._config.events);
};

/**
 * Create events dispatchers that the engine needs.
 * @method FORGE.Plugin#_createEvents
 * @private
 * @param {Object=} events - The events config of the plugin engine.
 */
FORGE.Plugin.prototype._createEvents = function(events)
{
    this._events = {};

    var event;
    for(var e in events)
    {
        event = new FORGE.ActionEventDispatcher(this._viewer, e);
        event.addActions(events[e]);
        this._events[e] = event;
    }
};

/**
 * Clear all plugin events.
 * @method FORGE.Plugin#_clearEvents
 * @private
 */
FORGE.Plugin.prototype._clearEvents = function()
{
    for(var e in this._events)
    {
        this._events[e].destroy();
        this._events[e] = null;
    }
};

/**
 * Instantiate the plugin engine if it is loaded, if not listen to the engine load complete to retry to instantiate.
 * @method FORGE.Plugin#instantiate
 */
FORGE.Plugin.prototype.instantiate = function()
{
    if(this._engine.loaded === true)
    {
        this._instantiatePlugin();
    }
    else
    {
        this._engine.onLoadComplete.addOnce(this._engineLoadComplete, this);
    }
};

/**
 * Handler for engine load complete.
 * @method FORGE.Plugin#_engineLoadComplete
 * @private
 */
FORGE.Plugin.prototype._engineLoadComplete = function()
{
    this._instantiatePlugin();
};

/**
 * Parse the config then instantiate the plugin.
 * @method FORGE.Plugin#_instantiatePlugin
 * @private
 */
FORGE.Plugin.prototype._instantiatePlugin = function()
{
    this.log("Plugin._instantiatePlugin(); "+this._uid);

    this._parseConfig(this._instanceConfig);

    this._instance = this._engine.getNewInstance(this._viewer, this);

    this._instance.boot.call(this._instance);

    if(this._onInstanceCreate !== null)
    {
        this._onInstanceCreate.dispatch();
    }
};

/**
 * Instance can notify with this method that the instance is ready
 * @method FORGE.Plugin.notifyInstanceReady
 */
FORGE.Plugin.prototype.notifyInstanceReady = function()
{
    this._instanceReady = true;

    if(this._onInstanceReady !== null)
    {
        this._onInstanceReady.dispatch();
    }
};

/**
 * Update method called by the plugin manager.
 * @method FORGE.Plugin#update
 */
FORGE.Plugin.prototype.update = function()
{
    if(this._instance !== null && typeof this._instance.update === "function")
    {
        this._instance.update.call(this._instance);
    }
};

/**
 * Reset the plugin to a specified configuration.
 * @method FORGE.Plugin#reset
 */
FORGE.Plugin.prototype.resetInstance = function()
{
    this._mergeConfigurations();

    if(this._instance !== null)
    {
        if(typeof this._instance.reset === "function")
        {
            this._instance.reset.call(this._instance);
        }
        else
        {
            this.log("There is no reset function on plugin "+this._engine.uid);
        }
    }
};

/**
 * Destroy method.
 * @method FORGE.Plugin#destroy
 */
FORGE.Plugin.prototype.destroy = function()
{
    if(this._alive === false)
    {
        return;
    }

    if(this._instance !== null)
    {
        this._instance.viewer = null;
        this._instance.plugin = null;

        this._instance.destroy();
        this._instance = null;
    }

    if(this._create !== null)
    {
        this._create.destroy();
        this._create = null;
    }

    if(this._container !== null)
    {
        this._container.destroy();
        this._container = null;
    }

    if(this._onInstanceCreate !== null)
    {
        this._onInstanceCreate.destroy();
        this._onInstanceCreate = null;
    }

    if(this._onInstanceReady !== null)
    {
        this._onInstanceReady.destroy();
        this._onInstanceReady = null;
    }

    this._clearEvents();
    this._events = null;

    this._viewer = null;

    this._persistentData = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the plugin full url.
 * @name FORGE.Plugin#fullUrl
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.Plugin.prototype, "fullUrl",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._engine.fullUrl;
    }
});

/**
 * Get the plugin options.
 * @name FORGE.Plugin#options
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "options",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._options;
    }
});

/**
 * Get the plugin actions.
 * @name FORGE.Plugin#actions
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "actions",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._actions;
    }
});

/**
 * Get the plugin events.
 * @name FORGE.Plugin#events
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "events",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._events;
    }
});

/**
 * Get the scenes array. This plugin will be alive only on these scenes.
 * @name FORGE.Plugin#scenes
 * @readonly
 * @type {Array}
 */
Object.defineProperty(FORGE.Plugin.prototype, "scenes",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._scenes;
    }
});

/**
 * Get and set pesistent data.
 * @name FORGE.Plugin#persistentData
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "persistentData",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._persistentData;
    },

    /** @this {FORGE.Plugin} */
    set: function(value)
    {
        this._persistentData = value;
    }
});

/**
 * Get the plugin instance.
 * @name FORGE.Plugin#instance
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "instance",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._instance;
    }
});

/**
 * Get the plugin instance ready flag.
 * @name FORGE.Plugin#instanceReady
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Plugin.prototype, "instanceReady",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._instanceReady;
    }
});

/**
 * Get the plugin container.
 * @name FORGE.Plugin#container
 * @readonly
 * @type {FORGE.DisplayObjectContainer}
 */
Object.defineProperty(FORGE.Plugin.prototype, "container",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        if(this._container === null)
        {
            if(typeof this._parent === "string" && this._parent !== "")
            {
                this._parent = document.getElementById(this._parent);

                if(typeof this._parent === "undefined" || this._parent === null || FORGE.Dom.isHtmlElement(this._parent) === false)
                {
                    throw "FORGE.Plugin.boot : Plugin parent is invalid";
                }

                this._container = new FORGE.DisplayObjectContainer(this._viewer, null, null, this._parent);
            }
            else
            {
                this._container = new FORGE.DisplayObjectContainer(this._viewer);
                this._viewer.pluginContainer.addChild(this._container);
                this._container.maximize(true);
            }

            this._container.id = this._uid;
            this._container.index = this._index;
        }

        return this._container;
    }
});

/**
 * Get the plugin object factory.
 * @name FORGE.Plugin#create
 * @readonly
 * @type {FORGE.PluginObjectFactory}
 */
Object.defineProperty(FORGE.Plugin.prototype, "create",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        if(this._create === null)
        {
            this._create = new FORGE.PluginObjectFactory(this._viewer, this);
        }

        return this._create;
    }
});

/**
 * Get the plugin keep flag.
 * @name FORGE.Plugin#keep
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Plugin.prototype, "keep",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._keep;
    }
});

/**
 * Get the plugin reset flag.
 * @name FORGE.Plugin#reset
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.Plugin.prototype, "reset",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._reset;
    }
});

/**
 * Get the contextual config.
 * @name FORGE.Plugin#contextualConfig
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "contextualConfig",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._contextualConfig;
    },

    /** @this {FORGE.Plugin} */
    set: function(value)
    {
        this._contextualConfig = value;
    }
});

/**
 * Get the instance config.
 * @name FORGE.Plugin#instanceConfig.
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "instanceConfig",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._instanceConfig;
    }
});

/**
 * Get the final merged config.
 * @name FORGE.Plugin#config
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.Plugin.prototype, "config",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        return this._config;
    }
});


/**
 * Get the "onInstanceCreate" {@link FORGE.EventDispatcher} of the Plugin.
 * @name FORGE.Plugin#onInstanceCreate
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Plugin.prototype, "onInstanceCreate",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        if(this._onInstanceCreate === null)
        {
            this._onInstanceCreate = new FORGE.EventDispatcher(this, true);
        }

        return this._onInstanceCreate;
    }
});

/**
 * Get the "onInstanceReady" {@link FORGE.EventDispatcher} of the Plugin.
 * @name FORGE.Plugin#onInstanceReady
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.Plugin.prototype, "onInstanceReady",
{
    /** @this {FORGE.Plugin} */
    get: function()
    {
        if(this._onInstanceReady === null)
        {
            this._onInstanceReady = new FORGE.EventDispatcher(this, true);
        }

        return this._onInstanceReady;
    }
});

/**
 * A plugin Engine handle the javascript files to load, and the constructor.
 *
 * @constructor FORGE.PluginEngine
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 *
 * @todo  In the sources list (in plugin manifest), add possibility to specify type (script or css) for special cases.
 */
FORGE.PluginEngine = function(viewer)
{
    FORGE.BaseObject.call(this, "PluginEngine");

    /**
     * The viewer reference.
     * @name FORGE.PluginEngine#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The config of the plugin engine.
     * @name  FORGE.PluginEngine#_config
     * @type {?PluginEngineConfig}
     * @private
     */
    this._config = null;

    /**
     * The URL prefix for the engine, if not specified, will be the global URL prefix.
     * @name FORGE.PluginEngine#_prefix
     * @type {string}
     * @private
     */
    this._prefix = "";

    /**
     * The url of the plugin engine relative to the prefix location.
     * @name  FORGE.PluginEngine#_url
     * @type {?string}
     * @private
     */
    this._url = "";

    /**
     * The manifest json file name.
     * @name  FORGE.PluginEngine#_manifest
     * @type {string}
     * @private
     */
    this._manifest = "";

    /**
     * The full url of the plugin folder (prefix + url)
     * @name FORGE.PluginEngine#_fullUrl
     * @type {string}
     * @private
     */
    this._fullUrl = "";

    /**
     * The manifest is a json data that describes the plugin engine.
     * @name FORGE.PluginEngine#_manifestData
     * @type {?PluginManifest}
     * @private
     */
    this._manifestData = null;

    /**
     * The name of the plugin engine.
     * @name FORGE.PluginEngine#_name
     * @type {?string}
     * @private
     */
    this._name = "";

    /**
     * The version of the plugin engine.
     * @name FORGE.PluginEngine#_version
     * @type {?string}
     * @private
     */
    this._version = "";

    /**
     * The sources of the plugin engine.
     * @name FORGE.PluginEngine#_sources
     * @type {Array<string>}
     * @private
     */
    this._sources = null;

    /**
     * The name of the constructor of the plugin engine.
     * @name FORGE.PluginEngine#_constructorName
     * @type {string}
     * @private
     */
    this._constructorName = "";

    /**
     * The constructor function reference of the plugin engine.
     * @name  FORGE.PluginEngine#_constructorFunction
     * @type {Function}
     * @private
     */
    this._constructorFunction = null;

    /**
     * i18n configuration.
     * @name FORGE.PluginEngine#_i18n
     * @type {Object}
     * @private
     */
    this._i18n = null;

    /**
     * Object that handles customization options.
     * @name FORGE.PluginEngine#_options
     * @type {Object}
     * @private
     */
    this._options = null;

    /**
     * The events of the engine on which you can bind an action.
     * @name FORGE.PluginEngine#_events
     * @type {Object}
     * @private
     */
    this._events = null;

    /**
     * On a plugin engine, actions is an array of exposed method names.
     * @name FORGE.PluginEngine#_actions
     * @type {Array<Object>}
     * @private
     */
    this._actions = null;

    /**
     * Internal counter for loaded source files.
     * @name FORGE.PluginEngine#_sourcesLoadedCount
     * @type {number}
     * @private
     */
    this._sourcesLoadedCount = 0;

    /**
     * Number of instances that are created from this engine.
     * @name FORGE.PluginEngine#_instanceCount
     * @type {number}
     * @private
     */
    this._instancesCount = 0;

    /**
     * Is this plugin engine is considered as loaded.
     * @name FORGE.PluginEngine#_loaded
     * @type {boolean}
     * @private
     */
    this._loaded = false;

    /**
     * Event dipatcher for load complete.
     * @name FORGE.PluginEngine#_onLoadComplete
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onLoadComplete = null;
};

FORGE.PluginEngine.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PluginEngine.prototype.constructor = FORGE.PluginEngine;

/**
 * List of required method on a plugn engine.
 * @name FORGE.PluginEngine._REQUIRED
 * @type {Array<string>}
 * @const
 * @private
 */
FORGE.PluginEngine._REQUIRED = ["boot", "destroy"];

/**
 * List of reserved keyword on a plugn engine.
 * @name FORGE.PluginEngine._RESERVED
 * @type {Array<string>}
 * @const
 * @private
 */
FORGE.PluginEngine._RESERVED = ["viewer", "plugin"];

/**
 * Load a configuration for the plugin engine,<br>
 * the config describes the manifest url and the engine identification.
 * @method FORGE.PluginEngine#load
 * @param {PluginEngineConfig} config - The config of the plugin engine.
 * @return {boolean} Returns true if the engine is loaded.
 */
FORGE.PluginEngine.prototype.load = function(config)
{
    if(typeof config !== "undefined")
    {
        this._config = config;

        this._parseConfig(this._config);

        var manifestUrl = this._fullUrl + this._manifest;

        this._loadManifest(manifestUrl);

        return true;
    }

    return false;
};

/**
 * Get a new instance of the engine and associate it to a {@link FORGE.Plugin} Object.
 * @method FORGE.PluginEngine#getNewInstance
 * @param  {FORGE.Viewer} viewer - The viewer attached to this instance.
 * @param  {FORGE.Plugin} plugin - The plugin who will represent this instance.
 * @return {?PluginStructure} Returns the instance of the engine.
 */
FORGE.PluginEngine.prototype.getNewInstance = function(viewer, plugin)
{
    var instance = /** @type {PluginStructure} */ (new this._constructorFunction());

    if(this._validateInstance(instance))
    {
        instance._viewer = viewer;
        instance._plugin = plugin;

        this._instancesCount++;

        return instance;
    }

    return null;
};

/**
 * Parse a plugin engine configuration.
 * @method FORGE.PluginEngine#_parseConfig
 * @param  {PluginEngineConfig} config - The configuration to parse.
 * @private
 */
FORGE.PluginEngine.prototype._parseConfig = function(config)
{
    if(typeof config.uid === "undefined")
    {
        throw "Can't load a plugin engine, missing id in config";
    }

    this._uid = config.uid;
    this._register();

    this._prefix = (typeof config.prefix === "string") ? config.prefix : this._viewer.plugins.prefix;

    if(typeof config.url === "undefined")
    {
        throw "Can't load a plugin engine, missing url in config";
    }
    this._url = config.url;

    this._fullUrl = this._prefix + this._url;

    this._manifest = (typeof config.manifest === "string") ? config.manifest : "manifest.json";
};

/**
 * Internal method to load the engine manifest.
 * @method FORGE.PluginEngine#_loadManifest
 * @param  {string} url - The url of the manifest to load.
 * @private
 */
FORGE.PluginEngine.prototype._loadManifest = function(url)
{
    this._viewer.load.json(this._uid, url, this._manifestLoadComplete, this);
};

/**
 * Handler for the manifest load complete.
 * @method FORGE.PluginEngine#_manifestLoadComplete
 * @private
 * @param  {FORGE.File} file - The  manifest {@link FORGE.File}.
 */
FORGE.PluginEngine.prototype._manifestLoadComplete = function(file)
{
    var json = this._viewer.cache.get(FORGE.Cache.types.JSON, file.key);

    this._manifestData = /** @type {PluginManifest} */ (json.data);
    this._parseManifest(this._manifestData);

    this.log("FORGE.Plugin._manifestLoadComplete();");
};

/**
 * Handler for the manifest load error.
 * @method FORGE.PluginEngine#_manifestLoadError
 * @private
 * @param  {FORGE.File} file - The  manifest {@link FORGE.File}.
 */
FORGE.PluginEngine.prototype._manifestLoadError = function(file)
{
    throw "Can't load the plugin engine manifest "+file.url;
};

/**
 * Internal method to parse the loaded manifest file.
 * @method FORGE.PluginEngine#_parseManifest
 * @private
 * @param  {PluginManifest} manifest - Manifest object to parse.
 * @suppress {checkTypes}
 * @todo  need to change manifest.constructor to another keyword (not reserved)
 */
FORGE.PluginEngine.prototype._parseManifest = function(manifest)
{
    if(this._uid !== manifest.uid)
    {
        throw "Plugin Engine UID doesn't match with manifest UID";
    }

    if(this._versionCheck(manifest.viewer) === false)
    {
        this.warn("Version compatibility check for plugin "+manifest.uid+" failed!");
        this.warn(manifest.viewer);
        return;
    }

    if(FORGE.Device.check(manifest.device) === false)
    {
        this.warn("Device compatibility check for plugin "+manifest.uid+" failed!");
        this.warn(manifest.device);
        return;
    }

    if(typeof manifest.options !== "undefined")
    {
        this._options = manifest.options;
    }

    if(typeof manifest.data !== "undefined")
    {
        this._data = manifest.data;
    }

    if(typeof manifest.actions !== "undefined")
    {
        this._actions = manifest.actions;
    }

    if(typeof manifest.events !== "undefined")
    {
        this._events = manifest.events;
    }

    this._name = manifest.name;
    this._version = manifest.version;
    this._sources = manifest.sources;
    this._constructorName = manifest.constructor;
    this._i18n = this._parseLocales(manifest);

    this._loadSources(this._sources);
};

/**
 * Compare FORGE.VERSION with a minimal and maximal compatibility version.
 * @method FORGE.PluginEngine#_versionCheck
 * @private
 * @param  {Object} config - The viewer version configuration object that contain min and max compatible version of the viewer for the plugin.
 * @return {boolean} Returns true if the plugin is compatible with the current FORGE.VERSION, false if not.
 */
FORGE.PluginEngine.prototype._versionCheck = function(config)
{
    var min = "0.0.0";
    var max = "9.9.9";

    if(typeof config !== "undefined")
    {
        if(typeof config.min !== "undefined")
        {
            min = config.min;
        }

        if(typeof config.max !== "undefined")
        {
            max = config.max;
        }
    }

    var viewerVersion = FORGE.VERSION.split(".");
    var minVersion = min.split(".");
    var maxVersion = max.split(".");
    var maxLength = Math.max(viewerVersion.length, minVersion.length, maxVersion.length);

    var viewerN = 0;
    var minN = 0;
    var maxN = 0;
    var inc = 1;
    for(var i = maxLength - 1; i >= 0; i--)
    {
        if(typeof viewerVersion[i] !== "undefined")
        {
            viewerN += parseInt(viewerVersion[i], 10) * inc;
        }
        if(typeof minVersion[i] !== "undefined")
        {
            minN += parseInt(minVersion[i], 10) * inc;
        }
        if(typeof maxVersion[i] !== "undefined")
        {
            maxN += parseInt(maxVersion[i], 10) * inc;
        }

        inc *= 10;
    }

    if(viewerN < minN || viewerN > maxN)
    {
        return false;
    }

    return true;
};

/**
 * Internal method to load a list of sources files of the engine.
 * @method FORGE.PluginEngine#_loadSources
 * @private
 * @param  {Array<string>} sources - Array of urls of files to load relative sto the base url of the engine.
 */
FORGE.PluginEngine.prototype._loadSources = function(sources)
{
    var sourceUrl;
    for(var i = 0, ii = sources.length; i < ii; i++)
    {
        //If source is an absolute URL
        if(String(sources[i]).substring(0, 7) === "http://" || String(sources[i]).substring(0, 8) === "https://")
        {
            sourceUrl = sources[i];
        }
        // Else it is relative to the plugin folder
        else
        {
            sourceUrl = this._fullUrl + sources[i];
        }

        this._loadSource(sourceUrl);
    }
};

/**
 * Load a single source file file.
 * @method  FORGE.PluginEngine#_loadSource
 * @private
 * @param {string} url - Url of the file to load.
 */
FORGE.PluginEngine.prototype._loadSource = function(url)
{
    var parsedURL = FORGE.URL.parse(url);

    if( parsedURL.extension === "css")
    {
        this._viewer.load.css(url, this._loadSourceComplete, this);
    }
    else
    {
        this._viewer.load.script(url, this._loadSourceComplete, this);
    }
};

/**
 * Handler for the load source complete.
 * @method FORGE.PluginEngine#_loadSourceComplete
 * @private
 */
FORGE.PluginEngine.prototype._loadSourceComplete = function()
{
    this._sourcesLoadedCount++;

    if(this._sourcesLoadedCount === this._sources.length)
    {
        this._constructorFunction = this._parseConstructor(this._constructorName);

        if(this._validateEngine() === true)
        {
            this._addLocales(this._i18n);

            this._loaded = true;

            if(this._onLoadComplete !== null)
            {
                this._onLoadComplete.dispatch();
            }
        }
    }
};

/**
 * Add the locale configuration to the i18n global manager.
 * @method FORGE.PluginEngine#_addlocales
 * @private
 * @param {Object} i18n - The i18n config to add.
 */
FORGE.PluginEngine.prototype._addLocales = function(i18n)
{
    if(i18n !== null)
    {
        this._viewer.i18n.addConfig(i18n);
    }
};

/**
 * For i18n, we check nodes for files and redefine the url for the loading of files<br>
 * with a path relative to the root of the project.
 * @method  FORGE.PluginEngine#_parseLocales
 * @private
 * @param  {Object} manifest - The manifest passed by the main parser.
 * @return {Object} The updated i18n object with paths relative to the project.
 */
FORGE.PluginEngine.prototype._parseLocales = function(manifest)
{
    var i18n = null;

    if(typeof manifest.i18n !== "undefined")
    {
        i18n = manifest.i18n;
        if(typeof i18n.locales !== "undefined")
        {
            var locale;
            for(var i = 0, ii = i18n.locales.length; i < ii; i++)
            {
                locale = i18n.locales[i];
                if(typeof locale.files !== "undefined")
                {
                    var file;
                    for(var j = 0, jj = locale.files.length; j < jj; j++)
                    {
                        file = locale.files[j];
                        file.key = this._uid + "-" + file.key;
                        file.url = this._fullUrl + file.url;
                    }
                }
            }
        }
    }

    return i18n;
};

/**
 * Parse the constructor string.
 * @method FORGE.PluginEngine#_parseConstructor
 * @param {string} constructorName - The constructor name to parse. (ex: "Namespace.PluginConstructor")
 * @return {Function} Return the found constructor function, null if not found
 * @private
 */
FORGE.PluginEngine.prototype._parseConstructor = function(constructorName)
{
    if(typeof constructorName === "string" && constructorName !== "")
    {
        var pathArray = constructorName.split(".");
        var currentObject = window;
        for(var i = 0, ii = pathArray.length; i < ii; i++)
        {
            if(i === pathArray.length - 1 && typeof currentObject[pathArray[i]] === "function")
            {
                return currentObject[pathArray[i]];
            }
            else if(typeof currentObject[pathArray[i]] === "object")
            {
                currentObject = currentObject[pathArray[i]];
            }
            else
            {
                return null;
            }
        }
    }

    return null;
};

/**
 * Internal method to validate the instance.
 * @method FORGE.PluginEngine#_validateEngine
 * @private
 * @return {boolean} Returns true if the engine instance is valid.
 */
FORGE.PluginEngine.prototype._validateEngine = function()
{
    if(this._constructorFunction === null || typeof this._constructorFunction !== "function")
    {
        throw "Plugin engine "+this._uid+" can't find it's constructor";
    }

    for(var i = 0, ii = FORGE.PluginEngine._REQUIRED.length; i < ii; i++)
    {
        if(typeof this._constructorFunction.prototype[FORGE.PluginEngine._REQUIRED[i]] !== "function")
        {
            throw "FORGE.PluginEngine engine validation failed, missing required method "+FORGE.PluginEngine._REQUIRED[i]+" on plugin engine "+this._name;
        }
    }

    //Add a readonly reference of the viewer to the plugin engine instance.
    Object.defineProperty(this._constructorFunction.prototype, "viewer",
    {
        /** @this {FORGE.PluginEngine} */
        get: function()
        {
            return this._viewer;
        }
    });

    //Add a readonly reference of the plugin to the plugin engine instance.
    Object.defineProperty(this._constructorFunction.prototype, "plugin",
    {
        // This annotation is not really true
        /** @this {FORGE.PluginObjectFactory} */
        get: function()
        {
            return this._plugin;
        }
    });

    return true;
};

/**
 * Internal method to validate that the instance do not use one of the reserved keywords.
 * @method FORGE.PluginEngine#_validateReserved
 * @private
 * @param  {Object} instance - The instance to validate.
 * @return {boolean} Returns true if no reserved keywords are in use on the instance.
 */
FORGE.PluginEngine.prototype._validateReserved = function(instance)
{
    for(var i = 0, ii = FORGE.PluginEngine._RESERVED.length; i < ii; i++)
    {
        if(typeof instance[FORGE.PluginEngine._RESERVED[i]] !== "undefined")
        {
            throw "FORGE.PluginEngine instance validation failed, "+FORGE.PluginEngine._RESERVED[i]+" is reserved";
        }
    }

    return true;
};

/**
 * Internal method to validate that the actions listed on the manifest are part of the instance.
 * @method FORGE.PluginEngine#_validateActions
 * @private
 * @param  {Object} instance - The instance to validate.
 * @return {boolean} Returns true if actions are valid on the instance.
 */
FORGE.PluginEngine.prototype._validateActions = function(instance)
{
    if(this._actions === null)
    {
        return true;
    }

    for(var i = 0, ii = this._actions.length; i < ii; i++)
    {
        if(typeof instance[this._actions[i]] !== "function")
        {
            throw "FORGE.PluginEngine instance validation failed, action "+this._actions[i]+" is undefined on plugin "+this._constructorName;
        }
    }

    return true;
};

/**
 * This method validate the instace by testing both reserved keywords and actions.
 * @method FORGE.PluginEngine#_validateEngine
 * @private
 * @param  {Object} instance - The instance to validate.
 * @return {boolean} Returns true if the instance is valid.
 */
FORGE.PluginEngine.prototype._validateInstance = function(instance)
{
    return (this._validateReserved(instance) === true && this._validateActions(instance) === true);
};

/**
 * Destroy method.
 * @method FORGE.PluginEngine#destroy
 */
FORGE.PluginEngine.prototype.destroy = function()
{
    this._viewer = null;
    this._config = null;
    this._manifestData = null;
    this._name = null;
    this._version = null;
    this._sources = null;
    this._constructorFunction = null;
    this._i18n = null;

    if(this._onLoadComplete !== null)
    {
        this._onLoadComplete.destroy();
        this._onLoadComplete = null;
    }

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Get the name of this plugin engine.
 * @name FORGE.PluginEngine#name
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "name",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._name;
    }
});

/**
 * Get the loaded status of this plugin engine.
 * True if the plugin is considered as loaded.
 * @name FORGE.PluginEngine#loaded
 * @readonly
 * @type {boolean}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "loaded",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._loaded;
    }
});

/**
 * Get the base url of the plugin engine.
 * @name FORGE.PluginEngine#url
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "url",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._url;
    }
});

/**
 * Get the base url of the plugin engine.
 * @name FORGE.PluginEngine#fullUrl
 * @readonly
 * @type {string}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "fullUrl",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._fullUrl;
    }
});

/**
 * Get the options of the plugin engine.
 * @name FORGE.PluginEngine#options
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "options",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._options;
    }
});

/**
 * Get the available actions of the plugin engine.
 * @name FORGE.PluginEngine#actions
 * @readonly
 * @type {Array<String>}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "actions",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._actions;
    }
});

/**
 * Get the available events of the plugin engine.
 * @name FORGE.PluginEngine#events
 * @readonly
 * @type {Object}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "events",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        return this._events;
    }
});

/**
 * Get the default plugin configuration that contains (data + options + actions + events).
 * @name FORGE.PluginEngine#defaultConfig
 * @readonly
 * @type {PluginConfiguration}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "defaultConfig",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        /** @type {PluginConfiguration} */
        var config =
        {
            data: /** @type {Object} */ (this._data),
            options: this._options,
            actions: this._actions,
            events: this._events
        };

        return config;
    }
});

/**
 * Get the on load complete {@link FORGE.EventDispatcher} of the plugin engine.
 * @name FORGE.PluginEngine#onLoadComplete
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PluginEngine.prototype, "onLoadComplete",
{
    /** @this {FORGE.PluginEngine} */
    get: function()
    {
        if(this._onLoadComplete === null)
        {
            this._onLoadComplete = new FORGE.EventDispatcher(this);
        }

        return this._onLoadComplete;
    }
});


/**
 * The FORGE.PluginManager manages {@link FORGE.PluginEngine} and {@link FORGE.Plugin}.
 *
 * @constructor FORGE.PluginManager
 * @param {FORGE.Viewer} viewer - The {@link FORGE.Viewer} reference.
 * @extends {FORGE.BaseObject}
 */
FORGE.PluginManager = function(viewer)
{
    FORGE.BaseObject.call(this, "PluginManager");

    /**
     * The viewer reference.
     * @name FORGE.PluginManager#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * Config of the plugins
     * @name  FORGE.PluginManager#_config
     * @type {PluginsConfig}
     * @private
     */
    this._config;

    /**
     * Does the plugin manager is enabled ?
     * @name  FORGE.PluginManager#_enabled
     * @type {boolean}
     * @private
     */
    this._enabled = true;

    /**
     * Prefix url for the global plugins folder.
     * @name FORGE.PluginManager#_prefix
     * @type {string}
     * @private
     */
    this._prefix = "";

    /**
     * The engines list.
     * @name FORGE.PluginManager#_engines
     * @type {FORGE.Collection<FORGE.PluginEngine>}
     * @private
     */
    this._engines = null;

    /**
     * The plugins list.
     * @name FORGE.PluginManager#_plugins
     * @type {FORGE.Collection<FORGE.Plugin>}
     * @private
     */
    this._plugins = null;

    /**
     * Event dispatcher for instance creation
     * @name  FORGE.PluginManager#_onInstanceCreate
     * @type {FORGE.EventDispatcher}
     * @private
     */
    this._onInstanceCreate = null;
};

FORGE.PluginManager.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PluginManager.prototype.constructor = FORGE.PluginManager;

/**
 * Parse the main plugin config.
 * @method FORGE.PluginManager#_parseMainConfig
 * @param {PluginsConfig} config - The configuration of the main plugins node to parse.
 * @private
 */
FORGE.PluginManager.prototype._parseMainConfig = function(config)
{
    this._config = config;

    this._enabled = (typeof config.enabled === "boolean") ? config.enabled : true;
    this._prefix = (typeof config.prefix === "string") ? config.prefix : "./plugins/";
};

/**
 * Parse a plugin config.
 * @method FORGE.PluginManager#_parseConfig
 * @param {PluginsConfig} config - The configuration of the main plugins node to parse.
 * @private
 */
FORGE.PluginManager.prototype._parseConfig = function(config)
{
    this._parseEngines(config.engines);
    this._parseInstances(config.instances);
    this._parseConfigurations(config.configurations);
};


/**
 * Event handler for scene load start.
 * @method FORGE.PluginManager#_sceneLoadStartHandler
 * @private
 */
FORGE.PluginManager.prototype._sceneLoadStartHandler = function()
{
    //Remove plugin that have keep = false and plugin that have scene restrictions.

    this._removeUnkeptPlugins(this._viewer.story.sceneUid);

    //Add plugins =============================================

    if(this._enabled === true && typeof this._config !== "undefined")
    {
        this._parseConfig(this._config);
    }

    if(this._enabled === true && typeof this._viewer.story.scene.config.plugins !== "undefined")
    {
        this._parseConfig(this._viewer.story.scene.config.plugins);
    }

    //Reset plugins ============================================

    this._resetPlugins();
};

/**
 * Remove plugins that have a keep value @ false OR if the plugin have a scene restriction.
 * @method FORGE.PluginManager#_removeUnkeptPlugins
 * @param {string} [sceneUid] The scene uid received on scene load start.
 * @private
 */
FORGE.PluginManager.prototype._removeUnkeptPlugins = function(sceneUid)
{
    var plugin;
    var count = this._plugins.size;

    while(count--)
    {
        plugin = this._plugins.get(count);

        if(plugin.keep === false || (plugin.scenes !== null && plugin.scenes.indexOf(sceneUid) === -1))
        {
            this.remove(plugin.uid);
        }
    }
};

/**
 * This method will reset plugins that have a reset value @ true between each scene.
 * @method  FORGE.PluginManager#_resetPlugins
 * @private
 */
FORGE.PluginManager.prototype._resetPlugins = function()
{
    var plugin;
    var count = this._plugins.size;

    while(count--)
    {
        plugin = this._plugins.get(count);

        if(plugin.reset === true)
        {
            plugin.resetInstance();
        }
    }
};

/**
 * Parse plugins engines in main config.
 * @method FORGE.PluginManager#_parseEngines
 * @param  {Array<PluginEngineConfig>} engines - The engines object to parse.
 * @private
 */
FORGE.PluginManager.prototype._parseEngines = function(engines)
{
    if(typeof engines !== "undefined")
    {
        for(var i = 0, ii = engines.length; i < ii; i++)
        {
            this._addEngine(engines[i]);
        }
    }
};

/**
 * Parse plugins instances in main config.
 * @method FORGE.PluginManager#_parseInstances
 * @param  {Array<PluginInstanceConfig>} instances - The instances object to parse.
 * @private
 */
FORGE.PluginManager.prototype._parseInstances = function(instances)
{
    if(typeof instances !== "undefined")
    {
        for(var i = 0, ii = instances.length; i < ii; i++)
        {
            var instanceUid = instances[i].uid;
            var enabled = (instances[i].enabled !== false) ? true : false;
            var plugin = this.getById(instanceUid);

            var sceneUid = this._viewer.story.sceneUid;
            var scenes = (FORGE.Utils.isArrayOf(instances[i].scenes, "string")) ? instances[i].scenes : [];

            //Scene valid is true if there is no current scene AND there are no scene restriction for the instance,
            //OR if there is a scene restriction and the current scene UID is in the restriction array!
            var sceneValid = Boolean( scenes.length === 0 || (sceneUid === "" && scenes.length === 0) || scenes.indexOf(sceneUid) !== -1 );

            if(enabled === true && plugin === null && sceneValid === true)
            {
                this._addInstance(instances[i]);
            }
        }
    }
};

/**
 * Parse plugins configurations in main config.
 * @method FORGE.PluginManager#_parseConfigurations
 * @param  {Array<PluginConfigurationConfig>} configurations - The configurations object to parse.
 * @private
 */
FORGE.PluginManager.prototype._parseConfigurations = function(configurations)
{
    if(typeof configurations !== "undefined")
    {
        for(var i = 0, ii = configurations.length; i < ii; i++)
        {
            var uid = configurations[i].instance;
            var enabled = (configurations[i].enabled !== false) ? true : false;
            var plugin = this.getById(uid);
            var config = null;

            if(enabled === true)
            {
                config = configurations[i];
            }

            if(plugin !== null)
            {
                plugin.contextualConfig = config;
            }
        }
    }
};

/**
 * Add an engine.
 * @method FORGE.PluginManager#_addEngine
 * @param {PluginEngineConfig} config - The config to parse.
 * @private
 */
FORGE.PluginManager.prototype._addEngine = function(config)
{
    if(typeof config.uid === "string")
    {
        if(this._getEngine(config.uid) !== null)
        {
            return;
        }

        if(FORGE.UID.exists(config.uid) === false)
        {
            var engine = new FORGE.PluginEngine(this._viewer);
            this._engines.add(engine);

            engine.load(config);
        }
        else
        {
            // In some multiple viewer instance case the engine already exists, just get it from the UID
            this._engines.add(FORGE.UID.get(config.uid));
        }
    }
};

/**
 * Get an engine.
 * @method FORGE.PluginManager#_getEngine
 * @param {string} uid - The uid to search for.
 * @return {?FORGE.PluginEngine} Returns the engine.
 * @private
 */
FORGE.PluginManager.prototype._getEngine = function(uid)
{
    var engine = null;
    for(var i = 0, ii = this._engines.size; i < ii; i++)
    {
        engine = /** @type {FORGE.PluginEngine} */ (this._engines.get(i));

        if(engine.uid === uid)
        {
            return engine;
        }
    }

    return null;
};

/**
 * Add a plugin instance.
 * @method FORGE.PluginManager#_addInstance
 * @param {PluginInstanceConfig} config - The config to parse.
 * @private
 * @return {FORGE.Plugin} The created plugin object.
 */
FORGE.PluginManager.prototype._addInstance = function(config)
{
    if(typeof config.engine === "undefined")
    {
        throw "Can't create plugin instance, engineUid is undefined";
    }

    var engine = this._getEngine(config.engine);

    if(engine === null)
    {
        throw "Plugin Engine "+config.engine+" doesn't exist";
    }

    var index = typeof config.index === "number" ? config.index : this._plugins.size;

    var plugin = new FORGE.Plugin(this._viewer, engine, config, index);
    plugin.onInstanceCreate.addOnce(this._onInstanceCreateHandler, this);
    this._plugins.add(plugin);

    plugin.instantiate();

    return plugin;
};

/**
 * Instance create handler, wil lredispatch from the plugin manager
 * @method FORGE.PluginManager#_onInstanceCreateHandler
 * @param {FORGE.Event} event
 * @private
 */
FORGE.PluginManager.prototype._onInstanceCreateHandler = function(event)
{
    var plugin = event.emitter;

    if(this._onInstanceCreate !== null)
    {
        this._onInstanceCreate.dispatch(plugin);
    }
};

/**
 * Boot sequence.
 * @method FORGE.PluginManager#_boot
 * @private
 */
FORGE.PluginManager.prototype.boot = function()
{
    this._engines = new FORGE.Collection();

    this._plugins = new FORGE.Collection();

    this._viewer.story.onSceneLoadStart.add(this._sceneLoadStartHandler, this);
};

/**
 * Add plugins configuration.
 * @method FORGE.PluginManager#addConfig
 * @param {PluginsConfig} config - The config to add
 */
FORGE.PluginManager.prototype.addConfig = function(config)
{
    this._parseMainConfig(config);
};

/**
 * Get a plugin instance by uid.
 * @method FORGE.PluginManager#getById
 * @param {string} uid - The uid to search for.
 * @return {FORGE.Plugin} Returns the plugin.
 */
FORGE.PluginManager.prototype.getById = function(uid)
{
    var plugin;
    for(var i = 0, ii = this._plugins.size; i < ii; i++)
    {
        plugin = /** @type {FORGE.Plugin} */ (this._plugins.get(i));

        if(plugin.uid === uid)
        {
            return plugin;
        }
    }

    return null;
};

/**
 * Get a plugin instance by index.
 * @method FORGE.PluginManager#getByIndex
 * @param {number} index - The index to search for.
 * @return {FORGE.Plugin} Returns the plugin.
 */
FORGE.PluginManager.prototype.getByIndex = function(index)
{
    var plugin = /** @type {FORGE.Plugin} */ (this._plugins.get(index));

    if(typeof plugin !== "undefined")
    {
        return plugin;
    }
    else
    {
        return null;
    }
};

/**
 * Get a plugin instance by value.
 * @method FORGE.PluginManager#get
 * @param {number|string} value - The index or uid to search for.
 * @return {FORGE.Plugin} Returns the plugin.
 */
FORGE.PluginManager.prototype.get = function(value)
{
    var plugin = null;

    if(typeof value === "string")
    {
        plugin = this.getById(value);
    }
    else if(typeof value === "number")
    {
        plugin = this.getByIndex(value);
    }

    return plugin;
};

/**
 * Remove a plugin instance.
 * @method FORGE.PluginManager#remove
 * @param {string} uid - The uid to search for.
 */
FORGE.PluginManager.prototype.remove = function(uid)
{
    var plugin = this.get(uid);

    plugin.destroy();

    this._plugins.remove(plugin);
};

/**
 * Update main loop for plugins.
 * @method FORGE.PluginManager#update
 */
FORGE.PluginManager.prototype.update = function()
{
    var plugin;
    for(var i = 0, ii = this._plugins.size; i < ii; i++)
    {
        plugin = this._plugins.get(i);
        plugin.update();
    }
};

/**
 * Render main loop for plugins.
 * @method FORGE.PluginManager#render
 */
FORGE.PluginManager.prototype.render = function()
{

};

/**
 * Destroy method.
 * @method FORGE.PluginManager#destroy
 */
FORGE.PluginManager.prototype.destroy = function()
{

    this._viewer = null;

    if(this._onInstanceCreate !== null)
    {
        this._onInstanceCreate.destroy();
        this._onInstanceCreate = null;
    }

    for(var i = 0, ii = this._engines.size; i < ii; i++)
    {
        this._engines.get(i).destroy();
    }
    this._engines = null;

    for(var j = 0, jj = this._plugins.size; j < jj; j++)
    {
        this._plugins.get(j).destroy();
    }
    this._plugins = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
* Get the all plugins.
* @name FORGE.PluginManager#all
* @type {Array<FORGE.Plugin>}
* @readonly
*/
Object.defineProperty(FORGE.PluginManager.prototype, "all",
{
    /** @this {FORGE.PluginManager} */
    get: function()
    {
        return this._plugins;
    }
});

/**
* Get the enabled flag of plugins.
* @name FORGE.PluginManager#enabled
* @type {boolean}
* @readonly
*/
Object.defineProperty(FORGE.PluginManager.prototype, "enabled",
{
    /** @this {FORGE.PluginManager} */
    get: function()
    {
        return this._enabled;
    }
});

/**
* Get the global prefix of plugins engines.
* @name FORGE.PluginManager#prefix
* @type {string}
* @readonly
*/
Object.defineProperty(FORGE.PluginManager.prototype, "prefix",
{
    /** @this {FORGE.PluginManager} */
    get: function()
    {
        return this._prefix;
    }
});

/**
 * Get the "onInstanceCreate" {@link FORGE.EventDispatcher} of the PluginManager.
 * @name FORGE.PluginManager#onInstanceCreate
 * @readonly
 * @type {FORGE.EventDispatcher}
 */
Object.defineProperty(FORGE.PluginManager.prototype, "onInstanceCreate",
{
    /** @this {FORGE.PluginManager} */
    get: function()
    {
        if(this._onInstanceCreate === null)
        {
            this._onInstanceCreate = new FORGE.EventDispatcher(this, true);
        }

        return this._onInstanceCreate;
    }
});




/**
 * Factory helper to create object inside plugins.<br>
 * The factory knows what objects are created for a plugin, so it can destroy all object of the plugin at plugin destroy.
 *
 * @constructor FORGE.PluginObjectFactory
 * @param {FORGE.Viewer} viewer - {@link FORGE.Viewer} reference.
 * @param {FORGE.Plugin} plugin - The {@link FORGE.Plugin} that will use this factory.
 * @extends {FORGE.BaseObject}
 *
 * @todo add the file key restriction for i18n stuff
 */
FORGE.PluginObjectFactory = function(viewer, plugin)
{
    /**
     * The viewer reference.
     * @name FORGE.PluginEngine#_viewer
     * @type {FORGE.Viewer}
     * @private
     */
    this._viewer = viewer;

    /**
     * The plugin reference.
     * @name FORGE.PluginEngine#_plugin
     * @type {FORGE.Plugin}
     * @private
     */
    this._plugin = plugin;

    /**
     * The type of object list.
     * @name FORGE.PluginEngine#_objects
     * @type {Array<Object>}
     * @private
     */
    this._objects = [];

    FORGE.BaseObject.call(this, "PluginObjectFactory");
};

FORGE.PluginObjectFactory.prototype = Object.create(FORGE.BaseObject.prototype);
FORGE.PluginObjectFactory.prototype.constructor = FORGE.PluginObjectFactory;

/**
 * Search for an object into the list.
 * @method FORGE.PluginObjectFactory#_indexOfObject
 * @param  {Object} obj - The object to look for.
 * @return {number} Returns the index of the object found, if not, returns -1.
 * @private
 */
FORGE.PluginObjectFactory.prototype._indexOfObject = function(obj)
{
    var n = this._objects.length;
    var o;

    while(n--)
    {
        o = this._objects[n];

        if(o === obj)
        {
            return n;
        }
    }

    return -1;
};

/**
 * Internal handler on object destroy.
 * @method FORGE.PluginObjectFactory#_destroyObjectHandler
 * @param  {FORGE.Event} event - The event.
 * @private
 */
FORGE.PluginObjectFactory.prototype._destroyObjectHandler = function(event)
{
    var o = event.emitter;
    var i = this._indexOfObject(o);
    this._objects.splice(i, 1);
};

/**
 * Add a {@link FORGE.LocaleString}.
 * @method FORGE.PluginObjectFactory#string
 * @param  {string} key - Object key.
 * @return {FORGE.LocaleString} Returns the created FORGE.LocaleString.
 */
FORGE.PluginObjectFactory.prototype.string = function(key)
{
    var str = new FORGE.LocaleString(this._viewer, key);
    str.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(str);
    return str;
};

/**
 * Add a {@link FORGE.Textfield}.
 * @method FORGE.PluginObjectFactory#textField
 * @param  {(string|TextFieldConfig)} config - Object configuration.
 * @return {FORGE.TextField} Returns the created FORGE.Textfield.
 */
FORGE.PluginObjectFactory.prototype.textField = function(config)
{
    var textField = new FORGE.TextField(this._viewer, config);
    textField.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(textField);
    return textField;
};

/**
 * Add a {@link FORGE.Sound}.
 * @method FORGE.PluginObjectFactory#sound
 * @param {string} key - Object key.
 * @param {string} url - The sound url.
 * @return {FORGE.Sound} Returns the created FORGE.Sound.
 */
FORGE.PluginObjectFactory.prototype.sound = function(key, url)
{
    var sound = new FORGE.Sound(this._viewer, key, url);
    sound.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(sound);
    return sound;
};

/**
 * Add a {@link FORGE.DisplayObjectContainer}.
 * @method FORGE.PluginObjectFactory#displayObjectContainer
 * @param  {Element|HTMLElement} dom - Dom object.
 * @return {FORGE.DisplayObjectContainer} Returns the created FORGE.DisplayObjectContainer.
 */
FORGE.PluginObjectFactory.prototype.displayObjectContainer = function(dom)
{
    var displayObjectContainer = new FORGE.DisplayObjectContainer(this._viewer, dom);
    displayObjectContainer.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(displayObjectContainer);
    return displayObjectContainer;
};

/**
 * Add a {@link FORGE.DisplayObject}.
 * @method FORGE.PluginObjectFactory#displayObject
 * @param  {Element|HTMLElement} dom - Dom object.
 * @return {FORGE.DisplayObject} Returns the created FORGE.DisplayObject.
 */
FORGE.PluginObjectFactory.prototype.displayObject = function(dom)
{
    var displayObject = new FORGE.DisplayObject(this._viewer, dom);
    displayObject.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(displayObject);
    return displayObject;
};

/**
 * Add a {@link FORGE.Image}.
 * @method FORGE.PluginObjectFactory#image
 * @param  {string|Object} config - URL of the image or an image configuration object.
 * @param  {boolean} relativeToPluginPath
 * @return {FORGE.Image} Returns the created FORGE.Image.
 */
FORGE.PluginObjectFactory.prototype.image = function(config, relativeToPluginPath)
{
    var c = config;

    if(typeof config === "string")
    {
        c = {"key": null, "url": config};
    }

    c.key = this._plugin.uid + "-" + c.key;

    if(relativeToPluginPath !== false)
    {
        c.url = this._plugin.fullUrl + c.url;
    }

    var img = new FORGE.Image(this._viewer, c);
    img.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(img);
    return img;
};

/**
 * Add a {@link FORGE.Sprite}.
 * @method FORGE.PluginObjectFactory#sprite
 * @param  {Object} config - Sprite configuration object.
 * @param  {boolean} relativeToPluginPath
 * @return {FORGE.Image} Returns the created FORGE.Sprite.
 */
FORGE.PluginObjectFactory.prototype.sprite = function(config, relativeToPluginPath)
{
    var c = config;

    c.key = this._plugin.uid + "-" + c.key;

    if(relativeToPluginPath !== false)
    {
        c.url = this._plugin.fullUrl + c.url;
        c.frames = this._plugin.fullUrl + c.frames;
    }

    var sprite = new FORGE.Sprite(this._viewer, c);
    sprite.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(sprite);
    return sprite;
};

/**
 * Add a {@link FORGE.Button}.
 * @method FORGE.PluginObjectFactory#button
 * @param  {ButtonConfig} config - The button configuration object.
 * @return {FORGE.Button} Returns the created FORGE.Button.
 */
FORGE.PluginObjectFactory.prototype.button = function(config)
{
    var button = new FORGE.Button(this._viewer, config);
    button.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(button);
    return button;
};

/**
 * Add a {@link FORGE.Video}.
 * @method FORGE.PluginObjectFactory#video
 * @param {string} key - The video Id reference.
 * @param {?(string|FORGE.VideoQuality|Array<(FORGE.VideoQuality|string)>)=} config - The video configuration object
 * @param {string=} streaming - The video streaming format. Can be "HTML5" or "DASH".
 * @param {string=} qualityMode - The video quality mode. Can be "auto" or "manual".
 * @param {boolean=} ambisonic - 3D sound including ambisonics. For "HTML5" video only.
 * @return {(FORGE.VideoHTML5|FORGE.VideoDash)} Returns the created FORGE.Video object.
 */
FORGE.PluginObjectFactory.prototype.video = function(key, config, streaming, qualityMode, ambisonic)
{
    var video;

    if(typeof streaming !== "undefined" && streaming.toLowerCase() === FORGE.VideoFormat.DASH)
    {
        video = new FORGE.VideoDash(this._viewer, key, config, qualityMode);
    }
    else
    {
        video = new FORGE.VideoHTML5(this._viewer, key, config, qualityMode, ambisonic);
    }

    video.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(video);
    return video;
};

/**
 * Add a {@link FORGE.Canvas}.
 * @method FORGE.PluginObjectFactory#canvas
 * @return {FORGE.Canvas} Returns the created FORGE.Canvas.
 */
FORGE.PluginObjectFactory.prototype.canvas = function()
{
    var canvas = new FORGE.Canvas(this._viewer);
    canvas.onDestroy.addOnce(this._destroyObjectHandler, this);

    this._objects.push(canvas);
    return canvas;
};

/**
 * Add a {@link FORGE.Tween}.
 * @method FORGE.PluginObjectFactory#tween
 * @param {Object} object
 * @return {FORGE.Tween} Returns the created FORGE.Tween.
 */
FORGE.PluginObjectFactory.prototype.tween = function(object)
{
    var tween = new FORGE.Tween(this._viewer, object);
    tween.onDestroy.addOnce(this._destroyObjectHandler, this);
    this._viewer.tween.add(tween);

    this._objects.push(tween);
    return tween;
};

/**
 * Destroy all objects.
 * @method FORGE.PluginObjectFactory#destroyAllObjects
 */
FORGE.PluginObjectFactory.prototype.destroyAllObjects = function()
{
    this.log("destroyAllObjects();");

    var n = this._objects.length;

    while(n--)
    {
        this._objects[n].destroy();
    }
};

/**
 * Destroy method.
 * @method FORGE.PluginObjectFactory#destroy
 */
FORGE.PluginObjectFactory.prototype.destroy = function()
{
    this.destroyAllObjects();

    this._viewer = null;
    this._plugin = null;
    this._objects = null;

    FORGE.BaseObject.prototype.destroy.call(this);
};

/**
 * Manage Devices inside FORGE.<br>
 * Device is singleton, so if you have multiple instances in the same page you MUST avoid UID conflict.
 * @constructor
 * @extends {FORGE.BaseObject}
 */
FORGE.Device = (function(c)
 {
    var Tmp = c();
    Tmp.prototype = Object.create(FORGE.BaseObject.prototype);
    Tmp.prototype.constructor = Tmp;

    /**
     * Check which OS is running.
     * @method FORGE.Device#_checkOS
     * @suppress {checkRegExp}
     * @private
     */
    Tmp.prototype._checkOS = function()
    {
        if (/Playstation Vita/.test(this._ua))
        {
            this._os = "playstationvita";
            this._vita = true;
        }
        else if (/Xbox/.test(this._ua))
        {
            this._os = "xbox";
            this._xbox = true;
        }
        else if (/Kindle/.test(this._ua) || /\bKF[A-Z][A-Z]+/.test(this._ua) || /Silk.*Mobile Safari/.test(this._ua))
        {
            this._os = "kindle";
            this._kindle = true;
        }
        else if ((/Windows Phone/i).test(this._ua) || (/IEMobile/i).test(this._ua))
        {
            this._os = "windowsphone";
            this._windowsPhone = true;
            if (/Windows Phone (\d+)/.test(this._ua))
            {
                this._osVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (/Android/.test(this._ua))
        {
            this._os = "android";
            this._android = true;
            if (/Android ([\.\_\d]+)/.test(this._ua))
            {
                this._osVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (/CrOS/.test(this._ua))
        {
            this._os = "chromeos";
            this._chromeOS = true;
        }
        else if (/iP[ao]d|iPhone/i.test(this._ua))
        {
            this._os = "ios";
            this._iOS = true;
            if (/OS (\d+)/.test(navigator.appVersion))
            {
                this._osVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (/(Linux|X11)/.test(this._ua))
        {
            this._os = "linux";
            this._linux = true;
        }
        else if (/Mac OS X/.test(this._ua))
        {
            this._os = "macosx";
            this._macOS = true;
            if (/Mac OS X (10[\.\_\d]+)/.test(this._ua))
            {
                this._osVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (/Windows/.test(this._ua) || (/WPDesktop/i).test(this._ua))
        {
            if ((/WPDesktop/i).test(this._ua))
            {
                this._os = "windowsphone";
                this._windowsPhone = true;
            }
            else
            {
                this._os = "windows";
                this._windows = true;
            }
            if (/(Windows 10.0|Windows NT 10.0)/.test(this._ua))
            {
                this._osVersion = 10;
            }
            else if (/(Windows 8.1|Windows NT 6.3)/.test(this._ua) || /(Windows 8|Windows NT 6.2)/.test(this._ua))
            {
                this._osVersion = 8;
            }
            else if (/(Windows 7|Windows NT 6.1)/.test(this._ua))
            {
                this._osVersion = 7;
            }
        }
    };

    /**
     * Check which browser is running.
     * @method FORGE.Device#_checkBrowsers
     * @suppress {checkRegExp}
     * @private
     */
    Tmp.prototype._checkBrowsers = function()
    {
        this._ie = /** @type {boolean} */ (/*@cc_on!@*/ false || typeof document["documentMode"] !== "undefined");
        this._edge = this._ie === false && Boolean(window["StyleMedia"]) === true;
        this._firefox = typeof window["InstallTrigger"] !== "undefined";
        this._opera = Boolean(window["opr"]) === true || this._ua.indexOf(" OPR/") >= 0 || this._ua.indexOf("Opera") >= 0;
        this._safari = /^((?!chrome|android).)*safari/i.test(navigator.userAgent);
        this._chrome = this._ie === false && this._edge === false && this._opera === false && (Boolean(window["chrome"]) === true || this._ua.indexOf("CriOS") >= 0);

        if (this._edge)
        {
            this._browser = "edge";
            if (/Edge\/(\d+)/.test(this._ua))
            {
                this._browserVersion = this._edgeVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (this._chrome)
        {
            this._browser = "chrome";
            if (/CriOS\/(\d+)/.test(this._ua))
            {
                this._browserVersion = this._chromeVersion = parseInt(RegExp.$1, 10);
            }
            else if (/Chrome\/(\d+)/.test(this._ua))
            {
                this._browserVersion = this._chromeVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (this._firefox)
        {
            this._browser = "firefox";
            if (/Firefox\D+(\d+)/.test(this._ua))
            {
                this._browserVersion = this._firefoxVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (this._kindle)
        {
            // Silk gets its own if clause because its ua also contains 'Safari'
            if (/Silk/.test(this._ua))
            {
                this._browser = "silk";
                this._silk = true;
            }
        }
        else if (this._ie)
        {
            this._browser = "internetexplorer";
            if (/MSIE (\d+\.\d+);/.test(this._ua))
            {
                this._browserVersion = this._ieVersion = parseInt(RegExp.$1, 10);
            }
            else if (/Trident\/(\d+\.\d+)(.*)rv:(\d+\.\d+)/.test(this._ua))
            {
                this._browserVersion = this._ieVersion = parseInt(RegExp.$3, 10);
            }
        }
        else if (this._opera)
        {
            this._browser = "opera";
            if (/OPR\/(\d+)/.test(this._ua))
            {
                this._browserVersion = this._operaVersion = parseInt(RegExp.$1, 10);
            }
            else if (this._ua.indexOf("Opera/") >= 0)
            {
                if (/Version\/(\d+)/.test(this._ua))
                {
                    this._browserVersion = this._operaVersion = parseInt(RegExp.$1, 10);
                }
            }
            else if (/Opera (\d+)/.test(this._ua))
            {
                this._browserVersion = this._operaVersion = parseInt(RegExp.$1, 10);
            }
        }
        else if (this._safari)
        {
            this._browser = "safari";
            if ((/version\/(\d+(\.\d+)?)/i).test(this._ua))
            {
                this._browserVersion = this._safariVersion = parseInt(RegExp.$1, 10);
            }
        }
        else
        {
            var matches = this._ua.match(/Android.*AppleWebKit\/([\d.]+)/);
            if (matches && matches[1] < 537)
            {
                this._browser = "androidstock";
                this._isAndroidStockBrowser = true;
                this._browserVersion = this._androidStockBrowserVersion = parseFloat(this._ua.slice(this._ua.indexOf("Android") + 8));
            }
        }
        //  WebApp mode
        if (navigator.standalone)
        {
            this._webApp = true;
        }

        this._quirksMode = (document.compatMode === "CSS1Compat") ? false : true;
    };

    /**
     * Check the device informations.
     * @method FORGE.Device#_checkDevice
     * @private
     */
    Tmp.prototype._checkDevice = function()
    {
        this._pixelRatio = window.devicePixelRatio || 1;

        this._iPhone = this._iOS === true && this._ua.toLowerCase().indexOf("iphone") !== -1;
        this._iPod = this._iOS === true && this._ua.toLowerCase().indexOf("ipod") !== -1;
        this._iPad = this._iOS === true && this._ua.toLowerCase().indexOf("ipad") !== -1;
        this._retina = this._pixelRatio >= 2 && this._iOS === true;

        if ((this._windows && !this._windowsPhone) || this._macOS || (this._linux && !this._silk) || this._chromeOS)
        {
            this._desktop = true;
        }
        else if (/Mobi/i.test(this._ua) && this._iPad === false)
        {
            this._mobile = true;
        }
        else
        {
            this._tablet = true;
        }

        //smart TV, Playstation, Table Windows
        if (/TV/i.test(this._ua) || this._vita === true || this._xbox === true || (this._desktop && /Windows NT/i.test(this._ua) && /Touch/i.test(this._ua)))
        {
            this._other = true;
            this._mobile = false;
            this._tablet = false;
            this._desktop = false;
        }
    };

    /**
     * Check video support.
     * @method FORGE.Device#_checkVideo
     * @private
     */
    Tmp.prototype._checkVideo = function()
    {
        var videoElement = document.createElement("video");

        try
        {
            if (typeof videoElement.canPlayType === "function")
            {
                if (videoElement.canPlayType("video/ogg; codecs=\"theora\"").replace(/^no$/, ""))
                {
                    this._oggVideo = true;
                }

                if (videoElement.canPlayType("video/mp4; codecs=\"avc1.42E01E\"").replace(/^no$/, ""))
                {
                    // without QuickTime, this value will be "undefined"
                    this._h264Video = true;
                    this._mp4Video = true;
                }

                if (videoElement.canPlayType("video/webm; codecs=\"vp8, vorbis\"").replace(/^no$/, ""))
                {
                    this._webmVideo = true;
                }

                if (videoElement.canPlayType("video/webm; codecs=\"vp9\"").replace(/^no$/, ""))
                {
                    this._vp9Video = true;
                }

                if (videoElement.canPlayType("application/x-mpegURL; codecs=\"avc1.42E01E\"").replace(/^no$/, ""))
                {
                    this._hlsVideo = true;
                }
            }
        }
        catch (e)
        {}
    };

    /**
     * Check audio support.
     * @method FORGE.Device#_checkAudio
     * @private
     */
    Tmp.prototype._checkAudio = function()
    {
        this._audioTag = (typeof window.Audio !== "undefined");
        this._webAudio = (typeof window.AudioContext !== "undefined" || typeof window.webkitAudioContext !== "undefined");

        var audioElement = document.createElement("audio");

        try
        {
            if (typeof audioElement.canPlayType === "function")
            {
                if (audioElement.canPlayType("audio/ogg; codecs=\"vorbis\"").replace(/^no$/, ""))
                {
                    this._ogg = true;
                }

                if (audioElement.canPlayType("audio/mpeg;").replace(/^no$/, ""))
                {
                    this._mp3 = true;
                }

                if (audioElement.canPlayType("audio/ogg; codecs=\"opus\"").replace(/^no$/, "") || audioElement.canPlayType("audio/opus;").replace(/^no$/, ""))
                {
                    this._opus = true;
                }

                if (audioElement.canPlayType("audio/wav; codecs=\"1\"").replace(/^no$/, ""))
                {
                    this._wav = true;
                }

                if (audioElement.canPlayType("audio/aac;").replace(/^no$/, ""))
                {
                    this._aac = true;
                }

                if (audioElement.canPlayType("audio/x-m4a;") || audioElement.canPlayType("audio/m4a;") || audioElement.canPlayType("audio/aac;").replace(/^no$/, ""))
                {
                    this._m4a = true;
                }

                if (audioElement.canPlayType("audio/x-mp4;") || audioElement.canPlayType("audio/mp4;") || audioElement.canPlayType("audio/aac;").replace(/^no$/, ""))
                {
                    this._mp4 = true;
                }

                if (audioElement.canPlayType("audio/webm; codecs=\"vorbis\"").replace(/^no$/, ""))
                {
                    this._webm = true;
                    this._weba = true;
                }

            }
        }
        catch (e)
        {}
    };

    /**
     * Check the device features.
     * @method FORGE.Device#_checkDeviceFeatures
     * @private
     */
    Tmp.prototype._checkDeviceFeatures = function()
    {
        this._vibrate = (typeof navigator.vibrate !== "undefined" || typeof navigator.webkitVibrate !== "undefined" || typeof navigator.mozVibrate !== "undefined" || typeof navigator.msVibrate !== "undefined");

        this._battery = (typeof navigator.getBattery === "function");
    };

    /**
     * Check HTML5 features.
     * @method FORGE.Device#_checkFeatures
     * @private
     */
    Tmp.prototype._checkFeatures = function()
    {
        this._canvas = (typeof window.CanvasRenderingContext2D !== "undefined");
        if (this._canvas === true)
        {
            this._canvasText = (typeof document.createElement("canvas").getContext("2d").fillText === "function");
            var canvasCtx = document.createElement("canvas").getContext("2d");
            canvasCtx.rect(0, 0, 10, 10);
            canvasCtx.rect(2, 2, 6, 6);
            this._canvasWinding = (canvasCtx.isPointInPath(5, 5, "evenodd") === false);
        }

        try
        {
            this._localStorage = (typeof window.localStorage === "object" && typeof window.localStorage.getItem === "function");
        }
        catch (error)
        {
            this._localStorage = false;
        }

        this._mediaSource = (typeof window.MediaSource === "function");

        this._encryptedMedia = (typeof window.HTMLMediaElement === "function" && typeof window.MediaKeys === "function" && typeof window.MediaKeySystemAccess === "function" && typeof navigator.requestMediaKeySystemAccess === "function");

        this._applicationCache = (typeof window.applicationCache === "object");

        this._addEventListener = (typeof window.addEventListener === "function");

        this.__raf = (typeof window.requestAnimationFrame === "function" || typeof window.webkitRequestAnimationFrame === "function" || typeof window.mozRequestAnimationFrame === "function");

        try
        {
            var canvas = document.createElement("canvas");
            /* Force screencanvas to false */
            canvas.screencanvas = false;
            this._webGL = (typeof canvas.getContext === "function" && typeof window.WebGLRenderingContext !== "undefined" && Boolean(canvas.getContext("webgl") || canvas.getContext("experimental-webgl")));
        }
        catch (e)
        {
            this._webGL = false;
        }

        if (typeof navigator.getVRDisplays === "function")
        {
            this._webVR = true;
        }
        else
        {
            this._webVR = false;
        }

        this._JSON = (typeof window.JSON === "object" && typeof window.JSON.parse === "function" && typeof window.JSON.stringify === "function");

        this._geolocation = (typeof navigator.geolocation === "object");

        this._history = (typeof window.history === "object" && typeof window.history.pushState === "function");

        this._svg = (typeof document.createElementNS === "function" && typeof document.createElementNS("http://www.w3.org/2000/svg", "svg").createSVGRect === "function");

        this._contextMenu = (typeof document.documentElement.contextMenu !== "undefined" && typeof window.HTMLMenuItemElement === "function");
    };

    /**
     * Check the URL environment.
     * @method FORGE.Device#_checkEnvironment
     * @private
     */
    Tmp.prototype._checkEnvironment = function()
    {
        this._isSecure = /^https/i.test(window.location.protocol);

        try
        {
            this._isIframe = (window.self !== window.top);
        }
        catch (e)
        {
            this._isIframe = true;
        }
    };

    /**
     * Check the various inputs.
     * @method FORGE.Device#_checkInput
     * @private
     */
    Tmp.prototype._checkInput = function()
    {
        this._touch = (typeof window.ontouchstart !== "undefined" || typeof window.DocumentTouch !== "undefined" && document instanceof window.DocumentTouch || (typeof navigator.maxTouchPoints === "number" && navigator.maxTouchPoints > 0) || (typeof navigator.msMaxTouchPoints === "number" && navigator.msMaxTouchPoints > 0));

        // Test for Safari iOS touch force feature
        if (typeof window.onmouseforcewillbegin !== "undefined" || typeof window.onwebkitmouseforcewillbegin !== "undefined")
        {
            // Test if the browser provides thresholds defining a "force touch" from a normal touch/click event
            this._touchForce = Boolean(MouseEvent.WEBKIT_FORCE_AT_MOUSE_DOWN && MouseEvent.WEBKIT_FORCE_AT_FORCE_MOUSE_DOWN);
        }

        this._gamepad = (typeof navigator.getGamepads === "function" || typeof navigator.webkitGetGamepads === "function");
    };

    /**
     * Check the support of the full screen API.
     * @method FORGE.Device#_checkFullscreenSupport
     * @private
     */
    Tmp.prototype._checkFullscreenSupport = function()
    {
        var requestFullscreen =
            [
                "requestFullscreen",
                "requestFullScreen",
                "webkitRequestFullscreen",
                "webkitRequestFullScreen",
                "mozRequestFullScreen",
                "mozRequestFul